use serde::{Serialize};
use std::fmt::{Display, Formatter, Result as FmtResult};
use serde_json::{to_string_pretty};
use std::collections::HashMap;
use crate::model::Value;
use crate::build_sql;
use crate::oracle::connect_db;

#[derive(Debug, Serialize)]
pub enum OraDaoError {

    ConnectionError(String),

    StatementError(String),

    QueryError(String),

    ExecutionError(String),

    CommitError(String),

}

impl Display for OraDaoError {
    fn fmt(&self, f: &mut Formatter) -> FmtResult {
        let error = to_string_pretty(self).unwrap();
        error!("{}", &error);
        write!(f, "{}", error)
    }
}

pub fn cud_dao(sql_id: &str, item: &HashMap<String, Value>) -> Result<usize, OraDaoError> {

    let sql = build_sql(sql_id, "", item);
    return match connect_db() {

        Ok(conn) => {
            match conn.execute(&sql, &[]) {
                Ok(_s) => {
                    match conn.commit() {
                        Ok(_t) => {
                            debug!("Commit OK");
                            Ok(1)
                        }
                        Err(e) => {
                            error!("CommitError:\t{}", e);
                            Err(OraDaoError::CommitError(format!("{}", e)))
                        }
                    }
                }
                Err(e) => {
                    error!("ExecutionError:\t{}", e);
                    Err(OraDaoError::ExecutionError(format!("{}", e)))
                }
            }

        }
        Err(e) => {
            error!("ConnectionError:\t{}", e);
            Err(OraDaoError::ConnectionError(format!("{}", e)))
        }
    }
}
pub fn cud_multi_entries_dao(sql_id: &str, list : Vec<HashMap<String, Value>>) -> Result<usize, OraDaoError> {

    return match connect_db() {
        Ok(conn) => {
            let mut count = 0;

            for item in list{

                let sql = build_sql(sql_id, "", &item);

                match conn.execute(&sql, &[]) {
                    Ok(_t)=> {
                        count = count + 1;
                    }
                    Err(e) => {
                        error!("ExecutionError:\t{}", e);
                        match conn.rollback() {
                            Ok(_t) => {}
                            Err(_e) => {}
                        }
                        return Err(OraDaoError::ExecutionError(format!("{}", e)))
                    }
                }
            }
            match conn.commit() {
                Ok(_t) => {
                    debug!("Commit OK");
                    Ok(count)
                }
                Err(e) => {
                    error!("CommitError:\t{}", e);
                    Err(OraDaoError::CommitError(format!("{}", e)))
                }
            }
        }
        Err(e) => {
            error!("ConnectionError:\t{}", e);
            Err(OraDaoError::ConnectionError(format!("{}", e)))
        }
    }
}