
use r2d2_oracle::OracleConnectionManager;
use r2d2::{Pool, PooledConnection};
use parking_lot::{ RwLock};
use std::time::Duration;

use crate::{DB_PROPERTIES};

lazy_static!{

    static ref ORACLE_POOL: RwLock<Pool<OracleConnectionManager>> = {
        RwLock::new(create_oracle_connection_manager())
    };
}

pub fn connect_db() -> Result<PooledConnection <OracleConnectionManager>, r2d2::Error> {

    let conn = ORACLE_POOL.read();
    return match conn.get() {
        Ok(t) => Ok(t),
        Err(e) => {
            error!("==> {:?}", e);

            drop(conn);

            let mut conn = ORACLE_POOL.write();
            *conn = create_oracle_connection_manager();
            conn.get()
        }
    };
}
pub fn create_oracle_connection_manager() -> Pool<OracleConnectionManager> {

    let username = match DB_PROPERTIES.get("db.oracle.username") {
        Some(v) => v,
        _ => panic!("No such property: db.oracle.username"),
    };
    let password = match DB_PROPERTIES.get("db.oracle.password") {
        Some(v) => v,
        _ => panic!("No such property: db.oracle.password"),
    };
    let database = match DB_PROPERTIES.get("db.oracle.database") {
        Some(v) => v,
        _ => panic!("No such property: db.oracle.database"),
    };

    let pool_size = match DB_PROPERTIES.get("db.oracle.pool_size") {
        Some(t) =>   t.parse::<u32>().unwrap(),
        None => panic!("No such property: db.oracle.pool_size")
    };
    let idle_timeout_sec = match DB_PROPERTIES.get("db.oracle.idle_timeout_sec") {
        Some(t) =>   t.parse::<u64>().unwrap_or(15),
        None => 15,
    };

    let manager = OracleConnectionManager::new(username, password, database);
    let pool = r2d2::Pool::builder()
        .idle_timeout(Some(Duration::from_secs(idle_timeout_sec)))
        .max_size(pool_size)
        .build(manager)
        .unwrap();
    pool
}

#[cfg(test)]
mod tests{
}