
use r2d2_oracle::OracleConnectionManager;
use r2d2::{Pool, PooledConnection};
use std::collections::HashMap;
use std::fs;
use parking_lot::{ RwLock};
use std::time::Duration;

lazy_static!{

    static ref ORACLE_POOL: RwLock<Pool<OracleConnectionManager>> = {
        RwLock::new(create_oracle_connection_manager())
    };
}

pub fn connect_db() -> Result<PooledConnection <OracleConnectionManager>, r2d2::Error> {

    let conn = ORACLE_POOL.read();
    return match conn.get() {
        Ok(t) => Ok(t),
        Err(e) => {
            error!("==> {:?}", e);

            drop(conn);

            let mut conn = ORACLE_POOL.write();
            *conn = create_oracle_connection_manager();
            conn.get()
        }
    };
}
pub fn create_oracle_connection_manager() -> Pool<OracleConnectionManager> {
    let filepath = "resources/application.properties";
    let contents = fs::read_to_string(filepath).expect("Something went wrong reading the file");
    let mut ds_map = HashMap::new();

    for line in contents.lines() {
        if line.starts_with("//") || line.starts_with("#"){
            continue;
        }
        let tokens: Vec<&str> = line.split("=").collect();

        if tokens.len() >= 2 {
            ds_map.insert(String::from(tokens[0]), String::from(tokens[1]));
        }
    }

    let username = match ds_map.get("db.username") {
        Some(v) => v,
        _ => panic!("No such property: db.username"),
    };
    let password = match ds_map.get("db.password") {
        Some(v) => v,
        _ => panic!("No such property: db.password"),
    };
    let database = match ds_map.get("db.database") {
        Some(v) => v,
        _ => panic!("No such property: db.database"),
    };

    let manager = OracleConnectionManager::new(username, password, database);
    let pool = r2d2::Pool::builder()
        .idle_timeout(Some(Duration::from_secs(10)))
        .max_size(15)
        .build(manager)
        .unwrap();
    pool
}

#[cfg(test)]
mod tests{
    use std::thread;
    use r2d2_oracle::OracleConnectionManager;
    use crate::oracle::{connect_db, switch_db};

    #[test]
    fn test_01() {
    }
    #[test]
    fn test_02() {
        let conn = connect_db();
        match conn.unwrap().ping() {
            Ok(t) => {
                println!("OK");
            }
            _ =>{
                println!("ERROR");
            }
        }
    }
    #[test]
    fn test_03() {
        switch_db();
    }
}