use std::{
    mem::ManuallyDrop,
    thread::{self, JoinHandle},
    time::Duration,
};

use cancellable_timer::{Canceller, Timer};
use gpio::{
    sysfs::{SysFsGpioInput, SysFsGpioOutput},
    GpioIn, GpioOut,
};

/// Trait regrouping gpio reader and writer.
pub trait UnbotheredGpioPin {
    /// Create a new instance with the specified gpio pin.
    fn new(pin: usize) -> Self;
}

/// An unbothered single gpio pin reader.
pub struct UnbotheredGpioPinReader(SysFsGpioInput);

/// An unbothered single gpio pin writer.
pub struct UnbotheredGpioPinWriter(SysFsGpioOutput);

impl UnbotheredGpioPinReader {
    /// Reads the pin's current value.
    pub fn read(&mut self) -> bool {
        self.0.read_value().unwrap().into()
    }
}
impl UnbotheredGpioPinWriter {
    /// Writes the specified value to the pin.
    pub fn write(&mut self, v: bool) {
        self.0.set_value(v).unwrap()
    }
}

impl UnbotheredGpioPin for UnbotheredGpioPinReader {
    fn new(pin: usize) -> Self {
        UnbotheredGpioPinReader(SysFsGpioInput::open(pin as u16).unwrap())
    }
}
impl UnbotheredGpioPin for UnbotheredGpioPinWriter {
    fn new(pin: usize) -> Self {
        UnbotheredGpioPinWriter(SysFsGpioOutput::open(pin as u16).unwrap())
    }
}

/// An unbothered single gpio pin listener, invoking a callback on gpio state change.
/// Will stop listening upon drop by default, look at [`UnbotheredGpioPinListener::keep_alive`] to change this behavior.
/// Dropping this object is time consuming, it is intended to be long lived.
pub struct UnbotheredGpioPinListener {
    thread_handle: ManuallyDrop<JoinHandle<()>>,
    sleep_canceller: Canceller,
}

impl UnbotheredGpioPinListener {
    /// Delay between each gpio pin poll.
    const POLL_DELAY: Duration = Duration::from_millis(20);

    /// Create a new [`UnbotheredGpioPinListener`] with a callback.
    pub fn new<F>(gpio_pin: usize, mut onchange: F) -> UnbotheredGpioPinListener
    where
        F: FnMut(bool) + Send + 'static,
    {
        let (mut timer, canceller) = Timer::new2().unwrap();
        let mut pin_reader = UnbotheredGpioPinReader::new(gpio_pin);

        let loop_thread_handle = thread::spawn(move || {
            let mut last_state = pin_reader.read();

            // cancelable sleep +- 150_000 ns
            while timer.sleep(UnbotheredGpioPinListener::POLL_DELAY).is_ok() {
                let state = pin_reader.read();
                if state != last_state {
                    last_state = state;
                    onchange(state);
                }
            }
        });

        UnbotheredGpioPinListener {
            thread_handle: ManuallyDrop::new(loop_thread_handle),
            sleep_canceller: canceller,
        }
    }

    /// Makes the object live until program termination.
    pub fn keep_alive(self)
    where
        Self: Sized,
    {
        std::mem::forget(self)
    }
}

impl Drop for UnbotheredGpioPinListener {
    fn drop(&mut self) {
        let thread_handle = unsafe { ManuallyDrop::take(&mut self.thread_handle) };

        self.sleep_canceller.cancel().unwrap();
        thread_handle.join().unwrap();
    }
}
