use std::collections::HashMap;
use std::sync::Arc;
use std::io::Write;

use maplit::hashmap;

use crate::{Error, Type, Callable, Value, TVal, RefTVal, FnArgs, FnReturn, umcore::_parse_fargs, ummod::fmt::format, tprintln};

pub fn init(vars: &mut HashMap<String, RefTVal>) {
    vars.extend(hashmap!{
        "print".into() => print_init(),
        "printf".into() => printf_init(),
    });
}

pub fn print_init() -> RefTVal {
    Value::Function {
        args: Arc::new(vec![
            ("*vargs".into(), Type::any()),
        ]),
        vars: hashmap!{}.into(),
        body: Callable::Native(print),
    }.into()
}
pub fn print(args: FnArgs) -> FnReturn {
    let (_this, pos, args) = match _parse_fargs("function io.print", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    if let Value::List(ref args) = args.clone_out().val {
        if args.is_empty() {
            tprintln!();
        } else {
            let p = args[1..].iter()
                .map(|a| a.clone_out().val.to_string())
                .collect::<Vec<String>>()
                .join(" ");
            tprintln!("{}", p);
        }
        return (None, Ok(Value::none().into()));
    }

    (None, Err(Error::Script(format!("function io.print: expected args [Any], got {}", args.clone_out().val), pos)))
}

pub fn printf_init() -> RefTVal {
    Value::Function {
        args: Arc::new(vec![
            ("f".into(), Type::string()),
            ("*vargs".into(), Type::any()),
        ]),
        vars: hashmap!{}.into(),
        body: Callable::Native(printf),
    }.into()
}
pub fn printf(args: FnArgs) -> FnReturn {
    let (this, pos, targs) = match _parse_fargs("function io.printf", args.clone()) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    if let Value::List(l) = targs.clone_out().val {
        if l.is_empty() {
            tprintln!();
            return (None, Ok(Value::none().into()));
        }
    }

    let fs = format(args);
    match fs {
        (_, Ok(v)) => print(FnArgs::Normal {
            this,
            args: TVal {
                ttype: Type::list(),
                attr: hashmap!{}.into(),
                val: Value::List(vec![Value::none().into(), v]),
            }.into(),
            pos,
        }),
        _ => fs,
    }
}
