use std::sync::Arc;

use num::ToPrimitive;

use maplit::hashmap;

use crate::{Error, Env, Type, Value, TVal, RefTVal, FnArgs, FnReturn, Callable, run};

use super::_parse_fargs;

// Builtin Functions: eval, exit

pub fn eval_init() -> RefTVal {
    Value::Function {
        args: Arc::new(vec![
            ("code".into(), Type::string()),
            ("vars".into(), Type::map()),
        ]),
        vars: hashmap!{}.into(),
        body: Callable::Native(eval),
    }.into()
}
pub fn eval(args: FnArgs) -> FnReturn {
    let (_this, mut pos, args) = match _parse_fargs("function eval", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    if let Value::List(ref args) = args.clone_out().val {
        if let Some(rv0) = args.get(0) {
            if let TVal { val: Value::String(s), .. } = rv0.clone_out() {
                if let Some(vars) = args.get(1) {
                    if let Value::Map(vars) = vars.clone_out().val {
                        let filename = match &mut pos {
                            Some(p) => {
                                let filename = p.filename.clone() + "<eval>";
                                p.filename = filename.clone();
                                filename
                            },
                            None => "<eval>".into(),
                        };

                        let p = Env::prelude();
                        let mut tenv = Env::child(&p);
                        tenv.update(
                            vars.iter()
                            .filter_map(|(k, v)| {
                                if let Value::String(k) = k.clone_out().val {
                                    return Some((k, v.clone()));
                                }
                                None
                            }).collect()
                        );
                        let (_, vals) = run(&filename, &s, &tenv, pos, (0, false));
                        return (None, vals);
                    }
                }
            }
        }
    }

    (None, Err(Error::Script(format!("function eval: expected args [String], got {}", args.clone_out().val), pos)))
}
pub fn exit_init() -> RefTVal {
    Value::Function {
        args: Arc::new(vec![
            ("status".into(), Type::number()),
        ]),
        vars: hashmap!{}.into(),
        body: Callable::Native(exit),
    }.into()
}
pub fn exit(args: FnArgs) -> FnReturn {
    let (_this, pos, args) = match _parse_fargs("function exit", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    if let Value::List(ref args) = args.clone_out().val {
        let ec = match args.get(0) {
            Some(rv0) => {
                if let TVal { val: Value::Number(n), .. } = rv0.clone_out() {
                    n.to_integer().to_i32().unwrap_or(128)
                } else {
                    128
                }
            },
            _ => 0,
        };
        std::process::exit(ec);
    }

    (None, Err(Error::Script(format!("function exit: expected args [Number], got {}", args.clone_out().val), pos)))
}
