use std::path::Path;
use std::collections::HashMap;

use maplit::hashmap;

use crate::{Error, Env, Value, TVal, RefTVal, TokenType, FnArgs, FnReturn, AST, run_path, unescape, ummod};

use super::_parse_margs;

// Base Macros: use, let

pub fn r#use(args: FnArgs) -> FnReturn {
    let (env, pos, tokens) = match _parse_margs("macro use", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    // Compute module path
    let m = &tokens[1];
    let modpath = match m.ttype {
        TokenType::Identifier => env.get(&m.data).cloned(),
        TokenType::String => Some(Value::from(&*unescape(&m.data[1..m.data.len()-1])).into()),
        _ => None,
    };
    let mp = match &modpath {
        Some(rv) => {
            if let TVal { val: Value::String(mn), .. } = rv.clone_out() {
                mn
            } else {
                return (None, Err(Error::Script(format!("macro use: invalid module path {}", m), pos)))
            }
        },
        _ => return (None, Err(Error::Script(format!("macro use: invalid module path {}", m), pos))),
    };
    let modpath = Path::new(&mp);
    let modname: String = match m.ttype {
        TokenType::Identifier => m.data.clone(),
        _ => modpath.file_name().expect("macro use: expected module name in path").to_string_lossy().to_string(),
    };
    let modas: Vec<String> = if tokens.len() > 3 && tokens[2].data == "as" {
        match tokens[3].ttype {
            TokenType::Identifier => vec![tokens[3].data.clone()],
            TokenType::Container if tokens[3].data.starts_with('[') => {
                let l = tokens[3].data.len();
                tokens[3].data[1..l-1]
                    .split(',')
                    .map(|s| s.trim().to_string())
                    .collect()
            },
            _ => return (None, Err(Error::Script(format!("macro use: invalid 'as' name {}", tokens[3].data), pos))),
        }
    } else {
        vec![]
    };
    let modpath = Path::new(&tokens[0].pos.filename)
        .parent().expect("macro use: expected parent for the current module path")
        .join(modpath);

    // Import module
    let menv = if env.has("__prelude") {
        Env::core()
    } else {
        Env::prelude()
    };
    let (vars, val): FnReturn = if let Some(vars) = ummod::get(&modname) {
        (Some(vars), Ok(Value::none().into()))
    } else {
        let mpath = Path::new("mod").join(modname.clone());
        match run_path(&mpath, &menv, false) {
            (vars, Ok(val)) => (vars, Ok(val)),
            (_, Err(Error::Argument(_))) => run_path(&modpath, &menv, false),
            (vars, Err(m)) => (vars, Err(m)),
        }
    };
    if let Err(m) = val {
        return (None, Err(m));
    }

    // Construct module struct with modas
    let vars = if let Some(vars) = vars {
        vars
    } else {
        hashmap!{}
    };
    let modst: RefTVal = Value::Struct(vars.clone().into()).into();
    if modas.is_empty() {
        (Some(hashmap!{ modname => modst.clone() }), Ok(modst))
    } else {
        let mut hm: HashMap<String, RefTVal> = hashmap!{};
        for a in modas {
            match vars.get(&a) {
                Some(v) => {
                    hm.insert(a, v.clone());
                },
                None => {
                    if a == "self" {
                        hm.insert(modname.clone(), modst.clone());
                    } else {
                        return (None, Err(Error::Script(format!("macro use: failed to find name {} in module {}", a, modname), pos)))
                    }
                },
            }
        }
        (Some(hm), Ok(modst))
    }
}
pub fn r#let(args: FnArgs) -> FnReturn {
    let (env, _pos, tokens) = match _parse_margs("macro let", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    let varname = &tokens[1].data;
    let ast = match AST::parse(tokens[3..].iter().cloned(), &env) {
        Ok(a) => a,
        Err(m) => return (None, Err(m)),
    };
    let (_, val) = ast.run(&env);

    match val {
        Ok(v) => match v.clone_out().val {
            Value::Reference(rv) => (Some(hashmap!{ varname.clone() => rv.clone() }), Ok(rv)),
            _ => (Some(hashmap!{ varname.clone() => v.clone() }), Ok(v)),
        },
        Err(m) => (None, Err(m)),
    }
}
