use std::thread::{self, JoinHandle};
use std::hash::{Hash, Hasher};
use std::cmp::Ordering;
use std::sync::{Arc, Mutex};

use num::ToPrimitive;

#[cfg(feature = "compile")]
use std::fmt;
#[cfg(feature = "compile")]
use serde::{Serialize, Deserialize, Serializer, Deserializer, ser::SerializeStruct, de};

use crate::{Callable, Env, Error, FnArgs, FnReturn, HashableMap, RefTVal, Value, thrio};

thread_local! {
    static OUT: Arc<Mutex<thrio::WriteIO>> = Arc::new(Mutex::new(thrio::WriteIO::new()));
    static ERR: Arc<Mutex<thrio::WriteIO>> = Arc::new(Mutex::new(thrio::WriteIO::new()));
}

/// The context for a running `Thread`. Can be joined from multiple threads.
#[derive(Debug)]
pub struct ThreadHandle {
    pub vars: HashableMap<String, RefTVal>,
    pub body: Callable,
    pub handle: Arc<Mutex<Option<JoinHandle<FnReturn>>>>,
}
impl ThreadHandle {
    /// The default stack size based on common ulimit values: 8MiB
    pub const STACK_SIZE: usize = 8 * 1024 * 1024;

    /// Constructs a `ThreadHandle` and starts the thread.
    ///
    /// # Panics
    /// Will panic if the thread output fails to lock.
    #[must_use]
    pub fn new(vars: HashableMap<String, RefTVal>, body: Callable) -> Self {
        let name: String = if let Some(name) = vars.get("__name") {
            if let Value::String(name) = name.clone_out().val {
                name
            } else {
                "<unknown>".into()
            }
        } else {
            "<unknown>".into()
        };
        let stack_size: usize = if let Some(stack_size) = vars.get("__stack_size") {
            if let Value::Number(stack_size) = stack_size.clone_out().val {
                stack_size.to_usize().unwrap()
            } else {
                ThreadHandle::STACK_SIZE
            }
        } else {
            ThreadHandle::STACK_SIZE
        };

        ThreadHandle {
            vars: vars.clone(),
            body: body.clone(),
            handle: Arc::new(Mutex::new(Some(
                thread::Builder::new()
                    .name(name)
                    .stack_size(stack_size)
                    .spawn(
                        move || {
                            // Redirect thread output
                            OUT.with(|out| {
                                thrio::set_out(out);
                            });
                            ERR.with(|err| {
                                thrio::set_err(err);
                            });

                            // Run thread body
                            let env = Env::from(vars);
                            let (vars, val) = body.call(&env, FnArgs::Normal {
                                this: Box::new(Value::none().into()),
                                pos: None,
                                args: Value::List(vec![]).into(),
                            });

                            // Reset thread output
                            thrio::reset_out();
                            thrio::reset_err();

                            // Forward thread output to caller
                            let mut tout = Value::none();
                            let mut terr = Value::none();
                            OUT.with(|out| {
                                let out: Vec<u8> = match &*out.lock().unwrap() {
                                    thrio::WriteIO::Cursor(c) => c.clone().into_inner(),
                                    _ => vec![],
                                };
                                tout = Value::String(String::from_utf8_lossy(&out).into());
                            });
                            ERR.with(|err| {
                                let err: Vec<u8> = match &*err.lock().unwrap() {
                                    thrio::WriteIO::Cursor(c) => c.clone().into_inner(),
                                    _ => vec![],
                                };
                                terr = Value::String(String::from_utf8_lossy(&err).into());
                            });

                            (vars, Ok(Value::List(vec![Value::from_result(&val).into(), tout.into(), terr.into()]).into()))
                        }
                    ).unwrap()
            ))),
        }
    }

    /// Joins the thread and returns its value.
    ///
    /// # Panics
    /// Will panic if the internal `Mutex` is poisoned or if the `Thread` fails
    /// to join.
    pub fn join(&self) -> FnReturn {
        let mut th = self.handle.lock().unwrap();
        match th.take() {
            Some(th) => th.join().unwrap(),
            None => (None, Err(Error::Script("Thread failed to join".into(), None))),
        }
    }
}

impl Clone for ThreadHandle {
    fn clone(&self) -> Self {
        ThreadHandle {
            vars: self.vars.clone(),
            body: self.body.clone(),
            handle: Arc::clone(&self.handle),
        }
    }
}

impl PartialEq for ThreadHandle {
    fn eq(&self, other: &Self) -> bool {
        self.vars == other.vars && self.body == other.body
    }
}
impl Eq for ThreadHandle {}
impl Ord for ThreadHandle {
    fn cmp(&self, other: &Self) -> Ordering {
        self.body.cmp(&other.body)
    }
}
impl PartialOrd for ThreadHandle {
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        Some(self.cmp(other))
    }
}

impl Hash for ThreadHandle {
    fn hash<H: Hasher>(&self, state: &mut H) {
        self.vars.hash(state);
        self.body.hash(state);
    }
}

#[cfg(feature = "compile")]
impl Serialize for ThreadHandle {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer
    {
        let mut state = serializer.serialize_struct("ThreadHandle", 2)?;
        state.serialize_field("vars", &self.vars)?;
        state.serialize_field("body", &self.body)?;
        state.end()
    }
}

#[cfg(feature = "compile")]
impl<'de> Deserialize<'de> for ThreadHandle {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>
    {
        #[derive(Deserialize)]
        #[serde(field_identifier, rename_all = "lowercase")]
        enum Field {
            Vars,
            Body,
        }

        struct ThreadHandleVisitor;
        impl<'de> de::Visitor<'de> for ThreadHandleVisitor {
            type Value = ThreadHandle;

            fn expecting<'a>(&self, formatter: &mut fmt::Formatter<'a>) -> fmt::Result {
                formatter.write_str("struct ThreadHandle")
            }

            fn visit_seq<A>(self, mut seq: A) -> Result<Self::Value, A::Error>
            where
                A: de::SeqAccess<'de>,
            {
                let vars = seq.next_element()?
                    .ok_or_else(|| de::Error::invalid_length(0, &self))?;
                let body = seq.next_element()?
                    .ok_or_else(|| de::Error::invalid_length(1, &self))?;
                Ok(ThreadHandle::new(vars, body))
            }

            fn visit_map<A>(self, mut map: A) -> Result<Self::Value, A::Error>
            where
                A: de::MapAccess<'de>,
            {
                let mut vars = None;
                let mut body = None;
                while let Some(key) = map.next_key()? {
                    match key {
                        Field::Vars => {
                            if vars.is_some() {
                                return Err(de::Error::duplicate_field("vars"));
                            }
                            vars = Some(map.next_value()?);
                        },
                        Field::Body => {
                            if body.is_some() {
                                return Err(de::Error::duplicate_field("body"));
                            }
                            body = Some(map.next_value()?);
                        },
                    }
                }
                let vars = vars.ok_or_else(|| de::Error::missing_field("vars"))?;
                let body = body.ok_or_else(|| de::Error::missing_field("body"))?;
                Ok(ThreadHandle::new(vars, body))
            }
        }

        const FIELDS: &[&str] = &["vars", "body"];
        deserializer.deserialize_struct("ThreadHandle", FIELDS, ThreadHandleVisitor)
    }
}
