#[cfg(feature = "compile")]
use serde::{Serialize, Deserialize, Serializer, Deserializer, ser, de};

#[cfg(feature = "compile")]
use maplit::hashmap;

use crate::{Env, AST};

#[cfg(feature = "compile")]
use crate::{Callable, Pos, TokenType, Token, Value};

/// A collection of all data required for final script execution.
#[derive(Debug, Clone)]
#[cfg_attr(feature = "compile", derive(Serialize, Deserialize))]
pub struct Runnable<'a> {
    /// The hash used by the `compile` feature to determine when to recompile.
    pub hash: u64,
    /// The `Env` the script will access variables from, typically the Prelude.
    pub env: Env<'a>,
    /// The parsed representation of the script.
    pub ast: AST,
}

// Custom serde impl for Callable
#[cfg(feature = "compile")]
impl Serialize for Callable {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer
    {
        match self {
            Callable::Native(fp) => {
                let env = Env::core();
                for nf in env.vars.iter()
                    .flat_map(|(k, v)| match v.clone_out().val {
                        Value::Struct(btm) => btm,
                        _ => hashmap!{ k.clone() => v.clone() }.into(),
                    })
                {
                    if let Value::Function { body: Callable::Native(nfp), .. } = nf.1.clone_out().val {
                        if nfp == *fp {
                            let token = Token {
                                ttype: TokenType::Identifier,
                                pos: Pos::start("<runnable>"),
                                data: nf.0,
                            };
                            let ast = AST::Value { token };
                            return ast.serialize(serializer);
                        }
                    }
                }
                Err(ser::Error::custom(format!("Callable::Native serialization failed: couldn't find associated function in prelude {:?}", fp)))
            },
            Callable::Value(rv) => rv.serialize(serializer),
            Callable::AST(ast) => ast.serialize(serializer),
        }
    }
}
#[cfg(feature = "compile")]
impl<'de> Deserialize<'de> for Callable {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>
    {
        match AST::deserialize(deserializer) {
            Ok(ast) => {
                if let AST::Value { token } = &ast {
                    if token.pos.filename == "<runnable>" {
                        let env = Env::core();
                        for nf in env.vars.iter()
                            .flat_map(|(k, v)| match v.clone_out().val {
                                Value::Struct(btm) => btm,
                                _ => hashmap!{ k.clone() => v.clone() }.into(),
                            })
                        {
                            if nf.0 == token.data {
                                if let Value::Function { body, .. } =  nf.1.clone_out().val {
                                    return Ok(body);
                                }
                            }
                        }

                        return Err(de::Error::custom(
                            format!("Callable::Native deserialization failed: couldn't find associated function in prelude {:?}", token.data)
                        ));
                    }
                }
                Ok(Callable::AST(ast))
            },
            Err(m) => Err(m),
        }
    }
}
