use std::fmt;

use num::ToPrimitive;

#[cfg(feature = "compile")]
use serde::{Serialize, Deserialize};

use crate::Value;

/// Denotes a specific position within a script.
#[derive(Debug, Clone, PartialEq, Eq, Hash, PartialOrd, Ord)]
#[cfg_attr(feature = "compile", derive(Serialize, Deserialize))]
pub struct Pos {
    pub filename: String,
    pub line: usize,
    pub col: usize,
}
impl Pos {
    /**
    Constructs a `Pos` at the beginning of the given filename, i.e. line 1
    column 1.
    */
    #[must_use]
    pub fn start(filename: &str) -> Self {
        Pos {
            filename: filename.into(),
            line: 1,
            col: 1,
        }
    }
    pub(crate) fn from_value(v: &Value) -> Option<Self> {
        if let Value::Struct(hm) = v {
            return Some(Pos {
                filename: match &hm.get("filename")?.clone_out().val {
                    Value::String(s) => s.clone(),
                    _ => return None,
                },
                line: match &hm.get("line")?.clone_out().val {
                    Value::Number(n) => n.to_usize()?,
                    _ => return None,
                },
                col: match &hm.get("col")?.clone_out().val {
                    Value::Number(n) => n.to_usize()?,
                    _ => return  None,
                },
            });
        }
        None
    }
}
impl fmt::Display for Pos {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "{}:{}:{}", self.filename, self.line, self.col)
    }
}

enum SplitAmount {
    Set(usize),
    Increase(isize),
}
enum SplitItem<'a> {
    Lines(&'a mut Lines),
    Exprs(&'a mut Exprs),
}
impl<'a> SplitItem<'a> {
    fn get_pos(&self) -> Pos {
        match self {
            SplitItem::Lines(ls) => ls.pos.clone(),
            SplitItem::Exprs(es) => es.pos.clone(),
        }
    }
    fn get_remainder(&self) -> String {
        match self {
            SplitItem::Lines(ls) => ls.remainder.clone(),
            SplitItem::Exprs(es) => es.remainder.clone(),
        }
    }

    fn set_remainder(&mut self, rem: String) {
        match self {
            SplitItem::Lines(ls) => ls.remainder = rem,
            SplitItem::Exprs(es) => es.remainder = rem,
        }
    }
    fn pos_set_line(&mut self, amount: SplitAmount) {
        let mut line = match self {
            SplitItem::Lines(ls) => ls.pos.line,
            SplitItem::Exprs(es) => es.pos.line,
        };
        match amount {
            SplitAmount::Set(a) => line = a,
            SplitAmount::Increase(a) => {
                let iline = line.to_isize().unwrap();
                if a <= iline {
                    line = (iline + a).to_usize().unwrap();
                }
            },
        };
        match self {
            SplitItem::Lines(ls) => ls.pos.line = line,
            SplitItem::Exprs(es) => es.pos.line = line,
        }
    }
    fn pos_set_col(&mut self, amount: SplitAmount) {
        let mut col = match self {
            SplitItem::Lines(ls) => ls.pos.col,
            SplitItem::Exprs(es) => es.pos.col,
        };
        match amount {
            SplitAmount::Set(a) => col = a,
            SplitAmount::Increase(a) => {
                let icol = col.to_isize().unwrap();
                if a <= icol {
                    col = (icol + a).to_usize().unwrap();
                }
            },
        }
        match self {
            SplitItem::Lines(ls) => ls.pos.col = col,
            SplitItem::Exprs(es) => es.pos.col = col,
        }
    }
}
trait SplitResult {
    fn new(pos: Pos, data: String) -> Self;
}
fn split<'a, SR>(si: &mut SplitItem<'a>, schar: char) -> Option<SR>
where
    SR: SplitResult,
{
    let remainder = si.get_remainder();
    if remainder.is_empty() {
        return None;
    }

    let remainder: Vec<char> = remainder.chars().collect();
    let mut sc: usize = 0;
    // Eat whitespace before token
    while sc < remainder.len() && remainder[sc].is_whitespace() {
        match remainder.get(sc).unwrap() {
            '\n' => {
                si.pos_set_line(SplitAmount::Increase(1));
                si.pos_set_col(SplitAmount::Set(1));
            },
            '\t' => si.pos_set_col(SplitAmount::Increase(4)),
            _ => si.pos_set_col(SplitAmount::Increase(1)),
        }
        sc += 1;
    }
    let startpos = si.get_pos();

    // Find the next schar that's not inside a container
    let mut containers: Vec<char> = vec![];
    let mut prev: Option<char> = None;
    while sc < remainder.len() {
        match remainder.get(sc).unwrap() {
            c if c == &schar => {
                if containers.is_empty() {
                    break;
                }
            },
            '/' => { // Skip comments
                if let Some(p) = prev {
                    if p == '/' && containers.is_empty() {
                        loop {
                            sc += 1;
                            match remainder.get(sc) {
                                Some('\n') => {
                                    si.pos_set_line(SplitAmount::Increase(1));
                                    break;
                                },
                                None => break,
                                _ => {},
                            }
                        }
                        break;
                    }
                }
            },
            '\n' => si.pos_set_line(SplitAmount::Increase(1)),

            '(' => containers.push('('),
            '{' => containers.push('{'),
            '[' => containers.push('['),
            ')' => {
                if containers.last() == Some(&'(') {
                    containers.pop();
                } else {
                    panic!("mismatched containers at {}: {:?}", si.get_pos(), containers);
                }
            },
            '}' => {
                if containers.last() == Some(&'{') {
                    containers.pop();
                } else {
                    panic!("mismatched containers at {}: {:?}", si.get_pos(), containers);
                }
            },
            ']' => {
                if containers.last() == Some(&'[') {
                    containers.pop();
                } else {
                    panic!("mismatched containers at {}: {:?}", si.get_pos(), containers);
                }
            },

            _ => {},
        }
        prev = Some(remainder[sc]);
        sc += 1;
    }

    // Split off the front up to the found schar
    let data = si.get_remainder().get(0..sc)?.trim().to_string();
    si.set_remainder(
        si.get_remainder().get(sc+1..)
            .unwrap_or("")
            .to_string()
    );
    if data.is_empty() {
        return None;
    }
    Some(SR::new(startpos, data))
}

pub struct Line {
    pub pos: Pos,
    pub data: String,
}
impl fmt::Display for Line {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "{}: {}", self.pos, self.data)
    }
}
impl SplitResult for Line {
    fn new(pos: Pos, data: String) -> Self {
        Self { pos, data }
    }
}
#[derive(Debug)]
pub struct Lines {
    pos: Pos,
    remainder: String,
}
impl Lines {
    pub fn split(scriptname: &str, script: &str, pos: Option<Pos>) -> Lines {
        Lines {
            pos: match pos {
                Some(pos) => pos,
                None => Pos::start(scriptname)
            },
            remainder: script.into(),
        }
    }
}
impl Iterator for Lines {
    type Item = Line;
    fn next(&mut self) -> Option<Self::Item> {
        split(&mut SplitItem::Lines(self), ';')
    }
}

pub struct Expr {
    pub pos: Pos,
    pub data: String,
}
impl SplitResult for Expr {
    fn new(pos: Pos, data: String) -> Self {
        Self { pos, data }
    }
}
pub struct Exprs {
    pos: Pos,
    remainder: String,
}
impl Exprs {
    pub fn split(line: &Line) -> Exprs {
        Exprs {
            pos: line.pos.clone(),
            remainder: line.data.clone(),
        }
    }
}
impl Iterator for Exprs {
    type Item = Expr;
    fn next(&mut self) -> Option<Self::Item> {
        split(&mut SplitItem::Exprs(self), ',')
    }
}

/// Easily differentiates `Token`s.
#[derive(Debug, Clone, PartialEq, Eq, Hash, PartialOrd, Ord)]
#[cfg_attr(feature = "compile", derive(Serialize, Deserialize))]
pub enum TokenType {
    /// Anything alphanumeric not beginning with a number.
    Identifier,
    /// Anything numeric containing at most one `.` and `e` each.
    Number,
    /**
    Anything enclosed by double quotes.
    May contain escaped quotes, newlines, tabs, and backslashes.
    */
    String,
    /**
    A collection of untokenized data.
    Enclosed in either `()`, `[]`, or `{}`.
    */
    Container,
    /// Another symbol, typically an operator. May have multiple characters.
    Symbol,
    Comment,
}
impl TokenType {
    fn from_value(v: &Value) -> Option<Self> {
        if let Value::Enum(_es, e) = v {
            return Some(match &*(**e).0 {
                "Identifier" => TokenType::Identifier,
                "Number" => TokenType::Number,
                "String" => TokenType::String,
                "Container" => TokenType::Container,
                "Symbol" => TokenType::Symbol,
                "Comment" => TokenType::Comment,
                _ => return None,
            });
        }
        None
    }
}

/**
A piece of the tokenized script.

Usually a singular piece but may be a `Container`.
*/
#[derive(Debug, Clone, PartialEq, Eq, Hash, PartialOrd, Ord)]
#[cfg_attr(feature = "compile", derive(Serialize, Deserialize))]
pub struct Token {
    pub ttype: TokenType,
    pub pos: Pos,
    pub data: String,
}
impl Token {
    pub(crate) fn from_value(v: &Value) -> Option<Self> {
        if let Value::Struct(hm) = v {
            return Some(Token {
                ttype: TokenType::from_value(&hm.get("ttype")?.clone_out().val)?,
                pos: Pos::from_value(&hm.get("pos")?.clone_out().val)?,
                data: match &hm.get("data")?.clone_out().val {
                    Value::String(s) => s.clone(),
                    _ => return None,
                },
            });
        }
        None
    }
}
impl fmt::Display for Token {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "{}", self.data)
    }
}
#[derive(Debug, Clone)]
pub struct Tokens {
    pos: Pos,
    remainder: String,
}
impl Tokens {
    pub fn tokenize(expr: &Expr) -> Tokens {
        Tokens {
            pos: expr.pos.clone(),
            remainder: expr.data.clone(),
        }
    }
}
impl Iterator for Tokens {
    type Item = Token;
    fn next(&mut self) -> Option<Self::Item> {
        if self.remainder.is_empty() {
            return None;
        }

        let remainder: Vec<char> = self.remainder.chars().collect();
        let mut tc: usize = 0;
        // Eat whitespace before token
        while tc < remainder.len() && remainder[tc].is_whitespace() {
            match remainder.get(tc).unwrap() {
                '\t' => self.pos.col += 4,
                _ => self.pos.col += 1,
            }
            tc += 1;
        }
        let startpos = self.pos.clone();

        // Find the next token type
        let ttype: TokenType = {
            match remainder.get(tc) {
                Some(&nc) => {
                    if nc.is_alphabetic() || nc == '_' {
                        TokenType::Identifier
                    } else if nc.is_numeric() {
                        TokenType::Number
                    } else if nc == '"' {
                        TokenType::String
                    } else if nc == '/' {
                        match remainder.get(tc+1) {
                            Some('/') => TokenType::Comment,
                            _ => TokenType::Symbol,
                        }
                    } else {
                        match nc {
                            '(' | ')' | '{' | '}' | '[' | ']' => TokenType::Container,
                            _ => TokenType::Symbol,
                        }
                    }
                },
                None => return None,
            }
        };
        tc += 1;

        // Find the next token change
        while tc < remainder.len() {
            let nc = remainder[tc];
            match nc {
                '\n' => {
                    self.pos.line += 1;
                    self.pos.col = 1;
                },
                '\t' => self.pos.col += 4,
                _ => self.pos.col += 1,
            }

            match ttype {
                TokenType::Identifier => {
                    // Allow alphanumeric and underscores
                    if !nc.is_alphanumeric() && nc != '_' {
                        break;
                    }
                },
                TokenType::Number => {
                    // Allow numbers, a single period, and a single 'e'/'E'
                    // TODO add support for hex and binary numbers
                    if !nc.is_numeric() {
                        let token = self.remainder.get(0..=tc)?.trim();
                        if token.matches('.').count() > 1
                            || token.to_lowercase().matches('e').count() > 1
                            || !token.to_lowercase()
                                .replace('.', "")
                                .replace('e', "")
                                .chars().all(char::is_numeric)
                        {
                            // Check that if there's a dot that it's not for Op::Dot
                            let dot = token.find('.');
                            if let Some(dot) = dot {
                                if let Some(dec) =  token.get(dot+1..) {
                                    if !dec.chars().all(char::is_numeric) {
                                        tc -= 1;
                                    }
                                }
                            }
                            break;
                        }
                    }
                },
                TokenType::String => {
                    // Remove escaped backslashes then remove escaped quotes to determine actual quotes
                    let token = self.remainder.get(0..=tc)?.trim();
                    let unescaped = token.replace(r"\\", "")
                        .replace(r#"\""#, "");

                    if unescaped.len() > 1 && unescaped.ends_with('"') {
                        tc += 1;
                        break;
                    }
                },
                TokenType::Container => {
                    // Check container matching
                    let mut containers = vec![];
                    let token: Vec<char> = self.remainder.get(0..=tc)?.trim()
                        .chars().collect();
                    for c in &token {
                        match c {
                            '(' | '{' | '[' => containers.push(c),
                            ')' => {
                                if containers.is_empty() || containers.last().unwrap() != &&'(' {
                                    panic!("mismatched containers at {}: {:?}", self.pos, token);
                                }
                                containers.pop();
                            },
                            '}' => {
                                if containers.is_empty() || containers.last().unwrap() != &&'{' {
                                    panic!("mismatched containers at {}: {:?}", self.pos, token);
                                }
                                containers.pop();
                            },
                            ']' => {
                                if containers.is_empty() || containers.last().unwrap() != &&'[' {
                                    panic!("mismatched containers at {}: {:?}", self.pos, token);
                                }
                                containers.pop();
                            },
                            _ => {},
                        }
                    }

                    // If the original container is closed then we're done
                    if containers.is_empty() {
                        tc += 1;
                        break;
                    }
                },
                TokenType::Symbol => {
                    let token = self.remainder.get(0..=tc)?.trim();
                    match token {
                        "::" |
                        "**" |
                        "<<" |
                        ">>" |
                        "<=>" |
                        "<=" |
                        ">=" |
                        "==" |
                        "!=" |
                        "&&" |
                        "||" |
                        ".." |
                        "..=" |
                        "->" |
                        "<-" |
                        "=>" |
                        "|>" => {},
                        _ => break,
                    }
                },
                TokenType::Comment => {},
            }
            tc += 1;
        }

        // Split off the front up to the found token change
        let token = self.remainder.get(0..tc)?.trim().to_string();
        self.remainder = self.remainder.get(tc..)
            .unwrap_or("")
            .to_string();
        if token.is_empty() {
            return None;
        }
        Some(Token {
            ttype,
            pos: startpos,
            data: token,
        })
    }
}

pub fn unescape(s: &str) -> String {
    s.chars()
        .fold((String::new(), None), |(mut ns, prev), c| {
            if let Some('\\') = prev {
                match c {
                    '\\' => {
                        return (ns, None);
                    },
                    '"' => {
                        ns.pop();
                        ns.push(c);
                        return (ns, None);
                    },
                    'n' => {
                        ns.pop();
                        ns.push('\n');
                        return (ns, None);
                    },
                    't' => {
                        ns.pop();
                        ns.push('\t');
                        return (ns, None);
                    },
                    _ => {},
                }
            }
            ns.push(c);
            (ns, Some(c))
        }).0
}
