use std::collections::HashMap;
use std::sync::Arc;

use itertools::Itertools;
use itertools::EitherOrBoth::{Both, Right, Left};
use num::{BigInt, BigRational, FromPrimitive, One, ToPrimitive, Zero};
use num::pow::Pow;

use maplit::hashmap;

use crate::{Env, Error, FnArgs, FnReturn, Pos, TVal, RefTVal, Token, Type, Value, hashablemap::HashableMap, umcore};

#[allow(unused)] // TODO implement all ops
#[derive(Debug, Clone, Copy, PartialEq, Eq)]
pub enum Op {
    ColonColon,

    Dot,

    Call,
    Index,
    Construct,

    Question,
    Tilde,
    Exclaim,

    Ampersat,
    Hash,
    Dollar,
    Backslash,
    Colon,

    AsteriskAsterisk,
    Asterisk,
    Slash,
    Percent,
    Plus,
    Dash,

    LangleLangle,
    RangleRangle,

    Ampersand,
    Caret,
    Vbar,

    LangleEqualRangle,
    Langle,
    LangleEqual,
    Rangle,
    RangleEqual,

    EqualEqual,
    ExclaimEqual,

    AmpersandAmpersand,
    VbarVbar,

    DotDot,
    DotDotEqual,

    DashRangle,
    LangleDash,

    EqualRangle,
    VbarRangle,
    Equal,
}
impl Op {
    pub fn op<'a>(env: &'a Env<'a>, op: Token, lval: Option<&'a RefTVal>, rval: Option<&'a RefTVal>) -> FnReturn {
        match (lval, rval) {
            (None, Some(rv)) => match &*op.data {
                "!" => return OpTrait::<'a, {Op::Exclaim}>::op(rv, env, op.pos, None),

                "*" => return OpTrait::<'a, {Op::Asterisk}>::op(rv, env, op.pos, None),
                "+" => return OpTrait::<'a, {Op::Plus}>::op(rv, env, op.pos, None),
                "-" => return OpTrait::<'a, {Op::Dash}>::op(rv, env, op.pos, None),

                "&" => return OpTrait::<'a, {Op::Ampersand}>::op(rv, env, op.pos, None),

                _ => {},
            },
            (Some(lv), _) => match &*op.data {
                "." => return OpTrait::<{Op::Dot}>::op(lv, env, op.pos, rval),

                "(" => return OpTrait::<{Op::Call}>::op(lv, env, op.pos, rval),
                "[" => return OpTrait::<{Op::Index}>::op(lv, env, op.pos, rval),
                "{" => return OpTrait::<{Op::Construct}>::op(lv, env, op.pos, rval),

                "@" => return OpTrait::<{Op::Ampersat}>::op(lv, env, op.pos, rval),
                ":" => return OpTrait::<{Op::Colon}>::op(lv, env, op.pos, rval),

                "**" => return OpTrait::<{Op::AsteriskAsterisk}>::op(lv, env, op.pos, rval),
                "*" => return OpTrait::<{Op::Asterisk}>::op(lv, env, op.pos, rval),
                "/" => return OpTrait::<{Op::Slash}>::op(lv, env, op.pos, rval),
                "%" => return OpTrait::<{Op::Percent}>::op(lv, env, op.pos, rval),
                "+" => return OpTrait::<{Op::Plus}>::op(lv, env, op.pos, rval),
                "-" => return OpTrait::<{Op::Dash}>::op(lv, env, op.pos, rval),

                "<<" => return OpTrait::<{Op::LangleLangle}>::op(lv, env, op.pos, rval),
                ">>" => return OpTrait::<{Op::RangleRangle}>::op(lv, env, op.pos, rval),

                "&" => return OpTrait::<{Op::Ampersand}>::op(lv, env, op.pos, rval),
                "^" => return OpTrait::<{Op::Caret}>::op(lv, env, op.pos, rval),
                "|" => return OpTrait::<{Op::Vbar}>::op(lv, env, op.pos, rval),

                "<" => return OpTrait::<{Op::Langle}>::op(lv, env, op.pos, rval),
                "<=" => return OpTrait::<{Op::LangleEqual}>::op(lv, env, op.pos, rval),
                ">" => return OpTrait::<{Op::Rangle}>::op(lv, env, op.pos, rval),
                ">=" => return OpTrait::<{Op::RangleEqual}>::op(lv, env, op.pos, rval),

                "==" => return OpTrait::<{Op::EqualEqual}>::op(lv, env, op.pos, rval),
                "!=" => return OpTrait::<{Op::ExclaimEqual}>::op(lv, env, op.pos, rval),

                "&&" => return OpTrait::<{Op::AmpersandAmpersand}>::op(lv, env, op.pos, rval),
                "||" => return OpTrait::<{Op::VbarVbar}>::op(lv, env, op.pos, rval),

                "->" => return OpTrait::<{Op::DashRangle}>::op(lv, env, op.pos, rval),

                "=" => return OpTrait::<{Op::Equal}>::op(lv, env, op.pos, rval),

                _ => {},
            },
            (None, None) => {},
        }
        (None, Err(Error::Script(format!("invalid operator {} for {:?} and {:?}", op, lval, rval), Some(op.pos))))
    }
}

pub trait OpTrait<'a, const OP: Op> {
    fn op(&self, env: &'a Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn;
}
// TODO impl more ops
impl<'a> OpTrait<'a, {Op::Dot}> for RefTVal {
    fn op(&self, env: &'a Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        let bind = |name: &str, os: &str, f: TVal| -> FnReturn {
            let bound = umcore::function_rebind(FnArgs::Normal {
                this: Box::new(f.clone().into()),
                pos: Some(pos.clone()),
                args: Value::List(vec![
                    f.into(),
                    self.clone(),
                ]).into(),
            });
            let (bargs, bvars, bbody) = match bound {
                (_, Ok(brv)) => {
                    if let TVal { val: Value::Function { args: bargs, vars: bvars, body: bbody }, .. } = brv.clone_out() {
                        (bargs, bvars, bbody)
                    } else {
                        return (None, Err(Error::Script(format!("failed to bind {}.{}: {}", name, os, brv), Some(pos.clone()))))
                    }
                },
                (_, Err(Error::Script(m, _))) => {
                    return (None, Err(Error::Script(format!("failed to bind {}.{}: {}", name, os, m), Some(pos.clone()))))
                },
                _ => {
                    return (None, Err(Error::Script(format!("failed to bind {}.{}", name, os), Some(pos.clone()))))
                },
            };
            (None, Ok(
                Value::Function {
                    args: bargs,
                    vars: bvars,
                    body: bbody,
                }.into()
            ))
        };
        let get_impl = |ts: &str, os: &str| -> FnReturn {
            if let Some(impls) = self.clone_out().attr.get("impls") {
                if let Value::Map(impls) = impls.clone_out().val {
                    for im in impls {
                        if let Value::Struct(imap) = im.1.clone_out().val {
                            if imap.contains_key(os) {
                                match imap[os].clone_out().val {
                                    Value::Function { args, vars: fvars, body } => {
                                        return bind("IMPLS", os, Value::Function {
                                            args,
                                            vars: fvars,
                                            body,
                                        }.into());
                                    },
                                    _ => return (None, Ok(imap[os].clone())),
                                }
                            }
                        }
                    }
                }
            }
            (None, Err(Error::Script(format!("{} access failed for {:?} with {}", ts, self.clone_out().val, other.as_ref().unwrap()), Some(pos.clone()))))
        };

        if let Some(orv) = other {
            if let TVal { val: Value::String(os), .. } = orv.clone_out() {
                let name: &str = match &self.clone_out().val {
                    Value::Type(_, ts) => match ts.get(&os) {
                        Some(v) => return (None, Ok(v.clone())),
                        None => return get_impl("type", &os),
                    },
                    Value::Reference(_) => {
                        if let Value::Reference(rv) = self.clone_out().val {
                            return OpTrait::<{Op::Dot}>::op(&rv, env, pos, other);
                        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
                            return OpTrait::<{Op::Dot}>::op(self, env, pos, Some(&orv));
                        }
                        "Reference"
                    },
                    Value::Number(_) => "Number",
                    Value::String(_) => "String",

                    Value::List(_) => "List",
                    Value::Map(_) => "Map",
                    Value::Enum(_, _) => "Enum",
                    Value::Struct(st) => match st.get(&os) {
                        Some(v) => return (None, Ok(Value::Reference(v.clone()).into())),
                        None => return get_impl("struct", &os),
                    },

                    Value::Function { .. } => "Function",
                    Value::Thread(_) => "Thread",
                };

                if let Some(lhs) = env.get(name) {
                    // Attempt to bind self to the found lhs
                    match OpTrait::<{Op::Dot}>::op(lhs, env, pos.clone(), other) {
                        (_, Ok(oprv)) => {
                            if let Value::Function { args, vars: fvars, body } = oprv.clone_out().val {
                                return bind(name, &os, Value::Function {
                                    args,
                                    vars: fvars,
                                    body,
                                }.into());
                            }
                            todo!("{:?}", oprv);
                        },
                        (_, Err(m)) => return (None, Err(m)),
                    }
                }
            }
        }

        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", ".", self, other), Some(pos))))
    }
}

impl<'a> OpTrait<'a, {Op::Call}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::Call}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::Call}>::op(self, env, pos, Some(&orv));
        }

        match &self.clone_out().val {
            Value::Function { args, vars: fvars, body } => {
                // Apply bound args
                if let Some(other) = other {
                    if let TVal { val: Value::List(l1), .. } = other.clone_out() {
                        let mut fenv = Env::from(fvars.clone());
                        fenv.set("this", &self.clone());

                        // Set arg names in env
                        let mut l2 = vec![];
                        let mut mi = 0;
                        let mut bi = 0;
                        for (i, a) in args.iter().enumerate() {
                            if a.0.starts_with('*') {
                                break;
                            }
                            if let Some(v) = fvars.get(&a.0) { // Check for bound vars
                                // TODO only check immediate scope
                                if !a.1.equiv(&v.clone_out().ttype) {
                                    return (None, Err(Error::Script(format!("operator {}: function bound arg type mismatch: expected {}:{}, got {}", "()", a.0, a.1, v), Some(pos))));
                                }
                                l2.push(v.clone());
                                bi += 1;
                            } else if let Some(v) = l1.get(i-bi) { // Check provided vars
                                let ttype = match v.clone_out() {
                                    TVal { val: Value::Reference(r), .. } => r.clone_out().ttype,
                                    v => v.ttype,
                                };
                                if !a.1.equiv(&ttype) {
                                    // eprintln!("{:#?}", l);
                                    return (None, Err(Error::Script(format!("operator {}: function provided arg type mismatch: expected {}:{}, got {}", "()", a.0, a.1, v), Some(pos))));
                                }
                                fenv.set(&a.0, &v.clone());
                                l2.push(v.clone());
                            }
                            mi += 1;
                        }
                        if let Some(a) = args.last() {
                            if a.0.starts_with('*') {
                                let mut vargs = vec![];
                                for i in mi..l1.len() {
                                    if let Some(v) = l1.get(i-bi) {
                                        l2.push(v.clone());
                                        vargs.push(v.clone());
                                    }
                                }
                                if a.0.len() > 1 {
                                    fenv.set(&a.0[1..], &Value::List(vargs).into());
                                }
                            } else if mi+1 < l1.len() {
                                return (None, Err(Error::Script(format!("operator {}: function got excess args {:?}", "()", &l1[mi+1..]), Some(pos))));
                            }
                        }

                        let (vars, mut val) = body.call(&fenv, FnArgs::Normal {
                            this: Box::new(self.clone()),
                            pos: Some(pos),
                            args: Value::List(l2).into(),
                        });
                        if self.clone_out().attr.contains_key("test") {
                            val = Ok(Value::from_result(&val).into());
                        }
                        // TODO check return type
                        return (vars, val);
                    }
                }
                return (None, Err(Error::Script(format!("operator {}: missing function arguments", "()"), Some(pos))));
            },
            Value::Enum(vs, v) => {
                if let Some(other) = other {
                    let l = if let TVal { val: Value::List(l), .. } = other.clone_out() {
                        l
                    } else {
                        vec![other.clone()]
                    };

                    if let Type::Enum(es) = &*self.clone_out().ttype {
                        let ets = es.iter()
                            .find_map(|(n, e)| {
                                if n == &v.0 {
                                    match &**e {
                                        Type::Enum(es) => Some(Ok(es)),
                                        _ => Some(Err(())),
                                    }
                                } else {
                                    None
                                }
                            });
                        let ets = if let Some(Ok(ets)) = ets {
                            ets
                        } else {
                            return (None, Err(Error::Script(format!("operator {}: invalid enum variant {} for enum {:?}", "()", v.0, es), Some(pos))));
                        };
                        let mut vals: Vec<RefTVal> = vec![];
                        for etpt in ets.iter().zip_longest(l) {
                            match etpt {
                                Both(et, pt) => {
                                    if !et.1.equiv(&pt.clone_out().ttype) {
                                        // TODO properly instantiate generic parameters
                                        if env.has(&et.0) {
                                            // eprintln!("ets {:?}", ets);
                                            return (None, Err(Error::Script(format!("enum param type mismatch: expected {:?}, got {}", et.1, pt), Some(pos))));
                                        }
                                    }
                                    vals.push(pt);
                                },
                                Right(pt) => {
                                    vals.push(pt);
                                },
                                Left(et) => return (None, Err(Error::Script(format!("enum param missing: expected {:?}", et), Some(pos)))),
                            }
                        }

                        return (None, Ok(Value::Enum(Arc::clone(vs), Box::new((v.0.clone(), vals))).into()));
                    }
                }
            },
            _ => {},
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "()", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::Index}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::Index}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::Index}>::op(self, env, pos, Some(&orv));
        }

        match &self.clone_out().val {
            Value::Type(subt, ts) => {
                let mut nsubt: Type = (**subt).clone();
                match nsubt {
                    Type::List(ref mut lt) => {
                        if let Some(orv) = other {
                            if let TVal { val: Value::List(l), .. } = orv.clone_out() {
                                if let Some(rv0) = l.get(0) {
                                    if let TVal { val: Value::Type(tt, _), .. } = rv0.clone_out() {
                                        *lt = Arc::clone(&tt);
                                        return (None, Ok(Value::Type(Arc::new(nsubt), Arc::clone(&ts)).into()));
                                    }
                                }
                            }
                        }
                    },
                    Type::Map(ref mut k, ref mut v) => {
                        if let Some(orv) = other {
                            if let TVal { val: Value::List(l), .. } = orv.clone_out() {
                                if let Some(rv0) = l.get(0) {
                                    if let TVal { val: Value::Type(kt, _), .. } = rv0.clone_out() {
                                        if let Some(rv1) = l.get(1) {
                                            if let TVal { val: Value::Type(vt, _), .. } = rv1.clone_out() {
                                                *k = Arc::clone(&kt);
                                                *v = Arc::clone(&vt);
                                                return (None, Ok(Value::Type(Arc::new(nsubt), Arc::clone(&ts)).into()));
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    },
                    Type::Enum(_) => {
                        // TODO make generic indexing actually generic
                        if let Type::Generic(ut, tp) = &*self.clone_out().ttype {
                            if let Some(orv) = other {
                                if let TVal { val: Value::List(l), .. } = orv.clone_out() {
                                    let mut ntp: HashableMap<String, Arc<Type>> = (**tp).clone();
                                    for ((_, v), t) in ntp.iter_mut().zip(l.iter()) {
                                        *v = match &t.clone_out().val {
                                            Value::Type(t, _) => Arc::clone(&t),
                                            _ => return (None, Err(Error::Script(format!("invalid type parameter {}", t.clone_out().val), Some(pos)))),
                                        };
                                    }
                                    if let Value::Type(subt, ts) = &self.clone_out().val {
                                        let mut nts: HashableMap<String, RefTVal> = (**ts).clone();
                                        for v in nts.map.values_mut() {
                                            *v = TVal {
                                                ttype: Arc::new(Type::Enum(Arc::new({
                                                    if let Type::Enum(es) = &*v.clone_out().ttype {
                                                        let nes = es.iter().zip(l.iter())
                                                            .map(|((k, _), t)| {
                                                                match &t.clone_out().val {
                                                                    Value::Type(t, _) => Ok((k.clone(), Arc::clone(t))),
                                                                    _ => return Err(Error::Script(format!("invalid type parameter {}", t.clone_out().val), Some(pos.clone()))),
                                                                }
                                                            }).collect::<Result<Vec<(String, Arc<Type>)>, Error>>();
                                                        match nes {
                                                            Ok(es) => es,
                                                            Err(m) => return (None, Err(m)),
                                                        }
                                                    } else {
                                                        return (None, Err(Error::Script(format!("invalid type parameter {}", v.clone_out().ttype), Some(pos))));
                                                    }
                                                }))),
                                                attr: hashmap!{}.into(),
                                                val: v.clone_out().val.clone(),
                                            }.into();
                                        }

                                        return (None, Ok(TVal {
                                            ttype: Arc::new(Type::Generic(Arc::clone(ut), Arc::new(ntp))),
                                            attr: hashmap!{}.into(),
                                            val: Value::Type(subt.clone(), Arc::new(nts)),
                                        }.into()));
                                    }
                                }
                            }
                        }
                    },
                    _ => {},
                }
                return (None, Err(Error::Script(format!("type index mismatch for {} and {:?}", self, other), Some(pos))));
            },
            Value::String(s) => {
                if let Some(orv) = other {
                    if let TVal { val: Value::List(l), .. } = orv.clone_out() {
                        if let Some(rv0) = l.get(0) {
                            if let TVal { val: Value::Number(ref idx), .. } = rv0.clone_out() {
                                let mut idx: usize = match idx.to_usize() {
                                    Some(n) => n,
                                    None => match (-idx).to_usize() {
                                        Some(n) => s.len().wrapping_sub(n),
                                        None => return (None, Err(Error::Script(format!("string index doesn't fit in usize: {}", idx), Some(pos))))
                                    },
                                };
                                // Handle range
                                if let Some(rv1) = l.get(1) {
                                    if let TVal { val: Value::Number(end), .. } = rv1.clone_out() {
                                        let mut end: usize = end.to_usize().unwrap();
                                        if idx >= s.len() {
                                            idx = s.len();
                                        }
                                        if end >= s.len() {
                                            end = s.len();
                                        }
                                        if end < idx {
                                            end = idx;
                                        }
                                        return (None, Ok(Value::String(s[idx..end].to_string()).into()));
                                    }
                                }
                                if idx < s.len() {
                                    return (None, Ok(s[idx..=idx].into()));
                                }
                                return (None, Err(Error::Script(format!("string index out of range: {}", idx), Some(pos))));
                            }
                        }
                    }
                }
            },
            Value::List(l1) => {
                if let Some(orv) = other {
                    if let TVal { val: Value::List(l2), .. } = orv.clone_out() {
                        if let Some(rv0) = l2.get(0) {
                            if let TVal { val: Value::Number(ref idx), .. } = rv0.clone_out() {
                                let mut idx: usize = match idx.to_usize() {
                                    Some(n) => n,
                                    None => match (-idx).to_usize() {
                                        Some(n) => l1.len().wrapping_sub(n),
                                        None => return (None, Err(Error::Script(format!("list index doesn't fit in usize: {}", idx), Some(pos))))
                                    },
                                };
                                // Handle range
                                if let Some(rv1) = l2.get(1) {
                                    if let TVal { val: Value::Number(end), .. } = rv1.clone_out() {
                                        let mut end: usize = end.to_usize().unwrap();
                                        if idx >= l1.len() {
                                            idx = l1.len();
                                        }
                                        if end >= l1.len() {
                                            end = l1.len();
                                        }
                                        if end < idx {
                                            end = idx;
                                        }
                                        return (None, Ok(Value::List(l1[idx..end].to_vec()).into()));
                                    }
                                }
                                return match l1.get(idx) {
                                    Some(val) => (None, Ok(val.clone())),
                                    None => (None, Err(Error::Script(format!("list index out of range: {}", idx), Some(pos)))),
                                };
                            }
                        }
                    }
                }
            },
            Value::Map(hm) => {
                if let Some(orv) = other {
                    if let TVal { val: Value::List(l), .. } = orv.clone_out() {
                        if let Some(rv0) = l.get(0) {
                            let TVal { val, .. } = rv0.clone_out();
                            return match hm.get(&val.clone().into()) {
                                Some(val) => (None, Ok(val.clone())),
                                None => (None, Err(Error::Script(format!("map index not found: {}", val), Some(pos)))),
                            };
                        }
                    }
                }
            },
            _ => {},
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "[]", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::Construct}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::Construct}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::Construct}>::op(self, env, pos, Some(&orv));
        }

        if let Value::Type(st1, m1) = &self.clone_out().val {
            if let Some(orv) = other {
                match orv.clone_out().val {
                    Value::Map(m2) => {
                        let mut map = hashmap!{};
                        for (k, v) in &m2.map {
                            if let TVal { val: Value::String(ks), .. } = k.clone_out() {
                                if let Some(evt) = m1.get(&ks) {
                                    if let Value::Type(tt, _) = &evt.clone_out().val {
                                        let v = v.clone_out();
                                        if v.ttype.equiv(tt) {
                                            map.insert(ks.clone(), v.into());
                                            continue;
                                        }
                                    }
                                    return (None, Err(Error::Script(format!("struct value {} type mismatch: expected {}, got {}", k, evt.clone_out().val, v.clone_out().ttype), Some(pos))));
                                }
                            }
                            return (None, Err(Error::Script(format!("struct key {} not found in {}", k, self), Some(pos))));
                        }
                        let mut st: TVal = Value::Struct(map.into()).into();
                        st.attr = self.clone_out().attr;
                        return (None, Ok(st.into()));
                    },
                    Value::Type(t, _st2) if t == Type::none() => {
                        if st1 == &Type::map() {
                            return (None, Ok(Value::Map(hashmap!{}.into()).into()));
                        }
                    },
                    _ => {},
                }
            }
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "{}", self, other), Some(pos))))
    }
}

impl<'a> OpTrait<'a, {Op::Exclaim}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::Exclaim}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::Exclaim}>::op(self, env, pos, Some(&orv));
        }

        if let Value::Number(r1) = &self.clone_out().val {
            if other.is_none() {
                if r1 == &BigRational::one() {
                    return (None, Ok(Value::Number(BigRational::zero()).into()));
                } else if r1 == &BigRational::zero() {
                    return (None, Ok(Value::Number(BigRational::one()).into()));
                }
            }
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "!", self, other), Some(pos))))
    }
}

impl<'a> OpTrait<'a, {Op::Ampersat}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::Ampersat}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::Ampersat}>::op(self, env, pos, Some(&orv));
        }

        if let Some(orv) = other {
            if let TVal { val: Value::String(s), .. } = orv.clone_out() {
                let tself = &mut (*self.try_lock().unwrap());
                let attr: &mut HashMap<String, RefTVal> = &mut tself.attr;
                if let Some(a) = attr.get(&s) {
                    return (None, Ok(Value::Reference(a.clone()).into()));
                }

                if let Value::Struct(st) = &tself.val {
                    let mut attrs: Vec<RefTVal> = vec![];
                    for (_, v) in st.iter() {
                        if v.clone_out().attr.contains_key(&s) {
                            attrs.push(v.clone());
                        }
                    }
                    return (None, Ok(Value::Reference(Value::List(attrs).into()).into()));
                }

                attr.insert(s.clone(), Value::none().into());
                return (None, Ok(Value::Reference(attr[&s].clone()).into()));
            }
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", ":", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::Colon}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::Colon}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::Colon}>::op(self, env, pos, Some(&orv));
        }

        if let Some(orv) = other {
            let TVal { val: _, .. } = orv.clone_out();
            return (None, Ok(Value::List(vec![self.clone(), other.unwrap().clone()]).into()));
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", ":", self, other), Some(pos))))
    }
}

impl<'a> OpTrait<'a, {Op::AsteriskAsterisk}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::AsteriskAsterisk}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::AsteriskAsterisk}>::op(self, env, pos, Some(&orv));
        }

        if let Value::Number(r1) = &self.clone_out().val {
            if let Some(orv) = other {
                if let TVal { val: Value::Number(r2), .. } = orv.clone_out() {
                    return (None, Ok(Value::Number({
                        // TODO add proper support for fractional powers
                        let r3 = Pow::pow(r1, r2.numer());
                        let dr: u32 = ToPrimitive::to_u32(r2.denom()).expect("op ** denom can't fit in u32");
                        BigRational::new(
                            r3.numer().nth_root(dr),
                            r3.denom().nth_root(dr),
                        )
                    }).into()));
                }
            }
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "**", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::Asterisk}> for RefTVal {
    fn op(&self, _env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        match &self.clone_out().val {
            Value::Reference(rv) => return (None, Ok(rv.deep_clone(true))),
            Value::Number(r1) => if let Some(orv) = other {
                match orv.clone_out() {
                    TVal { val: Value::Number(r2), .. } => return (None, Ok(Value::Number(r1 * r2).into())),
                    TVal { val: Value::String(s), .. } => return (None, Ok(Value::String(
                        s.repeat(
                            match r1.to_usize() {
                                Some(n) => n,
                                None => return (None, Err(Error::Script(format!("failed to repeat string {} times", r1), Some(pos))))
                            }
                        )
                    ).into())),
                    _ => {},
                }
            },
            Value::String(s) => if let Some(orv) = other {
                if let TVal { val: Value::Number(r), .. } = orv.clone_out() {
                    return (None, Ok(Value::String(
                        s.repeat(
                            match r.to_usize() {
                                Some(n) => n,
                                None => return (None, Err(Error::Script(format!("failed to repeat string {} times", r), Some(pos))))
                            }
                        )
                    ).into()));
                }
            },
            Value::Struct(hm) => {
                let hm: HashMap<RefTVal, RefTVal> = hm.iter()
                    .map(|(k, v)| (Value::String(k.into()).into(), v.clone()))
                    .collect();
                return (None, Ok(Value::Map(hm.into()).into()));
            },
            _ => {},
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "*", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::Slash}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::Slash}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::Slash}>::op(self, env, pos, Some(&orv));
        }

        match &self.clone_out().val {
            Value::Number(r1) => if let Some(orv) = other {
                if let TVal { val: Value::Number(r2), .. } = orv.clone_out() {
                    return (None, Ok(Value::Number(r1 / r2).into()));
                }
            },
            Value::String(s1) => if let Some(orv) = other {
                if let TVal { val: Value::String(s2), .. } = orv.clone_out() {
                    return (None, Ok(Value::String(s1.clone() + "/" + &s2).into()));
                }
            },
            _ => {},
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "/", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::Percent}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::Percent}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::Percent}>::op(self, env, pos, Some(&orv));
        }

        if let Value::Number(r1) = &self.clone_out().val {
            if let Some(orv) = other {
                if let TVal { val: Value::Number(r2), .. } = orv.clone_out() {
                    return (None, Ok(Value::Number(r1 % r2).into()));
                }
            }
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "%", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::Plus}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::Plus}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::Plus}>::op(self, env, pos, Some(&orv));
        }

        match &self.clone_out().val {
            Value::Number(r1) => match other {
                Some(orv) => if let TVal { val: Value::Number(r2), .. } = orv.clone_out() {
                    return (None, Ok(Value::Number(r1 + r2).into()))
                },
                None => return (None, Ok(Value::Number(r1.clone()).into())),
            },
            Value::String(s1) => if let Some(orv) = other {
                if let TVal { val: Value::String(s2), .. } = orv.clone_out() {
                    return (None, Ok(Value::String(s1.to_string()+&s2).into()));
                }
            }
            Value::List(l1) => if let Some(orv) = other {
                if let TVal { val: Value::List(l2), .. } = orv.clone_out() {
                    let mut l3 = Vec::with_capacity(l1.len() + l2.len());
                    l3.extend(l1.iter().cloned());
                    l3.extend(l2.iter().cloned());
                    return (None, Ok(Value::List(l3).into()));
                }
            },
            Value::Map(m1) => if let Some(orv) = other {
                if let TVal { val: Value::Map(m2), .. } = orv.clone_out() {
                    let mut m3 = HashMap::with_capacity(m1.len() + m2.len());
                    m3.extend(m1.clone().into_iter());
                    m3.extend(m2.into_iter());
                    return (None, Ok(Value::Map(m3.into()).into()));
                }
            },
            _ => {},
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "+", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::Dash}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::Dash}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::Dash}>::op(self, env, pos, Some(&orv));
        }

        if let Value::Number(r1) = &self.clone_out().val {
            match other {
                Some(orv) => if let TVal { val: Value::Number(r2), .. } = orv.clone_out() {
                    return (None, Ok(Value::Number(r1 - r2).into()))
                },
                None => return (None, Ok(Value::Number(-r1).into())),
            }
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "-", self, other), Some(pos))))
    }
}

impl<'a> OpTrait<'a, {Op::LangleLangle}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::LangleLangle}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::LangleLangle}>::op(self, env, pos, Some(&orv));
        }

        if let Value::Number(r1) = &self.clone_out().val {
            if let Some(orv) = other {
                if let TVal { val: Value::Number(r2), .. } = orv.clone_out() {
                    if r1.denom() == &BigInt::one() && r2.denom() == &BigInt::one() {
                        if let Some(u1) = r1.numer().to_usize() {
                            if let Some(u2) = r2.numer().to_usize() {
                                return (None, Ok(Value::Number(BigRational::new(
                                    BigInt::from_usize(u1 << u2).unwrap(),
                                    BigInt::one(),
                                )).into()));
                            }
                        }
                    }
                }
            }
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "<<", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::RangleRangle}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::RangleRangle}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::RangleRangle}>::op(self, env, pos, Some(&orv));
        }

        if let Value::Number(r1) = &self.clone_out().val {
            if let Some(orv) = other {
                if let TVal { val: Value::Number(r2), .. } = orv.clone_out() {
                    if r1.denom() == &BigInt::one() && r2.denom() == &BigInt::one() {
                        if let Some(u1) = r1.numer().to_usize() {
                            if let Some(u2) = r2.numer().to_usize() {
                                return (None, Ok(Value::Number(BigRational::new(
                                    BigInt::from_usize(u1 >> u2).unwrap(),
                                    BigInt::one(),
                                )).into()));
                            }
                        }
                    }
                }
            }
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", ">>", self, other), Some(pos))))
    }
}

impl<'a> OpTrait<'a, {Op::Ampersand}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::Ampersand}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::Ampersand}>::op(self, env, pos, Some(&orv));
        }

        if let Value::Number(r1) = &self.clone_out().val {
            if let Some(orv) = other {
                if let TVal { val: Value::Number(r2), .. } = orv.clone_out() {
                    if r1.denom() == &BigInt::one() && r2.denom() == &BigInt::one() {
                        if let Some(u1) = r1.numer().to_usize() {
                            if let Some(u2) = r2.numer().to_usize() {
                                return (None, Ok(Value::Number(BigRational::new(
                                    BigInt::from_usize(u1 & u2).unwrap(),
                                    BigInt::one(),
                                )).into()));
                            }
                        }
                    }
                }
            }
        }

        // Create reference
        if other.is_none() {
            return (None, Ok(Value::Reference(self.clone()).into()));
        }

        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "&", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::Caret}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::Caret}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::Caret}>::op(self, env, pos, Some(&orv));
        }

        if let Value::Number(r1) = &self.clone_out().val {
            if let Some(orv) = other {
                if let TVal { val: Value::Number(r2), .. } = orv.clone_out() {
                    if r1.denom() == &BigInt::one() && r2.denom() == &BigInt::one() {
                        if let Some(u1) = r1.numer().to_usize() {
                            if let Some(u2) = r2.numer().to_usize() {
                                return (None, Ok(Value::Number(BigRational::new(
                                    BigInt::from_usize(u1 ^ u2).unwrap(),
                                    BigInt::one(),
                                )).into()));
                            }
                        }
                    }
                }
            }
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "^", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::Vbar}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::Vbar}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::Vbar}>::op(self, env, pos, Some(&orv));
        }

        if let Value::Number(r1) = &self.clone_out().val {
            if let Some(orv) = other {
                if let TVal { val: Value::Number(r2), .. } = orv.clone_out() {
                    if r1.denom() == &BigInt::one() && r2.denom() == &BigInt::one() {
                        if let Some(u1) = r1.numer().to_usize() {
                            if let Some(u2) = r2.numer().to_usize() {
                                return (None, Ok(Value::Number(BigRational::new(
                                    BigInt::from_usize(u1 | u2).unwrap(),
                                    BigInt::one(),
                                )).into()));
                            }
                        }
                    }
                }
            }
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "|", self, other), Some(pos))))
    }
}

impl<'a> OpTrait<'a, {Op::Langle}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::Langle}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::Langle}>::op(self, env, pos, Some(&orv));
        }

        match &self.clone_out().val {
            Value::Number(r1) => if let Some(orv) = other {
                if let TVal { val: Value::Number(r2), .. } = orv.clone_out() {
                    if let Some(r) = BigRational::from_u32((r1 < &r2) as u32) {
                        return (None, Ok(Value::Number(r).into()));
                    }
                }
            },
            Value::String(s1) => if let Some(orv) = other {
                if let TVal { val: Value::String(s2), .. } = orv.clone_out() {
                    if let Some(r) = BigRational::from_u32((s1 < &s2) as u32) {
                        return (None, Ok(Value::Number(r).into()));
                    }
                }
            },
            _ => {},
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "<", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::LangleEqual}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::LangleEqual}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::LangleEqual}>::op(self, env, pos, Some(&orv));
        }

        match &self.clone_out().val {
            Value::Number(r1) => if let Some(orv) = other {
                if let TVal { val: Value::Number(r2), .. } = orv.clone_out() {
                    if let Some(r) = BigRational::from_u32((r1 <= &r2) as u32) {
                        return (None, Ok(Value::Number(r).into()));
                    }
                }
            },
            Value::String(s1) => if let Some(orv) = other {
                if let TVal { val: Value::String(s2), .. } = orv.clone_out() {
                    if let Some(r) = BigRational::from_u32((s1 <= &s2) as u32) {
                        return (None, Ok(Value::Number(r).into()));
                    }
                }
            },
            _ => {},
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "<=", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::Rangle}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::Rangle}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::Rangle}>::op(self, env, pos, Some(&orv));
        }

        match &self.clone_out().val {
            Value::Number(r1) => if let Some(orv) = other {
                if let TVal { val: Value::Number(r2), .. } = orv.clone_out() {
                    if let Some(r) = BigRational::from_u32((r1 > &r2) as u32) {
                        return (None, Ok(Value::Number(r).into()));
                    }
                }
            },
            Value::String(s1) => if let Some(orv) = other {
                if let TVal { val: Value::String(s2), .. } = orv.clone_out() {
                    if let Some(r) = BigRational::from_u32((s1 > &s2) as u32) {
                        return (None, Ok(Value::Number(r).into()));
                    }
                }
            },
            _ => {},
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", ">", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::RangleEqual}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::RangleEqual}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::RangleEqual}>::op(self, env, pos, Some(&orv));
        }

        match &self.clone_out().val {
            Value::Number(r1) => if let Some(orv) = other {
                if let TVal { val: Value::Number(r2), .. } = orv.clone_out() {
                    if let Some(r) = BigRational::from_u32((r1 >= &r2) as u32) {
                        return (None, Ok(Value::Number(r).into()));
                    }
                }
            },
            Value::String(s1) => if let Some(orv) = other {
                if let TVal { val: Value::String(s2), .. } = orv.clone_out() {
                    if let Some(r) = BigRational::from_u32((s1 >= &s2) as u32) {
                        return (None, Ok(Value::Number(r).into()));
                    }
                }
            },
            _ => {},
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", ">=", self, other), Some(pos))))
    }
}

impl<'a> OpTrait<'a, {Op::EqualEqual}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::EqualEqual}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::EqualEqual}>::op(self, env, pos, Some(&orv));
        }

        match &self.clone_out().val {
            Value::Type(t1, _st1) => if let Some(orv) = other {
                if let Value::Type(t2, _st2) = orv.clone_out().val {
                    if let Some(r) = BigRational::from_u32((t1.equiv(t2)) as u32) {
                        return (None, Ok(Value::Number(r).into()));
                    }
                }
            },
            Value::Number(r1) => if let Some(orv) = other {
                if let TVal { val: Value::Number(ref r2), .. } = orv.clone_out() {
                    if let Some(r) = BigRational::from_u32((r1 == r2) as u32) {
                        return (None, Ok(Value::Number(r).into()));
                    }
                }
            },
            Value::String(s1) => if let Some(orv) = other {
                if let TVal { val: Value::String(ref s2), .. } = orv.clone_out() {
                    if let Some(r) = BigRational::from_u32((s1 == s2) as u32) {
                        return (None, Ok(Value::Number(r).into()));
                    }
                }
            },
            Value::List(l1) => if let Some(orv) = other {
                if let Value::List(l2) = orv.clone_out().val {
                    let mut b = true;
                    if l1.len() != l2.len() {
                        b = false;
                    }

                    for (e1, e2) in l1.iter().zip(&l2) {
                        let (vars, val) = OpTrait::<{Op::EqualEqual}>::op(e1, env, pos.clone(), Some(e2));
                        match val {
                            Ok(rv) => if let Value::Number(n) = rv.clone_out().val {
                                if n != BigRational::one() {
                                    b = false;
                                    break;
                                }
                            },
                            Err(m) => return (vars, Err(m)),
                        }
                    }

                    if let Some(r) = BigRational::from_u32(b as u32) {
                        return (None, Ok(Value::Number(r).into()));
                    }
                }
            },
            Value::Map(m1) => if let Some(orv) = other {
                if let Value::Map(m2) = orv.clone_out().val {
                    let mut b = true;
                    if m1.len() != m2.len() {
                        b = false;
                    }

                    for (e1, e2) in m1.iter()
                        .sorted()
                        .zip(m2.iter().sorted())
                    {
                        let (vars, val) = OpTrait::<{Op::EqualEqual}>::op(
                            &RefTVal::from(Value::List(vec![e1.0.clone(), e1.1.clone()])),
                            env,
                            pos.clone(),
                            Some(&Value::List(vec![e2.0.clone(), e2.1.clone()]).into())
                        );
                        match val {
                            Ok(rv) => if let Value::Number(n) = rv.clone_out().val {
                                if n != BigRational::one() {
                                    b = false;
                                    break;
                                }
                            },
                            Err(m) => return (vars, Err(m)),
                        }
                    }

                    if let Some(r) = BigRational::from_u32(b as u32) {
                        return (None, Ok(Value::Number(r).into()));
                    }
                }
            },
            Value::Enum(_m1, e1) => if let Some(orv) = other {
                if let TVal { val: Value::Enum(ref _m2, ref e2), .. } = orv.clone_out() {
                    // let b = m1 == m2 && e1 == e2;
                    let b = e1 == e2;
                    if let Some(r) = BigRational::from_u32(b as u32) {
                        return (None, Ok(Value::Number(r).into()));
                    }
                }
            },
            _ => {},
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "==", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::ExclaimEqual}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        let (vars, val) = OpTrait::<{Op::EqualEqual}>::op(self, env, pos.clone(), other);
        match val {
            Ok(rv) => {
                if let Value::Number(n) = rv.clone_out().val {
                    if n == BigRational::one() {
                        (vars, Ok(Value::Number(BigRational::zero()).into()))
                    } else {
                        (vars, Ok(Value::Number(BigRational::one()).into()))
                    }
                } else {
                    (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "!=", self, other), Some(pos))))
                }
            },
            Err(m) => (vars, Err(m)),
        }
    }
}

impl<'a> OpTrait<'a, {Op::AmpersandAmpersand}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::AmpersandAmpersand}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::AmpersandAmpersand}>::op(self, env, pos, Some(&orv));
        }

        if let Value::Number(r1) = &self.clone_out().val {
            if let Some(orv) = other {
                if let TVal { val: Value::Number(r2), .. } = orv.clone_out() {
                    if r1.denom() == &BigInt::one() && r2.denom() == &BigInt::one()
                        && r1.numer() <= &BigInt::one() && r2.numer() <= &BigInt::one()
                        && r1.numer() >= &BigInt::zero() && r2.numer() >= &BigInt::zero()
                    {
                        let v = r1.numer() == &BigInt::one() && r2.numer() == &BigInt::one();
                        return (None, Ok(Value::Number(BigRational::from_usize(v as usize).unwrap()).into()));
                    }
                }
            }
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "&&", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::VbarVbar}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::VbarVbar}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::VbarVbar}>::op(self, env, pos, Some(&orv));
        }

        if let Value::Number(r1) = &self.clone_out().val {
            if let Some(orv) = other {
                if let TVal { val: Value::Number(r2), .. } = orv.clone_out() {
                    if r1.denom() == &BigInt::one() && r2.denom() == &BigInt::one()
                        && r1.numer() <= &BigInt::one() && r2.numer() <= &BigInt::one()
                        && r1.numer() >= &BigInt::zero() && r2.numer() >= &BigInt::zero()
                    {
                        let v = r1.numer() == &BigInt::one() || r2.numer() == &BigInt::one();
                        return (None, Ok(Value::Number(BigRational::from_usize(v as usize).unwrap()).into()));
                    }
                }
            }
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "||", self, other), Some(pos))))
    }
}

impl<'a> OpTrait<'a, {Op::DashRangle}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = self.clone_out().val {
            return OpTrait::<{Op::DashRangle}>::op(&rv, env, pos, other);
        } else if let Some(Value::Reference(orv)) = other.map(|o| o.clone_out().val) {
            return OpTrait::<{Op::DashRangle}>::op(self, env, pos, Some(&orv));
        }

        match self.clone_out().val {
            Value::Type(t1, _st1) => {
                if let Some(orv) = other {
                    if let Value::Type(t2, _st2) = orv.clone_out().val {
                        let args = vec![("".into(), t1)];
                        let t3 = Type::Function {
                            args: Arc::new(args),
                            ret: t2,
                        };
                        return (None, Ok(Value::Type(t3.into(), HashableMap::arc()).into()));
                    }
                }
            },
            Value::List(l) => {
                if let Some(orv) = other {
                    if let Value::Type(t2, _st2) = orv.clone_out().val {
                        let args: Result<Vec<(String, Arc<Type>)>, Error> = l.iter()
                            .map(|v| match v.clone_out().val {
                                Value::Type(t, _st) => Ok(("".into(), t)),
                                _ => Err(Error::Script(format!("operator {}: expected Type, got {}", "->", v), Some(pos.clone()))),
                            })
                            .collect();
                        let args: Vec<(String, Arc<Type>)> = match args {
                            Ok(v) => v,
                            Err(m) => return (None, Err(m)),
                        };
                        let t3 = Type::Function {
                            args: Arc::new(args),
                            ret: t2,
                        };
                        return (None, Ok(Value::Type(t3.into(), HashableMap::arc()).into()));
                    }
                }
            },
            _ => {},
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "->", self, other), Some(pos))))
    }
}

impl<'a> OpTrait<'a, {Op::Equal}> for RefTVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&RefTVal>) -> FnReturn {
        if let Value::Reference(rv) = &self.clone_out().val {
            if let Some(orv) = other {
                let v: &mut TVal = &mut (*rv.try_lock().unwrap());
                *v = orv.clone_out();
                return (None,  Ok(self.clone()));
            }
        } else if let Value::String(varname) = &self.clone_out().val {
            if !env.has(varname) {
                return (None, Err(Error::Script(format!("operator {}: varname {} not found in env, define it with the let macro", "=", varname), Some(pos))));
            }
            if let Some(v) = other {
                return (Some(hashmap!{ varname.clone() => v.clone() }), Ok(v.clone()));
            }
        }
        (None, Err(Error::Script(format!("operator {} not implemented for {} and {:?}", "=", self, other), Some(pos))))
    }
}
