use umbra_lang as umbra;
use umbra::{Args, Error, Env};

#[cfg(feature = "compile")]
use std::{fs, io::Write, path::Path};
#[cfg(feature = "compile")]
use umbra::Runnable;

#[cfg(feature = "compile")]
fn compile(args: Args) -> Result<(), Error> {
    // Load file
    let path = match args.script.get(0) {
        Some(p) => p,
        None => return Err(Error::Argument("Failed to compile: missing filename".into())),
    };
    let script = match fs::read_to_string(path) {
        Ok(s) => s,
        Err(m) => return Err(Error::Script(format!("Failed to load {}: {}", path, m), None)),
    };

    // Compile script
    let env = Env::prelude();
    let runnable: Runnable = umbra::compile(path, &script, &env, None, 0)?;

    // Serialize runnable
    let objdata = bincode::serialize(&runnable)
        .map_err(|e| Error::Script(
            format!("Failed to serialize object file for {}: ", path)
                + &e.to_string(),
            None,
        ))?;

    // Create bin directory
    fs::create_dir_all("bin").map_err(|e| Error::Script(
        "Failed to create bin directory: ".to_string()
            + &e.to_string(),
        None,
    ))?;

    // Create and write file
    let objpath = match Path::new(path).with_extension("obj").file_name() {
        Some(p) => Path::new("bin").join(p.to_owned()),
        None => return Err(Error::Script(format!("Failed to save object file for {}: invalid path", path), None)),
    };
    {
        let mut file = fs::File::create(objpath.clone())
            .map_err(|e| Error::Script(
                format!("Failed to create object file {} for {}: ", objpath.to_string_lossy(), path)
                    + &e.to_string(),
                None,
            ))?;
        file.write_all(&objdata)
            .map_err(|e| Error::Script(
                format!("Failed to write object file {} for {}: ", objpath.to_string_lossy(), path)
                    + &e.to_string(),
                None,
            ))?;
    }

    Ok(())
}
#[cfg(feature = "compile")]
fn exec(args: Args) -> Result<(), Error> {
    // Load file
    let path = match args.script.get(0) {
        Some(p) => p,
        None => return Err(Error::Argument("Failed to exec: missing filename".into())),
    };
    let binscript = match fs::read(path) {
        Ok(b) => b,
        Err(m) => return Err(Error::Script(format!("Failed to load {}: {}", path, m), None)),
    };

    // Deserialize runnable
    let runnable: Runnable = bincode::deserialize(&binscript).map_err(|e| Error::Script(
        format!("Failed to deserialize object file for {}: ", path)
            + &e.to_string(),
        None,
    ))?;

    // println!("{:#?}", runnable.ast);

    // Run script
    let (_, val) = runnable.ast.run(&runnable.env);
    val?;

    Ok(())
}

fn help(_args: Args) {
    eprintln!("{} v{} by {}", env!("CARGO_PKG_NAME"), env!("CARGO_PKG_VERSION"), env!("CARGO_PKG_AUTHORS"));
    eprintln!();
    eprintln!("USAGE:");
    eprintln!("    umbra [OPTIONS] [SCRIPT] [ARGS]");
    eprintln!();
    eprintln!("OPTIONS:");
    eprintln!("    --help\t\tShow this help text");
    eprintln!("    --version\t\tShow the version info");
    eprintln!("    --inspect\t\tStart an interpreter when the script ends");
}

fn main() -> Result<(), Error> {
    let args = match Args::handle() {
        Ok(args) => args,
        Err(m) => return Err(m),
    };

    // Handle flags
    let mut inspect = false;
    for f in &args.longflags {
        match f.as_ref() {
            #[cfg(feature = "compile")]
            "compile" => return compile(args),
            #[cfg(feature = "compile")]
            "exec" => return exec(args),

            "inspect" => inspect = true,
            "version" => {
                eprintln!("{} v{}", env!("CARGO_PKG_NAME"), env!("CARGO_PKG_VERSION"));
                return Ok(());
            },
            "help" => {
                help(args);
                return Ok(());
            },
            _ => return Err(Error::Argument(format!("Unsupported flag: {}", f))),
        }
    }

    // If no script, start the interactive interpreter
    let env = Env::prelude();
    if args.script.is_empty() {
        umbra::run_interactive("um", &env);
        return Ok(())
    }

    // Otherwise run the script
    let f = args.script[0].clone();
    let (vars, val) = umbra::run_path(&f, &env, true);
    if let Err(m) = &val {
        eprintln!("{}", m);
    }

    if inspect {
        let mut nenv = Env::child(&env);
        if let Some(vars) = vars {
            nenv.update(vars);
        }

        eprintln!("inspecting {} post-execution:", f);
        umbra::run_interactive("post", &nenv);
    }

    Ok(())
}
