use std::sync::Arc;

use num::{BigRational, BigInt, One, Signed, FromPrimitive};

use maplit::hashmap;

use crate::{Error, Type, Value, TVal, Callable, FnArgs, FnReturn};

use super::_parse_fargs;

pub fn type_init() -> TVal {
    Value::Type(Type::ttype(), Arc::new(hashmap!{}.into())).into()
}
pub fn number_init() -> TVal {
    Value::Type(Type::number(), Arc::new(hashmap!{
        "abs".into() => Value::Function {
            args: Arc::new(vec![
                ("self".into(), Type::number()),
            ]),
            vars: hashmap!{}.into(),
            body: Callable::Native(number_abs),
        }.into(),
        "sqrt".into() => Value::Function {
            args: Arc::new(vec![
                ("self".into(), Type::number()),
            ]),
            vars: hashmap!{}.into(),
            body: Callable::Native(number_sqrt),
        }.into(),
    }.into())).into()
}
pub fn number_abs(args: FnArgs) -> FnReturn {
    let (pos, args) = match _parse_fargs("function Number.abs", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    use Value::*;
    if let List(ref args) = args.val {
        if let Some(TVal { val: Number(x), .. }) = args.get(1) {
            return (None, Ok(Value::Number(x.abs()).into()));
        }
    }

    (None, Err(Error::ScriptError(format!("function Number.abs: expected args [Number], got {}", args.val), pos)))
}
pub fn number_sqrt(args: FnArgs) -> FnReturn {
    let (pos, args) = match _parse_fargs("function Number.sqrt", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    use Value::*;
    if let List(ref args) = args.val {
        if let Some(TVal { val: Number(x), .. }) = args.get(1) {
            return (None, Ok(Value::Number(BigRational::new(
                x.numer().sqrt(),
                x.denom().sqrt(),
            )).into()));
        }
    }

    (None, Err(Error::ScriptError(format!("function Number.sqrt: expected args [Number], got {}", args.val), pos)))
}
pub fn string_init() -> TVal {
    Value::Type(Type::string(), Arc::new(hashmap!{}.into())).into()
}

pub fn list_init() -> TVal {
    Value::Type(Type::list(), Arc::new(hashmap!{
        "len".into() => Value::Function {
            args: Arc::new(vec![
                ("self".into(), Type::list()),
            ]),
            vars: hashmap!{}.into(),
            body: Callable::Native(list_len),
        }.into(),
    }.into())).into()
}
pub fn list_len(args: FnArgs) -> FnReturn {
    let (pos, args) = match _parse_fargs("function List.len", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    use Value::*;
    if let List(ref args) = args.val {
        if let Some(TVal { val: List(l), .. }) = args.get(1) {
            return (None, Ok(Value::Number(BigRational::new(
                BigInt::from_usize(l.len()).unwrap(),
                BigInt::one(),
            )).into()));
        }
    }

    (None, Err(Error::ScriptError(format!("function List.len: expected args [List], got {}", args.val), pos)))
}
pub fn map_init() -> TVal {
    Value::Type(Type::map(), Arc::new(hashmap!{
        "set".into() => Value::Function {
            args: Arc::new(vec![
                ("self".into(), Type::map()),
                ("key".into(), Type::any()),
                ("value".into(), Type::any()),
            ]),
            vars: hashmap!{}.into(),
            body: Callable::Native(map_set),
        }.into(),
        "get".into() => Value::Function {
            args: Arc::new(vec![
                ("self".into(), Type::map()),
                ("key".into(), Type::any()),
            ]),
            vars: hashmap!{}.into(),
            body: Callable::Native(map_get),
        }.into(),
    }.into())).into()
}
pub fn map_set(args: FnArgs) -> FnReturn {
    let (pos, mut args) = match _parse_fargs("function List.len", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    use Value::*;
    if let List(ref mut args) = &mut args.val {
        if let Some(key) = args.get(2) {
            let key = key.clone();
            if let Some(val) = args.get(3) {
                let val = val.clone();
                if let Some(TVal { val: Map(hm), .. }) = args.get_mut(1) {
                    match hm.insert(key, val) {
                        Some(old) => return (None, Ok(old)),
                        None => return (None, Ok(Value::none().into())),
                    }
                }
            }
        }
    }

    (None, Err(Error::ScriptError(format!("function List.len: expected args [Map, Any, Any], got {}", args.val), pos)))
}
pub fn map_get(args: FnArgs) -> FnReturn {
    let (pos, args) = match _parse_fargs("function List.len", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    use Value::*;
    if let List(ref args) = args.val {
        if let Some(TVal { val: Map(hm), .. }) = args.get(1) {
            if let Some(key) = args.get(2) {
                match hm.get(key) {
                    Some(v) => return (None, Ok(v.clone())),
                    None => return (None, Ok(Value::none().into())),
                }
            }
        }
    }

    (None, Err(Error::ScriptError(format!("function List.len: expected args [Map, Any], got {}", args.val), pos)))
}
pub fn enum_init() -> TVal {
    Value::Type(Type::tenum(), Arc::new(hashmap!{}.into())).into()
}
pub fn struct_init() -> TVal {
    Value::Type(Type::tstruct(), Arc::new(hashmap!{}.into())).into()
}

pub fn function_init() -> TVal {
    Value::Type(Type::function(), Arc::new(hashmap!{
        "bind".into() => Value::Function {
            args: Arc::new(vec![
                ("self".into(), Arc::new(Type::Function {
                    args: Arc::new(vec![]),
                    ret: Type::function(),
                })),
                ("val".into(), Type::number()),
            ]),
            vars: hashmap!{}.into(),
            body: Callable::Native(function_bind),
        }.into(),
    }.into())).into()
}
pub fn function_bind(fnargs: FnArgs) -> FnReturn {
    let (pos, args) = match _parse_fargs("function Function.bind", fnargs) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    use Value::*;
    if let List(l) = &args.val {
        if let Some(TVal { val: Function { args: fargs, vars, body }, .. }) = l.get(0) {
            let mut nvars = vars.clone();
            if let Some(a) = fargs.iter().take(1).next() {
                // TODO type checking
                nvars.insert(a.0.clone(), l[1].clone());
            }
            return (None, Ok(Function {
                args: Arc::clone(fargs),
                vars: nvars,
                body: body.clone(),
            }.into()));
        }
    }

    (None, Err(Error::ScriptError(format!("function Function.bind: expected args [Function, Any], got {}", args.val), pos)))
}
