use std::collections::HashMap;
use std::sync::Arc;

use lazy_static::lazy_static;

use maplit::hashmap;
use num::{BigRational, FromPrimitive};

use crate::{Error, Env, Type, Value, TVal, Pos, Token, FnArgs, FnReturn, Callable};

// Macros
mod base;
pub use base::*;
mod control;
pub use control::*;
mod construct;
pub use construct::*;

// Builtins
mod builtins;
pub use builtins::*;
mod types;
pub use types::*;
mod composite;
pub use composite::*;

pub fn _init(vars: &mut HashMap<String, TVal>) {
    vars.extend(hashmap!{
        ////////////
        // Macros //
        ////////////
        // Base
        "use".into() => _macro_init(r#use),
        "let".into() => _macro_init(r#let),
        "print".into() => _macro_init(r#print),

        // Control
        "if".into() => _macro_init(r#if),
        "match".into() => _macro_init(r#match),
        "loop".into() => _macro_init(r#loop),
        "break".into() => _macro_init(r#break),
        "continue".into() => _macro_init(r#continue),
        "return".into() => _macro_init(r#return),

        // Construct
        "generic".into() => _macro_init(r#generic),
        "fn".into() => _macro_init(r#fn),
        "macro".into() => _macro_init(r#macro),
        "struct".into() => _macro_init(r#struct),
        "enum".into() => _macro_init(r#enum),

        //////////////
        // Builtins //
        //////////////
        // Values
        "true".into() => Value::Number(BigRational::from_u32(1u32).unwrap()).into(),
        "false".into() => Value::Number(BigRational::from_u32(0u32).unwrap()).into(),

        // Functions
        "eval".into() => eval_init(),
        "exit".into() => exit_init(),

        // Base Types
        "Type".into() => type_init(),
        "Number".into() => number_init(),
        "String".into() => string_init(),
        "List".into() => list_init(),
        "Map".into() => map_init(),
        "Enum".into() => enum_init(),
        "Struct".into() => struct_init(),
        "Function".into() => function_init(),

        // Composite Types
        "Env".into() => env_init(),
        "AST".into() => ast_init(),
    });
}

lazy_static! {
    static ref MACRO_ARGS: Arc<Vec<(String, Arc<Type>)>> = Arc::new(vec![
        ("env".into(), Type::tstruct()),
        ("pos".into(), Type::tstruct()),
        ("tokens".into(), Type::list()),
    ]);
}

// Util functions
fn _macro_init(func: fn(FnArgs) -> FnReturn) -> TVal {
    Value::Function {
        args: _macro_args(),
        vars: hashmap!{}.into(),
        body: Callable::Native(func),
    }.into()
}
pub fn _macro_args() -> Arc<Vec<(String, Arc<Type>)>> {
    Arc::clone(&MACRO_ARGS)
}
pub fn _parse_margs<'a>(name: &str, args: FnArgs) -> Result<(Env<'a>, Option<Pos>, Vec<Token>), Error> {
    let (vars, pos, tokens) = match args {
        FnArgs::Macro { vars, pos, tokens } => (vars, pos, tokens),
        _ => return Err(Error::ScriptError(format!("{}: {}", name, "invalid FnArgs"), None)),
    };
    Ok((Env::from(vars), pos, tokens))
}
pub fn _parse_fargs(name: &str, args: FnArgs) -> Result<(Option<Pos>, TVal), Error> {
    let (pos, args) = match args {
        FnArgs::Normal { pos, args } => (pos, args),
        _ => return Err(Error::ScriptError(format!("{}: {}", name, "invalid FnArgs"), None)),
    };
    Ok((pos, args))
}
