use num::{BigRational, One, Zero};

use crate::{AST, Env, Error, FnArgs, FnReturn, TVal, TokenType, Value, match_pat};

use super::_parse_margs;

// Macros: if, match, loop, break, continue, return

pub fn r#if(args: FnArgs) -> FnReturn {
    let (env, pos, tokens) = match _parse_margs("macro if", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    let mut i = 0;
    loop {
        match &tokens.get(i) {
            Some(t) => match t.ttype {
                TokenType::Container if tokens[i].data.starts_with('{') => break,
                _ => {
                    i += 1;
                    continue;
                },
            },
            None => return (None, Err(Error::ScriptError("macro if: missing body".into(), pos))),
        }
    }
    let cond = &tokens[1..i];
    let body = &tokens[i];

    let cond_ast = match AST::parse(cond.iter().cloned(), &env) {
        Ok(a) => a,
        Err(m) => return (None, Err(m)),
    };
    let (_, cond_val) = cond_ast.run(&env);

    if let Ok(TVal { val: Value::Number(n), .. }) = cond_val {
        if n == BigRational::one() {
            // Cond succeeds, run body
            let body_ast = match AST::parse(vec![body.clone()], &env) {
                Ok(a) => a,
                Err(m) => return (None, Err(m)),
            };
            let (_, body_val) = body_ast.run(&env);
            return (Some(env.vars), body_val);
        } else if n == BigRational::zero() {
            // Cond fails, run else blocks
            if let Some(el) = tokens.get(i+1) {
                if el.data == "else" {
                    if let Some(ifblock) = tokens.get(i+2) {
                        if ifblock.data == "if" {
                            // Else if
                            return r#if(FnArgs::Macro {
                                vars: env.vars.into(),
                                pos: Some(ifblock.pos.clone()),
                                tokens: tokens[i+2..].to_vec(),
                            });
                        } else {
                            // Lone else
                            let else_ast = match AST::parse(vec![ifblock.clone()], &env) {
                                Ok(a) => a,
                                Err(m) => return (None, Err(m)),
                            };
                            let (_, else_val) = else_ast.run(&env);
                            return (Some(env.vars), else_val);
                        }
                    } else {
                        return (None, Err(Error::ScriptError("macro if: invalid else statement".into(), pos)));
                    }
                }
            }

            return (None, Ok(Value::none().into()));
        } else {
            // Non-boolean cond
            return (None, Err(Error::ScriptError("macro if: non-boolean cond".into(), pos)));
        }
    }

    let condstr = cond.iter()
        .map(|t| t.data.clone())
        .collect::<Vec<String>>()
        .join(" ");
    (None, Err(Error::ScriptError(format!("macro if: failed to evaluate condition {}: {:?}", condstr, cond_val), pos)))
}
pub fn r#match(args: FnArgs) -> FnReturn {
    let (env, pos, tokens) = match _parse_margs("macro match", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    let mitem = &tokens[1..tokens.len()-1];
    let body = &tokens[tokens.len()-1];

    let mitem_ast = match AST::parse(mitem.iter().cloned(), &env) {
        Ok(a) => a,
        Err(m) => return (None, Err(m)),
    };
    let mitem_val = match mitem_ast.run(&env) {
        (_, Ok(v)) => v,
        (_, Err(m)) => return (None, Err(m)),
    };

    let body_ast = match AST::parse(vec![body.clone()], &env) {
        Ok(a) => a,
        Err(m) => return (None, Err(m)),
    };

    match body_ast {
        AST::Container { children, .. } => {
            for c in children {
                match c {
                    AST::Operator { token, lhs, rhs } if token.data == ":" => {
                        if let Ok(vs) = match_pat(&mitem_val, &*lhs, pos.clone()) {
                            let mut nenv = Env::child(&env);
                            if let Some(vs) = vs {
                                nenv.update(vs);
                            }
                            return rhs.run(&nenv);
                        }
                    },
                    AST::Container { token, .. } if token.data.starts_with("//") => {},
                    _ => return (None, Err(Error::ScriptError(format!("macro match: invalid row {}", c.to_string_lossy()), pos))),
                }
            }
        },
        _ => return (None, Err(Error::ScriptError(format!("macro match: invalid block {}", body), pos))),
    }

    (None, Err(Error::ScriptError(format!("macro match: failed to match {}", mitem_val), pos)))
}

pub fn r#loop(args: FnArgs) -> FnReturn {
    let (env, _pos, tokens) = match _parse_margs("macro loop", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    let body = tokens[1].clone();
    let ast = match AST::parse(vec![body], &env) {
        Ok(a) => a,
        Err(m) => return (None, Err(m)),
    };
    let mut nenv = Env::child(&env);
    loop {
        let (vars, val) = ast.run(&nenv);
        match val {
            Ok(_) => if let Some(vars) = vars {
                nenv.update(vars);
            },
            Err(Error::ControlError(s, v)) if s == "break" => match v {
                Some(v) => return (None, Ok(v)),
                None => break,
            },
            Err(Error::ControlError(s, _)) if s == "continue" => {
                if let Some(vars) = vars {
                    nenv.update(vars);
                }
                continue;
            },
            Err(m) => return (None, Err(m)),
        }
    }

    (None, Ok(Value::none().into()))
}
pub fn r#break(args: FnArgs) -> FnReturn {
    let (env, _pos, tokens) = match _parse_margs("macro break", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    if tokens.len() == 1 {
        return (None, Err(Error::ControlError("break".into(), Some(Value::none().into()))));
    }

    let ast = match AST::parse(tokens[1..].iter().cloned(), &env) {
        Ok(a) => a,
        Err(m) => return (None, Err(m)),
    };
    let (_, val) = ast.run(&env);

    match val {
        Ok(v) => (None, Err(Error::ControlError("break".into(), Some(v)))),
        Err(m) => (None, Err(m)),
    }
}
pub fn r#continue(args: FnArgs) -> FnReturn {
    let (env, pos, tokens) = match _parse_margs("macro continue", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    if tokens.len() == 1 {
        return (Some(env.vars), Err(Error::ControlError("continue".into(), Some(Value::none().into()))));
    }

    (None, Err(Error::ScriptError(format!("macro continue: extra tokens {:?}", &tokens[1..]), pos)))
}
pub fn r#return(args: FnArgs) -> FnReturn {
    let (env, _pos, tokens) = match _parse_margs("macro return", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    if tokens.len() == 1 {
        return (None, Err(Error::ControlError("return".into(), Some(Value::none().into()))));
    }

    let ast = match AST::parse(tokens[1..].iter().cloned(), &env) {
        Ok(a) => a,
        Err(m) => return (None, Err(m)),
    };
    let (_, val) = ast.run(&env);

    match val {
        Ok(v) => (None, Err(Error::ControlError("return".into(), Some(v)))),
        Err(m) => (None, Err(m)),
    }
}
