use std::collections::HashMap;
use std::sync::Arc;
use std::convert::TryFrom;

use maplit::hashmap;

use crate::{Error, Env, Token, TokenType, Type, Value, TVal, Callable, FnArgs, FnReturn, AST, HashableMap};

use super::_parse_margs;

pub fn r#generic(args: FnArgs) -> FnReturn {
    let (env, pos, tokens) = match _parse_margs("macro generic", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    let params_tok = tokens[1].clone();
    let params_ast = match AST::parse(vec![params_tok], &env) {
        Ok(a) => a,
        Err(m) => return (None, Err(m)),
    };

    let val_toks = tokens[2..].iter().cloned();
    let val_ast = match AST::parse(val_toks, &env) {
        Ok(a) => a,
        Err(m) => return (None, Err(m)),
    };

    let mut nenv = Env::child(&env);
    let mut hm: HashMap<String, Arc<Type>> = hashmap!{};
    if let AST::Container { children, .. } = params_ast {
        for c in children {
            if let AST::Value { token: vt } = c {
                hm.insert(vt.data.clone(), Type::any());
                nenv.set(&vt.data, Value::none().into());
            } else {
                return (None, Err(Error::ScriptError(format!("macro generic: invalid type parameter {}", c.to_string_lossy()), pos)))
            }
        }
    } else {
        return (None, Err(Error::ScriptError(format!("macro generic: invalid type params {}", params_ast.to_string_lossy()), pos)))
    }

    let (venv, val) = match val_ast.run(&nenv) {
        (e, Ok(v)) => (e, v),
        (e, Err(m)) => return (e, Err(m)),
    };

    let vt = Type::Generic(Arc::clone(&val.ttype), Arc::new(hm.into()));

    (venv, Ok(TVal {
        ttype: Arc::new(vt),
        val: val.val,
    }))
}

pub fn r#fn(args: FnArgs) -> FnReturn {
    let (env, _pos, tokens) = match _parse_margs("macro fn", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    let fargs = tokens[1].clone();
    let _ret;
    let body;
    if tokens.len() == 3 {
        _ret = Token {
            ttype: TokenType::Comment,
            pos: fargs.pos.clone(),
            data: "".into(),
        };
        body = tokens[2].clone();
    } else {
        _ret = tokens[3].clone();
        body = tokens[4].clone();
    }

    let fargs_val: Result<Vec<(String, Arc<Type>)>, Error> = fargs.data[1..fargs.data.len()-1].split(',')
        .filter_map(|vt| {
            if vt.is_empty() {
                None
            } else {
                let mut vts = vt.split(':');
                match (vts.next(), vts.next()) {
                    (Some(vn), Some(t)) => Some(Ok((
                        // TODO check env
                        vn.trim().into(),
                        Arc::new(Type::try_from(t.trim()).ok()?)
                    ))),
                    _ => Some(Err(Error::ScriptError(format!("macro fn: parameter missing type {}", vt), Some(fargs.pos.clone())))),
                }
            }
        }).collect();
    let fargs_val = match fargs_val {
        Ok(v) => v,
        Err(m) => return (None, Err(m)),
    };

    let body_ast = match AST::parse(vec![body], &env) {
        Ok(a) => a,
        Err(m) => return (None, Err(m)),
    };

    (None, Ok(Value::Function {
        args: Arc::new(fargs_val),
        vars: Env::flatten(&env).vars.into(),
        body: Callable::AST(body_ast),
    }.into()))
}
pub fn r#macro(args: FnArgs) -> FnReturn {
    let (env, _pos, tokens) = match _parse_margs("macro macro", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    let fargs = tokens[1].clone();
    let _ret;
    let body;
    if tokens.len() == 3 {
        _ret = Token {
            ttype: TokenType::Comment,
            pos: fargs.pos.clone(),
            data: "".into(),
        };
        body = tokens[2].clone();
    } else {
        _ret = tokens[3].clone();
        body = tokens[4].clone();
    }

    let fargs_val: Result<Vec<(String, Arc<Type>)>, Error> = fargs.data[1..fargs.data.len()-1].split(',')
        .filter_map(|vt| -> Option<Result<(String, Arc<Type>), Error>> {
            if vt.is_empty() {
                None
            } else {
                let mut vts = vt.split(':');
                match (vts.next(), vts.next()) {
                    (Some(vn), Some(t)) => Some(Ok((
                        vn.trim().into(),
                        // TODO fix type determination
                        match env.get(t.trim()) {
                            Some(tv) => Arc::clone(&tv.ttype),
                            None => {
                                if t.trim().starts_with('[') && t.trim().ends_with(']') {
                                    match env.get(&t.trim()[1..t.trim().len()-1]) {
                                        Some(tv) => Arc::clone(&tv.ttype),
                                        None => return Some(Err(Error::ScriptError(format!("macro macro: parameter type {} not found", t), Some(fargs.pos.clone())))),
                                    }
                                } else {
                                    return Some(Err(Error::ScriptError(format!("macro macro: parameter type {} not found", t), Some(fargs.pos.clone()))))
                                }
                            },
                        },
                    ))),
                    _ => Some(Err(Error::ScriptError(format!("macro macro: parameter missing type {}", vt), Some(fargs.pos.clone())))),
                }
            }
        }).collect();
    let fargs_val = match fargs_val {
        Ok(v) => v,
        Err(m) => return (None, Err(m)),
    };

    let body_ast = match AST::parse(vec![body], &env) {
        Ok(a) => a,
        Err(m) => return (None, Err(m)),
    };

    (None, Ok(Value::Function {
        args: Arc::new(fargs_val),
        vars: Env::flatten(&env).vars.into(),
        body: Callable::AST(body_ast),
    }.into()))
}

pub fn r#struct(args: FnArgs) -> FnReturn {
    let (env, pos, tokens) = match _parse_margs("macro struct", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    let ast = match AST::parse(tokens[1..].iter().cloned(), &env) {
        Ok(a) => a,
        Err(m) => return (None, Err(m)),
    };
    let mut map = HashableMap::new();

    // Build map of struct items
    match ast {
        AST::Container { token, children } if token.data == "[" => {
            for c in children {
                match c {
                    AST::Operator { token, lhs, rhs } if token.data == ":" => {
                        if let AST::Value { token: ltok } = &*lhs {
                            if let AST::Value { token: rtok } = &*rhs {
                                if let Some(ttype) = env.get(&rtok.data) {
                                    map.insert(ltok.data.clone(), ttype.clone());
                                    continue;
                                }
                            }
                        }
                        return (None, Err(Error::ScriptError(format!("macro struct: invalid item {} : {}", lhs.to_string_lossy(), rhs.to_string_lossy()), pos)));
                    },
                    _ => return (None, Err(Error::ScriptError(format!("macro struct: invalid line {}", c.to_string_lossy()), c.get_first_pos().or(pos)))),
                }
            }
        },
        _ => return (None, Err(Error::ScriptError(format!("macro struct: invalid container {:?}", &tokens[1..]), pos))),
    }

    (None, Ok(Value::Type(Type::tstruct(), Arc::new(map)).into()))
}
pub fn r#enum(args: FnArgs) -> FnReturn {
    let (env, pos, tokens) = match _parse_margs("macro enum", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    let ast = match AST::parse(tokens[1..].iter().cloned(), &env) {
        Ok(a) => a,
        Err(m) => return (None, Err(m)),
    };
    let mut map: HashableMap<String, TVal> = HashableMap::new();
    let empty = Arc::new(HashableMap::new());

    // Build map of enum variants
    match ast {
        AST::Container { token, children } if token.data == "[" => {
            for c in children {
                match &c {
                    AST::Value { token: ctok } => {
                        map.insert(ctok.data.clone(), Value::Enum(
                            Arc::clone(&empty),
                            Box::new((ctok.data.clone(), None)
                        )).into());
                        continue;
                    },
                    AST::Container { token, .. } if token.data.starts_with("//") => {
                        continue;
                    },
                    AST::Operator { token, lhs, rhs } if token.data == "(" => {
                        if let AST::Value { token: Token { data, .. }} = &**lhs {
                            if let AST::Container { children, .. } = &**rhs {
                                let tv = TVal {
                                    val: Value::Enum(
                                        Arc::clone(&empty),
                                        Box::new((data.clone(), None)),
                                    ),
                                    ttype: {
                                        let mut hm: HashableMap<String, Arc<Type>> = HashableMap::new();
                                        for c in children {
                                            if let AST::Value { token } = c {
                                                hm.insert(token.data.clone(), Type::any());
                                            } else {
                                                return (None, Err(Error::ScriptError(format!("macro enum: invalid child {}", c.to_string_lossy()), pos)));
                                            }
                                        }
                                        Arc::new(Type::Enum(Arc::new(hm)))
                                    },
                                };
                                map.insert(data.clone(), tv);
                                continue;
                            }
                        }
                    },
                    _ => {},
                }
                return (None, Err(Error::ScriptError(format!("macro enum: invalid item {}", c.to_string_lossy()), pos)));
            }
        },
        _ => return (None, Err(Error::ScriptError(format!("macro enum: invalid container {:?}", &tokens[1..]), pos))),
    }

    // Copy enum map to each variant
    let enums: HashableMap<String, Arc<Type>> = HashableMap::from(
        map.iter()
            .map(|(k, v)| {
                if let TVal { val: Value::Enum(_, b), .. } = v {
                    return (k.clone(), match &b.1 {
                        Some(v) => Arc::clone(&v.ttype),
                        None => Type::none(),
                    });
                }
                unreachable!();
            }).collect::<HashMap<String, Arc<Type>>>()
    );
    let enums = Arc::new(enums);
    for mut v in map.map.values_mut() {
        if let TVal { val: Value::Enum(em, _), .. } = &mut v {
            *em = Arc::clone(&enums);
        }
    }

    (None, Ok(Value::Type(Type::tenum(), Arc::new(map)).into()))
}
