use std::collections::HashMap;
use std::sync::Arc;

use maplit::hashmap;

use crate::{HashableMap, Error, Env, Type, Value, TVal, Token, FnArgs, FnReturn, Callable, AST};

use super::_parse_fargs;

pub fn env_init() -> TVal {
    Value::Type(Type::tstruct(), Arc::new(hashmap!{
        "child".into() => Value::Function {
            args: Arc::new(vec![
                ("parent".into(), Type::tstruct()),
            ]),
            vars: hashmap!{}.into(),
            body: Callable::Native(env_child),
        }.into(),
    }.into())).into()
}
pub fn env_child(fnargs: FnArgs) -> FnReturn {
    let (pos, args) = match _parse_fargs("function Env.child", fnargs) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    use Value::*;
    if let List(l) = &args.val {
        if let Some(TVal { val: Struct(penv), .. }) = l.get(1) {
            let pvars = match penv.get("vars") {
                Some(vs) => match vs {
                    TVal { val: Map(vs), .. } => vs,
                    _ => return (None, Err(Error::ScriptError(format!("function Env.child: env.vars is not a map {}", vs), pos))),
                },
                None => return (None, Err(Error::ScriptError(format!("function Env.child: env doesn't contain vars {}", penv), pos))),
            };
            let vars: Result<HashMap<std::string::String, TVal>, Error> = pvars.iter()
                .map(|(k, v)| Ok((match k {
                    TVal { val: String(s), .. } => s.clone(),
                    _ => return Err(Error::ScriptError(format!("function Env.child: expected Env[String, Any], got {}", penv), pos.clone())),
                }, v.clone())))
                .collect();
            let vars = match vars {
                Ok(vs) => vs,
                Err(m) => return (None, Err(m)),
            };
            let tenv = Env::from(vars);
            let mut ntenv = Env::child(&tenv);

            // Optionally set vars
            if let Some(TVal { val: Map(hm), .. }) = l.get(2) {
                for (k, v) in hm.iter() {
                    if let TVal { val: String(k), ..} = k {
                        ntenv.set(k, v.clone());
                    }
                }
            }

            return (None, Ok(Value::from_env(&ntenv).into()));
        }
    }

    (None, Err(Error::ScriptError(format!("function Env.child: expected args [Env], got {}", args.val), pos)))
}
pub fn ast_init() -> TVal {
    Value::Type(Type::tenum(), Arc::new(hashmap!{
        "parse".into() => Value::Function {
            args: Arc::new(vec![
                ("tokens".into(), Type::list()),
                ("env".into(), Type::tstruct()),
            ]),
            vars: hashmap!{}.into(),
            body: Callable::Native(ast_parse),
        }.into(),
        "run".into() => Value::Function {
            args: Arc::new(vec![
                ("ast".into(), Type::tenum()),
                ("env".into(), Type::tstruct()),
            ]),
            vars: hashmap!{}.into(),
            body: Callable::Native(ast_run),
        }.into(),
    }.into())).into()
}
pub fn ast_parse(fnargs: FnArgs) -> FnReturn {
    let (pos, args) = match _parse_fargs("function AST.parse", fnargs) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    use Value::*;
    if let List(l) = &args.val {
        if let Some(TVal { val: List(tokens), .. }) = l.get(1) {
            if let Some(TVal { val: Struct(env), .. }) = l.get(2) {
                let tokens: Result<Vec<Token>, Error> = tokens.iter()
                    .map(|t| Token::from_value(&t.val).ok_or_else(|| Error::ScriptError(format!("function AST.parse: expected token {}", t), pos.clone())))
                    .collect();
                let tokens = match tokens {
                    Ok(ts) => ts,
                    Err(m) => return (None, Err(m)),
                };

                let vars = match env.get("vars") {
                    Some(vs) => match vs {
                        TVal { val: Map(vs), .. } => vs,
                        _ => return (None, Err(Error::ScriptError(format!("function AST.parse: env.vars is not a map {}", vs), pos))),
                    },
                    None => return (None, Err(Error::ScriptError(format!("function AST.parse: env doesn't contain vars {}", env), pos))),
                };
                let tenv: Result<HashableMap<std::string::String, TVal>, Error> = vars.iter()
                    .map(|(k, v)| Ok((match &k.val {
                        String(s) => s.clone(),
                        _ => return Err(Error::ScriptError(format!("function AST.parse: env contains non-string key {}", k), pos.clone())),
                    }, v.clone()))).collect();
                let tenv = match tenv {
                    Ok(e) => e,
                    Err(m) => return (None, Err(m)),
                };

                let ast = match AST::parse(tokens, &Env::from(tenv)) {
                    Ok(a) => a,
                    Err(m) => return (None, Err(m)),
                };
                return (None, Ok(Value::from_ast(&ast).into()));
            }
        }
    }

    (None, Err(Error::ScriptError(format!("function AST.parse: expected args [List, Struct], got {}", args.val), pos)))
}
pub fn ast_run(fnargs: FnArgs) -> FnReturn {
    let (pos, args) = match _parse_fargs("function AST.run", fnargs) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    use Value::*;
    if let List(l) = &args.val {
        if let Some(TVal { val: ast, .. }) = l.get(1) {
            if let Some(TVal { val: Struct(env), .. }) = l.get(2) {
                let ast = match AST::from_value(&ast) {
                    Some(a) => a,
                    None => return (None, Err(Error::ScriptError(format!("function AST.run: expected AST {}", ast), pos))),
                };

                let vars: &HashableMap<TVal, TVal> = match env.get("vars") {
                    Some(vs) => match vs {
                        TVal { val: Map(vs), .. } => vs,
                        _ => return (None, Err(Error::ScriptError(format!("function AST.run: env.vars is not a map {}", vs), pos))),
                    },
                    None => return (None, Err(Error::ScriptError(format!("function AST.run: env doesn't contain vars {}", env), pos))),
                };
                let pvars: HashableMap<TVal, TVal> = match env.get("parent") {
                    Some(vs) => match vs {
                        TVal { val: Map(pvs), .. } => pvs.clone(),
                        _ => return (None, Err(Error::ScriptError(format!("function AST.run: env.parent is not a struct {}", vs), pos))),
                    },
                    None => hashmap!{}.into(),
                };
                let tenv: Result<HashableMap<std::string::String, TVal>, Error> = vars.iter()
                    .chain(pvars.iter())
                    .map(|(k, v)| Ok((match &k.val {
                        String(s) => s.clone(),
                        _ => return Err(Error::ScriptError(format!("function AST.run: env contains non-string key {}", k), pos.clone())),
                    }, v.clone()))).collect();
                let tenv = match tenv {
                    Ok(e) => e,
                    Err(m) => return (None, Err(m)),
                };

                let (vars, val) = ast.run(&Env::from(tenv));
                return (vars, match &val {
                    Ok(v) => Ok(Value::Enum(
                        Arc::new(hashmap!{
                            "Ok".into() => crate::Type::any(),
                            "Err".into() => crate::Type::any(),
                        }.into()),
                        Box::new(("Ok".into(), Some(v.clone())))
                    ).into()),
                    Err(m) => Ok(Value::Enum(
                        Arc::new(hashmap!{
                            "Ok".into() => crate::Type::any(),
                            "Err".into() => crate::Type::any(),
                        }.into()),
                        Box::new(("Err".into(), Some(Value::Enum(
                            Arc::new(hashmap!{
                                "ArgumentError".into() => crate::Type::string(),
                                "ParseError".into() => crate::Type::list(),
                                "ScriptError".into() => crate::Type::list(),
                                "ControlError".into() => crate::Type::list(),
                                "CustomError".into() => crate::Type::any(),
                            }.into()),
                            Box::new(match m {
                                Error::ArgumentError(m) => ("ArgumentError".into(), Some(Value::String(m.clone()).into())),
                                Error::ParseError(m, p) => ("ParseError".into(), Some(Value::List(vec![
                                    Value::String(m.clone()).into(),
                                    match p {
                                        Some(p) => Value::from_pos(p).into(),
                                        None => Value::none().into(),
                                    }
                                ]).into())),
                                Error::ScriptError(m, p) => ("ScriptError".into(), Some(Value::List(vec![
                                    Value::String(m.clone()).into(),
                                    match p {
                                        Some(p) => Value::from_pos(p).into(),
                                        None => Value::none().into(),
                                    }
                                ]).into())),
                                Error::ControlError(m, v) => ("ControlError".into(), Some(Value::List(vec![
                                    Value::String(m.clone()).into(),
                                    match v {
                                        Some(v) => v.clone(),
                                        None => Value::none().into(),
                                    }
                                ]).into())),
                                Error::CustomError(v) => ("CustomError".into(), Some(v.clone())),
                            }),
                        ).into())),
                    )).into()),
                });
            }
        }
    }

    (None, Err(Error::ScriptError(format!("function AST.run: expected args [Enum, Struct], got {}", args.val), pos)))
}
