use std::path::Path;

use maplit::hashmap;

use crate::{Error, Env, Value, TVal, TokenType, FnArgs, FnReturn, AST, run_path, unescape};

use super::_parse_margs;

// Macros: use, let, print

pub fn r#use(args: FnArgs) -> FnReturn {
    let (env, pos, tokens) = match _parse_margs("macro use", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    // Compute module path
    let m = &tokens[1];
    let modpath = match m.ttype {
        TokenType::Identifier => env.get(&m.data).cloned(),
        TokenType::String => Some(TVal::from(Value::from(&*unescape(&m.data[1..m.data.len()-1])))),
        _ => None,
    };
    let modpath = match &modpath {
        Some(TVal { val: Value::String(mn), .. }) => Path::new(mn),
        _ => return (None, Err(Error::ScriptError(format!("macro use: invalid module path {}", m), pos))),
    };
    let modname: String = if tokens.len() > 2 {
        tokens[3].data.clone()
    } else {
        match m.ttype {
            TokenType::Identifier => m.data.clone(),
            _ => modpath.file_name().expect("macro use: expected module name in path").to_string_lossy().to_string(),
        }
    };
    let modpath = Path::new(&tokens[0].pos.filename)
        .parent().expect("macro use: expected parent for the current module path")
        .join(modpath);

    // Import module
    let menv = Env::prelude();
    let (vars, val) = run_path(&modpath, &menv, false);
    if let Err(m) = val {
        return (None, Err(m));
    }

    // Construct module struct
    let modst: TVal = match vars {
        Some(vars) => Value::Struct(vars.into()).into(),
        None => Value::Struct(hashmap!{}.into()).into(),
    };
    (Some(hashmap!{ modname => modst.clone() }), Ok(modst))
}
pub fn r#let(args: FnArgs) -> FnReturn {
    let (env, _pos, tokens) = match _parse_margs("macro let", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    let varname = &tokens[1].data;
    let ast = match AST::parse(tokens[3..].iter().cloned(), &env) {
        Ok(a) => a,
        Err(m) => return (None, Err(m)),
    };
    let (_, val) = ast.run(&env);

    match val {
        Ok(v) => (Some(hashmap!{ varname.clone() => v.clone() }), Ok(v)),
        Err(m) => (None, Err(m)),
    }
}

pub fn r#print(args: FnArgs) -> FnReturn {
    let (env, pos, tokens) = match _parse_margs("macro print", args) {
        Ok(t) => t,
        Err(m) => return (None, Err(m)),
    };

    let ast = match AST::parse(tokens[1..].iter().cloned(), &env) {
        Ok(a) => a,
        Err(m) => return (None, Err(m)),
    };
    let (_, val) = ast.run(&env);

    match val {
        Ok(TVal { val: Value::List(l), .. }) => {
            for e in l {
                print!("{} ", e.val);
            }
            println!();
        },
        Err(m) => return (None, Err(m)),
        _ => return (None, Err(Error::ScriptError(format!("macro print: invalid container {:?}", &tokens[1..]), pos))),
    }

    (None, Ok(Value::none().into()))
}
