#[cfg(feature = "compile")]
use serde::{Serialize, Deserialize, Serializer, Deserializer, ser, de};

#[cfg(feature = "compile")]
use maplit::hashmap;

use crate::{Env, AST};

#[cfg(feature = "compile")]
use crate::{Callable, Pos, TokenType, Token, Value, umcore};

#[derive(Debug, Clone)]
#[cfg_attr(feature = "compile", derive(Serialize, Deserialize))]
pub struct Runnable<'a> {
    pub hash: u64,
    pub env: Env<'a>,
    pub ast: AST,
}

// Custom serde impl for Callable
#[cfg(feature = "compile")]
impl Serialize for Callable {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer
    {
        match self {
            Callable::Native(fp) => {
                let mut vars = hashmap! {};
                umcore::_init(&mut vars);
                for nf in vars.iter()
                    .flat_map(|(k, v)| match &v.val {
                        Value::Struct(btm) => btm.clone(),
                        _ => hashmap!{ k.clone() => v.clone() }.into(),
                    })
                {
                    if let Value::Function { body: Callable::Native(nfp), .. } = nf.1.val {
                        if nfp == *fp {
                            let token = Token {
                                ttype: TokenType::Identifier,
                                pos: Pos::start("<runnable>"),
                                data: nf.0,
                            };
                            let ast = AST::Value { token };
                            return ast.serialize(serializer);
                        }
                    }
                }
                Err(ser::Error::custom(format!("Callable::Native serialization failed: couldn't find associated function in prelude {:?}", fp)))
            },
            Callable::AST(ast) => ast.serialize(serializer),
        }
    }
}
#[cfg(feature = "compile")]
impl<'de> Deserialize<'de> for Callable {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>
    {
        match AST::deserialize(deserializer) {
            Ok(ast) => {
                if let AST::Value { token } = &ast {
                    if token.pos.filename == "<runnable>" {
                        let mut vars = hashmap! {};
                        umcore::_init(&mut vars);
                        for nf in vars.iter()
                            .flat_map(|(k, v)| match &v.val {
                                Value::Struct(btm) => btm.clone(),
                                _ => hashmap!{ k.clone() => v.clone() }.into(),
                            })
                        {
                            if nf.0 == token.data {
                                if let Value::Function { body, .. } =  nf.1.val {
                                    return Ok(body);
                                }
                            }
                        }

                        return Err(de::Error::custom(
                            format!("Callable::Native deserialization failed: couldn't find associated function in prelude {:?}", token.data)
                        ));
                    }
                }
                Ok(Callable::AST(ast))
            },
            Err(m) => Err(m),
        }
    }
}
