use std::fmt;
use std::collections::HashMap;
use std::sync::Arc;
use std::convert::TryFrom;

use num::{BigRational, ToPrimitive, FromPrimitive};

use maplit::hashmap;

#[cfg(feature = "compile")]
use serde::{Serialize, Deserialize};

use lazy_static::lazy_static;

use crate::{HashableMap, Error, Env, Pos, TokenType, Token, AST};

#[derive(Debug, Clone, PartialEq, Eq)]
pub enum FnArgs {
    Macro {
        vars: HashableMap<String, TVal>,
        pos: Option<Pos>,
        tokens: Vec<Token>,
    },
    Normal {
        pos: Option<Pos>,
        args: TVal,
    },
    Op {
        vars: HashableMap<String, TVal>,
        pos: Pos,
        other: Option<TVal>,
    },
}
pub type FnReturn = (Option<HashMap<String, TVal>>, Result<TVal, Error>);
#[derive(Debug, Clone, PartialEq, Eq, Hash, PartialOrd, Ord)]
pub enum Callable {
    Native(fn(FnArgs) -> FnReturn),
    AST(AST),
}
impl Callable {
    pub fn call<'a>(&self, env: &'a Env<'a>, args: FnArgs) -> FnReturn {
        use Callable::*;
        match self {
            Native(fp) => fp(args),
            AST(ast) => {
                let mut nenv = Env::child(env);
                if let FnArgs::Macro { vars, pos, tokens } = &args {
                    nenv.set("env", Value::from_env(&Env::from(vars.clone())).into());
                    nenv.set("pos", match pos {
                        Some(p) => Value::from_pos(&p).into(),
                        None => Value::none().into(),
                    });
                    nenv.set("tokens", Value::from_tokens(&tokens).into());
                }

                let (vars, val) = ast.run(&nenv);
                let val = match val {
                    Err(Error::ControlError(s, cv)) if s == "return" => Ok(cv.unwrap_or_else(|| Value::none().into())),
                    _ => val,
                };

                match args {
                    FnArgs::Macro { .. } => {
                        match &val {
                            Ok(TVal { val: Value::List(l), .. }) => {
                                if let Some(TVal { val: Value::Enum(_, e1), .. }) = l.get(0) {
                                    match &*e1.0 {
                                        "Some" => {
                                            // TODO apply new env
                                        },
                                        "None" => {},
                                        _ => {},
                                    }

                                    if let Some(TVal { val: Value::Enum(_, e2), .. }) = l.get(1) {
                                        let e2v = e2.1.clone().unwrap_or_else(|| Value::none().into());
                                        match &*e2.0 {
                                            "Ok" => {
                                                if let Value::List(l) = e2v.val {
                                                    return (vars, Ok(l[0].clone()));
                                                }
                                                return (vars, Ok(e2v));
                                            },
                                            "Err" => {
                                                if let Value::List(l1) = &e2v.val {
                                                    if let Value::Enum(_, e) = &l1[0].val {
                                                        let err = match &*e.0 {
                                                            "ArgumentError" => match &e.1 {
                                                                Some(TVal { val: Value::List(l2), .. }) => Some(Error::ArgumentError(l2[0].val.to_string())),
                                                                _ => None,
                                                            },
                                                            "ParseError" => match &e.1 {
                                                                Some(TVal { val: Value::List(l2), .. }) => Some(Error::ParseError(l2[0].val.to_string(), Pos::from_value(&l2[1].val))),
                                                                _ => None,
                                                            },
                                                            "ScriptError" => match &e.1 {
                                                                Some(TVal { val: Value::List(l2), .. }) => Some(Error::ScriptError(l2[0].val.to_string(), Pos::from_value(&l2[1].val))),
                                                                _ => None,
                                                            },
                                                            "ControlError" => match &e.1 {
                                                                Some(TVal { val: Value::List(l2), .. }) => Some(Error::ControlError(l2[0].val.to_string(), l2.get(1).cloned())),
                                                                _ => None,
                                                            },
                                                            "CustomError" => match &e.1 {
                                                                Some(TVal { val: Value::List(l2), .. }) => Some(Error::CustomError(l2[0].clone())),
                                                                _ => None,
                                                            },
                                                            _ => None,
                                                        };
                                                        if let Some(err) = err {
                                                            return (vars, Err(err));
                                                        }
                                                    }
                                                }
                                                return (vars, Err(Error::CustomError(e2v)));
                                            },
                                            _ => {},
                                        }

                                        return (vars, val);
                                    }
                                }
                            },
                            Err(m) => return (vars, Err(m.clone())),
                            _ => {},
                        }
                        return (None, Err(Error::ScriptError(format!("failed to get FnReturn from user macro {:?}", val), None)));
                    },
                    _ => (None, val),
                }
            },
        }
    }
}

#[derive(Debug, Clone, PartialEq, Eq, Hash, PartialOrd, Ord)]
#[cfg_attr(feature = "compile", derive(Serialize, Deserialize))]
pub enum Type {
    None,
    Any,
    Type,
    Number,
    String,

    List(Arc<Type>),
    Map(Arc<Type>, Arc<Type>),
    Enum(Arc<HashableMap<String, Arc<Type>>>),
    Struct(Arc<HashableMap<String, Arc<Type>>>),
    Generic(Arc<Type>, Arc<HashableMap<String, Arc<Type>>>),

    Function {
        args: Arc<Vec<(String, Arc<Type>)>>,
        ret: Arc<Type>,
    },
}
lazy_static! {
    pub(self) static ref EMPTY_MAP:   Arc<HashableMap<String, Arc<Type>>> = Arc::new(hashmap!{}.into());

    pub(self) static ref TYPE_NONE:   Arc<Type> = Arc::new(Type::None);
    pub(self) static ref TYPE_ANY:    Arc<Type> = Arc::new(Type::Any);
    pub(self) static ref TYPE_TYPE:   Arc<Type> = Arc::new(Type::Type);
    pub(self) static ref TYPE_NUMBER: Arc<Type> = Arc::new(Type::Number);
    pub(self) static ref TYPE_STRING: Arc<Type> = Arc::new(Type::String);

    static ref TYPE_LIST:   Arc<Type> = Arc::new(Type::List(Arc::clone(&TYPE_ANY)));
    static ref TYPE_MAP:    Arc<Type> = Arc::new(Type::Map(Arc::clone(&TYPE_ANY), Arc::clone(&TYPE_ANY)));
    static ref TYPE_ENUM:   Arc<Type> = Arc::new(Type::Enum(Arc::clone(&EMPTY_MAP)));
    static ref TYPE_STRUCT: Arc<Type> = Arc::new(Type::Struct(Arc::clone(&EMPTY_MAP)));
    static ref TYPE_GENERIC: Arc<Type> = Arc::new(Type::Generic(Type::any(), Arc::clone(&EMPTY_MAP)));

    static ref TYPE_FUNCTION: Arc<Type> = Arc::new(Type::Function { args: Arc::new(vec![]), ret: Arc::clone(&TYPE_ANY) });
}
impl Type {
    pub fn none() -> Arc<Type> {
        Arc::clone(&TYPE_NONE)
    }
    pub fn any() -> Arc<Type> {
        Arc::clone(&TYPE_ANY)
    }
    pub fn ttype() -> Arc<Type> {
        Arc::clone(&TYPE_TYPE)
    }
    pub fn number() -> Arc<Type> {
        Arc::clone(&TYPE_NUMBER)
    }
    pub fn string() -> Arc<Type> {
        Arc::clone(&TYPE_STRING)
    }

    pub fn list() -> Arc<Type> {
        Arc::clone(&TYPE_LIST)
    }
    pub fn map() -> Arc<Type> {
        Arc::clone(&TYPE_MAP)
    }
    pub fn tenum() -> Arc<Type> {
        Arc::clone(&TYPE_ENUM)
    }
    pub fn tstruct() -> Arc<Type> {
        Arc::clone(&TYPE_STRUCT)
    }
    pub fn generic() -> Arc<Type> {
        Arc::clone(&TYPE_GENERIC)
    }

    pub fn function() -> Arc<Type> {
        Arc::clone(&TYPE_FUNCTION)
    }

    pub fn equiv<T: AsRef<Type>>(&self, other: T) -> bool {
        let other = other.as_ref();
        use crate::Type::*;
        // TODO do proper type checking
        if matches!(self, Any) || matches!(other, Any) {
            return true;
        }

        match self {
            None => matches!(other, None),
            Any => matches!(other, Any),
            Type => matches!(other, Type),
            Number => matches!(other, Number),
            String => matches!(other, String),

            List(_l1) => matches!(other, List(_l2)),
            Map(_mk1, _mv1) => matches!(other, Map(_mk2, _mv2)),
            Enum(_es1) => matches!(other, Enum(_es2)),
            Struct(_ss1) => matches!(other, Struct(_ss2)),
            Generic(_gu1, _gp1) => matches!(other, Generic(_gu2, _gp2)),

            Function { args: _args1, ret: _ret1 } => matches!(other, Function { args: _args2, ret: _ret2 }),
        }
    }
}
impl fmt::Display for Type {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        use self::Type::*;
        match self {
            None => write!(f, "None"),
            Any => write!(f, "Any"),
            Type => write!(f, "Type"),
            Number => write!(f, "Number"),
            String => write!(f, "String"),

            List(t) => write!(f, "List[{}]", *t),
            Map(k, v) => write!(f, "Map[{}, {}]", *k, *v),
            Enum(es) => write!(f, "Enum[{:?}]", es),
            Struct(ss) => write!(f, "Struct[{:?}]", ss),
            Generic(gu, gp) => write!(f, "Generic[{}, {:?}]", gu, gp),

            Function { args, ret } => write!(f, "Function[{:?} -> {}]", args, ret),
        }
    }
}
impl TryFrom<&str> for Type {
    type Error = Error;
    fn try_from(s: &str) -> Result<Self, Error> {
        use self::Type::*;
        Ok(match s {
            "None" => None,
            "Any" => Any,
            "Type" => Type,
            "Number" => Number,
            "String" => String,

            // TODO add complex type support
            "List" => List(Self::none()),
            "Map" => Map(Self::none(), Self::none()),

            "Function" => Function { args: Arc::new(vec![]), ret: Self::none()},

            _ => return Err(Error::ScriptError(format!("Type::try_from invalid typename: {}", s), Option::None)),
        })
    }
}

#[derive(Clone, PartialEq, Eq, Hash, PartialOrd, Ord)]
#[cfg_attr(feature = "compile", derive(Serialize, Deserialize))]
pub enum Value {
    Type(Arc<Type>, Arc<HashableMap<String, TVal>>),
    Number(BigRational),
    String(String),

    List(Vec<TVal>),
    Map(HashableMap<TVal, TVal>),
    Enum(Arc<HashableMap<String, Arc<Type>>>, Box<(String, Option<TVal>)>),
    Struct(HashableMap<String, TVal>),

    Function {
        args: Arc<Vec<(String, Arc<Type>)>>,
        vars: HashableMap<String, TVal>,
        body: Callable, // custom serde impl in runnable.rs
    },
}
lazy_static! {
    static ref ENUM_TOKEN_TYPE: Arc<HashableMap<String, Arc<Type>>> = Arc::new(hashmap!{
        "Identifier".into() => Type::none(),
        "Number".into() => Type::none(),
        "String".into() => Type::none(),
        "Container".into() => Type::none(),
        "Symbol".into() => Type::none(),
    }.into());
    static ref ENUM_AST: Arc<HashableMap<String, Arc<Type>>> = Arc::new(hashmap!{
        "Macro".into() => Type::none(),
        "Container".into() => Type::none(),
        "Operator".into() => Type::none(),
        "Value".into() => Type::none(),
        "Empty".into() => Type::none(),
    }.into());
}
impl Value {
    pub fn none() -> Self {
        Value::Type(Type::none(), HashableMap::arc())
    }

    pub fn from_pos(pos: &Pos) -> Self {
        Value::Struct(hashmap!{
            "filename".into() => Value::from(&*pos.filename).into(),
            "line".into() => Value::from(pos.line).into(),
            "col".into() => Value::from(pos.col).into(),
        }.into())
    }
    pub fn from_token_type(ttype: &TokenType) -> Self {
        use TokenType::*;
        let tt: std::string::String = match ttype {
            Identifier => "Identifier".into(),
            Number => "Number".into(),
            String => "String".into(),
            Container => "Container".into(),
            Symbol => "Symbol".into(),
            Comment => "Comment".into(),
        };
        Value::Enum(Arc::clone(&ENUM_TOKEN_TYPE), Box::from((tt, None)))
    }
    pub fn from_token(token: &Token) -> Self {
        Value::Struct(hashmap!{
            "ttype".into() => Value::from_token_type(&token.ttype).into(),
            "pos".into() => Value::from_pos(&token.pos).into(),
            "data".into() => Value::from(&*token.data).into(),
        }.into())
    }
    pub fn from_tokens(tokens: &[Token]) -> Self {
        tokens.iter()
            .map(|t| Value::from_token(t))
            .collect::<Vec<Value>>()
            .into()
    }

    pub fn from_env<'a>(env: &Env<'a>) -> Self {
        let p: HashableMap<TVal, TVal> = match &env.parent {
            Some(p) => Env::flatten(p).vars.iter()
                .map(|(k, v)| (k.clone().into(), v.clone()))
                .collect(),
            None => hashmap!{}.into(),
        };
        let vs: HashableMap<TVal, TVal> = env.vars.iter()
            .map(|(k, v)| (k.clone().into(), v.clone()))
            .collect();
        Value::Struct(HashableMap::from(hashmap!{
            "parent".into() => Value::Map(p).into(),
            "vars".into() => Value::Map(vs).into(),
        }))
    }
    pub fn from_ast(ast: &AST) -> Self {
        let (at, av): (String, Value) = match ast {
            AST::Macro { tokens } => ("Macro".into(), Value::Struct(hashmap!{
                "tokens".into() => Value::from_tokens(tokens).into(),
            }.into())),
            AST::Container { token, children } => ("Container".into(), Value::Struct(hashmap!{
                "token".into() => Value::from_token(token).into(),
                "children".into() => Value::List(children.iter()
                    .map(|c| Value::from_ast(c).into())
                    .collect()
                ).into(),
            }.into())),
            AST::Operator { token, lhs, rhs } => ("Operator".into(), Value::Struct(hashmap!{
                "token".into() => Value::from_token(token).into(),
                "lhs".into() => Value::from_ast(lhs).into(),
                "rhs".into() => Value::from_ast(rhs).into(),
            }.into())),
            AST::Value { token } => ("Value".into(), Value::Struct(hashmap!{
                "token".into() => Value::from_token(token).into(),
            }.into())),
            AST::Empty => ("Empty".into(), Value::none()),
        };
        Value::Enum(Arc::clone(&ENUM_AST), Box::from((at, Some(av.into()))))
    }
}
impl From<usize> for Value {
    fn from(n: usize) -> Self {
        Value::Number(BigRational::from_usize(n).expect("expected usize to fit in BigRational"))
    }
}
impl From<&str> for Value {
    fn from(s: &str) -> Self {
        Value::String(s.into())
    }
}
impl From<Vec<Value>> for Value {
    fn from(vs: Vec<Value>) -> Self {
        let mut tvs = vec![];
        for v in vs {
            tvs.push(v.into());
        }
        Value::List(tvs)
    }
}
impl fmt::Debug for Value {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        use num::{BigInt, One};
        use Value::*;
        match self {
            Type(t, _) => {
                write!(f, "{}", t)
            },
            Number(r) => {
                if r.denom() == &BigInt::one() {
                    write!(f, "{}", r.numer())
                } else {
                    write!(f, "{}", r.to_f64().ok_or(std::fmt::Error)?)
                }
            },
            String(s) => {
                write!(f, "{}", s)
            },

            List(l) => write!(f, "{:?}", {
                l.iter()
                    .map(|x| &x.val)
                    .collect::<Vec<&Value>>()
            }),
            Map(m) => write!(f, "{:?}", {
                m.iter()
                    .map(|(k, v)| (&k.val, &v.val))
                    .collect::<HashMap<&Value, &Value>>()
            }),
            Enum(_, e) => write!(f, "Enum Variant {{{}: {:?}}}", &e.0, &e.1),
            Struct(st) => write!(f, "Struct {:?}", {
                st.iter()
                    .map(|(n, v)| (n, &v.val))
                    .collect::<HashMap<&std::string::String, &Value>>()
            }),

            Function { args, body, .. } => {
                match body {
                    // Callable::Native(fp) => write!(f, "Function {{ args: {}, vars: {{...}}, body: Native({:?}) }}", args, fp),
                    // Callable::AST(_) => write!(f, "Function {{ args: {}, vars: {{...}}, body: AST(...) }}", args),
                    Callable::Native(fp) => write!(f, "Function {{ args: {:?}, body: Native({:?}), .. }}", args, fp),
                    Callable::AST(_) => write!(f, "Function {{ args: {:?}, .. }}", args),
                }
            },
        }
    }
}
impl fmt::Display for Value {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        <Self as fmt::Debug>::fmt(self, f)
    }
}

#[derive(Clone, PartialEq, Eq, Hash, PartialOrd, Ord)]
#[cfg_attr(feature = "compile", derive(Serialize, Deserialize))]
pub struct TVal {
    pub ttype: Arc<Type>,
    pub val: Value,
}
impl From<String> for TVal {
    fn from(s: String) -> Self {
        TVal {
            ttype: Type::string(),
            val: Value::String(s),
        }
    }
}
impl From<Value> for TVal {
    fn from(val: Value) -> Self {
        use self::Type as T;
        use Value as V;
        match &val {
            V::Type(_, _) => TVal {
                ttype: T::ttype(),
                val,
            },
            V::Number(_) => TVal {
                ttype: T::number(),
                val,
            },
            V::String(_) => TVal {
                ttype: T::string(),
                val,
            },

            V::List(l) => TVal {
                ttype: Arc::new(T::List({
                    if let Some(a) = l.first() {
                        Arc::clone(&a.ttype)
                    } else {
                        T::none()
                    }
                })),
                val,
            },
            V::Map(m) => TVal {
                ttype: Arc::new(if let Some((k, v)) = m.iter().next() {
                    T::Map(Arc::clone(&k.ttype), Arc::clone(&v.ttype))
                } else {
                    T::Map(T::none(), T::none())
                }),
                val,
            },
            V::Enum(es, ..) => TVal {
                ttype: Arc::new(T::Enum(Arc::clone(&es))),
                val,
            },
            V::Struct(ss) => TVal {
                ttype: Arc::new(T::Struct({
                    let mut sts = hashmap!{};
                    for s in &ss.map {
                        sts.insert(s.0.clone(), Arc::clone(&s.1.ttype));
                    }
                    Arc::new(sts.into())
                })),
                val,
            },

            V::Function { args, .. } => TVal {
                ttype: Arc::new(T::Function {
                    args: Arc::clone(&args),
                    ret: T::none(), // TODO find return type
                }),
                val,
            },
        }
    }
}
impl fmt::Debug for TVal {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match &self.val {
            Value::String(s) => write!(f, "<\"{}\": {}>", s.replace('\\', r#"\\"#).replace('"', r#"\""#), self.ttype),
            Value::Function { .. } => write!(f, "<{}>", self.ttype),
            _ => write!(f, "<{}: {}>", self.val, self.ttype),
        }
    }
}
impl fmt::Display for TVal {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        <Self as fmt::Debug>::fmt(self, f)
    }
}

pub trait VecMap<T> {
    fn contains_key(&self, k: &str) -> bool;
    fn get(&self, k: &str) -> Option<(String, T)>;
    fn set(&mut self, k: &str, v: T) -> Option<(String, T)>;
}
impl<T> VecMap<T> for Vec<(String, T)>
where
    T: Clone,
{
    fn contains_key(&self, k: &str) -> bool {
        self.iter()
            .any(|(s, _v)| s == k)
    }
    fn get(&self, k: &str) -> Option<(String, T)> {
        for (s, v) in self {
            if s == k {
                return Some((s.clone(), v.clone()));
            }
        }
        None
    }
    fn set(&mut self, k: &str, nv: T) -> Option<(String, T)> {
        for (s, v) in self.iter_mut() {
            if s == k {
                let tv = v.clone();
                *v = nv;
                return Some((k.to_string(), tv));
            }
        }
        self.push((k.to_string(), nv));
        None
    }
}

pub fn match_pat(tv: &TVal, pat: &AST, pos: Option<Pos>) -> Result<Option<HashMap<String, TVal>>, Error> {
    let enumval = match &tv.val {
        Value::Enum(_es, e) => e.clone(),
        _ => return Err(Error::ScriptError(format!("match_pat failed: not an enum {}", tv), pos)),
    };

    match pat {
        AST::Operator { token, lhs, rhs } if token.data == "(" => {
            if let AST::Value { token } = &**lhs {
                if token.data == enumval.0 {
                    let mut hm: HashMap<String, TVal> = hashmap!{};
                    match &**rhs {
                        AST::Container { token, children } if token.data == "(" => {
                            if children.len() == 1 {
                                match &children[0] {
                                    AST::Value { token } => if let Some(ev) = enumval.1 {
                                        hm.insert(token.data.clone(), ev);
                                    },
                                    c => return Err(Error::ScriptError(format!("match_pat failed: invalid RHS child {:?}", c), pos)),
                                }
                            } else {
                                match &enumval.1 {
                                    Some(TVal { val: Value::List(evl), .. }) => {
                                        for (c, ev) in children.iter().zip(evl) {
                                            match c {
                                                AST::Value { token } => {
                                                    hm.insert(token.data.clone(), ev.clone());
                                                },
                                                _ => return Err(Error::ScriptError(format!("match_pat failed: invalid RHS child {:?}", c), pos)),
                                            }
                                        }
                                    },
                                    _ => return Err(Error::ScriptError(format!("match_pat failed: no enum value list to unpack {:?}", enumval), pos)),
                                }
                            }
                        },
                        _ => return Err(Error::ScriptError(format!("match_pat failed: invalid RHS for {}", token.data), pos)),
                    }
                    return Ok(Some(hm));
                }
            }
        },
        AST::Value { token } => {
            if token.data == enumval.0 {
                return Ok(None);
            }
        },
        _ => {},
    }
    Err(Error::ScriptError(format!("match_pat failed for {}", tv), pos))
}
