use std::collections::HashMap;
use std::str::FromStr;
use std::cmp;
use std::sync::Arc;

use num::{BigRational, FromPrimitive};

#[cfg(feature = "compile")]
use serde::{Serialize, Deserialize};

use crate::{Env, Error, Exprs, FnArgs, FnReturn, Lines, Op, OpTrait, Pos, TVal, Token, TokenType, Tokens, Type, Value, VecMap, umcore, unescape};

#[derive(Debug, Clone, PartialEq, Eq, Hash, PartialOrd, Ord)]
#[cfg_attr(feature = "compile", derive(Serialize, Deserialize))]
pub enum AST {
    Macro {
        tokens: Vec<Token>,
    },
    Container {
        token: Token,
        children: Vec<AST>,
    },
    Operator {
        token: Token,
        lhs: Box<AST>,
        rhs: Box<AST>,
    },
    Value {
        token: Token,
    },
    Empty,
}
impl AST {
    pub fn from_value(v: &Value) -> Option<Self> {
        if let Value::Enum(_es, e) = v {
            if e.0 == "Empty" {
                return Some(AST::Empty);
            }

            let st = match &e.1 {
                Some(TVal { val: Value::Struct(st), .. }) => st,
                _ => return None,
            };
            match &*e.0 {
                "Macro" => return Some(AST::Macro {
                    tokens: match st.get("tokens") {
                        Some(TVal { val: Value::List(l), .. }) => l.iter().map(|tv| Token::from_value(&tv.val)).collect::<Option<Vec<Token>>>()?,
                        _ => return None,
                    },
                }),
                "Container" => return Some(AST::Container {
                    token: match st.get("token") {
                        Some(tv) => Token::from_value(&tv.val)?,
                        _ => return None,
                    },
                    children: match st.get("children") {
                        Some(TVal { val: Value::List(l), ..}) => l.iter().map(|tv| AST::from_value(&tv.val)).collect::<Option<Vec<AST>>>()?,
                        _ => return None,
                    },
                }),
                "Operator" => return Some(AST::Operator {
                    token: match st.get("token") {
                        Some(tv) => Token::from_value(&tv.val)?,
                        _ => return None,
                    },
                    lhs: match st.get("lhs") {
                        Some(tv) => Box::new(AST::from_value(&tv.val)?),
                        _ => return None,
                    },
                    rhs: match st.get("rhs") {
                        Some(tv) => Box::new(AST::from_value(&tv.val)?),
                        _ => return None,
                    },
                }),
                "Value" => return Some(AST::Value {
                    token: match st.get("token") {
                        Some(tv) => Token::from_value(&tv.val)?,
                        _ => return None,
                    },
                }),
                _ => {},
            }
        }
        None
    }

    pub fn get_first_token(&self) -> Option<Token> {
        use AST::*;
        match self {
            Macro { tokens } => tokens.first().cloned(),
            Container { token, .. } => Some(token.clone()),
            Operator { token, lhs, .. } => match **lhs {
                Empty => Some(token.clone()),
                _ => lhs.get_first_token(),
            },
            Value { token } => Some(token.clone()),
            Empty => None,
        }
    }
    pub fn get_first_pos(&self) -> Option<Pos> {
        match self.get_first_token() {
            Some(t) => Some(t.pos),
            None => None,
        }
    }
    pub fn to_string_lossy(&self) -> String {
        match self {
            AST::Macro { tokens } => tokens.iter()
                .map(|t| t.data.clone())
                .collect::<Vec<String>>()
                .join(" "),
            AST::Container { token, children } => match &*token.data {
                "(" | "[" => {
                    let cs = children.iter()
                        .map(|a| a.to_string_lossy())
                        .collect::<Vec<String>>()
                        .join(", ");
                    format!("{}{}{}", token.data, cs, if token.data == "(" { ")" } else { "]" })
                },
                "{" => {
                    if children.len() > 1 {
                        let cs = children.iter()
                            .map(|a| a.to_string_lossy())
                            .collect::<Vec<String>>()
                            .join(";\n\t");
                        format!("{}\n\t{};\n{}", token.data, cs, "}")
                    } else {
                        "{}".into()
                    }
                },
                s if s.starts_with("//") => token.data.clone(),
                _ => format!("{{{{Invalid container: {}}}}}", token.data),
            },
            AST::Operator { token, lhs, rhs } => match &*token.data {
                "(" | "[" | "{" => format!("{}{}", lhs.to_string_lossy(), rhs.to_string_lossy()),
                _ => format!("{}{}{}", lhs.to_string_lossy(), token.data, rhs.to_string_lossy()),
            },
            AST::Value { token } => token.data.clone(),
            AST::Empty => "EMPTY".into(),
        }
    }

    pub fn parse<'a, I>(tokens: I, env: &Env<'a>) -> Result<AST, Error>
    where
        I: IntoIterator<Item = Token>
    {
        // Map tokens into flat ASTs
        let tokens: Vec<Token> = tokens.into_iter().collect();
        let mut asts: Vec<Result<AST, Error>> = tokens.iter().enumerate()
            .fold(vec![], |mut acc: Vec<Result<AST, Error>>, (i, t)| {
                if let Some(Ok(AST::Macro { .. })) = acc.last() {
                    return acc;
                }

                use TokenType::*;
                match t.ttype {
                    Identifier => {
                        if let Some(TVal { val: Value::Function { args, .. }, .. }) = env.get(&t.data) {
                            if args == &umcore::_macro_args()
                                || (args.contains_key("env") && args.contains_key("pos") && args.contains_key("tokens"))
                            {
                                acc.push(Ok(AST::Macro { tokens: tokens[i..].to_vec() }));
                                return acc;
                            }
                        }
                        acc.push(Ok(AST::Value { token: t.clone() }));
                        acc
                    },
                    Number | String => {
                        acc.push(Ok(AST::Value { token: t.clone() }));
                        acc
                    },
                    Container => {
                        let children = Lines::split(&t.pos.filename, &t.data[1..cmp::max(1, t.data.len()-1)], Some(Pos {
                                filename: t.pos.filename.clone(),
                                line: t.pos.line,
                                col: t.pos.col+1,
                            })).flat_map(|l| Exprs::split(&l))
                            .map(|e| Tokens::tokenize(&e))
                            .map(|t| AST::parse(t, env))
                            .collect::<Result<Vec<AST>, Error>>();
                        match children {
                            Ok(cs) => acc.push(Ok(AST::Container {
                                token: Token {
                                    ttype: Container,
                                    pos: t.pos.clone(),
                                    data: std::string::String::from(t.data.as_bytes()[0] as char),
                                },
                                children: cs,
                            })),
                            Err(m) => acc.push(Err(m)),
                        }
                        acc
                    },
                    Symbol => {
                        acc.push(Ok(AST::Operator {
                            token: t.clone(),
                            lhs: Box::new(AST::Empty),
                            rhs: Box::new(AST::Empty),
                        }));
                        acc
                    },
                    Comment => {
                        acc.push(Ok(AST::Container {
                            token: t.clone(),
                            children: vec![],
                        }));
                        acc
                    },
                }
            });
        for a in &asts {
            if let Err(m) = a {
                return Err(m.clone());
            }
        }
        let asts = asts.drain(0..)
            .map(|a| a.unwrap())
            .collect();

        AST::group_precedence(asts)
    }
    fn group_precedence(mut asts: Vec<AST>) -> Result<AST, Error> {
        // TODO actually determine precedence
        while asts.len() > 1 {
            let mut i: usize = 0;
            let mut unchanged = 0;
            while i < asts.len() {
                match asts[i].clone() {
                    AST::Operator { token, lhs, rhs } if token.data == "." && *lhs == AST::Empty && *rhs == AST::Empty => {
                        let lhs = if i > 0 {
                            asts.remove(i-1)
                        } else {
                            i += 1;
                            AST::Empty
                        };
                        if i > 0 {
                            let _op = asts.remove(i-1); // index changed from previous remove
                        } else {
                            let _op = asts.remove(i);
                            i += 1;
                        }
                        let rhs = if i > 0 && i <= asts.len() {
                            asts.remove(i-1)
                        } else {
                            AST::Empty
                        };

                        unchanged = 0;
                        asts.insert(i-1, AST::Operator {
                            token,
                            lhs: Box::new(lhs),
                            rhs: Box::new(rhs),
                        });
                    },
                    AST::Container { token, children } if i > 0 => {
                        match &asts[i-1] {
                            AST::Operator { rhs, .. } if **rhs == AST::Empty =>  {},
                            _ => {
                                let lhs = asts.remove(i-1);
                                let _cont = asts.remove(i-1); // index changed from previous remove

                                unchanged = 0;
                                asts.insert(i-1, AST::Operator {
                                    token: token.clone(),
                                    lhs: Box::new(lhs),
                                    rhs: Box::new(AST::Container {
                                        token: Token {
                                            ttype: TokenType::Symbol, // Indicate function call with Symbol
                                            ..token
                                        },
                                        children,
                                    }),
                                });
                            },
                        }
                    },
                    AST::Operator { token, lhs, rhs } if token.data != "=" && *lhs == AST::Empty && *rhs == AST::Empty => {
                        // TODO operator precedence
                        let lhs = if i > 0 {
                            asts.remove(i-1)
                        } else {
                            i += 1;
                            AST::Empty
                        };
                        if i > 0 {
                            let _op = asts.remove(i-1); // index changed from previous remove
                        } else {
                            let _op = asts.remove(i);
                            i += 1;
                        }
                        let rhs = if i > 0 && i <= asts.len() {
                            asts.remove(i-1)
                        } else {
                            AST::Empty
                        };

                        unchanged = 0;
                        asts.insert(i-1, AST::Operator {
                            token,
                            lhs: Box::new(lhs),
                            rhs: Box::new(rhs),
                        });
                    },
                    AST::Operator { token, lhs, rhs } if token.data == "=" && *lhs == AST::Empty && *rhs == AST::Empty => {
                        let lhs = if i > 0 {
                            asts.remove(i-1)
                        } else {
                            i += 1;
                            AST::Empty
                        };
                        if i > 0 {
                            let _op = asts.remove(i-1); // index changed from previous remove
                        } else {
                            let _op = asts.remove(i);
                            i += 1;
                        }
                        let rhs = if i > 0 && i <= asts.len() {
                            asts.remove(i-1)
                        } else {
                            AST::Empty
                        };

                        unchanged = 0;
                        asts.insert(i-1, AST::Operator {
                            token,
                            lhs: Box::new(lhs),
                            rhs: Box::new(rhs),
                        });
                    },

                    // TODO handle other cases as needed
                    // AST::Macro { .. } => {},
                    // AST::Container { .. } => {},
                    // AST::Operator { .. } => {},
                    AST::Value { .. } => {
                        unchanged = 0;
                    },
                    // AST::Empty { .. } => {},
                    _ => {
                        // println!("{:?}", asts[i]);
                    },
                }
                if unchanged > 1 {
                    // TODO fix fib.um
                    return Err(Error::ParseError(
                        format!("Unreducible values [{}/{}]: {:?}", i+1, asts.len(), asts.iter()
                            .map(|a| "<".to_string()+&a.to_string_lossy()+">")
                            .intersperse(" ".into())
                            .collect::<String>()),
                        asts[0].get_first_pos())
                    );
                }
                unchanged += 1;
                i += 1;
            }
        }
        Ok(asts.remove(0))
    }

    pub fn run<'a>(&self, env: &'a Env<'a>) -> FnReturn {
        match self {
            AST::Macro { tokens } => {
                let t0 = tokens.first().expect("expected macro token");
                if let Some(TVal { val: Value::Function { body, .. }, .. }) = env.get(&t0.data) {
                    return body.call(&*env, FnArgs::Macro {
                        vars: Env::flatten(env).vars.into(),
                        pos: self.get_first_pos(),
                        tokens: tokens.clone(),
                    });
                }
                return (None, Err(Error::ScriptError(format!("failed to find macro {} in env", t0.data), self.get_first_pos())));
            },
            AST::Container { token, children } => {
                let mut nenv = Env::child(&env);
                let mut should_map = false;
                let vals: Result<Vec<TVal>, Error> = children.iter()
                    .map(|ast| {
                        if let AST::Operator { token, .. } = &ast {
                            if token.data == ":" {
                                should_map = true;
                            }
                        }

                        // Run child AST
                        let (vars, v) = ast.run(&nenv);
                        if let Some(vars) = vars {
                            nenv.update(vars);
                        }
                        v
                    }).collect();
                let diff = env.diff(&mut nenv);
                let vals: Vec<TVal> = match vals {
                    Ok(vs) => vs,
                    Err(m) => return (Some(diff), Err(m)),
                };
                match &*token.data {
                    "(" => {
                        // Return a list if it's for a function call
                        // if vals.len() > 1 || vals.is_empty() {
                        if token.ttype == TokenType::Symbol {
                            return (Some(diff), Ok(Value::List(vals).into()))
                        }
                        return (Some(diff), Ok(vals.last().unwrap_or(&Value::none().into()).clone()))
                    },
                    "[" => return (Some(diff), Ok(Value::List(vals).into())),
                    "{" => {
                        // Attempt to build Map from kv pairs
                        if should_map {
                            let map: Result<Vec<(TVal, TVal)>, Error> = vals.iter()
                                .map(|tv| {
                                    if let TVal { val: Value::List(l), .. } = tv {
                                        if l.len() == 2 {
                                            return Ok((l[0].clone(), l[1].clone()));
                                        }
                                    }
                                    return Err(Error::ScriptError(format!("failed to build Map from Container{{}}: {} is not a List", tv), Some(token.pos.clone())));
                                }).collect();
                            if let Ok(m) = map {
                                return (Some(diff), Ok(Value::Map(m.into_iter().collect::<HashMap<TVal, TVal>>().into()).into()));
                            }
                        }
                        return (Some(diff), Ok(vals.last().unwrap_or(&Value::none().into()).clone()));
                    },
                    _ => {
                        // Ignore comments
                        if token.data.starts_with("//") {
                            return (None, Ok(Value::none().into()));
                        }
                        return (None, Err(Error::ScriptError(format!("unsupported Container {}", token.data), self.get_first_pos())));
                    },
                }
            },
            AST::Operator { token, lhs, rhs } => {
                let mut lval = None;
                let mut rval = None;

                // Handle the Equal operator without evaluation the LHS
                if token.data == "=" {
                    let lval: TVal = match &**lhs {
                        AST::Value { token: lt } => Value::String(lt.data.clone()).into(),
                        _ => return (None, Err(Error::ScriptError(format!("invalid LHS for Equal: {:?}", lhs), Some(token.pos.clone())))),
                    };
                    let rval = match rhs.run(env).1 {
                        Ok(v) => Some(v),
                        Err(m) => return (None, Err(m)),
                    };
                    return OpTrait::<'a, {Op::Equal}>::op(&lval, &env, token.pos.clone(), rval.as_ref());
                }

                // Compute lval
                if **lhs != AST::Empty {
                    lval = match lhs.run(env).1 {
                        Ok(v) => Some(v),
                        Err(m) => return (None, Err(m)),
                    };
                }

                // Handle Dot operator without evaluating the RHS
                if token.data == "." {
                    let rstr = rhs.get_first_token().expect("expected RHS token").data;
                    let rval = Some(Value::String(rstr.clone()).into());
                    match lval {
                        Some(lval) => return OpTrait::<'a, {Op::Dot}>::op(&lval, &env, token.pos.clone(), rval.as_ref()),
                        None => {
                            // If no LHS then fetch from the Enums in the current env
                            for (_pek, pev) in Env::iter_flatten(&env) {
                                if let TVal { val: Value::Type(en, enr), .. } = pev {
                                    if let Type::Enum(envar) = &**en {
                                        if enr.get(&rstr).is_some() {
                                            let v = Value::Enum(Arc::clone(envar), Box::new((rstr, None)));
                                            return (None, Ok(v.into()));
                                        }
                                    }
                                }
                            }
                        },
                    }
                }

                // Compute rval
                if **rhs != AST::Empty {
                    rval = match rhs.run(env).1 {
                        Ok(v) => Some(v),
                        Err(m) => return (None, Err(m)),
                    };
                }

                // Compute operation
                let (vars, val) = Op::op(&env, token.clone(), lval.as_ref(), rval.as_ref());
                let nvars = match vars {
                    Some(vars) => Some(env.diff(&mut Env::from(vars))),
                    None => None,
                };
                return (nvars, val);
            },
            AST::Value { token } => {
                use TokenType::*;
                match token.ttype {
                    Identifier => match env.get(&token.data) {
                        Some(v) => return (None, Ok(v.clone())),
                        None => return (None, Err(Error::ScriptError(format!("failed to find '{}' in this env", token.data), self.get_first_pos()))),
                    },
                    Number => {
                        match BigRational::from_str(&token.data) {
                            Ok(r) => return (None, Ok(Value::Number(r).into())),
                            Err(_) => if let Ok(f) = f64::from_str(&token.data) {
                                if let Some(r) = BigRational::from_f64(f) {
                                    return (None, Ok(Value::Number(r).into()));
                                }
                            },
                        }
                        return (None, Err(Error::ScriptError(format!("failed to parse '{}' as number", token.data), self.get_first_pos())));
                    },
                    String => return (None, Ok(Value::String(unescape(&token.data[1..token.data.len()-1])).into())),
                    _ => {},
                }
            },
            _ => {},
        }
        (None, Err(Error::ScriptError(format!("failed to execute AST: {:?}", self), self.get_first_pos())))
    }
}
