use std::sync::Arc;

use itertools::Itertools;
use itertools::EitherOrBoth::{Both, Right};
use num::{BigRational, ToPrimitive, FromPrimitive};
use num::pow::Pow;

use maplit::hashmap;

use crate::{Env, Error, FnArgs, FnReturn, Pos, TVal, Token, Type, Value, hashablemap::HashableMap, umcore};

#[allow(unused)] // TODO implement all ops
#[derive(Debug, Clone, Copy, PartialEq, Eq)]
pub enum Op {
    ColonColon,

    Dot,

    Call,
    Index,
    Construct,

    Question,
    Tilde,
    Exclaim,

    Ampersat,
    Hash,
    Dollar,
    Backslash,
    Colon,

    AsteriskAsterisk,
    Asterisk,
    Slash,
    Percent,
    Plus,
    Dash,

    LangleLangle,
    RangleRangle,

    Ampersand,
    Caret,
    Vbar,

    LangleEqualRangle,
    Langle,
    LangleEqual,
    Rangle,
    RangleEqual,

    EqualEqual,
    ExclaimEqual,

    AmpersandAmpersand,
    VbarVbar,

    DotDot,
    DotDotDot,

    DashRangle,
    LangleDash,

    EqualRangle,
    VbarRangle,
    Equal,
}
impl Op {
    pub fn op<'a>(env: &'a Env<'a>, op: Token, lval: Option<&'a TVal>, rval: Option<&'a TVal>) -> FnReturn {
        match (lval, rval) {
            (None, Some(rv)) => match &*op.data {
                "+" => return OpTrait::<'a, {Op::Plus}>::op(rv, env, op.pos, None),
                "-" => return OpTrait::<'a, {Op::Dash}>::op(rv, env, op.pos, None),
                _ => {},
            },
            (Some(_lv), None) => {},
            (Some(lv), Some(_)) => match &*op.data {
                "(" => return OpTrait::<{Op::Call}>::op(lv, env, op.pos, rval),
                "[" => return OpTrait::<{Op::Index}>::op(lv, env, op.pos, rval),
                "{" => return OpTrait::<{Op::Construct}>::op(lv, env, op.pos, rval),

                ":" => return OpTrait::<{Op::Colon}>::op(lv, env, op.pos, rval),

                "**" => return OpTrait::<{Op::AsteriskAsterisk}>::op(lv, env, op.pos, rval),
                "*" => return OpTrait::<{Op::Asterisk}>::op(lv, env, op.pos, rval),
                "+" => return OpTrait::<{Op::Plus}>::op(lv, env, op.pos, rval),
                "-" => return OpTrait::<{Op::Dash}>::op(lv, env, op.pos, rval),

                "<=" => return OpTrait::<{Op::LangleEqual}>::op(lv, env, op.pos, rval),
                "<" => return OpTrait::<{Op::LangleEqual}>::op(lv, env, op.pos, rval),
                ">" => return OpTrait::<{Op::Rangle}>::op(lv, env, op.pos, rval),
                ">=" => return OpTrait::<{Op::RangleEqual}>::op(lv, env, op.pos, rval),

                "==" => return OpTrait::<{Op::EqualEqual}>::op(lv, env, op.pos, rval),

                "=" => return OpTrait::<{Op::Equal}>::op(lv, env, op.pos, rval),

                _ => {},
            },
            (None, None) => {},
        }
        (None, Err(Error::ScriptError(format!("invalid operator {} for {:?} and {:?}", op, lval, rval), Some(op.pos))))
    }
}

pub trait OpTrait<'a, const OP: Op> {
    fn op(&self, env: &'a Env<'a>, pos: Pos, other: Option<&TVal>) -> FnReturn;
}
// TODO impl more ops
impl<'a> OpTrait<'a, {Op::Call}> for TVal {
    fn op(&self, _env: &Env<'a>, pos: Pos, other: Option<&TVal>) -> FnReturn {
        match &self.val {
            Value::Function { args, vars: fvars, body } => {
                // Apply bound args
                if let Some(other) = other {
                    if let TVal { val: Value::List(mut l1), .. } = other.clone() {
                        l1.insert(0, self.clone());
                        let mut other = Value::List(l1);

                        // Set arg names in env
                        let mut fenv = Env::from(fvars.clone());
                        for (i, a) in std::iter::once(&("this".to_string(), Arc::clone(&self.ttype))).chain(args.iter()).enumerate() {
                            if let Value::List(ref mut l) = &mut other {
                                if let Some(v) = l.get(i) {
                                    if !a.1.equiv(&v.ttype) {
                                        // println!("{:#?}", l);
                                        return (None, Err(Error::ScriptError(format!("function arg type mismatch: expected {}, got {}", a.1, v.ttype), Some(pos))));
                                    }
                                    fenv.set(&a.0, v.clone());
                                } else if let Some(v) = fvars.get(&a.0) {
                                    // TODO only check immediate scope
                                    if !a.1.equiv(&v.ttype) {
                                        return (None, Err(Error::ScriptError(format!("function arg type mismatch: expected {}, got {}", a.1, v.ttype), Some(pos))));
                                    }
                                    l.push(v.clone());
                                }
                            }
                        }

                        let (vars, val) = body.call(&fenv, FnArgs::Normal {
                            pos: Some(pos),
                            args: other.into(),
                        });
                        return (vars, val);
                    }
                }
                return (None, Err(Error::ScriptError(format!("operator {}: missing function arguments", "()"), Some(pos))));
            },
            Value::Enum(vs, v) => {
                if let Some(other) = other {
                    let l = if let TVal { val: Value::List(l), .. } = other.clone() {
                        l
                    } else {
                        vec![other.clone()]
                    };

                    if let Type::Enum(hm) = &*self.ttype {
                        let mut vals: Vec<TVal> = vec![];
                        for etpt in hm.map.iter().zip_longest(l) {
                            match etpt {
                                Both(et, pt) => {
                                    if !et.1.equiv(&pt.ttype) {
                                        return (None, Err(Error::ScriptError(format!("enum param type mismatch: expected {}, got {}", et.1, pt.ttype), Some(pos))));
                                    }
                                    vals.push(pt);
                                },
                                Right(pt) => {
                                    vals.push(pt);
                                },
                                _ => break,
                            }
                        }

                        let val = match vals.len() {
                            0 => None,
                            // 1 => Some(vals[0].clone()),
                            _ => Some(Value::List(vals).into()),
                        };
                        return (None, Ok(Value::Enum(Arc::clone(vs), Box::new((v.0.clone(), val))).into()));
                    }
                }
            },
            _ => {},
        }
        (None, Err(Error::ScriptError(format!("operator {} not implemented for {} and {:?}", "()", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::Index}> for TVal {
    fn op(&self, _env: &Env<'a>, pos: Pos, other: Option<&TVal>) -> FnReturn {
        match &self.val {
            Value::Type(subt, ts) => {
                let mut nsubt: Type = (**subt).clone();
                match nsubt {
                    Type::List(ref mut lt) => {
                        if let Some(TVal { val: Value::List(l), .. }) = other {
                            if let Some(TVal { val: Value::Type(tt, _), .. }) = l.get(0) {
                                *lt = Arc::clone(tt);
                                return (None, Ok(Value::Type(Arc::new(nsubt), Arc::clone(&ts)).into()));
                            }
                        }
                    },
                    Type::Map(ref mut k, ref mut v) => {
                        if let Some(TVal { val: Value::List(l), .. }) = other {
                            if let Some(TVal { val: Value::Type(kt, _), .. }) = l.get(0) {
                                if let Some(TVal { val: Value::Type(vt, _), .. }) = l.get(1) {
                                    *k = Arc::clone(kt);
                                    *v = Arc::clone(vt);
                                    return (None, Ok(Value::Type(Arc::new(nsubt), Arc::clone(&ts)).into()));
                                }
                            }
                        }
                    },
                    Type::Enum(_) => {
                        // TODO make generic indexing actually generic
                        if let Type::Generic(ut, tp) = &*self.ttype {
                            if let Some(TVal { val: Value::List(l), .. }) = other {
                                let mut ntp: HashableMap<String, Arc<Type>> = (**tp).clone();
                                for ((_, v), t) in ntp.iter_mut().zip(l) {
                                    *v = match &t.val {
                                        Value::Type(t, _) => Arc::clone(&t),
                                        _ => return (None, Err(Error::ScriptError(format!("invalid type parameter {}", t.val), Some(pos)))),
                                    };
                                }
                                if let Value::Type(subt, ts) = &self.val {
                                    let mut nts: HashableMap<String, TVal> = (**ts).clone();
                                    for v in nts.map.values_mut() {
                                        *v = TVal {
                                            ttype: Arc::new(Type::Enum(Arc::new({
                                                if let Type::Enum(hm) = &*v.ttype {
                                                    let nhm = hm.map.iter().zip(l)
                                                        .map(|((k, _), t)| {
                                                            match &t.val {
                                                                Value::Type(t, _) => Ok((k.clone(), Arc::clone(t))),
                                                                _ => return Err(Error::ScriptError(format!("invalid type parameter {}", t.val), Some(pos.clone()))),
                                                            }
                                                        }).collect::<Result<HashableMap<String, Arc<Type>>, Error>>();
                                                    match nhm {
                                                        Ok(hm) => hm,
                                                        Err(m) => return (None, Err(m)),
                                                    }
                                                } else {
                                                    return (None, Err(Error::ScriptError(format!("invalid type parameter {}", v.ttype), Some(pos))));
                                                }
                                            }))),
                                            val: v.val.clone(),
                                        };
                                    }

                                    return (None, Ok(TVal {
                                        ttype: Arc::new(Type::Generic(Arc::clone(ut), Arc::new(ntp))),
                                        val: Value::Type(subt.clone(), Arc::new(nts)),
                                    }));
                                }
                            }
                        }
                    },
                    _ => {},
                }
                return (None, Err(Error::ScriptError(format!("type index mismatch for {} and {:?}", self, other), Some(pos))));
            },
            Value::List(l1) => {
                if let Some(TVal { val: Value::List(l2), .. }) = other {
                    if let Some(TVal { val: Value::Number(idx), .. }) = l2.get(0) {
                        let idx: usize = match idx.to_usize() {
                            Some(n) => n,
                            None => match (-idx).to_usize() {
                                Some(n) => l1.len().wrapping_sub(n),
                                None => return (None, Err(Error::ScriptError(format!("list index doesn't fit in usize: {}", idx), Some(pos))))
                            },
                        };
                        return match l1.get(idx) {
                            Some(val) => (None, Ok(val.clone())),
                            None => (None, Err(Error::ScriptError(format!("list index out of range: {}", idx), Some(pos)))),
                        };
                    }
                }
            },
            Value::Map(hm) => {
                if let Some(TVal { val: Value::List(l), .. }) = other {
                    if let Some(TVal { val, .. }) = l.get(0) {
                        return match hm.get(&TVal::from(val.clone())) {
                            Some(val) => (None, Ok(val.clone())),
                            None => (None, Err(Error::ScriptError(format!("map index not found: {}", val), Some(pos)))),
                        };
                    }
                }
            },
            _ => {},
        }
        (None, Err(Error::ScriptError(format!("operator {} not implemented for {} and {:?}", "[]", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::Construct}> for TVal {
    fn op(&self, _env: &Env<'a>, pos: Pos, other: Option<&TVal>) -> FnReturn {
        if let Value::Type(_st, m1) = &self.val {
            if let Some(TVal { val: Value::Map(m2), .. }) = other {
                let mut map = hashmap!{};
                for (k, v) in &m2.map {
                    if let TVal { val: Value::String(ks), .. } = k {
                        if let Some(evt) = m1.get(ks) {
                            if let Value::Type(tt, _) = &evt.val {
                                if &v.ttype == tt {
                                    map.insert(ks.clone(), v.clone());
                                    continue;
                                }
                            }
                            return (None, Err(Error::ScriptError(format!("struct value {} type mismatch: got {}, expected {}", k, v.ttype, evt.val), Some(pos))));
                        }
                    }
                    return (None, Err(Error::ScriptError(format!("struct key {} not found in {}", k, self), Some(pos))));
                }
                return (None, Ok(Value::Struct(map.into()).into()));
            }
        }
        (None, Err(Error::ScriptError(format!("operator {} not implemented for {} and {:?}", "{}", self, other), Some(pos))))
    }
}

impl<'a> OpTrait<'a, {Op::Dot}> for TVal {
    fn op(&self, env: &'a Env<'a>, pos: Pos, other: Option<&TVal>) -> FnReturn {
        if let Some(TVal { val: Value::String(os), .. }) = other {
            let name = match &self.val {
                Value::Type(_, ts) => match ts.get(os) {
                    Some(v) => return (None, Ok(v.clone())),
                    None => return (None, Err(Error::ScriptError(format!("type access failed for {:?} with {}", self.val, other.as_ref().unwrap()), Some(pos)))),
                },
                Value::Number(_) => "Number",
                Value::String(_) => "String",

                Value::List(_) => "List",
                Value::Map(_) => "Map",
                Value::Enum(_, _) => "Enum",
                Value::Struct(st) => match st.get(os) {
                    Some(v) => return (None, Ok(v.clone())),
                    None => return (None, Err(Error::ScriptError(format!("struct access failed for {:?} with {}", self.val, other.as_ref().unwrap()), Some(pos)))),
                },

                Value::Function { .. } => "Function",
            };

            if let Some(lhs) = env.get(name) {
                // Attempt to bind self to the found lhs
                match OpTrait::<{Op::Dot}>::op(lhs, env, pos.clone(), other) {
                    (_, Ok(TVal { val: Value::Function { args, vars: fvars, body }, .. })) => {
                        let bound = umcore::function_bind(FnArgs::Normal {
                            pos: Some(pos.clone()),
                            args: Value::List(vec![
                                Value::Function {
                                    args,
                                    vars: fvars,
                                    body
                                }.into(),
                                self.clone(),
                            ]).into(),
                        });
                        let (bargs, bvars, bbody) = match bound {
                            (_, Ok(TVal { val: Value::Function { args: bargs, vars: bvars, body: bbody }, .. })) => (bargs, bvars, bbody),
                            (_, Err(Error::ScriptError(m, _))) => {
                                return (None, Err(Error::ScriptError(format!("failed to bind {}.{}: {}", name, other.unwrap().val, m), Some(pos))))
                            },
                            _ => {
                                return (None, Err(Error::ScriptError(format!("failed to bind {}.{}", name, other.unwrap().val), Some(pos))))
                            },
                        };
                        return (None, Ok(
                            Value::Function {
                                args: bargs,
                                vars: bvars,
                                body: bbody,
                            }.into()
                        ))
                    },
                    (_, Err(m)) => return (None, Err(m)),
                    v => todo!("{:?}", v),
                }
            }
        }
        (None, Err(Error::ScriptError(format!("operator {} not implemented for {} and {:?}", ".", self, other), Some(pos))))
    }
}

impl<'a> OpTrait<'a, {Op::Colon}> for TVal {
    fn op(&self, _env: &Env<'a>, pos: Pos, other: Option<&TVal>) -> FnReturn {
        if let Some(TVal { val: _, .. }) = other {
            return (None, Ok(Value::List(vec![self.clone(), other.unwrap().clone()]).into()));
        }
        (None, Err(Error::ScriptError(format!("operator {} not implemented for {} and {:?}", ":", self, other), Some(pos))))
    }
}

impl<'a> OpTrait<'a, {Op::AsteriskAsterisk}> for TVal {
    fn op(&self, _env: &Env<'a>, pos: Pos, other: Option<&TVal>) -> FnReturn {
        if let Value::Number(r1) = &self.val {
            if let Some(TVal { val: Value::Number(r2), .. }) = other {
                return (None, Ok(Value::Number({
                    // TODO add proper support for fractional powers
                    let r3 = Pow::pow(r1, r2.numer());
                    let dr: u32 = ToPrimitive::to_u32(r2.denom()).expect("op ** denom can't fit in u32");
                    BigRational::new(
                        r3.numer().nth_root(dr),
                        r3.denom().nth_root(dr),
                    )
                }).into()));
            }
        }
        (None, Err(Error::ScriptError(format!("operator {} not implemented for {} and {:?}", "**", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::Asterisk}> for TVal {
    fn op(&self, _env: &Env<'a>, pos: Pos, other: Option<&TVal>) -> FnReturn {
        match &self.val {
            Value::Number(r1) => match other {
                Some(TVal { val: Value::Number(r2), .. }) => return (None, Ok(Value::Number(r1 * r2).into())),
                Some(TVal { val: Value::String(s), .. }) => return (None, Ok(Value::String(
                    s.repeat(
                        match r1.to_usize() {
                            Some(n) => n,
                            None => return (None, Err(Error::ScriptError(format!("failed to repeat string {} times", r1), Some(pos))))
                        }
                    )
                ).into())),
                _ => {},
            },
            Value::String(s) => if let Some(TVal { val: Value::Number(r), .. }) = other {
                return (None, Ok(Value::String(
                    s.repeat(
                        match r.to_usize() {
                            Some(n) => n,
                            None => return (None, Err(Error::ScriptError(format!("failed to repeat string {} times", r), Some(pos))))
                        }
                    )
                ).into()));
            },
            _ => {},
        }
        (None, Err(Error::ScriptError(format!("operator {} not implemented for {} and {:?}", "*", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::Plus}> for TVal {
    fn op(&self, _env: &Env<'a>, pos: Pos, other: Option<&TVal>) -> FnReturn {
        match &self.val {
            Value::Number(r1) => match other {
                Some(TVal { val: Value::Number(r2), .. }) => return (None, Ok(Value::Number(r1 + r2).into())),
                None => return (None, Ok(Value::Number(r1.clone()).into())),
                _ => {},
            },
            Value::String(s1) => if let Some(TVal { val: Value::String(s2), .. }) = other {
                return (None, Ok(Value::String(s1.to_string()+&s2).into()));
            }
            _ => {},
        }
        (None, Err(Error::ScriptError(format!("operator {} not implemented for {} and {:?}", "+", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::Dash}> for TVal {
    fn op(&self, _env: &Env<'a>, pos: Pos, other: Option<&TVal>) -> FnReturn {
        if let Value::Number(r1) = &self.val {
            match other {
                Some(TVal { val: Value::Number(r2), .. }) => return (None, Ok(Value::Number(r1 - r2).into())),
                None => return (None, Ok(Value::Number(-r1).into())),
                _ => {},
            }
        }
        (None, Err(Error::ScriptError(format!("operator {} not implemented for {} and {:?}", "-", self, other), Some(pos))))
    }
}

impl<'a> OpTrait<'a, {Op::LangleEqual}> for TVal {
    fn op(&self, _env: &Env<'a>, pos: Pos, other: Option<&TVal>) -> FnReturn {
        match &self.val {
            Value::Number(r1) => if let Some(TVal { val: Value::Number(r2), .. }) = &other {
                if let Some(r) = BigRational::from_u32((r1 <= r2) as u32) {
                    return (None, Ok(Value::Number(r).into()));
                }
            },
            Value::String(s1) => if let Some(TVal { val: Value::String(s2), .. }) = &other {
                if let Some(r) = BigRational::from_u32((s1 <= s2) as u32) {
                    return (None, Ok(Value::Number(r).into()));
                }
            },
            _ => {},
        }
        (None, Err(Error::ScriptError(format!("operator {} not implemented for {} and {:?}", "<=", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::Langle}> for TVal {
    fn op(&self, _env: &Env<'a>, pos: Pos, other: Option<&TVal>) -> FnReturn {
        match &self.val {
            Value::Number(r1) => if let Some(TVal { val: Value::Number(r2), .. }) = &other {
                if let Some(r) = BigRational::from_u32((r1 < r2) as u32) {
                    return (None, Ok(Value::Number(r).into()));
                }
            },
            Value::String(s1) => if let Some(TVal { val: Value::String(s2), .. }) = &other {
                if let Some(r) = BigRational::from_u32((s1 < s2) as u32) {
                    return (None, Ok(Value::Number(r).into()));
                }
            },
            _ => {},
        }
        (None, Err(Error::ScriptError(format!("operator {} not implemented for {} and {:?}", "<=", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::Rangle}> for TVal {
    fn op(&self, _env: &Env<'a>, pos: Pos, other: Option<&TVal>) -> FnReturn {
        match &self.val {
            Value::Number(r1) => if let Some(TVal { val: Value::Number(r2), .. }) = &other {
                if let Some(r) = BigRational::from_u32((r1 > r2) as u32) {
                    return (None, Ok(Value::Number(r).into()));
                }
            },
            Value::String(s1) => if let Some(TVal { val: Value::String(s2), .. }) = &other {
                if let Some(r) = BigRational::from_u32((s1 > s2) as u32) {
                    return (None, Ok(Value::Number(r).into()));
                }
            },
            _ => {},
        }
        (None, Err(Error::ScriptError(format!("operator {} not implemented for {} and {:?}", "<=", self, other), Some(pos))))
    }
}
impl<'a> OpTrait<'a, {Op::RangleEqual}> for TVal {
    fn op(&self, _env: &Env<'a>, pos: Pos, other: Option<&TVal>) -> FnReturn {
        match &self.val {
            Value::Number(r1) => if let Some(TVal { val: Value::Number(r2), .. }) = &other {
                if let Some(r) = BigRational::from_u32((r1 >= r2) as u32) {
                    return (None, Ok(Value::Number(r).into()));
                }
            },
            Value::String(s1) => if let Some(TVal { val: Value::String(s2), .. }) = &other {
                if let Some(r) = BigRational::from_u32((s1 >= s2) as u32) {
                    return (None, Ok(Value::Number(r).into()));
                }
            },
            _ => {},
        }
        (None, Err(Error::ScriptError(format!("operator {} not implemented for {} and {:?}", "<=", self, other), Some(pos))))
    }
}

impl<'a> OpTrait<'a, {Op::EqualEqual}> for TVal {
    fn op(&self, _env: &Env<'a>, pos: Pos, other: Option<&TVal>) -> FnReturn {
        match &self.val {
            Value::Number(r1) => if let Some(TVal { val: Value::Number(r2), .. }) = &other {
                if let Some(r) = BigRational::from_u32((r1 == r2) as u32) {
                    return (None, Ok(Value::Number(r).into()));
                }
            },
            Value::String(s1) => if let Some(TVal { val: Value::String(s2), .. }) = &other {
                if let Some(r) = BigRational::from_u32((s1 == s2) as u32) {
                    return (None, Ok(Value::Number(r).into()));
                }
            },
            Value::Enum(m1, e1) => if let Some(TVal { val: Value::Enum(m2, e2), .. }) = &other {
                let b = m1 == m2 && e1 == e2;
                if let Some(r) = BigRational::from_u32(b as u32) {
                    return (None, Ok(Value::Number(r).into()));
                }
            },
            _ => {},
        }
        (None, Err(Error::ScriptError(format!("operator {} not implemented for {} and {:?}", "==", self, other), Some(pos))))
    }
}

impl<'a> OpTrait<'a, {Op::Equal}> for TVal {
    fn op(&self, env: &Env<'a>, pos: Pos, other: Option<&TVal>) -> FnReturn {
        if let Value::String(varname) = &self.val {
            if !env.has(varname) {
                return (None, Err(Error::ScriptError(format!("operator {}: varname {} not found in env, define it with the let macro", "=", varname), Some(pos))));
            }
            if let Some(v) = other {
                return (Some(hashmap!{ varname.clone() => v.clone() }), Ok(v.clone()));
            }
        }
        (None, Err(Error::ScriptError(format!("operator {} not implemented for {} and {:?}", "=", self, other), Some(pos))))
    }
}
