use umbra_lang as umbra;
use umbra::{Args, Error, Env};

#[cfg(feature = "compile")]
use std::{fs, io::Write, path::Path};
#[cfg(feature = "compile")]
use umbra::Runnable;

#[cfg(feature = "compile")]
fn compile(args: Args) -> Result<(), Error> {
    // Load file
    let path = match args.files.get(0) {
        Some(p) => p,
        None => return Err(Error::ArgumentError("Failed to compile: missing filename".into())),
    };
    let script = match fs::read_to_string(path) {
        Ok(s) => s,
        Err(m) => return Err(Error::ScriptError(format!("Failed to load {}: {}", path, m), None)),
    };

    // Compile script
    let env = Env::prelude();
    let runnable: Runnable = umbra::compile(path, &script, &env, None, 0)?;

    // Serialize runnable
    let objdata = bincode::serialize(&runnable)
        .map_err(|e| Error::ScriptError(
            format!("Failed to serialize object file for {}: ", path)
                + &e.to_string(),
            None,
        ))?;

    // Create bin directory
    fs::create_dir_all("bin").map_err(|e| Error::ScriptError(
        "Failed to create bin directory: ".to_string()
            + &e.to_string(),
        None,
    ))?;

    // Create and write file
    let objpath = match Path::new(path).with_extension("obj").file_name() {
        Some(p) => Path::new("bin").join(p.to_owned()),
        None => return Err(Error::ScriptError(format!("Failed to save object file for {}: invalid path", path), None)),
    };
    {
        let mut file = fs::File::create(objpath.clone())
            .map_err(|e| Error::ScriptError(
                format!("Failed to create object file {} for {}: ", objpath.to_string_lossy(), path)
                    + &e.to_string(),
                None,
            ))?;
        file.write_all(&objdata)
            .map_err(|e| Error::ScriptError(
                format!("Failed to write object file {} for {}: ", objpath.to_string_lossy(), path)
                    + &e.to_string(),
                None,
            ))?;
    }

    Ok(())
}
#[cfg(feature = "compile")]
fn exec(args: Args) -> Result<(), Error> {
    // Load file
    let path = match args.files.get(0) {
        Some(p) => p,
        None => return Err(Error::ArgumentError("Failed to exec: missing filename".into())),
    };
    let binscript = match fs::read(path) {
        Ok(b) => b,
        Err(m) => return Err(Error::ScriptError(format!("Failed to load {}: {}", path, m), None)),
    };

    // Deserialize runnable
    let runnable: Runnable = bincode::deserialize(&binscript).map_err(|e| Error::ScriptError(
        format!("Failed to deserialize object file for {}: ", path)
            + &e.to_string(),
        None,
    ))?;

    // println!("{:#?}", runnable.ast);

    // Run script
    let (_, val) = runnable.ast.run(&runnable.env);
    val?;

    Ok(())
}

fn main() -> Result<(), Error> {
    let args = match Args::handle() {
        Ok(args) => args,
        Err(m) => return Err(m),
    };
    // If no filename args, start the interactive interpreter
    if args.files.is_empty() {
        let env = Env::prelude();
        umbra::run_interactive(&env);
        return Ok(())
    }

    // Handle flags
    let mut inspect = false;
    for f in &args.longflags {
        match f.as_ref() {
            #[cfg(feature = "compile")]
            "compile" => return compile(args),
            #[cfg(feature = "compile")]
            "exec" => return exec(args),

            "inspect" => inspect = true,
            _ => return Err(Error::ArgumentError(format!("Unsupported flag: {}", f))),
        }
    }

    // Otherwise run the scripts in the provided order
    let env = Env::prelude();
    for f in args.files.iter() {
        let (vars, val) = umbra::run_path(&f, &env, true);
        if let Err(m) = &val {
            eprintln!("{}", m);
        }

        let mut nenv = Env::child(&env);
        if let Some(vars) = vars {
            nenv.update(vars);
        }

        if inspect {
            println!("inspecting {} post-execution:", f);
            umbra::run_interactive(&nenv);
        }

        if val.is_err() {
            break;
        }
    }


    Ok(())
}
