#![allow(incomplete_features)]
#![feature(const_generics)]

#![feature(iter_intersperse)]

#![deny(rust_2018_idioms)]

// #![warn(clippy::pedantic)]
#![allow(clippy::needless_lifetimes)]
#![allow(clippy::upper_case_acronyms)]

use std::{fmt, fs};
use std::collections::{HashMap, hash_map::DefaultHasher};
use std::hash::{Hash, Hasher};
use std::path::{Path, PathBuf};

use lazy_static::lazy_static;
use maplit::hashmap;

#[cfg(feature = "readline")]
use rustyline as rl;
#[cfg(not(feature = "readline"))]
use std::io::{self, BufRead, Write};

#[cfg(feature = "compile")]
use serde::{Serialize, Deserialize, Serializer, Deserializer, ser, de};

#[cfg(feature = "compile")]
use std::fmt::Display;

mod hashablemap;
use hashablemap::*;

mod parsing;
use parsing::*;
mod runnable;
pub use runnable::*;

mod ops;
use ops::*;
mod umcore;
mod umstd;

#[derive(Debug, Clone)]
pub enum Error {
    ArgumentError(String),
    ParseError(String, Option<Pos>),
    ScriptError(String, Option<Pos>),
    ControlError(String, Option<TVal>),

    CustomError(TVal),
}
impl fmt::Display for Error {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        use Error::*;
        match self {
            ArgumentError(m) => write!(f, "Argument Error: {}", m),
            ParseError(m, pos) => match pos {
                Some(pos) => write!(f, "Parse Error at {}: {}", pos, m),
                None => write!(f, "Parse Error: {}", m),
            },
            ScriptError(m, pos) => match pos {
                Some(pos) => write!(f, "Script Error at {}: {}", pos, m),
                None => write!(f, "Script Error: {}", m),
            },
            ControlError(m, v) => write!(f, "Control Error: {}: {:?}", m, v),
            CustomError(v) => write!(f, "Other Error: {}", v.val),
        }
    }
}
#[cfg(feature = "compile")]
impl ser::Error for Error {
    fn custom<T: Display>(msg: T) -> Self
    {
        Error::ScriptError(msg.to_string(), None)
    }
}
#[cfg(feature = "compile")]
impl de::Error for Error {
    fn custom<T: Display>(msg: T) -> Self
    {
        Error::ScriptError(msg.to_string(), None)
    }
}
impl std::error::Error for Error {}

pub struct Args {
    pub longflags: Vec<String>,
    pub files: Vec<String>,
}
impl Args {
    pub fn handle() -> Result<Args, Error> {
        let mut args = Args {
            longflags: vec![],
            files: vec![],
        };

        let eargs: Vec<String> = std::env::args().collect();
        let mut i = 1;
        while i < eargs.len() {
            // Parse single dash flags
            if eargs[i].starts_with('-') {
                // Parse double dash flags
                if eargs[i].starts_with("--") {
                    args.longflags.push(eargs[i][2..].into());
                } else {
                    return Err(Error::ArgumentError(format!("Unsupported arg: {}", eargs[i])))
                }
            } else {
                // All other args are scripts
                args.files.push(eargs[i].clone());
            }

            i += 1;
        }

        Ok(args)
    }
}

#[derive(Clone, PartialEq, Eq)]
pub struct Env<'a> {
    parent: Option<&'a Env<'a>>,
    vars: HashMap<String, TVal>,
}
lazy_static! {
    static ref PRELUDE: Env<'static> = {
        let mut vars: HashMap<String, TVal> = hashmap!{
            "_".into() => Value::Type(Type::any(), HashableMap::arc()).into(),
        };

        // Import Umbra Core
        umcore::_init(&mut vars);

        // Import Umbra STD
        umstd::_init(&mut vars);

        Env {
            parent: None,
            vars,
        }
    };
}
impl<'a> Env<'a> {
    pub fn prelude() -> Env<'a> {
        PRELUDE.clone()
    }
    pub fn empty() -> Env<'a> {
        Env {
            parent: None,
            vars: hashmap!{},
        }
    }
    pub fn child(parent: &'a Env<'a>) -> Env<'a> {
        Env {
            parent: Some(parent),
            vars: hashmap!{},
        }
    }

    pub fn iter_flatten<'b>(it: &'b Env<'a>) -> Box<dyn Iterator<Item = (&'b String, &'b TVal)> + 'b> {
        if let Some (p) = it.parent {
            return Box::new(Self::iter_flatten(p).chain(it.vars.iter()));
        } else {
            return Box::new(it.vars.iter());
        }
    }
    pub fn flatten<'b>(old: &'b Env<'a>) -> Env<'a> {
        Env::from(
            Self::iter_flatten(old)
                .map(|(k, v)| {
                    (k.clone(), v.clone())
                }).collect::<HashMap<String, TVal>>()
        )
    }

    pub fn get(&self, name: &str) -> Option<&TVal> {
        match self.vars.get(name) {
            Some(v) => Some(v),
            None => match &self.parent {
                Some(p) => p.get(name), // Recursively fetch variable values
                None => None,
            },
        }
    }
    pub fn has(&self, name: &str) -> bool {
        match self.vars.contains_key(name) {
            true => true,
            false => match &self.parent {
                Some(p) => p.has(name),
                None => false,
            },
        }
    }
    pub fn set(&mut self, name: &str, tval: TVal) {
        self.vars.insert(name.into(), tval);
    }
    pub fn update(&mut self, vars: HashMap<String, TVal>) {
        self.vars.extend(vars);
    }

    // Computes the changed vars in the nenv that exist in the outside env (self)
    pub fn diff(&self, nenv: &'a mut Env<'a>) -> HashMap<String, TVal> {
        let outer = match self.parent {
            Some(p) => p,
            None => self,
        };
        let mut vars = hashmap!{};
        for (k, v) in nenv.vars.drain() {
            if let Some(ov) = outer.get(&k) {
                if ov != &v {
                    vars.insert(k, v);
                }
            }
        }
        vars
    }
}
impl<'a> From<HashableMap<String, TVal>> for Env<'a> {
    fn from(vars: HashableMap<String, TVal>) -> Env<'a> {
        Env {
            parent: None,
            vars: vars.map,
        }
    }
}
impl<'a> From<HashMap<String, TVal>> for Env<'a> {
    fn from(vars: HashMap<String, TVal>) -> Env<'a> {
        Env {
            parent: None,
            vars,
        }
    }
}
impl<'a> fmt::Debug for Env<'a> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let parent = match self.parent {
            Some(_) => "Some(Env {...})",
            None => "None",
        };
        write!(f, "Env {{ parent: {}, vars: {{{:?}}} }}", parent, self.vars)
    }
}
#[cfg(feature = "compile")]
impl<'a> Serialize for Env<'a> {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer
    {
        let nenv = Env::flatten(self);
        nenv.vars.serialize(serializer)
    }
}
#[cfg(feature = "compile")]
impl<'a, 'de> Deserialize<'de> for Env<'a> {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>
    {
        Ok(Env::from(HashMap::deserialize(deserializer)?))
    }
}

pub fn compile<'a>(scriptname: &str, script: &str, env: &'a Env<'a>, pos: Option<Pos>, printout: usize) -> Result<Runnable<'a>, Error> {
    // Split script string into lines while preserving containers
    let lines: Vec<Line> = Lines::split(scriptname, script, pos)
        // Expression debug print
        .inspect(|l| {
            if printout > 0 && !l.data.starts_with("//") {
                println!("{}", l);
            }
        }).collect();

    // Convert the line into a series of expressions
    let exprs: Vec<Expr> = lines.iter().flat_map(|l| Exprs::split(&l)).collect();

    // Convert the expr into a series of tokens
    let mut tokens: Vec<Tokens> = exprs.iter().map(|e| Tokens::tokenize(&e))
        // Token debug prints
        .inspect(|ts| {
            if printout > 1 {
                print!("    tokens: [");
                for t in ts.clone() {
                    print!("{}, ", t);
                }
                println!("]");
            }
        }).collect();

    // Parse the tokens into an AST
    let asts: Result<Vec<AST>, Error> = tokens.drain(0..).map(|ts| {
            AST::parse(ts, &env)
        }).collect();

    let mut shash = DefaultHasher::new();
    env!("CARGO_PKG_VERSION").hash(&mut shash);
    script.hash(&mut shash);

    Ok(Runnable {
        hash: shash.finish(),
        env: Env::child(env),
        ast: AST::Container {
            token: Token {
                ttype: TokenType::Container,
                pos: Pos {
                    filename: scriptname.into(),
                    line: 0,
                    col: 0,
                },
                data: "{".into(),
            },
            children: asts?,
        },
    })
}
pub fn run<'a>(scriptname: &str, script: &str, env: &'a Env<'a>, pos: Option<Pos>, printout: (usize, bool)) -> FnReturn {
    // Compile the AST then execute it
    let run = match compile(scriptname, script, env, pos, printout.0) {
        Ok(r) => r,
        Err(m) => {
            if printout.1 {
                eprintln!("{}\n", m);
            }
            return (None, Err(m));
        },
    };

    match run.ast {
        AST::Container { children, .. } => {
            let mut nenv = Env::child(env);
            let vals: Result<Vec<TVal>, Error> = children.iter()
                .map(|ast| {

                    let (vars, val) = ast.run(&nenv);
                    if let Some(vars) = vars {
                        nenv.update(vars);
                    }

                    // Expression value debug print
                    if printout.1 {
                        match val {
                            Ok(ref v) => {
                                match ast {
                                    AST::Container { token, .. } if token.data.starts_with("//") => {},
                                    _ => println!("==> {}\n", v),
                                }
                            },
                            Err(ref m) => eprintln!("{}\n", m),
                        }
                    }

                    val
                }).collect();
            match vals {
                Ok(mut vals) => match vals.pop() {
                    Some(v) => (Some(nenv.vars), Ok(v)),
                    None => (None, Ok(Value::none().into())),
                },
                Err(m) => (None, Err(m)),
            }
        },
        _ => (None, Err(Error::ScriptError("expected script container AST".into(), None))),
    }
}
pub fn run_path<'a, P: AsRef<Path>>(path: &P, env: &'a Env<'a>, run_main: bool) -> FnReturn {
    // Correct path for modules
    let path = path.as_ref();
    let path: PathBuf = if path.is_dir() {
        path.join("main.um")
    } else if !path.is_file() {
        path.with_extension("um")
    } else {
        PathBuf::from(path)
    };

    // Read script into string
    let script = match fs::read_to_string(&path) {
        Ok(s) => s,
        Err(m) => return (None, Err(Error::ScriptError(format!("Failed to load {}: {}", path.display(), m), None))),
    };
    // Run script
    let (vars, val) = match run(&path.to_string_lossy(), &script, env, None, (0, false)) {
        (vars, Ok(v)) => (vars, v),
        (vars, Err(m)) => return (vars, Err(m)),
    };

    // Run main if exists and needed
    if run_main {
        if let Some(vars) = &vars {
            if let Some(main) = vars.get("main") {
                if let Value::Function { body, .. } = &main.val {
                    let mut nenv = Env::child(env);
                    nenv.update(vars.clone());
                    let (_, val) = body.call(&nenv, FnArgs::Normal {
                        pos: None,
                        args: TVal {
                            ttype: Type::none(),
                            val: Value::none(),
                        },
                    });
                    return (Some(nenv.vars), val);
                }
            }
        }
    }
    (vars, Ok(val))
}

#[cfg(feature = "readline")]
fn run_interactive_readline<'a>(env: &'a Env<'a>) {
    let mut nenv = Env::child(env);
    let mut pos = Pos::start("<stdin>");
    let mut ed = rl::Editor::<()>::new();
    loop {
        // Read a line from input
        use rl::error::ReadlineError::*;
        match ed.readline(&format!("um:{}> ", pos.line)) {
            Ok(line) => {
                if line.trim().is_empty() {
                    println!();
                    continue;
                }
                ed.add_history_entry(line.clone());

                // Run the line
                let (vars, _) = run("<stdin>", &line, &nenv, Some(pos.clone()), (0, true));
                if let Some(vars) = vars {
                    nenv.update(vars);
                }
                pos.line += 1;
            },
            Err(Interrupted) => {
                println!("^C");
                return
            },
            Err(Eof) => continue,
            Err(m) => {
                println!("Error: {:?}", m);
                continue;
            },
        }
    }
}
pub fn run_interactive<'a>(env: &'a Env<'a>) {
    #[cfg(feature = "readline")]
    return run_interactive_readline(env);

    #[cfg(not(feature = "readline"))]
    {
        let mut nenv = Env::child(env);
        let mut pos = Pos::start("<stdin>");
        let mut line = String::new();
        loop {
            // Read a line from input
            line.clear();
            print!("um:{}> ", pos.line);
            io::stdout().flush().unwrap();
            io::stdin().lock().read_line(&mut line).unwrap();
            if line.trim().is_empty() {
                println!();
                continue;
            }

            // Run the line
            let (vars, _) = run("<stdin>", &line, &nenv, Some(pos.clone()), (0, true));
            if let Some(vars) = vars {
                nenv.update(vars);
            }
            pos.line += 1;
        }
    }
}
