use futures::stream::try_unfold;
use futures::Stream;
use serde::Deserialize;
use std::collections::{HashMap, VecDeque};
use std::error::Error;

const VERSION: &str = "1.0.1";
const USER_AGENT: &str = "OxIonics rust UltraHook client/0.1";
const INIT_URL: &str = "https://www.ultrahook.com/init";
type Result<T> = std::result::Result<T, Box<dyn Error>>;

#[derive(Deserialize)]
pub struct Config {
    /// True in every example that I've seen
    pub success: bool,
    /// The url that's been connected to receive the events
    ///
    /// You don't need to do anything with this.
    pub url: String,
    /// The name of the host that the webhooks need to be sent to
    ///
    /// Any POST requests send to host will be forwarded to this client and
    /// appear as UltraHookEvent::Request entries in the stream returned by
    /// get_webhooks.
    pub host: String,
    /// The UltraHooks namespace that you chose
    pub namespace: String,
}

#[derive(Deserialize, Debug)]
#[serde(tag = "type")]
pub enum UltraHookEvent {
    /// Sent at the start of every connection
    ///
    /// You can probably be confident that any webhooks received by UltraHooks
    /// after you receive this will be forwarded.
    #[serde(alias = "init")]
    Init,
    /// A ping, ignore these
    #[serde(alias = "ping")]
    Ping,
    /// A forwarded webhook
    #[serde(alias = "request")]
    Request {
        path: String,
        body: String,
        query: String,
        headers: HashMap<String, String>,
    },
    #[serde(alias = "error")]
    Error { message: String },
    #[serde(alias = "warning")]
    Warning { message: String },
    #[serde(alias = "message")]
    Message { message: String },
}

struct ResponseState {
    response: reqwest::Response,
    buf: VecDeque<u8>,
    done: bool,
}

fn event_end<'a, T: IntoIterator<Item = &'a u8>>(data: T) -> Option<usize> {
    let mut newlines = 0;
    for (idx, ch) in data.into_iter().enumerate() {
        if *ch == b'\n' {
            newlines += 1;
        } else {
            newlines = 0;
        }
        if newlines == 2 {
            return Some(idx - 1);
        }
    }
    None
}

impl ResponseState {
    async fn fill_buf(&mut self) -> Result<Option<usize>> {
        if self.done {
            return Ok(None);
        }
        loop {
            let chunk = self.response.chunk().await?;
            if let Some(chunk) = chunk {
                let buflen = self.buf.len();
                let end = event_end(&chunk);
                self.buf.extend(chunk);
                if let Some(end) = end {
                    return Ok(Some(buflen + end));
                }
            } else {
                self.done = true;
                return Ok(None);
            }
        }
    }

    async fn next(&mut self) -> Result<Option<UltraHookEvent>> {
        let end = if let Some(idx) = event_end(&self.buf) {
            Some(idx)
        } else {
            self.fill_buf().await?
        };
        let end = match end {
            Some(end) => end,
            None if self.buf.is_empty() => return Ok(None),
            None => self.buf.len(),
        };
        let data = self
            .buf
            .drain(0..end)
            .filter(|c| *c != b'\n')
            .collect::<Vec<_>>();
        self.buf.drain(0..2);
        let raw = base64::decode(data)?;
        Ok(serde_json::from_slice(&raw)?)
    }
}

/// Get the webhooks
///
/// api_key is the api_key you got from https://www.ultrahooks.com
///
/// subdomain is combined with your namespace to create the host name that
/// webhooks are sent to. The result is returned in Config::host.
pub async fn get_webhooks(
    api_key: &str,
    subdomain: &str,
) -> Result<(Config, impl Stream<Item = Result<UltraHookEvent>>)> {
    let client = reqwest::Client::builder().user_agent(USER_AGENT).build()?;
    let request = client
        .post(INIT_URL)
        .form(&[("key", api_key), ("host", subdomain), ("version", VERSION)])
        .build()?;
    let config: Config = client.execute(request).await?.json().await?;
    let response = client.execute(client.get(&config.url).build()?).await?;
    Ok((
        config,
        try_unfold(
            ResponseState {
                response,
                buf: VecDeque::new(),
                done: false,
            },
            |mut state| async move {
                let result = state.next().await?;
                Ok(result.map(|result| (result, state)))
            },
        ),
    ))
}
