use reqwest::blocking::multipart;
use std::path::Path;

use std::cmp::min;
use std::fs::File;
use std::io::Write;
use tokio;

use futures_util::StreamExt;
use indicatif::{ProgressBar, ProgressStyle};
use reqwest::Client;

async fn download_file(client: &Client, url: &str, path: &str) -> Result<(), String> {
    // Download file from <url> to <path>
    let res = client
        .get(url)
        .send()
        .await
        .or(Err(format!("Failed to GET from '{}'", &url)))?;
    let total_size = res
        .content_length()
        .ok_or(format!("Failed to get content length from '{}'", &url))?;

    // Indicatif setup
    let pb = ProgressBar::new(total_size);
    pb.set_style(ProgressStyle::default_bar()
        .template("{msg}\n{spinner:.green} [{elapsed_precise}] [{wide_bar:.cyan/blue}] {bytes}/{total_bytes} ({bytes_per_sec}, {eta})")
        .progress_chars("#>-"));
    pb.set_message(&format!("Downloading {} to {}", url, path));

    // download chunks
    let mut file = File::create(path).or(Err(format!("Failed to create file '{}'", path)))?;
    let mut downloaded: u64 = 0;
    let mut stream = res.bytes_stream();

    while let Some(item) = stream.next().await {
        let chunk = item.or(Err(format!("Error while downloading file")))?;
        file.write(&chunk)
            .or(Err(format!("Error while writing to file")))?;
        let new = min(downloaded + (chunk.len() as u64), total_size);
        downloaded = new;
        pb.set_position(new);
    }

    pb.finish_with_message(&format!("Downloaded {} to {}", url, path));
    Ok(())
}

pub fn download(url: &str, path: &str) {
    let client = reqwest::Client::new();
    let rt = tokio::runtime::Runtime::new().unwrap();
    let future = download_file(&client, url, path);
    let resp = rt.block_on(future);
    match resp {
        Err(e) => println!("{:?}", e),
        _ => (),
    }
}

pub fn upload2olvps(filename: impl AsRef<Path> + std::fmt::Debug) {
    println!("Start uploading file {:?}", &filename);
    let client = reqwest::blocking::Client::new();
    let form = multipart::Form::new()
        .text("upload_folder", "/usr/share/nginx/html/files/")
        .file("file", &filename)
        .unwrap();

    let res = client
        .post("https://a.ddot.cc/upload")
        .multipart(form)
        .send()
        .unwrap();
    match &res.text() {
        Ok(_) => println!("Upload success: {:?}.", filename),
        _ => println!("Upload failed: {:?}", filename),
    }
}
