use super::lexer::{Lexer, LexicalError, Token};
use super::Node;

/// These are "stage 2 tokens", which only distinguish between the parts of the expression relevant
/// for the parser below. See collapse_tokens for the conversion.
#[derive(Debug, PartialEq, Copy, Clone)]
pub enum TokenS2 {
    Text,
    LBrace,
    RBrace,
    Equals,
    Newline,
    Comma,
}

/// Create stage 2 tokens from tokens generated by the lexer. Here, all structured expressions
/// (using `()`, `[]` or `<>`) will be collapsed into a single text token that can be handled much
/// more easilty by the parser.
fn collapse_tokens(
    tokens: impl Iterator<Item = (usize, Token, usize)>,
) -> Vec<(usize, TokenS2, usize)> {
    enum State {
        Free,
        SimpleText(usize, usize),
        StructuredText(usize, usize, Token, usize),
    }

    let mut ret = Vec::new();
    let mut state = State::Free;

    for (begin, tok, end) in tokens {
        state = match state {
            State::Free => match tok {
                Token::String
                | Token::Text
                | Token::RSquareBracket
                | Token::RBracket
                | Token::RPointyBracket => State::SimpleText(begin, end),
                Token::LSquareBracket => {
                    State::StructuredText(begin, end, Token::RSquareBracket, 1)
                }
                Token::LBracket => State::StructuredText(begin, end, Token::RBracket, 1),
                Token::LPointyBracket => {
                    State::StructuredText(begin, end, Token::RPointyBracket, 1)
                }
                Token::LBrace => {
                    ret.push((begin, TokenS2::LBrace, end));
                    State::Free
                }
                Token::RBrace => {
                    ret.push((begin, TokenS2::RBrace, end));
                    State::Free
                }
                Token::Equals => {
                    ret.push((begin, TokenS2::Equals, end));
                    State::Free
                }
                Token::Newline => {
                    ret.push((begin, TokenS2::Newline, end));
                    State::Free
                }
                Token::Comma => {
                    ret.push((begin, TokenS2::Comma, end));
                    State::Free
                }
            },
            State::SimpleText(b, e) => match tok {
                Token::String
                | Token::Text
                | Token::RSquareBracket
                | Token::RBracket
                | Token::RPointyBracket => State::SimpleText(b, end),
                Token::LSquareBracket => State::StructuredText(b, end, Token::RSquareBracket, 1),
                Token::LBracket => State::StructuredText(b, end, Token::RBracket, 1),
                Token::LPointyBracket => State::StructuredText(b, end, Token::RPointyBracket, 1),
                Token::LBrace => {
                    ret.push((b, TokenS2::Text, e));
                    ret.push((begin, TokenS2::LBrace, end));
                    State::Free
                }
                Token::RBrace => {
                    ret.push((b, TokenS2::Text, e));
                    ret.push((begin, TokenS2::RBrace, end));
                    State::Free
                }
                Token::Equals => {
                    ret.push((b, TokenS2::Text, e));
                    ret.push((begin, TokenS2::Equals, end));
                    State::Free
                }
                Token::Newline => {
                    ret.push((b, TokenS2::Text, e));
                    ret.push((begin, TokenS2::Newline, end));
                    State::Free
                }
                Token::Comma => {
                    ret.push((b, TokenS2::Text, e));
                    ret.push((begin, TokenS2::Comma, end));
                    State::Free
                }
            },
            State::StructuredText(b, _, endtoken, depth) => {
                if tok == endtoken {
                    if depth == 1 {
                        State::SimpleText(b, end)
                    } else {
                        State::StructuredText(b, end, endtoken, depth - 1)
                    }
                } else {
                    State::StructuredText(b, end, endtoken, depth)
                }
            }
        }
    }
    match state {
        State::Free => {}
        State::SimpleText(b, e) | State::StructuredText(b, e, _, _) => {
            ret.push((b, TokenS2::Text, e));
        }
    }
    ret
}

enum Struct<'s> {
    Array(Vec<Node<'s>>),
    Map(Vec<(&'s str, Node<'s>)>),
}

fn parse_structure<'s>(tokens: &mut Tokens, string: &'s str) -> Result<Struct<'s>, ()> {
    let mut kv = Vec::new();
    let mut single = Vec::new();
    let multiline = if let &[(_, TokenS2::Newline, _), ..] = tokens.tokens {
        tokens.consume(1);
        true
    } else {
        false
    };
    match tokens.tokens {
        &[(_, TokenS2::RBrace, _), ..] => {
            tokens.consume(1);
            return Ok(Struct::Array(Vec::new()));
        }
        _ => {}
    };
    loop {
        let key = match tokens.tokens {
            &[(b, TokenS2::Text, e), (_, TokenS2::Equals, _), ..] => {
                let res = Some((b, e));
                tokens.consume(2);
                res
            }
            _ => None,
        };
        let val = parse_value(
            tokens,
            string,
            if multiline {
                ValueContext::MultiLineStruct
            } else {
                ValueContext::SingleLineStruct
            },
        );
        if let Some((b, e)) = key {
            kv.push((&string[b..e], val));
        } else {
            single.push(val);
        }
        match (multiline, tokens.tokens) {
            (true, &[(_, TokenS2::Comma, _), (_, TokenS2::Newline, _), ..]) => {
                tokens.consume(2);
            }
            (false, &[(_, TokenS2::Comma, _), ..]) => {
                tokens.consume(1);
            }
            (false, &[(_, TokenS2::RBrace, _), ..])
            | (true, &[(_, TokenS2::Newline, _), (_, TokenS2::RBrace, _), ..]) => {
                tokens.consume(if multiline { 2 } else { 1 });
                return Ok(match (kv.is_empty(), single.len()) {
                    (true, 0) => Struct::Array(Vec::new()),
                    (true, _) => Struct::Array(single),
                    (false, 0) => Struct::Map(kv),
                    (false, 1) => {
                        kv.push((super::ANON_KEY, single.drain(..).next().unwrap()));
                        Struct::Map(kv)
                    }
                    (false, _) => {
                        kv.push((super::ANON_KEY, Node::Array(None, single)));
                        Struct::Map(kv)
                    }
                });
            }
            _ => {
                return Err(());
            }
        }
    }
}

struct Tokens<'a> {
    tokens: &'a [(usize, TokenS2, usize)],
}

impl Tokens<'_> {
    fn consume(&mut self, n: usize) {
        self.tokens = &self.tokens[n..];
    }
}

#[derive(Copy, Clone)]
enum ValueContext {
    Free,
    SingleLineStruct,
    MultiLineStruct,
}

pub type Error = LexicalError;

// TODO: Use or-patterns starting from 1.53?
fn parse_value<'s>(tokens: &mut Tokens, string: &'s str, context: ValueContext) -> Node<'s> {
    let mut text_begin = None;
    let mut text_end = 0;
    let update_range =
        |text_begin: &mut Option<usize>, text_end: &mut usize, b: usize, e: usize| {
            *text_begin = Some(text_begin.unwrap_or(b));
            *text_end = (*text_end).max(e);
        };
    let mut current_struct = None;
    loop {
        match (context, tokens.tokens) {
            (ValueContext::SingleLineStruct, &[(_, TokenS2::Comma, _), ..])
            | (
                ValueContext::MultiLineStruct,
                &[(_, TokenS2::Comma, _), (_, TokenS2::Newline, _), ..],
            )
            | (ValueContext::SingleLineStruct, &[(_, TokenS2::RBrace, _), ..])
            | (ValueContext::MultiLineStruct, &[(_, TokenS2::RBrace, _), ..])
            | (
                ValueContext::MultiLineStruct,
                &[(_, TokenS2::Newline, _), (_, TokenS2::RBrace, _), ..],
            )
            | (_, &[]) => {
                let text_begin = text_begin.unwrap_or(text_end);
                let text = &string[text_begin..text_end];
                let description = if text.is_empty() { None } else { Some(text) };
                return match current_struct {
                    None => Node::Leaf(text),
                    Some(Struct::Map(v)) => Node::Map(description, v),
                    Some(Struct::Array(v)) => Node::Array(description, v),
                };
            }
            (_, &[(b, TokenS2::Text, e), ..])
            | (_, &[(b, TokenS2::Equals, e), ..])
            | (ValueContext::MultiLineStruct, &[(b, TokenS2::Comma, e), ..])
            | (ValueContext::Free, &[(b, TokenS2::Comma, e), ..])
            | (ValueContext::Free, &[(b, TokenS2::RBrace, e), ..]) => {
                update_range(&mut text_begin, &mut text_end, b, e);
                current_struct = None;
                tokens.consume(1);
            }
            (_, &[(_, TokenS2::Newline, _), ..]) => {
                tokens.consume(1);
            }
            (_, &[(b, TokenS2::LBrace, _), ..]) => {
                if text_begin.is_none() {
                    text_begin = Some(b);
                    text_end = b;
                }
                tokens.consume(1);
                if let Ok(strct) = parse_structure(tokens, string) {
                    current_struct = Some(strct);
                }
            }
        }
    }
}

pub fn parse<'s>(lexer: Lexer, string: &'s str) -> Result<Node<'s>, Error> {
    let tokens = lexer.into_iter().collect::<Result<Vec<_>, _>>()?;
    let tokens = collapse_tokens(tokens.into_iter());
    let mut tokens = Tokens {
        tokens: &tokens[..],
    };
    Ok(parse_value(&mut tokens, string, ValueContext::Free))
}

#[cfg(test)]
mod test {
    use super::super::lexer::Token;
    use super::*;

    #[test]
    fn collapse_tokens_simple() {
        assert_eq!(
            collapse_tokens(
                vec![
                    (0, Token::Text, 1),
                    (1, Token::String, 4),
                    (4, Token::Text, 5)
                ]
                .into_iter()
            ),
            vec![(0, TokenS2::Text, 5)]
        );

        assert_eq!(
            collapse_tokens(
                vec![
                    (0, Token::Text, 1),
                    (1, Token::String, 4),
                    (4, Token::Comma, 5)
                ]
                .into_iter()
            ),
            vec![(0, TokenS2::Text, 4), (4, TokenS2::Comma, 5)]
        );
    }

    #[test]
    fn collapse_tokens_structured() {
        assert_eq!(
            collapse_tokens(
                vec![
                    (0, Token::Text, 1),
                    (1, Token::LSquareBracket, 4),
                    (4, Token::Text, 5),
                    (5, Token::RBrace, 6),
                    (6, Token::String, 7),
                    (9, Token::RSquareBracket, 10),
                    (10, Token::String, 11),
                ]
                .into_iter()
            ),
            vec![(0, TokenS2::Text, 11)]
        );
    }
}
