#![no_std]
pub mod register {
    /// Writes 8bit value to register
    ///
    /// # Arguments
    /// * `register_address` - address of the register found in the datasheet
    /// * `value` - 8bit byte
    ///
    /// # Examples
    /// ``` rust
    /// // Set TIMSK0 – Timer/Counter Interrupt Mask Register
	/// register::write(0x6E, 0b0000_0111)
    /// // or with hex
	/// register::write(0x6E, 0x07)
    /// // or by using integers in range 0 - 255
    /// register::write(0x6E, 7)
    /// ```
	pub fn write(register_address: u8, value: u8) {
		unsafe {
			core::ptr::write_volatile(register_address as *mut u8, value);
		}
	}
    /// Reads 8bit register
    ///
    /// # Arguments
    /// * `register_address` - address of the register found in the datasheet
    ///
    /// # Returns u8 panics otherwise
    ///
    /// # Examples
    /// ``` rust
    /// // Read TIMSK0 – Timer/Counter Interrupt Mask Register (atmega328p datasheet in example)
	/// let timsk0: u8 = register::read(0x6E);
    /// ```
	pub fn read(register_address: u8) -> u8 {
		unsafe {
			core::ptr::read_volatile(register_address as *mut u8)
		}
	}
}

pub mod bit {
    /// Reads bit from u8 type value (not the register directly)
    ///
    /// # Arguments
    /// * `position` - position of the bit in range  0..=7
    /// * `byte`     - byte to be read from
    /// # Returns u8 (either 0 or 1)
    ///
    /// # Examples
    /// ``` rust
    /// bit::read(0, 0b0000_0001) // returns 1
    /// bit::read(0, 0b1111_1110) // returns 0
    /// ```
	pub fn read(position: u8, byte: u8) -> u8 {
		(byte >> position) & 0x01
	}

    /// Sets bit in u8 type
    ///
    /// # Arguments
    /// * `position` - position of the bit in range  0..=7
    /// * `byte`     - byte value to be written to
    /// # Returns u8
    ///
    /// # Examples
    /// ``` rust
	/// bit::set(0, 0b0000_0000) returns 0b0000_0001
    /// ```
	pub fn set(position: u8, byte: u8) -> u8 {
		byte | (1 << position)
	}

	/// Unsets bit in u8 type
    ///
    /// # Arguments
    /// * `position` - position of the bit in range  0..=7
    /// * `byte`     - bye to be read from
    /// # Returns u8
    ///
    /// # Examples
    /// ``` rust
	/// bit::unset(0, 0b0000_0001) returns 0b0000_0000
    /// ```
	pub fn unset(position: u8, byte: u8) -> u8 {
		byte | (0 << position)
	}

    /// Toggle bit in u8 type
    ///
    /// # Arguments
    /// * `position` - position of the bit in range  0..=7
    /// * `byte`     - byte to be read from
    /// # Returns u8
    ///
    /// # Examples
    /// ``` rust
	/// bit::toggle(0, 0b0000_0000) // returns 0b0000_0001
    /// ```
	pub fn toggle(position: u8, byte: u8) -> u8 {
		byte ^ (1 << position)
	}
}


pub mod convert {
	use super::bit;

    /// Maps bit values into '0' or '1'
    ///
    /// # Arguments
    /// * `position` - position of the bit in range  0..=7
    /// * `byte`     - byte to be read from
    /// # Returns char
    ///
    /// # Examples
    /// ``` rust
    /// let bit_0: char = ueight::convert::bit_to_char(0, 0b0000_0001);
    /// ```
	pub fn bit_to_char(position: u8, byte: u8) -> char {
		match bit::read(position, byte) as u8 {
			0 => '0',
			1 => '1',
			_ => '\0'
		}
	}

    /// Maps whole byte into array of chars of '0' and '1'
    ///
    /// # Arguments
    /// * `byte`     - byte to be read from
    /// # Returns char array of length 8
    ///
    /// # Examples
    /// ``` rust
    /// let char_array: [char; 8] = convert::byte_to_char_array(0b0000_1111);
    /// // char_aaray contains ['0','0','0','0','1','1','1','1']
    /// ```
	pub fn byte_to_char_array(byte: u8) -> [char; 8] {
		let mut char_array: [char ; 8] = ['0'; 8];
		for position in 0..8 {
			char_array[position] = bit_to_char(position as u8, byte);
		}

		char_array
	}

    /// Represents byte in from 0bxxxx_xxxx
      ///
    /// # Arguments
    /// * `byte`     - byte to be read from
    /// # Returns char array of length 10
    ///
    /// # Examples
    /// ``` rust
    /// let repr = conver::byte_repr(0b0000_1111, false);
    /// // repr contains ['0','0','0','0','1','1','1','1','b','0']
    /// let repr = conver::byte_repr(0b0000_1111, true);
    /// // repr contains ['1','1','1','1','0','0','0','0','b','0']
    /// ```
	pub fn byte_repr(byte: u8, reverse: bool) -> [char; 10] {
		let mut repr: [char; 10]= ['0'; 10];
		let char_array: [char; 8] = byte_to_char_array(byte);

		repr[0] = '0';
		repr[1] = 'b';

        if reverse == false {
            for i in 2..=9 {
                repr[i] = char_array[9 - i];
            }
        }
        else {
            for i in 2..=9 {
                repr[i] = char_array[i - 2];
            }
        }

		repr
	}
}


#[cfg(test)]
mod tests {
    use super::bit;
    use super::convert;

    #[test]
    fn test_bit_read() {
        let byte: u8 = 0b1000_0001;
        assert_eq!(bit::read(0, byte), 1);
        assert_eq!(bit::read(1, byte), 0);
        assert_eq!(bit::read(7, byte), 1);
    }

    #[test]
    fn test_bit_set() {
        let mut byte: u8 = 0b0000_0000;

        byte = bit::set(0, byte);
        assert_eq!(bit::read(0, byte), 1);

        byte = bit::set(7, byte);
        assert_eq!(bit::read(7, byte), 1);
    }

    #[test]
    fn test_bit_toggle() {
        let mut byte: u8 = 0b0000_0000;

        byte = bit::toggle(0, byte);
        assert_eq!(bit::read(0, byte), 1);

        byte = bit::toggle(7, byte);
        assert_eq!(bit::read(7, byte), 1);

        byte = bit::toggle(0, byte);
        assert_eq!(bit::read(0, byte), 0);

        byte = bit::toggle(7, byte);
        assert_eq!(bit::read(7, byte), 0);
    }

    #[test]
    fn test_convert_bit_to_char() {
        let byte: u8 = 0b0000_0001;
        let mut bit: char = convert::bit_to_char(0, byte);

        assert_eq!(bit, '1');

        bit = convert::bit_to_char(1, byte);

        assert_eq!(bit, '0');
    }

    #[test]
    fn test_convert_byte_to_char_array() {
        let byte: u8 = 0b0101_0101;
        let char_array: [char; 8] = convert::byte_to_char_array(byte);

        assert_eq!(char_array[0], '1');
        assert_eq!(char_array[1], '0');
        assert_eq!(char_array[2], '1');
        assert_eq!(char_array[3], '0');
        assert_eq!(char_array[4], '1');
    }

    #[test]
    fn test_convert_byte_repr() {
        let byte: u8 = 0b0101_0101;

        assert_eq!(convert::byte_repr(byte, false), ['0','b','0','1','0','1','0','1','0','1']);
        assert_eq!(convert::byte_repr(byte, true), ['0','b','1','0','1','0','1','0','1','0']);
    }
}
