#![no_std]

pub mod register {
	/// register::write(0xCA, 0b0000_1111)
	pub fn write(register_address: u8, value: u8) {
		unsafe {
			core::ptr::write_volatile(register_address as *mut u8, value);
		}
	}
	/// register::read(0xCA)
	pub fn read(register_address: u8) -> u8 {
		unsafe {
			core::ptr::read_volatile(register_address as *mut u8)
		}
	}

	/// register::read_bit(0, 0b0000_0001) returns 1
	pub fn read_bit(position: u8, byte: u8) -> u8 {
		(byte >> position) & 0x01
	}

	/// register::set_bit(0, 0b0000_0000) returns 0b0000_0001
	pub fn set_bit(position: u8, byte: u8) -> u8 {
		byte | (1 << position)
	}

	/// register::unset_bit(0, 0b0000_0001) returns 0b0000_0000
	pub fn unset_bit(position: u8, byte: u8) -> u8 {
		byte | (0 << position)
	}

	/// register::toggle_bit(0, 0b0000_0000) returns 0b0000_0001
	pub fn toggle_bit(position: u8, byte: u8) -> u8 {
		byte ^ (1 << position)
	}
}


pub mod convert {
	use super::register;

    /// Maps bit values from byte to '0' or '1'
	pub fn bit_to_char(position: u8, byte: u8) -> char {
		match register::read_bit(position, byte) as u8 {
			0 => '0',
			1 => '1',
			_ => '\0'
		}
	}

    /// Maps whole byte into array of chars of '0' and '1'
	pub fn byte_to_char_array(byte: u8) -> [char; 8] {
		let mut char_array: [char ; 8] = ['0'; 8];
		for position in 0..8 {
			char_array[position] = bit_to_char(position as u8, byte);
		}

		char_array
	}

    /// Represents byte in from 0bxxxx_xxxx
	pub fn byte_repr(byte: u8) -> [char; 10] {
		let mut repr: [char; 10]= ['0'; 10];
		let char_array: [char; 8] = byte_to_char_array(byte);

		repr[0] = '0';
		repr[1] = 'b';

		for i in 2..=9 {
			repr[i] = char_array[9 - i];
		}

		repr
	}
}


#[cfg(test)]
mod tests {
    use super::register;
    use super::convert;

    #[test]
    fn test_register_read_bit() {
        let byte: u8 = 0b1000_0001;
        assert_eq!(register::read_bit(0, byte), 1);
        assert_eq!(register::read_bit(1, byte), 0);
        assert_eq!(register::read_bit(7, byte), 1);
    }

    #[test]
    fn test_register_set_bit() {
        let mut byte: u8 = 0b0000_0000;

        byte = register::set_bit(0, byte);
        assert_eq!(register::read_bit(0, byte), 1);

        byte = register::set_bit(7, byte);
        assert_eq!(register::read_bit(7, byte), 1);
    }

    #[test]
    fn test_register_toggle_bit() {
        let mut byte: u8 = 0b0000_0000;

        byte = register::toggle_bit(0, byte);
        assert_eq!(register::read_bit(0, byte), 1);

        byte = register::toggle_bit(7, byte);
        assert_eq!(register::read_bit(7, byte), 1);

        byte = register::toggle_bit(0, byte);
        assert_eq!(register::read_bit(0, byte), 0);

        byte = register::toggle_bit(7, byte);
        assert_eq!(register::read_bit(7, byte), 0);
    }

    #[test]
    fn test_convert_bit_to_char() {
        let byte: u8 = 0b0000_0001;
        let mut bit: char = convert::bit_to_char(0, byte);

        assert_eq!(bit, '1');

        bit = convert::bit_to_char(1, byte);

        assert_eq!(bit, '0');
    }

    #[test]
    fn test_convert_byte_to_char_array() {
        let byte: u8 = 0b0101_0101;
        let char_array: [char; 8] = convert::byte_to_char_array(byte);

        assert_eq!(char_array[0], '1');
        assert_eq!(char_array[1], '0');
        assert_eq!(char_array[2], '1');
        assert_eq!(char_array[3], '0');
        assert_eq!(char_array[4], '1');
    }

    #[test]
    fn test_convert_byte_repr() {
        let byte: u8 = 0b0101_0101;

        assert_eq!(convert::byte_repr(byte), ['0','b','0','1','0','1','0','1','0','1']);
    }
}
