use core::fmt::Debug;
use core::hash::Hash;
use alloc::borrow::Cow;
use alloc::vec::Vec;
use core::slice;
use nom::bytes::streaming::{tag, take_while, take};
use nom::combinator::{peek, opt};
use nom::branch::alt;
use nom::IResult;

pub const START_OF_HEADING: u8 = 0x01;
pub const START_OF_TEXT: u8 = 0x02;
pub const END_OF_TEXT: u8 = 0x03;
pub const ESCAPE: u8 = 0x1B;
pub const RECORD_SEPARATOR: u8 = 0x1E;
pub const UNIT_SEPARATOR: u8 = 0x1F;

#[derive(Debug)]
pub struct Unit<'a>(pub Cow<'a, [u8]>);

#[derive(Debug)]
pub struct Units<'a>(pub Vec<Unit<'a>>);

#[derive(Debug)]
pub struct Header<'a>(pub Units<'a>);

#[derive(Debug)]
pub struct Record<'a>(pub Units<'a>);

#[derive(Debug)]
pub struct Message<'a>{
    pub header: Option<Header<'a>>,
    pub records: Vec<Units<'a>>,
}

#[derive(Debug)]
pub struct Stream<'a>(pub Vec<Message<'a>>);

#[derive(Debug, Copy, Clone, PartialEq, Eq, Hash)]
pub struct Parser<T>
where T: Debug + Copy + Clone + PartialEq + Eq + Hash
{
    pub start_header: T,
    pub start_message: T,
    pub start_record: T,
    pub start_unit: T,
    pub escape: T,
    pub end_message: T,
    pub end_stream: T,
}

/// Return a default reference parser for binary data, using C0 control codes.
impl Default for Parser<u8> {
    fn default() -> Self {
        Parser {
            start_header: 0x01,
            start_message: 0x02,
            start_record: 0x1E,
            start_unit: 0x1F,
            escape: 0x1B,
            end_message: 0x03,
            end_stream: 0x04,
        }
    }
}

/// Return a default reference parser for text data.
impl Default for Parser<char> {
    fn default() -> Self {
        Parser {
            start_header: '#',
            start_message: '>',
            start_record: '\n',
            start_unit: ',',
            escape: '\\',
            end_message: '<',
            end_stream: '!',
        }
    }
}

impl Parser<char> {
    /// A text parser using c0 control sequences.
    pub fn new_c0_text() -> Self{
        Parser {
            start_header: '\u{01}',
            start_message: '\u{02}',
            start_record: '\u{1E}',
            start_unit: '\u{1F}',
            escape: '\u{1B}',
            end_message: '\u{03}',
            end_stream: '\u{04}',
        }
    }
}
