//! Implement UCAN-based authorization with conciseness and ease!
//!
//! [UCANs][UCAN docs] are an emerging pattern based on
//! [JSON Web Tokens][JWT docs] (aka JWTs) that facilitate distributed and/or
//! decentralized authorization flows in web applications. Visit
//! [https://ucan.xyz][UCAN docs] for an introduction to UCANs and ideas for
//! how you can use them in your application.
//!
//! # Examples
//!
//! This crate offers the [`builder::UcanBuilder`] abstraction to generate
//! signed UCAN tokens.
//!
//! To generate a signed token, you need to provide a [`crypto::SigningKey`]
//! implementation. For more information on providing a signing key, see the
//! [`crypto`] module documentation.
//!
//! ```rust
//! use ucan::{
//!   builder::UcanBuilder,
//!   crypto::SigningKey,
//! };
//!
//! fn generate_token<'a, K: SigningKey>(issuer_key: &'a K, audience_did: &'a str) -> Result<String, anyhow::Error> {
//!     UcanBuilder::new()
//!       .issued_by(issuer_key)
//!       .for_audience(audience_did)
//!       .with_lifetime(60)
//!       .build()?
//!       .sign()?
//!       .encode()
//! }
//! ```
//!
//! The crate also offers a validating parser to interpret UCAN tokens and
//! the capabilities they grant via their issuer and/or witnessing proofs.
//!
//! Most capabilities are closely tied to a specific application domain. See the
//! [`capability`] module documentation to read more about defining your own
//! domain-specific semantics.
//!
//! ```rust
//! use ucan::{
//!   chain::{ProofChain, CapabilityInfo},
//!   capability::{CapabilitySemantics, Scope, Action},
//!   crypto::did::{DidParser, KeyConstructorSlice}
//! };
//!
//! const SUPPORTED_KEY_TYPES: &KeyConstructorSlice = &[
//!     // You must bring your own key support
//! ];
//!
//! fn get_capabilities<'a, Semantics, S, A>(ucan_token: &'a str, semantics: &'a Semantics) -> Result<Vec<CapabilityInfo<S, A>>, anyhow::Error>
//!     where
//!         Semantics: CapabilitySemantics<S, A>,
//!         S: Scope,
//!         A: Action
//! {
//!     let did_parser = DidParser::new(SUPPORTED_KEY_TYPES);
//!
//!     Ok(ProofChain::try_from_token_string(ucan_token, &did_parser)?
//!         .reduce_capabilities(semantics))
//! }
//! ```
//!
//! [JWT docs]: https://jwt.io/
//! [UCAN docs]: https://ucan.xyz/
//! [DID spec]: https://www.w3.org/TR/did-core/
//! [DID Key spec]: https://w3c-ccg.github.io/did-method-key/

#[macro_use]
extern crate log;

pub mod crypto;
pub mod time;

pub mod builder;
pub mod capability;
pub mod chain;
pub mod ucan;

#[cfg(test)]
mod tests;
