use core::str::FromStr;

#[derive(Debug, PartialEq, defmt::Format)]
pub enum UbloxError {
    Generic,
    Cme(CmeError),
    Cms(CmsError),
}

impl From<atat::GenericError> for UbloxError {
    fn from(_: atat::GenericError) -> Self {
        Self::Generic
    }
}

impl FromStr for UbloxError {
    // This error will always get mapped to `atat::Error::Parse`
    type Err = ();

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        Ok(if let Some(err) = s.trim().strip_prefix("+CME ERROR:") {
            Self::Cme(err.parse().unwrap_or(CmeError::Unknown))
        } else if let Some(err) = s.trim().strip_prefix("+CMS ERROR:") {
            Self::Cms(err.parse().unwrap_or(CmsError::Unknown))
        } else {
            Self::Generic
        })
    }
}

/// Message service error result codes +CMS ERROR
#[derive(Debug, PartialEq, defmt::Format)]
pub enum CmsError {
    Empty,
    Unknown,
}

impl FromStr for CmsError {
    // This error will always get mapped to `atat::Error::Parse`
    type Err = ();

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s.trim() {
            _ => return Err(()),
        }
    }
}

/// Mobile termination error result codes +CME ERROR
#[derive(Debug, PartialEq, defmt::Format)]
pub enum CmeError {
    PhoneFailure,
    NoConnectionToPhone,
    PhoneAdaptorLinkReserved,
    OperationNotAllowed,
    OperationNotSupported,
    PhSimPinRequired,
    PhFsimPinRequired,
    PhFsimPukRequired,
    SimNotInserted,
    SimPinRequired,
    SimPukRequired,
    SimFailure,
    SimBusy,
    SimWrong,
    IncorrectPassword,
    SimPin2Required,
    SimPuk2Required,
    MemoryFull,
    InvalidIndex,
    NetworkNotFound,
    MemoryFailure,
    TextStringTooLong,
    InvalidCharactersInTextString,
    DialStringTooLong,
    InvalidCharactersInDialString,
    NoNetworkService,
    NetworkTimeout,
    NetworkNotAllowedEmergencyCallsOnly,
    NetworkPersonalisationPinRequired,
    NetworkPersonalisationPukRequired,
    NetworkSubsetPersonalisationPinRequired,
    NetworkSubsetPersonalisationPukRequired,
    ServiceProviderPersonalisationPinRequired,
    ServiceProviderPersonalisationPukRequired,
    CorporatePersonalisationPinRequired,
    CorporatePersonalisationPukRequired,
    IncorrectParameters,
    CommandImplementedButCurrentlyDisabled,
    CommandAbortedByUser,
    NotAttachedToNetworkDueToMtFunctionalityRestrictions,
    ModemNotAllowedMtRestrictedToEmergencyCallsOnly,
    OperationNotAllowedBecauseOfMtFunctionalityRestrictions,
    FixedDialNumberOnlyAllowedCalledNumberIsNotAFixedDialNumber,
    TemporarilyOutOfServiceDueToOtherMtUsage,
    Unknown,
    IllegalMs,
    IllegalMe,
    GprsServicesNotAllowed,
    GprsAndNonGprsServicesNotAllowed,
    PlmnNotAllowed,
    LocationAreaNotAllowed,
    RoamingNotAllowedInThisLocationArea,
    GprsServicesNotAllowedInThisPlmn,
    NoSuitableCellsInLocationArea,
    Congestion,
    NotAuthorizedForThisCsg,
    InsufficientResources,
    MissingOrUnknownApn,
    UnknownPdpAddressOrPdpType,
    UserAuthenticationFailed,
    RequestRejectedByServingGwOrPdnGw,
    RequestRejectedUnspecified,
    ServiceOptionNotSupported,
    RequestedServiceOptionNotSubscribed,
    ServiceOptionTemporarilyOutOfOrder,
    NsApiAlreadyUsed,
    EpsQoSNotAccepted,
    NetworkFailure,
    FeatureNotSupported,
    SemanticErrorInTheTftOperation,
    SyntacticalErrorInTheTftOperation,
    UnknownPdpContext,
    SemanticErrorsInPacketFilterS,
    SyntacticalErrorsInPacketFilterS,
    PdpContextWithoutTftAlreadyActivated,
    PtiMismatch,
    UnspecifiedGprsError,
    PdpAuthenticationFailure,
    InvalidMobileClass,
    EsmInformationNotReceived,
    PdnConnectionDoesNotExist,
    MultiplePdnConnectionsForAGivenApnNotAllowed,
    UserBusy,
    UplinkBusyFlowControl,
    BearerHandlingNotSupported,
    MaximumNumberOfEpsBearersReached,
    RequestedApnNotSupportedInCurrentRatAndPlmnCombination,
    ImsiUnknownInVlr,
    LastPdnDisconnectionNotAllowed,
    SemanticallyIncorrectMessage,
    MandatoryInformationElementError,
    InformationElementNonExistentOrNotImplemented,
    ConditionalIeError,
    ProtocolErrorUnspecified,
    OperatorDeterminedBarring,
    MaximumNumberOfPdpContextsReached,
    RequestRejectedBearerControlModeViolation,
    InvalidPtiValue,
    InvalidMandatoryIe,
    MessageTypeNonExistent,
    MessageTypeNotCompatible,
    IeNonExistent,
    MessageNotCompatible,
    InvalidErrorMapping,
    InternalError,
    SimBlocked,
    MeFailure,
    SmsServiceOfMeReserved,
    InvalidPduModeParameter,
    InvalidTextModeParameter,
    USimNotInserted,
    USimPinRequired,
    PhUSimPinRequired,
    USimFailure,
    USimBusy,
    USimWrong,
    USimPukRequired,
    USimPin2Required,
    USimPuk2Required,
    InvalidMemoryIndex,
    // SmscAddressUnknown,
    // NoCnmaAcknowledgementExpected,
    // UnknownError,
    // VoiceCallActive,
    // IncorrectSecurityCode,
    // MaxAttemptsReached,
    // UnassignedUnallocatedNumber,
    // NoRouteToDestination,
    // ChannelUnacceptable,
    // NormalCallClearing,
    // NoUserResponding,
    // UserAlertingNoAnswer,
    // CallRejected,
    // NumberChanged,
    // NonSelectedUserClearing,
    // DestinationOutOfOrder,
    // InvalidNumberFormatIncompleteNumber,
    // FacilityRejected,
    // ResponseToStatusEnquiry,
    // NormalUnspecified,
    // NoCircuitChannelAvailable,
    // NetworkOutOfOrder,
    // TemporaryFailure,
    // SwitchingEquipmentCongestion,
    // AccessInformationDiscarded,
    // RequestedCircuitChannelNotAvailable,
    // ResourcesUnavailableUnspecified,
    // QualityOfServiceUnavailable,
    // RequestedFacilityNotSubscribed,
    // IncomingCallsBarredWithinTheCug,
    // CollisionWithNetworkInitiatedRequest,
    // BearerCapabilityNotAuthorized,
    // BearerCapabilityNotPresentlyAvailable,
    // UnsupportedQciValue,
    // ServiceOrOptionNotAvailableUnspecified,
    // BearerServiceNotImplemented,
    // AcmEqualToOrGreaterThanAcMmax,
    // RequestedFacilityNotImplemented,
    // OnlyRestrictedDigitalInformationBearerCapabilityIsAvailable,
    // ServiceOrOptionNotImplementedUnspecified,
    // InvalidTransactionIdentifierValue,
    // UserNotMemberOfCug,
    // IncompatibleDestination,
    // InvalidTransitNetworkSelection,
    // InvalidMandatoryInformation,
    // MessageTypeNonExistentOrNotImplemented,
    // MessageTypeNotCompatibleWithProtocolState,
    // MessageNotCompatibleWithProtocolState,
    // RecoveryOnTimerExpiry,
    // ApnRestrictionValueIncompatibleWithActiveEpsBearerContext,
    // InterworkingUnspecified,
    // NetworkError,
    // InvalidEpsBearerIdentity,
    // EmmErrorUnspecified,
    // EsmErrorUnspecified,
    // NumberNotAllowed,
    // CcbsPossible,
    // WrongGpioIdentifier,
    // SetGpioDefaultError,
    // SelectGpioModeError,
    // ReadGpioError,
    // WriteGpioError,
    // GpioBusy,
    // WrongAdcIdentifier,
    // ReadAdcError,
    // IPv4OnlyAllowed,
    // IPv6OnlyAllowed,
    // WrongRingerIdentifier,
    // LlcOrSndcpFailure,
    // RegularDeactivation,
    // ReactivationRequested,
    // SingleAddressBearersOnlyAllowed,
    // ApnRestrictionValIncompatibleWithPdpContext,
    // PdpActivationRejected,
    // GprsGenericOperationError,
    // GprsInvalidApn,
    // GprsAuthenticationFailure,
    // GprsQoSParametersInconsistent,
    // GprsNetworkFailure,
    // GprsContextBusy,
    // CsdGenericOperationError,
    // CsdUndefinedProfile,
    // CsdContextBusy,
    // PlmnScanNotAllowed,
    // FfsError,
    // PdpTypeIPv4OnlyAllowed,
    // PdpTypeIPv6OnlyAllowed,
    // FileNotFound,
    // CannotOpenFile,
    // TacValueNotAllowed,
    // OtpFailure,
    // WrongCheckDigit,
    // BufferFull,
    // FfsInitializing,
    // FfsAlreadyOpenFile,
    // FfsNotOpenFile,
    // FfsFileNotFound,
    // FfsFileAlreadyCreated,
    // FfsIllegalId,
    // FfsIllegalFileHandle,
    // FfsIllegalType,
    // FfsIllegalMode,
    // FfsFileRange,
    // FfsOperationNotPossible,
    // FfsWriteError,
    // FfsUserIdError,
    // FfsInternalFatalError,
    // FfsMemoryResourceError,
    // FfsMaximumNumberOfFilesExceeded,
    // FfsMemoryNotAvailable,
    // FfsInvalidFilename,
    // FfsStreamingNotEnabled,
    // FfsOperationNotAllowedOnStaticFile,
    // FfsMemoryTableInconsistency,
    // FfsNotAFactoryDefaultFile,
    // FfsRequestedMemoryTemporaryNotAvailable,
    // FfsOperationNotAllowedForADirectory,
    // FfsDirectorySpaceNotAvailable,
    // FfsTooManyStreamingFilesOpen,
    // FfsRequestedDynamicMemoryTemporaryNotAvailable,
    // FfsUserProvidedANullParameterInsteadOfASuitableBuffer,
    // FfsTimeout,
    // CommandLineTooLong,
    // CallBarredFixedDialingNumbersOnly,
    // SecRemoteObjectWrongState,
    // SecRotNotPersonalized,
    // SecLossOfConnectivity,
    // SecServiceNotAuthorized,
    // SecFwPackageInstallationRequired,
    // SecFwPackageNotValid,
    // SecResourceNotAvailable,
    // SecDataNotAvailable,
    // SecTimeout,
    // SecDataInconsistentOrUnsupported,
    // SecPspkLockPending,
    // SecC2CAlreadyPaired,
    // SecC2CChannelsConsumed,
    // SecC2CPairingNotPresent,
    // SecBusy,
    // GpsGpioNotConfigured,
    // GpsGpioOwnershipError,
    // InvalidOperationWithGpsOn,
    // InvalidOperationWithGpsOff,
    // InvalidGpsAidingMode,
    // ReservedGpsAidingMode,
    // GpsAidingModeAlreadySet,
    // InvalidGpsTraceMode,
    // ParameterValidOnlyInCaseOfGpsOta,
    // GpsTraceInvalidServer,
    // InvalidTimeZone,
    // InvalidValue,
    // InvalidParameter,
    // InvalidOperationWithLocRunningGpsBusy,
    // NoOngoingCall,
    // IbmBusyECallAlreadyArmedActive,
    // IbmFeatureOffECallFeatureOff,
    // WrongIbmRequested,
    // AudioResourceNotAvailable,
    // EcallRestriction,
    // ECallInvalidDialNumber,
    // NoSapServerConnection,
    // SapProtocolError,
    // SapConnectionFailure,
    // SapServerDisconnection,
    // SapOtherTerminalUsingService,
    // UsecmngImportTimeoutExpired,
    // UsecmngImportFileSizeExceedsLimit,
    // UsecmngNoMemoryAvailable,
    // UsecmngInvalidCertificateKeyFormat,
    // UsecmngDatabaseFull,
    // CdcEcmIsNotAvailable,
    // CdcEcmIsBusy,
    // NoDhcpPacketsReceivedFromTheDte,
    // CommandTimeout,
    // CommandAborted,
    // ApnConfigurationMismatch,
    // IpTypeConfigurationMismatch,
    // FotaPackageDownloadStateOrNameMismatch,
    // FotaPackageDataCorrupted,
    // FotaMemoryIsInUse,
}

impl FromStr for CmeError {
    // This error will always get mapped to `atat::Error::Parse`
    type Err = ();

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        Ok(match s.trim().parse::<u16>() {
            Ok(0) => Self::PhoneFailure,
            Ok(1) => Self::NoConnectionToPhone,
            Ok(2) => Self::PhoneAdaptorLinkReserved,
            Ok(3) | Ok(302) => Self::OperationNotAllowed,
            Ok(4) | Ok(303) => Self::OperationNotSupported,
            Ok(5) => Self::PhSimPinRequired,
            Ok(6) => Self::PhFsimPinRequired,
            Ok(7) => Self::PhFsimPukRequired,
            Ok(10) => Self::SimNotInserted,
            Ok(11) => Self::SimPinRequired,
            Ok(12) => Self::SimPukRequired,
            Ok(13) => Self::SimFailure,
            Ok(14) => Self::SimBusy,
            Ok(15) => Self::SimWrong,
            Ok(16) => Self::IncorrectPassword,
            Ok(17) => Self::SimPin2Required,
            Ok(18) => Self::SimPuk2Required,
            Ok(20) | Ok(322) => Self::MemoryFull,
            Ok(21) => Self::InvalidIndex,
            Ok(22) => Self::NetworkNotFound,
            Ok(23) | Ok(320) => Self::MemoryFailure,
            Ok(24) => Self::TextStringTooLong,
            Ok(25) => Self::InvalidCharactersInTextString,
            Ok(26) => Self::DialStringTooLong,
            Ok(27) => Self::InvalidCharactersInDialString,
            Ok(30) | Ok(331) => Self::NoNetworkService,
            Ok(31) | Ok(332) => Self::NetworkTimeout,
            Ok(32) => Self::NetworkNotAllowedEmergencyCallsOnly,
            Ok(40) => Self::NetworkPersonalisationPinRequired,
            Ok(41) => Self::NetworkPersonalisationPukRequired,
            Ok(42) => Self::NetworkSubsetPersonalisationPinRequired,
            Ok(43) => Self::NetworkSubsetPersonalisationPukRequired,
            Ok(44) => Self::ServiceProviderPersonalisationPinRequired,
            Ok(45) => Self::ServiceProviderPersonalisationPukRequired,
            Ok(46) => Self::CorporatePersonalisationPinRequired,
            Ok(47) => Self::CorporatePersonalisationPukRequired,
            Ok(50) => Self::IncorrectParameters,
            Ok(51) => Self::CommandImplementedButCurrentlyDisabled,
            Ok(52) => Self::CommandAbortedByUser,
            Ok(53) => Self::NotAttachedToNetworkDueToMtFunctionalityRestrictions,
            Ok(54) => Self::ModemNotAllowedMtRestrictedToEmergencyCallsOnly,
            Ok(55) => Self::OperationNotAllowedBecauseOfMtFunctionalityRestrictions,
            Ok(56) => Self::FixedDialNumberOnlyAllowedCalledNumberIsNotAFixedDialNumber,
            Ok(57) => Self::TemporarilyOutOfServiceDueToOtherMtUsage,
            Ok(100) => Self::Unknown,
            Ok(103) => Self::IllegalMs,
            Ok(106) => Self::IllegalMe,
            Ok(107) => Self::GprsServicesNotAllowed,
            Ok(108) => Self::GprsAndNonGprsServicesNotAllowed,
            Ok(111) => Self::PlmnNotAllowed,
            Ok(112) => Self::LocationAreaNotAllowed,
            Ok(113) => Self::RoamingNotAllowedInThisLocationArea,
            Ok(114) => Self::GprsServicesNotAllowedInThisPlmn,
            Ok(115) => Self::NoSuitableCellsInLocationArea,
            Ok(122) | Ok(170) => Self::Congestion,
            Ok(125) => Self::NotAuthorizedForThisCsg,
            Ok(126) => Self::InsufficientResources,
            Ok(127) => Self::MissingOrUnknownApn,
            Ok(128) | Ok(1549) => Self::UnknownPdpAddressOrPdpType,
            Ok(129) => Self::UserAuthenticationFailed,
            Ok(130) => Self::RequestRejectedByServingGwOrPdnGw,
            Ok(131) => Self::RequestRejectedUnspecified,
            Ok(132) => Self::ServiceOptionNotSupported,
            Ok(133) => Self::RequestedServiceOptionNotSubscribed,
            Ok(134) => Self::ServiceOptionTemporarilyOutOfOrder,
            Ok(135) => Self::NsApiAlreadyUsed,
            Ok(137) => Self::EpsQoSNotAccepted,
            Ok(138) | Ok(168) => Self::NetworkFailure,
            Ok(140) => Self::FeatureNotSupported,
            Ok(141) => Self::SemanticErrorInTheTftOperation,
            Ok(142) => Self::SyntacticalErrorInTheTftOperation,
            Ok(143) => Self::UnknownPdpContext,
            Ok(144) => Self::SemanticErrorsInPacketFilterS,
            Ok(145) => Self::SyntacticalErrorsInPacketFilterS,
            Ok(146) => Self::PdpContextWithoutTftAlreadyActivated,
            Ok(147) => Self::PtiMismatch,
            Ok(148) => Self::UnspecifiedGprsError,
            Ok(149) => Self::PdpAuthenticationFailure,
            Ok(150) => Self::InvalidMobileClass,
            Ok(153) => Self::EsmInformationNotReceived,
            Ok(154) => Self::PdnConnectionDoesNotExist,
            Ok(155) => Self::MultiplePdnConnectionsForAGivenApnNotAllowed,
            Ok(156) | Ok(1017) => Self::UserBusy,
            Ok(159) => Self::UplinkBusyFlowControl,
            Ok(160) => Self::BearerHandlingNotSupported,
            Ok(165) => Self::MaximumNumberOfEpsBearersReached,
            Ok(166) | Ok(179) => Self::RequestedApnNotSupportedInCurrentRatAndPlmnCombination,
            Ok(169) => Self::ImsiUnknownInVlr,
            Ok(171) | Ok(1149) => Self::LastPdnDisconnectionNotAllowed,
            Ok(172) | Ok(189) | Ok(1095) => Self::SemanticallyIncorrectMessage,
            Ok(173) => Self::MandatoryInformationElementError,
            Ok(174) | Ok(1099) => Self::InformationElementNonExistentOrNotImplemented,
            Ok(175) | Ok(194) | Ok(1100) => Self::ConditionalIeError,
            Ok(176) | Ok(197) | Ok(1111) => Self::ProtocolErrorUnspecified,
            Ok(177) | Ok(1008) => Self::OperatorDeterminedBarring,
            Ok(178) => Self::MaximumNumberOfPdpContextsReached,
            Ok(180) => Self::RequestRejectedBearerControlModeViolation,
            Ok(181) => Self::InvalidPtiValue,
            Ok(190) => Self::InvalidMandatoryIe,
            Ok(191) => Self::MessageTypeNonExistent,
            Ok(192) => Self::MessageTypeNotCompatible,
            Ok(193) => Self::IeNonExistent,
            Ok(195) => Self::MessageNotCompatible,
            Ok(254) => Self::InvalidErrorMapping,
            Ok(255) => Self::InternalError,
            Ok(262) => Self::SimBlocked,
            Ok(300) => Self::MeFailure,
            Ok(301) => Self::SmsServiceOfMeReserved,
            Ok(304) => Self::InvalidPduModeParameter,
            Ok(305) => Self::InvalidTextModeParameter,
            Ok(310) => Self::USimNotInserted,
            Ok(311) => Self::USimPinRequired,
            Ok(312) => Self::PhUSimPinRequired,
            Ok(313) => Self::USimFailure,
            Ok(314) => Self::USimBusy,
            Ok(315) => Self::USimWrong,
            Ok(316) => Self::USimPukRequired,
            Ok(317) => Self::USimPin2Required,
            Ok(318) => Self::USimPuk2Required,
            Ok(321) => Self::InvalidMemoryIndex,
            // Ok(330) => Self::SmscAddressUnknown,
            // Ok(340) => Self::NoCnmaAcknowledgementExpected,
            // Ok(500) => Self::UnknownError,
            // Ok(608) => Self::VoiceCallActive,
            // Ok(701) => Self::IncorrectSecurityCode,
            // Ok(702) => Self::MaxAttemptsReached,
            // Ok(1001) => Self::UnassignedUnallocatedNumber,
            // Ok(1003) => Self::NoRouteToDestination,
            // Ok(1006) => Self::ChannelUnacceptable,
            // Ok(1016) => Self::NormalCallClearing,
            // Ok(1018) => Self::NoUserResponding,
            // Ok(1019) => Self::UserAlertingNoAnswer,
            // Ok(1021) => Self::CallRejected,
            // Ok(1022) => Self::NumberChanged,
            // Ok(1026) => Self::NonSelectedUserClearing,
            // Ok(1027) => Self::DestinationOutOfOrder,
            // Ok(1028) => Self::InvalidNumberFormatIncompleteNumber,
            // Ok(1029) => Self::FacilityRejected,
            // Ok(1030) => Self::ResponseToStatusEnquiry,
            // Ok(1031) => Self::NormalUnspecified,
            // Ok(1034) => Self::NoCircuitChannelAvailable,
            // Ok(1038) => Self::NetworkOutOfOrder,
            // Ok(1041) => Self::TemporaryFailure,
            // Ok(1042) => Self::SwitchingEquipmentCongestion,
            // Ok(1043) => Self::AccessInformationDiscarded,
            // Ok(1044) => Self::RequestedCircuitChannelNotAvailable,
            // Ok(1047) => Self::ResourcesUnavailableUnspecified,
            // Ok(1049) => Self::QualityOfServiceUnavailable,
            // Ok(1050) => Self::RequestedFacilityNotSubscribed,
            // Ok(1055) => Self::IncomingCallsBarredWithinTheCug,
            // Ok(1056) => Self::CollisionWithNetworkInitiatedRequest,
            // Ok(1057) => Self::BearerCapabilityNotAuthorized,
            // Ok(1058) => Self::BearerCapabilityNotPresentlyAvailable,
            // Ok(1059) => Self::UnsupportedQciValue,
            // Ok(1063) => Self::ServiceOrOptionNotAvailableUnspecified,
            // Ok(1065) => Self::BearerServiceNotImplemented,
            // Ok(1068) => Self::AcmEqualToOrGreaterThanAcMmax,
            // Ok(1069) => Self::RequestedFacilityNotImplemented,
            // Ok(1070) => Self::OnlyRestrictedDigitalInformationBearerCapabilityIsAvailable,
            // Ok(1079) => Self::ServiceOrOptionNotImplementedUnspecified,
            // Ok(1081) | Ok(1546) => Self::InvalidTransactionIdentifierValue,
            // Ok(1087) => Self::UserNotMemberOfCug,
            // Ok(1088) => Self::IncompatibleDestination,
            // Ok(1091) => Self::InvalidTransitNetworkSelection,
            // Ok(1096) => Self::InvalidMandatoryInformation,
            // Ok(1097) => Self::MessageTypeNonExistentOrNotImplemented,
            // Ok(1098) => Self::MessageTypeNotCompatibleWithProtocolState,
            // Ok(1101) => Self::MessageNotCompatibleWithProtocolState,
            // Ok(1102) => Self::RecoveryOnTimerExpiry,
            // Ok(1112) => Self::ApnRestrictionValueIncompatibleWithActiveEpsBearerContext,
            // Ok(1127) => Self::InterworkingUnspecified,
            // Ok(1142) => Self::NetworkError,
            // Ok(1143) => Self::InvalidEpsBearerIdentity,
            // Ok(1243) => Self::EmmErrorUnspecified,
            // Ok(1244) => Self::EsmErrorUnspecified,
            // Ok(1279) => Self::NumberNotAllowed,
            // Ok(1283) => Self::CcbsPossible,
            // Ok(1500) => Self::WrongGpioIdentifier,
            // Ok(1501) => Self::SetGpioDefaultError,
            // Ok(1502) => Self::SelectGpioModeError,
            // Ok(1503) => Self::ReadGpioError,
            // Ok(1504) => Self::WriteGpioError,
            // Ok(1505) => Self::GpioBusy,
            // Ok(1520) => Self::WrongAdcIdentifier,
            // Ok(1521) => Self::ReadAdcError,
            // Ok(1530) => Self::IPv4OnlyAllowed,
            // Ok(1531) => Self::IPv6OnlyAllowed,
            // Ok(1540) => Self::WrongRingerIdentifier,
            // Ok(1542) => Self::LlcOrSndcpFailure,
            // Ok(1543) => Self::RegularDeactivation,
            // Ok(1544) => Self::ReactivationRequested,
            // Ok(1545) => Self::SingleAddressBearersOnlyAllowed,
            // Ok(1547) => Self::ApnRestrictionValIncompatibleWithPdpContext,
            // Ok(1548) => Self::PdpActivationRejected,
            // Ok(1550) => Self::GprsGenericOperationError,
            // Ok(1551) => Self::GprsInvalidApn,
            // Ok(1552) => Self::GprsAuthenticationFailure,
            // Ok(1553) => Self::GprsQoSParametersInconsistent,
            // Ok(1554) => Self::GprsNetworkFailure,
            // Ok(1555) => Self::GprsContextBusy,
            // Ok(1556) => Self::CsdGenericOperationError,
            // Ok(1557) => Self::CsdUndefinedProfile,
            // Ok(1558) => Self::CsdContextBusy,
            // Ok(1559) => Self::PlmnScanNotAllowed,
            // Ok(1600) => Self::FfsError,
            // Ok(1560) => Self::PdpTypeIPv4OnlyAllowed,
            // Ok(1561) => Self::PdpTypeIPv6OnlyAllowed,
            // Ok(1612) => Self::FileNotFound,
            // Ok(1613) => Self::CannotOpenFile,
            // Ok(1614) => Self::TacValueNotAllowed,
            // Ok(1615) => Self::OtpFailure,
            // Ok(1616) => Self::WrongCheckDigit,
            // Ok(1620) => Self::BufferFull,
            // Ok(1621) => Self::FfsInitializing,
            // Ok(1622) => Self::FfsAlreadyOpenFile,
            // Ok(1623) => Self::FfsNotOpenFile,
            // Ok(1624) => Self::FfsFileNotFound,
            // Ok(1625) => Self::FfsFileAlreadyCreated,
            // Ok(1626) => Self::FfsIllegalId,
            // Ok(1627) => Self::FfsIllegalFileHandle,
            // Ok(1628) => Self::FfsIllegalType,
            // Ok(1629) => Self::FfsIllegalMode,
            // Ok(1630) => Self::FfsFileRange,
            // Ok(1631) => Self::FfsOperationNotPossible,
            // Ok(1632) => Self::FfsWriteError,
            // Ok(1633) => Self::FfsUserIdError,
            // Ok(1634) => Self::FfsInternalFatalError,
            // Ok(1635) => Self::FfsMemoryResourceError,
            // Ok(1636) => Self::FfsMaximumNumberOfFilesExceeded,
            // Ok(1637) => Self::FfsMemoryNotAvailable,
            // Ok(1638) => Self::FfsInvalidFilename,
            // Ok(1639) => Self::FfsStreamingNotEnabled,
            // Ok(1640) => Self::FfsOperationNotAllowedOnStaticFile,
            // Ok(1641) => Self::FfsMemoryTableInconsistency,
            // Ok(1642) => Self::FfsNotAFactoryDefaultFile,
            // Ok(1643) => Self::FfsRequestedMemoryTemporaryNotAvailable,
            // Ok(1644) => Self::FfsOperationNotAllowedForADirectory,
            // Ok(1645) => Self::FfsDirectorySpaceNotAvailable,
            // Ok(1646) => Self::FfsTooManyStreamingFilesOpen,
            // Ok(1647) => Self::FfsRequestedDynamicMemoryTemporaryNotAvailable,
            // Ok(1648) => Self::FfsUserProvidedANullParameterInsteadOfASuitableBuffer,
            // Ok(1649) => Self::FfsTimeout,
            // Ok(1650) => Self::CommandLineTooLong,
            // Ok(1660) => Self::CallBarredFixedDialingNumbersOnly,
            // Ok(1670) => Self::SecRemoteObjectWrongState,
            // Ok(1671) => Self::SecRotNotPersonalized,
            // Ok(1672) => Self::SecLossOfConnectivity,
            // Ok(1673) => Self::SecServiceNotAuthorized,
            // Ok(1674) => Self::SecFwPackageInstallationRequired,
            // Ok(1675) => Self::SecFwPackageNotValid,
            // Ok(1676) => Self::SecResourceNotAvailable,
            // Ok(1677) => Self::SecDataNotAvailable,
            // Ok(1678) => Self::SecTimeout,
            // Ok(1679) => Self::SecDataInconsistentOrUnsupported,
            // Ok(1680) => Self::SecPspkLockPending,
            // Ok(1681) => Self::SecC2CAlreadyPaired,
            // Ok(1682) => Self::SecC2CChannelsConsumed,
            // Ok(1683) => Self::SecC2CPairingNotPresent,
            // Ok(1684) => Self::SecBusy,
            // Ok(1700) => Self::GpsGpioNotConfigured,
            // Ok(1701) => Self::GpsGpioOwnershipError,
            // Ok(1702) => Self::InvalidOperationWithGpsOn,
            // Ok(1703) => Self::InvalidOperationWithGpsOff,
            // Ok(1704) => Self::InvalidGpsAidingMode,
            // Ok(1705) => Self::ReservedGpsAidingMode,
            // Ok(1706) => Self::GpsAidingModeAlreadySet,
            // Ok(1707) => Self::InvalidGpsTraceMode,
            // Ok(1708) => Self::ParameterValidOnlyInCaseOfGpsOta,
            // Ok(1709) => Self::GpsTraceInvalidServer,
            // Ok(1710) => Self::InvalidTimeZone,
            // Ok(1711) => Self::InvalidValue,
            // Ok(1712) => Self::InvalidParameter,
            // Ok(1713) => Self::InvalidOperationWithLocRunningGpsBusy,
            // Ok(1800) => Self::NoOngoingCall,
            // Ok(1801) => Self::IbmBusyECallAlreadyArmedActive,
            // Ok(1802) => Self::IbmFeatureOffECallFeatureOff,
            // Ok(1803) => Self::WrongIbmRequested,
            // Ok(1804) => Self::AudioResourceNotAvailable,
            // Ok(1805) => Self::EcallRestriction,
            // Ok(1806) => Self::ECallInvalidDialNumber,
            // Ok(1900) => Self::NoSapServerConnection,
            // Ok(1901) => Self::SapProtocolError,
            // Ok(1902) => Self::SapConnectionFailure,
            // Ok(1903) => Self::SapServerDisconnection,
            // Ok(1904) => Self::SapOtherTerminalUsingService,
            // Ok(1910) => Self::UsecmngImportTimeoutExpired,
            // Ok(1911) => Self::UsecmngImportFileSizeExceedsLimit,
            // Ok(1912) => Self::UsecmngNoMemoryAvailable,
            // Ok(1913) => Self::UsecmngInvalidCertificateKeyFormat,
            // Ok(1914) => Self::UsecmngDatabaseFull,
            // Ok(1950) => Self::CdcEcmIsNotAvailable,
            // Ok(1951) => Self::CdcEcmIsBusy,
            // Ok(1952) => Self::NoDhcpPacketsReceivedFromTheDte,
            // Ok(2000) => Self::CommandTimeout,
            // Ok(3000) => Self::CommandAborted,
            // Ok(4000) => Self::ApnConfigurationMismatch,
            // Ok(4001) => Self::IpTypeConfigurationMismatch,
            // Ok(5000) => Self::FotaPackageDownloadStateOrNameMismatch,
            // Ok(5001) => Self::FotaPackageDataCorrupted,
            // Ok(5002) => Self::FotaMemoryIsInUse,
            Ok(_) => return Err(()),
            Err(_) => return Err(()),
        })
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn unknown_error() {
        assert_eq!(
            "+CME ERROR: Something went wrong".parse::<UbloxError>(),
            Ok(UbloxError::Cme(CmeError::Unknown))
        );
        assert_eq!(
            "+CMS ERROR: Something went wrong".parse::<UbloxError>(),
            Ok(UbloxError::Cms(CmsError::Unknown))
        );
    }
    #[test]
    fn numeric_error() {
        assert_eq!(
            "+CME ERROR: 0".parse::<UbloxError>(),
            Ok(UbloxError::Cme(CmeError::PhoneFailure))
        );
        assert_eq!(
            "+CME ERROR: 138".parse::<UbloxError>(),
            Ok(UbloxError::Cme(CmeError::NetworkFailure))
        );
    }

    #[test]
    fn generic_error() {
        assert_eq!("ERROR".parse::<UbloxError>(), Ok(UbloxError::Generic));
    }
}
