//!
//! [![crates.io](https://img.shields.io/crates/v/ubits.svg)](https://crates.io/crates/ubits)
//! [![Downloads](https://img.shields.io/crates/d/ubits.svg)](https://crates.io/crates/ubits)
//! [![Build](https://github.com/manoadamro/ubits/actions/workflows/rust.yml/badge.svg)](https://github.com/manoadamro/ubits/actions/workflows/rust.yml)
//! [![Docs](https://img.shields.io/docsrs/ubits)](https://manoadamro.github.io/ubits/doc/ubits/)
//! [![Licence](https://img.shields.io/github/license/manoadamro/ubits)](https://github.com/manoadamro/ubits/blob/main/LICENSE)
//!
//! Bit fields and masks for rust!
//!
//! Provides a macro for generating [`bit field`](https://en.wikipedia.org/wiki/Bit_field) types
//! complete with flags and some helpful trait implementations.
//!
//! Supports field widths of `8`, `16`, `32`, `64` and `128` bits.
//!
//! - [`bitfield`] - macro documentation.
//! - [`examples`] - examples of the output generated by the [`bitfield`] macro.
//!
//! __Note: the example module is only complied with documentation builds
//! and is not available for importing in the wild.__
//!
//!
//! # Usage
//!
//! Generate a bitfield struct with a flag enum...
//! (The following examples all use this as a definition.)
//! ```rust
//! use ubits::bitfield;
//!
//!     bitfield! {
//!         pub u8 ExampleField
//!         ExampleFlags {
//!             0 : Flag0,
//!             1 : Flag1,
//!             2 : Flag2,
//!             3 : Flag3,
//!             4 : Flag4,
//!             5 : Flag5,
//!             6 : Flag6,
//!             7 : Flag7,
//!         }
//!     }
//! ```
//!
//! ## Instances
//!
//! From integer:
//! ```rust
//! # use ubits::bitfield;
//! #    bitfield! {
//! #        pub u8 ExampleField
//! #        ExampleFlags {
//! #            0 : Flag0,
//! #            1 : Flag1,
//! #            2 : Flag2,
//! #            3 : Flag3,
//! #            4 : Flag4,
//! #            5 : Flag5,
//! #            6 : Flag6,
//! #            7 : Flag7,
//! #        }
//! #    }
//!
//! let from_integer = ExampleField(123);
//! assert_eq!(ExampleField(123), from_integer);
//! ```
//!
//! From a binary string:
//! ```rust
//! # use ubits::bitfield;
//! #    bitfield! {
//! #        pub u8 ExampleField
//! #        ExampleFlags {
//! #            0 : Flag0,
//! #            1 : Flag1,
//! #            2 : Flag2,
//! #            3 : Flag3,
//! #            4 : Flag4,
//! #            5 : Flag5,
//! #            6 : Flag6,
//! #            7 : Flag7,
//! #        }
//! #    }
//!
//! let from_binary = ExampleField::from_binary_str("01111011");
//! assert_eq!(ExampleField(123), from_binary)
//! ```
//!
//! From ones:
//! ```rust
//! # use ubits::bitfield;
//! #    bitfield! {
//! #        pub u8 ExampleField
//! #        ExampleFlags {
//! #            0 : Flag0,
//! #            1 : Flag1,
//! #            2 : Flag2,
//! #            3 : Flag3,
//! #            4 : Flag4,
//! #            5 : Flag5,
//! #            6 : Flag6,
//! #            7 : Flag7,
//! #        }
//! #    }
//!
//! let from_ones = ExampleField::ones();
//! assert_eq!("11111111", from_ones.as_binary());
//! assert_eq!(255, from_ones.as_integer());
//! ```
//!
//! From zeros:
//! ```rust
//! # use ubits::bitfield;
//! #    bitfield! {
//! #        pub u8 ExampleField
//! #        ExampleFlags {
//! #            0 : Flag0,
//! #            1 : Flag1,
//! #            2 : Flag2,
//! #            3 : Flag3,
//! #            4 : Flag4,
//! #            5 : Flag5,
//! #            6 : Flag6,
//! #            7 : Flag7,
//! #        }
//! #    }
//!
//! let from_zeros = ExampleField::zeros();
//! assert_eq!("00000000", from_zeros.as_binary());
//! assert_eq!(0, from_zeros.as_integer());
//! ```
//!
//! ### Field Access
//!
//! Get bit value by field:
//! ```rust
//! # use ubits::bitfield;
//! #    bitfield! {
//! #        pub u8 ExampleField
//! #        ExampleFlags {
//! #            0 : Flag0,
//! #            1 : Flag1,
//! #            2 : Flag2,
//! #            3 : Flag3,
//! #            4 : Flag4,
//! #            5 : Flag5,
//! #            6 : Flag6,
//! #            7 : Flag7,
//! #        }
//! #    }
//!
//! let field = ExampleField::from_binary_str("01010101");
//! assert!(field.get(ExampleFlags::Flag0));
//! assert!(!field.get(ExampleFlags::Flag1));
//! ```
//!
//! Get bit value by index:
//!
//! ```rust
//! # use ubits::bitfield;
//! #    bitfield! {
//! #        pub u8 ExampleField
//! #        ExampleFlags {
//! #            0 : Flag0,
//! #            1 : Flag1,
//! #            2 : Flag2,
//! #            3 : Flag3,
//! #            4 : Flag4,
//! #            5 : Flag5,
//! #            6 : Flag6,
//! #            7 : Flag7,
//! #        }
//! #    }
//!
//! let field = ExampleField::from_binary_str("01010101");
//! assert!(field.get_index(0));
//! assert!(!field.get_index(1));
//! ```
//!
//! Set bit value by field:
//!
//! ```rust
//! # use ubits::bitfield;
//! #    bitfield! {
//! #        pub u8 ExampleField
//! #        ExampleFlags {
//! #            0 : Flag0,
//! #            1 : Flag1,
//! #            2 : Flag2,
//! #            3 : Flag3,
//! #            4 : Flag4,
//! #            5 : Flag5,
//! #            6 : Flag6,
//! #            7 : Flag7,
//! #        }
//! #    }
//!
//! let mut field = ExampleField::from_binary_str("01010101");
//! field.set(ExampleFlags::Flag1);
//! field.set(ExampleFlags::Flag3);
//! assert_eq!("01011111", field.as_binary());
//! ```
//!
//! Set bit value by index:
//!
//! ```rust
//! # use ubits::bitfield;
//! #    bitfield! {
//! #        pub u8 ExampleField
//! #        ExampleFlags {
//! #            0 : Flag0,
//! #            1 : Flag1,
//! #            2 : Flag2,
//! #            3 : Flag3,
//! #            4 : Flag4,
//! #            5 : Flag5,
//! #            6 : Flag6,
//! #            7 : Flag7,
//! #        }
//! #    }
//!
//! let mut field = ExampleField::from_binary_str("01010101");
//! field.set_index(1);
//! field.set_index(3);
//! assert_eq!("01011111", field.as_binary());
//! ```
//!
//! Clear bit value by field:
//!
//! ```rust
//! # use ubits::bitfield;
//! #    bitfield! {
//! #        pub u8 ExampleField
//! #        ExampleFlags {
//! #            0 : Flag0,
//! #            1 : Flag1,
//! #            2 : Flag2,
//! #            3 : Flag3,
//! #            4 : Flag4,
//! #            5 : Flag5,
//! #            6 : Flag6,
//! #            7 : Flag7,
//! #        }
//! #    }
//!
//! let mut field = ExampleField::from_binary_str("01010101");
//! field.clear(ExampleFlags::Flag0);
//! field.clear(ExampleFlags::Flag2);
//! assert_eq!("01010000", field.as_binary());
//! ```
//!
//! Clear bit value by index:
//!
//! ```rust
//! # use ubits::bitfield;
//! #    bitfield! {
//! #        pub u8 ExampleField
//! #        ExampleFlags {
//! #            0 : Flag0,
//! #            1 : Flag1,
//! #            2 : Flag2,
//! #            3 : Flag3,
//! #            4 : Flag4,
//! #            5 : Flag5,
//! #            6 : Flag6,
//! #            7 : Flag7,
//! #        }
//! #    }
//!
//! let mut field = ExampleField::from_binary_str("01010101");
//! field.clear_index(0);
//! field.clear_index(2);
//! assert_eq!("01010000", field.as_binary());
//! ```
//!
//! Toggle bit value by field:
//!
//! ```rust
//! # use ubits::bitfield;
//! #    bitfield! {
//! #        pub u8 ExampleField
//! #        ExampleFlags {
//! #            0 : Flag0,
//! #            1 : Flag1,
//! #            2 : Flag2,
//! #            3 : Flag3,
//! #            4 : Flag4,
//! #            5 : Flag5,
//! #            6 : Flag6,
//! #            7 : Flag7,
//! #        }
//! #    }
//!
//! let mut field = ExampleField::from_binary_str("01010101");
//! field.toggle(ExampleFlags::Flag0);
//! assert_eq!("01010100", field.as_binary());
//! field.toggle(ExampleFlags::Flag0);
//! assert_eq!("01010101", field.as_binary());
//! ```
//!
//! Toggle bit value by index:
//!
//! ```rust
//! # use ubits::bitfield;
//! #    bitfield! {
//! #        pub u8 ExampleField
//! #        ExampleFlags {
//! #            0 : Flag0,
//! #            1 : Flag1,
//! #            2 : Flag2,
//! #            3 : Flag3,
//! #            4 : Flag4,
//! #            5 : Flag5,
//! #            6 : Flag6,
//! #            7 : Flag7,
//! #        }
//! #    }
//!
//! let mut field = ExampleField::from_binary_str("01010101");
//! field.toggle_index(0);
//! assert_eq!("01010100", field.as_binary());
//! field.toggle_index(0);
//! assert_eq!("01010101", field.as_binary());
//! ```
//!
//! ### Named Getters and Setters
//!
//! Ubits can generate getter and setter methods for zero or more fields, if a name is provided.
//!
//! ```rust
//! use ubits::bitfield;
//!    bitfield! {
//!        pub u8 ExampleField
//!        ExampleFlags {
//!            0 : Flag0 : (field_0),  // is_field_0 & set_field_0 & clear_field_0 & toggle_field_0
//!            1 : Flag1 : (field_1),  // is_field_1 & set_field_1 & clear_field_1 & toggle_field_1
//!            2 : Flag2,
//!            3 : Flag3,
//!            4 : Flag4,
//!            5 : Flag5,
//!            6 : Flag6,
//!            7 : Flag7,
//!        }
//!    }
//!
//! let mut field = ExampleField::from_binary_str("01010101");
//! assert_eq!(true, field.is_field_0());
//! assert_eq!(false, field.is_field_1());
//!
//! field.set_field_1();
//! assert_eq!(true, field.is_field_1());
//!
//! field.clear_field_1();
//! assert_eq!(false, field.is_field_1());
//!
//! field.toggle_field_1();
//! assert_eq!(true, field.is_field_1());
//! ```
//!
//! ### Combinations
//!
//! Combine bit fields: <br>
//! (use `into_combined` to consume self)
//!
//! ```rust
//! # use ubits::bitfield;
//! #    bitfield! {
//! #        pub u8 ExampleField
//! #        ExampleFlags {
//! #            0 : Flag0,
//! #            1 : Flag1,
//! #            2 : Flag2,
//! #            3 : Flag3,
//! #            4 : Flag4,
//! #            5 : Flag5,
//! #            6 : Flag6,
//! #            7 : Flag7,
//! #        }
//! #    }
//!
//! let mut a = ExampleField::from_binary_str("01010101");
//! let b = ExampleField::from_binary_str("10101010");
//! assert_eq!("11111111", a.combine(b).as_binary());
//! ```
//!
//! Get the intersection of two bitfields: <br>
//! (use `into_intersection` to consume self)
//! ```rust
//! # use ubits::bitfield;
//! #    bitfield! {
//! #        pub u8 ExampleField
//! #        ExampleFlags {
//! #            0 : Flag0,
//! #            1 : Flag1,
//! #            2 : Flag2,
//! #            3 : Flag3,
//! #            4 : Flag4,
//! #            5 : Flag5,
//! #            6 : Flag6,
//! #            7 : Flag7,
//! #        }
//! #    }
//!
//! let mut a = ExampleField::from_binary_str("11000011");
//! let b = ExampleField::from_binary_str("01111110");
//! assert_eq!("01000010", a.intersect(b).as_binary());
//! ```
//!
//! Get the diff of two bitfields: <br>
//! (use `into_diff` to consume self)
//!
//! ```rust
//! # use ubits::bitfield;
//! #    bitfield! {
//! #        pub u8 ExampleField
//! #        ExampleFlags {
//! #            0 : Flag0,
//! #            1 : Flag1,
//! #            2 : Flag2,
//! #            3 : Flag3,
//! #            4 : Flag4,
//! #            5 : Flag5,
//! #            6 : Flag6,
//! #            7 : Flag7,
//! #        }
//! #    }
//!
//! let mut a = ExampleField::from_binary_str("11000011");
//! let b = ExampleField::from_binary_str("01100110");
//! assert_eq!("10100101", a.diff(b).as_binary());
//! ```
//!
//! ### Bitwise
//!
//! Both bit field instances and flags use bitwise operators to change bit values.
//!
//! ```rust
//! # use ubits::bitfield;
//! #    bitfield! {
//! #        pub u8 ExampleField
//! #        ExampleFlags {
//! #            0 : Flag0,
//! #            1 : Flag1,
//! #            2 : Flag2,
//! #            3 : Flag3,
//! #            4 : Flag4,
//! #            5 : Flag5,
//! #            6 : Flag6,
//! #            7 : Flag7,
//! #        }
//! #    }
//!
//! let mut from_zeros = ExampleField::zeros();
//! assert_eq!("00000000", from_zeros.as_binary());
//!
//! // set bit to 1
//! from_zeros |= ExampleFlags::Flag1;
//! assert_eq!("00000010", from_zeros.as_binary());
//!
//! // set bit back to 0
//! from_zeros &= ExampleFlags::Flag1;
//! assert_eq!("00000000", from_zeros.as_binary());
//!
//! // toggle a bit
//! from_zeros ^= ExampleFlags::Flag1;
//! assert_eq!("00000010", from_zeros.as_binary());
//!
//! from_zeros ^= ExampleFlags::Flag1;
//! assert_eq!("00000000", from_zeros.as_binary());
//! ```
//!
//! Operations can also be chained together:
//!
//! ```rust
//! # use ubits::bitfield;
//! #    bitfield! {
//! #        pub u8 ExampleField
//! #        ExampleFlags {
//! #            0 : Flag0,
//! #            1 : Flag1,
//! #            2 : Flag2,
//! #            3 : Flag3,
//! #            4 : Flag4,
//! #            5 : Flag5,
//! #            6 : Flag6,
//! #            7 : Flag7,
//! #        }
//! #    }
//!
//! let mut from_zeros = ExampleField::zeros() | ExampleFlags::Flag1 | ExampleFlags::Flag3;
//! assert_eq!("00001010", from_zeros.as_binary());
//!
//! ```
//!
//! Bitfield instances can also be created from combining flags:
//!
//! ```rust
//! # use ubits::bitfield;
//! #    bitfield! {
//! #        pub u8 ExampleField
//! #        ExampleFlags {
//! #            0 : Flag0,
//! #            1 : Flag1,
//! #            2 : Flag2,
//! #            3 : Flag3,
//! #            4 : Flag4,
//! #            5 : Flag5,
//! #            6 : Flag6,
//! #            7 : Flag7,
//! #        }
//! #    }
//!
//! let mut from_zeros = ExampleFlags::Flag1 | ExampleFlags::Flag3;
//! assert_eq!("00001010", from_zeros.as_binary());
//!
//! ```
//!
//! ## Fields named with flags
//!
//! The generated flags enum allows you to access bits by name.
//! The flag has an associated [`u8`] value,
//! which determines the index its target bit.
//! (See [`bitfield`] for more info)
//!
//! With the following input...
//! ```no_compile
//! 1 0 1 0 0 1 1 0
//! ```
//!
//! and the following flags...
//! ```no_compile
//! 0 : f1
//! 1 : f1
//! 2 : f2
//! 3 : f3
//! 4 : f4
//! 5 : f5
//! 6 : f6
//! 7 : f7
//! ```
//!
//! we end up with this layout.
//!
//! | name      | f7  | f6  | f5  | f4  | f3  | f2  | f1  | f0  |
//! |-----------|----|----|----|----|----|----|----|----|
//! | bit value | 1  | 0  | 1  | 0  | 0  | 1  | 1  | 0  |
//! | index     | 7  | 6  | 5  | 4  | 3  | 2  | 1  | 0  |
//!
//!
//! With the same input and only the first few flags:
//!
//! ```no_compile
//! 0 : f0
//! 1 : f1
//! 2 : f2
//! ```
//!
//! we end up with this layout.
//!
//! | name      |    |    |    |    |    | f2 | f1 | f0 |
//! |-----------|----|----|----|----|----|----|----|----|
//! | bit value | 1  | 0  | 1  | 0  | 0  | 1  | 1  | 0  |
//! | index     | 7  | 6  | 5  | 4  | 3  | 2  | 1  | 0  |
//!
//!
//! Using the same input, but with dispersed flags:
//!
//! ```no_compile
//! 1 : f0
//! 3 : f1
//! 6 : f2
//! ```
//!
//! we end up with this layout.
//!
//! | name      |    | f2 |    |    | f1 |    | f0 |    |
//! |-----------|----|----|----|----|----|----|----|----|
//! | bit value | 1  | 0  | 1  | 0  | 0  | 1  | 1  | 0  |
//! | index     | 7  | 6  | 5  | 4  | 3  | 2  | 1  | 0  |
//!

// -------------------------------------------------------------------------------------------------
// Imports

extern crate doc_comment;
extern crate paste;
extern crate safe_transmute;

#[doc(hidden)]
pub use doc_comment::doc_comment as __doc_comment;

#[doc(hidden)]
pub use safe_transmute::{
    transmute_one as __transmute_one, TriviallyTransmutable as __TriviallyTransmutable,
};

#[doc(hidden)]
pub use paste::paste as __paste;

// -------------------------------------------------------------------------------------------------
// Generator Macros

/// Generates a bitfield struct and a corresponding flag enum.
///
/// Can be used multiple times in a single codebase
/// as long as the identifiers are unique or are isolated so as not to clash.
///
/// __See the [`examples`] module for examples of the generated output.__
///
/// # Args
///
/// - `pub` - optional accessor: if provided, both generated items will public.
/// If omitted, both will be private. <br>
/// [see here](https://doc.rust-lang.org/reference/visibility-and-privacy.html#visibility-and-privacy) for possible visibility options.
///
/// - `ExampleField` - name for the generated bitfield struct.
/// (results in `struct ExampleField(...)`)
///
/// - `ExampleFlags` - name for the generated flag enum.
/// (results in `enum ExampleFlags { ... }`)
///
/// - `u8` - unsigned integer type to use as a bit array. *Must be an unsigned integer* <br>
/// valid options are: [`u8`],[`u16`], [`u32`], [`u64`], [`u128`], [`usize`]
///
/// - `0 : Flag0` - defines each member of the flag enum. <br>
/// The left hand side must be a [`u8`] `(0..=254)`
/// and must be less than the number of bits in the underlying unsigned type.
/// This value determines the index of the target bit within the field. <br>
/// The right hand side can be any valid identifier (unique to this enum).
/// These identifiers will be used to access the corresponding field.
///
/// - `/// docstrings` - Each element can take optional docstrings.
/// This includes the bit field struct, the flag enum, and each member within the flag enum.
/// See [`examples`] for more information.
///
/// # Examples:
///
/// Full Example - (see below for explanation for each parameter):
/// ```rust
/// use ubits::bitfield;
///
///     bitfield! {
///         /// Optional docstring
///         /// for [`ExampleField`] struct
///         pub u8 ExampleField
///         /// Optional docstring
///         /// for [`ExampleFlags`] enum
///         ExampleFlags {
///             /// Optional docstring for [`ExampleFlags::Flag0`]
///             0 : Flag0,
///             /// Optional docstring for [`ExampleFlags::Flag1`]
///             1 : Flag1,
///             /// Optional docstring for [`ExampleFlags::Flag2`]
///             2 : Flag2,
///             /// Optional docstring for [`ExampleFlags::Flag3`]
///             3 : Flag3,
///             /// Optional docstring for [`ExampleFlags::Flag4`]
///             4 : Flag4,
///             /// Optional docstring for [`ExampleFlags::Flag5`]
///             5 : Flag5,
///             /// Optional docstring for [`ExampleFlags::Flag6`]
///             6 : Flag6,
///             /// Optional docstring for [`ExampleFlags::Flag7`]
///             7 : Flag7,
///         }
///     }
/// ```
///
/// You don't have to name *all* the fields if you don't need them...
///
/// You can use just a few from the front:
/// ```rust
/// use ubits::bitfield;
///
///     bitfield! {
///         /// Optional docstring
///         /// for [`ExampleField`] struct
///         pub u8 ExampleField
///         /// Optional docstring
///         /// for [`ExampleFlags`] enum
///         ExampleFlags {
///             /// Optional docstring for [`ExampleFlags::Flag0`]
///             0 : Flag0,
///             /// Optional docstring for [`ExampleFlags::Flag1`]
///             1 : Flag1,
///             /// Optional docstring for [`ExampleFlags::Flag2`]
///             2 : Flag2,
///         }
///     }
/// ```
///
/// ... or any valid indices:
/// ```rust
/// use ubits::bitfield;
///
///     bitfield! {
///         /// Optional docstring
///         /// for [`ExampleField`] struct
///         pub u8 ExampleField
///         /// Optional docstring
///         /// for [`ExampleFlags`] enum
///         ExampleFlags {
///             /// Optional docstring for [`ExampleFlags::Flag0`]
///             0 : Flag0,
///             /// Optional docstring for [`ExampleFlags::Flag1`]
///             3 : Flag1,
///             /// Optional docstring for [`ExampleFlags::Flag2`]
///             6 : Flag2,
///         }
///     }
/// ```
/// ... or just omit the flags altogether and use indices only.:
/// ```rust
/// use ubits::bitfield;
///     bitfield! {
///         /// Optional docstring
///         /// for [`ExampleField`] struct
///         pub u8 ExampleField
///     }
/// ```
/// __Note: if you omit flags, no flag enum will be generated
/// and no methods for the bitfield struct that rely on flags will be generated.__
///
#[macro_export(local_inner_macros)]
macro_rules! bitfield {

    // U8
    (
        $(#[$doc:meta])*
        $access:vis u8 $name:ident
        $(
            $(#[$flag_doc:meta])*
            $flag:ident {
                $(
                    $(#[$member_doc:meta])*
                    $idx:literal : $field:ident $( : ( $named_getter:ident ) )?,
                )*
            }
        )?
    ) => {
        __bitfield_unchecked!( $(#[$doc])* $access [u8] [8] [u8::MAX] $name $($(#[$flag_doc])* $flag { $( $(#[$member_doc])* $idx : $field $(($named_getter))?, )* } )?);
    };

    // U16
    (
        $(#[$doc:meta])*
        $access:vis u16 $name:ident
        $(
            $(#[$flag_doc:meta])*
            $flag:ident {
                $(
                    $(#[$member_doc:meta])*
                    $idx:literal : $field:ident $( ( $named_getter:ident ) )?,
                )*
            }
        )?
    ) => {
        __bitfield_unchecked!( $(#[$doc])* $access [u16] [16] [u16::MAX] $name $($(#[$flag_doc])* $flag { $( $(#[$member_doc])* $idx : $field $(($named_getter))?, )* } )?);
    };

    // U32
    (
        $(#[$doc:meta])*
        $access:vis u32 $name:ident
        $(
            $(#[$flag_doc:meta])*
            $flag:ident {
                $(
                    $(#[$member_doc:meta])*
                    $idx:literal : $field:ident $( ( $named_getter:ident ) )?,
                )*
            }
        )?
    ) => {
        __bitfield_unchecked!( $(#[$doc])* $access [u32] [32] [u32::MAX] $name $($(#[$flag_doc])* $flag { $( $(#[$member_doc])* $idx : $field $(($named_getter))?, )* } )?);
    };

    // U64
    (
        $(#[$doc:meta])*
        $access:vis u64 $name:ident
        $(
            $(#[$flag_doc:meta])*
            $flag:ident {
                $(
                    $(#[$member_doc:meta])*
                    $idx:literal : $field:ident $( ( $named_getter:ident ) )?,
                )*
            }
        )?
    ) => {
        __bitfield_unchecked!( $(#[$doc])* $access [u64] [64] [u64::MAX] $name $($(#[$flag_doc])* $flag { $( $(#[$member_doc])* $idx : $field $(($named_getter))?, )* } )?);
    };

    // U128
    (
        $(#[$doc:meta])*
        $access:vis u128 $name:ident
        $(
            $(#[$flag_doc:meta])*
            $flag:ident {
                $(
                    $(#[$member_doc:meta])*
                    $idx:literal : $field:ident $( ( $named_getter:ident ) )?,
                )*
            }
        )?
    ) => {
        __bitfield_unchecked!( $(#[$doc])* $access [u128] [128] [u128::MAX] $name $($(#[$flag_doc])* $flag { $( $(#[$member_doc])* $idx : $field $(($named_getter))?, )* } )?);
    };

    // USIZE
    (
        $(#[$doc:meta])*
        $access:vis usize $name:ident
        $(
            $(#[$flag_doc:meta])*
            $flag:ident {
                $(
                    $(#[$member_doc:meta])*
                    $idx:literal : $field:ident $( ( $named_getter:ident ) )?,
                )*
            }
        )?
    ) => {
        __bitfield_unchecked!( $(#[$doc])* $access [usize] [usize::BITS as usize] [usize::MAX] $name $($(#[$flag_doc])* $flag { $( $(#[$member_doc])* $idx : $field $(($named_getter))?, )* } )?);
    };
}

/// Not to be used directly.
///
/// Generates a bitfield struct and flag enum
/// but does no type checking.
/// Passing types that aren't unsigned integer types is undefined behaviour.
///
/// Use the [`bitfield`] macro instead as it enforces a closed set of types.
#[doc(hidden)]
#[macro_export(local_inner_macros)]
macro_rules! __bitfield_unchecked {
    (
        $(#[$field_doc:meta])*
        $access:vis [$type:ty] [$($bits:tt)*] [$max:path] $name:ident
        $(
            $(#[$flag_doc:meta])*
            $flag:ident {
                $(
                    $(#[$member_doc:meta])*
                    $idx:literal : $field:ident $( ( $named_getter:ident ) )?,
                )*
            }
        )?
    ) => {

        $(
            __def_flag_enum! {
                $(#[$flag_doc])*
                $access $name
                $flag {
                    $(
                        $(#[$member_doc])*
                        $idx : $field $(($named_getter))?,
                    )*
                }
            }
        )?

        __def_field_struct! {
            $(#[$field_doc])*
            $access [$type] [$($bits)*] [$max] $name
            $(
                $flag {
                    $(
                        $(#[$member_doc])*
                        $idx : $field $(($named_getter))?,
                    )*
                }
            )?
        }
    };
}

// -------------------------------------------------------------------------------------------------
// Type Definitions

/// Defines a field struct with its members and implementations
#[doc(hidden)]
#[macro_export(local_inner_macros)]
macro_rules! __def_field_struct {
    (
        $(#[$field_doc:meta])*
        $access:vis [$type:ty] [$($bits:tt)*] [$max:path] $name:ident
        $(
            $flag:ident {
                $(
                    $(#[$member_doc:meta])*
                    $idx:literal : $field:ident $( ( $named_getter:ident ) )?,
                )*
            }
        )?
    ) => {

        // Struct: $name
        $(#[$field_doc])*
        #[derive(Copy, Clone, PartialEq, PartialOrd)]
        $access struct $name($type);

        $(
            impl $name {
                $(
                    $(
                        __paste! {
                            pub fn [< is_ $named_getter >](&self) -> bool {
                                self.get($flag::$field)
                            }
                        }

                        __paste! {
                            pub fn [< set_ $named_getter >](&mut self) {
                                self.set($flag::$field);
                            }
                        }

                        __paste! {
                            pub fn [< clear_ $named_getter >](&mut self) {
                                self.clear($flag::$field);
                            }
                        }

                        __paste! {
                            pub fn [< toggle_ $named_getter >](&mut self) {
                                self.toggle($flag::$field);
                            }
                        }
                    )?
                )*
            }
        )?

        // Constants
        __impl_field_constants! {
            $name { type: $type, max: $max, bits: [$($bits)*] }
        }

        // Constructors
        __impl_field_ctors! {
            $name : $type
        }

        // State
        __impl_field_state! {
            $name : $type
        }

        // Indexers
        __impl_field_index_accessors! {
            $name
        }

        $(
            // Flags
            __impl_field_flag_accessors! {
                $name $flag $type
            }
        )?

        // Combinators
        __impl_field_flag_combinators! {
            $name
        }

        // Converters
        __impl_field_flag_converters! {
            $name
        }

        // Name: $name
        __impl_default! {
            $name => { $name(0) }
        }

        // From: $type -> $name
        __impl_from! {
            /// Returns a field instance from value of its underlying type.
            $type as $name (value) => {
                $name(value)
            }
        }

        $(
            // From: $flag -> $name
            __impl_from! {
                /// Returns a field instance from a flag.
                $flag as $name (value) => {
                    Self(0 | (1 << (value as $type)) )
                }
            }
        )?

        // Bitwise: $name + $name
        __impl_bitwise_operators! {
            [Self] for $name : (self rhs -> Self)
            BitAnd => { Self(self.0 & rhs.0) }
            BitOr => { Self(self.0 | rhs.0) }
            BitXor => {  Self(self.0 ^ rhs.0) }
            BitAndAssign => { self.0 &= rhs.0; }
            BitOrAssign => { self.0 |= rhs.0; }
            BitXorAssign => { self.0 ^= rhs.0; }
        }

        $(
            // Bitwise: $name + $flag
            __impl_bitwise_operators! {
                [$flag] for $name : (self rhs -> $name)
                BitAnd => { $name(self.0 & !(1 << (rhs as $type))) }
                BitOr => { $name(self.0 | (1 << (rhs as $type))) }
                BitXor => { $name(self.0 ^ (1 << (rhs as $type))) }
                BitAndAssign => { self.clear(rhs); }
                BitOrAssign => { self.set(rhs); }
                BitXorAssign => { self.toggle(rhs); }
            }

            // Bitwise: &$name + $flag
            __impl_bitwise_operators! {
                [$flag] for & $name : (self rhs -> $name)
                BitAnd => { $name(self.0 & !(1 << (rhs as $type))) }
                BitOr => { $name(self.0 | (1 << (rhs as $type))) }
                BitXor => { $name(self.0 ^ (1 << (rhs as $type))) }
            }

            // Bitwise: $mut $name + $flag
            __impl_bitwise_operators! {
                [$flag] for &mut $name : (self rhs -> $name)
                BitAnd => { $name(self.0 & !(1 << (rhs as $type))) }
                BitOr => { $name(self.0 | (1 << (rhs as $type))) }
                BitXor => { $name(self.0 ^ (1 << (rhs as $type))) }
            }
        )?

        // Bitwise: Debug & Binary
        __impl_formatters! {
            $name (self f) {
                Debug => { core::write!(f, "{}({})", core::stringify!($name), self.as_binary()) }
                Binary => { core::write!(f, "{}", self.as_binary()) }
            }
        }
    };
}

/// Defines a flag enum with its members and implementations
#[doc(hidden)]
#[macro_export(local_inner_macros)]
macro_rules! __def_flag_enum {
    (
        $(#[$flag_doc:meta])*
        $access:vis $name:ident
        $flag:ident {
            $(
                $(#[$member_doc:meta])*
                $idx:literal : $field:ident $( ( $named_getter:ident ) )?,
            )*
        }
    ) => {

        // Enum: $flag
        $(#[$flag_doc])*
        #[derive(Debug, Copy, Clone, PartialEq, PartialOrd)]
        #[repr(u8)]
        $access enum $flag {
            $( $(#[$member_doc])* $field = $idx ),*
        }

        // Bitwise: $flag + $flag
        __impl_bitwise_operators! {
            [Self] for $flag : (self rhs -> $name)
            BitAnd => { $name(0).clear(self) & rhs }
            BitOr => { $name(0).set(self) | rhs }
            BitXor => { $name(0).toggle(self) ^ rhs }
        }

        // Bitwise: u8 -> $flag
        __impl_from! {
            /// Converts a [`u8`] to a flag
            ///
            /// # Errors
            ///
            /// Panics if the supplied [`u8`] is out of range for fields base type.
            /// eg: for a 16-bit field (using [`u16`]),
            /// providing an index of `20` will cause a panic.
            u8 as $flag (value) => {
                $crate::__transmute_one::<$flag>(&[value])
                .expect(std::format!(
                    "failed to transmute {} to {}. {} is not a valid index in {}.",
                    value,
                    core::stringify!($flag),
                    value,
                    core::stringify!($flag)
                ).as_str())
            }
        }

        // Bitwise: $flag -> u8
        __impl_from! {
            /// Returns a flag as its corresponding [`u8`] value.
            $flag as u8 (value) => {
                value as u8
            }
        }

        unsafe impl $crate::__TriviallyTransmutable for $flag {}
    };
}

// -------------------------------------------------------------------------------------------------
// Field Implementations

/// Generates constants for `$name`
#[doc(hidden)]
#[macro_export]
macro_rules! __impl_field_constants {
    ($name:ident { type: $type:ty, max: $max:path, bits: [$($bits:tt)*] }) => {
        $crate::__doc_comment! {
            core::concat!(
                "Constant values describing [`", core::stringify!($name), "`]."
            ),
            impl $name {
                $crate::__doc_comment! {
                    core::concat!(
                      "Number of bits in an instance of [`", core::stringify!($name), "`]."
                    ),
                    pub const BITS: usize = $($bits)*;
                }
                $crate::__doc_comment! {
                    core::concat!(
                      "Number of bytes used by an instance of [`", core::stringify!($name), "`]."
                    ),
                    pub const BYTES: usize = $($bits)* / 8;
                }
                $crate::__doc_comment! {
                    core::concat!(
                      "Maximum valid integer for [`", core::stringify!($name), "`]."
                    ),
                    pub const MAX: $type = $max;
                }
            }
        }
    };
}

/// Generates constructors for `$name`
#[doc(hidden)]
#[macro_export]
macro_rules! __impl_field_ctors {
    ( $name:ident : $type:ty ) => {
        $crate::__doc_comment! {
            core::concat!(
                "Constructors for creating instances of [`", core::stringify!($name), "`]."
            ),
            impl $name {
                $crate::__doc_comment! {
                    core::concat!(
                      "Create a new instance of [`", core::stringify!($name), "`] ",
                      "from a [`", core::stringify!($type), "`] value."
                    ),
                    pub fn new(value: $type) -> Self {
                        Self(value)
                    }
                }

                $crate::__doc_comment! {
                    core::concat!(
                      "Create a new instance of [`", core::stringify!($name), "`] ",
                      "from `0`."
                    ),
                    pub fn zeros() -> Self {
                        Self(0)
                    }
                }

                $crate::__doc_comment! {
                    core::concat!(
                      "Create a new instance of [`", core::stringify!($name), "`] ",
                      "from the maximum possible integer value."
                    ),
                    pub fn ones() -> Self {
                        Self(Self::MAX)
                    }
                }

                $crate::__doc_comment! {
                    core::concat!(
                      "Create a new instance of [`", core::stringify!($name), "`] ",
                      "from a binary string."
                    ),
                    pub fn from_binary_str(value: &str) -> Self {
                        Self(<$type>::from_str_radix(value, 2).unwrap())
                    }
                }

                $crate::__doc_comment! {
                    core::concat!(
                      "Create a new instance of [`", core::stringify!($name), "`] ",
                      "from a binary string."
                    ),
                    pub fn from_binary_string(value: String) -> Self {
                        Self(<$type>::from_str_radix(value.as_str(), 2).unwrap())
                    }
                }
            }
        }
    };
}

/// Generates getters for state fields in `$name`
#[doc(hidden)]
#[macro_export]
macro_rules! __impl_field_state {
    ($name:ident : $type:ty) => {
        $crate::__doc_comment! {
            core::concat!(
                "Current state of this bitfield."
            ),
            impl $name {
                $crate::__doc_comment! {
                    core::concat!(
                      "Returns the current field value as a [`", core::stringify!($type), "`]"
                    ),
                    pub fn as_integer(&self) -> $type {
                        self.0
                    }
                }

                $crate::__doc_comment! {
                    core::concat!(
                      "Returns the current field value as a binary formatted string."
                    ),
                    pub fn as_binary(&self) -> String {
                        std::format!("{:0width$b}", self.0, width = Self::BITS)
                    }
                }
            }
        }
    };
}

/// Generates getters by index for bit fields in `$name`
#[doc(hidden)]
#[macro_export]
macro_rules! __impl_field_index_accessors {
    ($name:ident) => {
        $crate::__doc_comment! {
            core::concat!(
                "Field accessors by index for [`", core::stringify!($name), "`]."
            ),
            impl $name {
                $crate::__doc_comment! {
                    core::concat!(
                        "Returns the value of the bit at the supplied index as a boolean."
                    ),
                    pub fn get_index(&self, index: u8) -> bool {
                        ((self.0 >> index) & 1) == 1
                    }
                }

                $crate::__doc_comment! {
                    core::concat!(
                        "Sets the value of the bit at the supplied index to `1`."
                    ),
                    pub fn set_index(&mut self, index: u8) {
                        self.0 |= (1 << index);
                    }
                }

                $crate::__doc_comment! {
                    core::concat!(
                        "Sets the value of the bit at the supplied index to `0`."
                    ),
                    pub fn clear_index(&mut self, index: u8) {
                        self.0 &= !(1 << index);
                    }
                }

                $crate::__doc_comment! {
                    core::concat!(
                        "Flips the value of the bit at the supplied index."
                    ),
                    pub fn toggle_index(&mut self, index: u8) {
                        self.0 ^= (1 << index);
                    }
                }
            }
        }
    };
}

/// Generates getters by flag for bit fields in `$name`
#[doc(hidden)]
#[macro_export]
macro_rules! __impl_field_flag_accessors {
    ($name:ident $flag:ident $type:ty) => {
        $crate::__doc_comment! {
            core::concat!(
                "Named field accessors by [`", core::stringify!($flag), "`] for [`", core::stringify!($name), "`]."
            ),
            impl $name {
                $crate::__doc_comment! {
                    core::concat!(
                        "Returns the value of the bit at the supplied flag as a boolean."
                    ),
                    pub fn get(&self, flag: $flag) -> bool {
                        ((self.0 >> (flag as $type)) & 1) == 1
                    }
                }

                $crate::__doc_comment! {
                    core::concat!(
                        "Sets the value of the bit at the supplied flag to `1`."
                    ),
                    pub fn set(&mut self, flag: $flag) -> &mut Self {
                        self.0 |= (1 << (flag as $type));
                        self
                    }
                }

                $crate::__doc_comment! {
                    core::concat!(
                        "Sets the value of the bit at the supplied flag to `0`."
                    ),
                    pub fn clear(&mut self, flag: $flag) -> &mut Self {
                        self.0 &= !(1 << (flag as $type));
                        self
                    }
                }

                $crate::__doc_comment! {
                    core::concat!(
                        "Flips the value of the bit at the supplied flag."
                    ),
                    pub fn toggle(&mut self, flag: $flag) -> &mut Self {
                        self.0 ^= (1 << (flag as $type));
                        self
                    }
                }
            }
        }
    };
}

/// Generates combinator methods for `$name`
#[doc(hidden)]
#[macro_export]
macro_rules! __impl_field_flag_combinators {
    ($name:ident) => {
        $crate::__doc_comment! {
            core::concat!(
                "Combinators for [`", core::stringify!($name), "`]."
            ),
            impl $name {
                $crate::__doc_comment! {
                    core::concat!(
                        "Returns a new [`", core::stringify!($name), "`]",
                        "with ones for flags that do not match. ",
                        "Does not consume `self`."
                    ),
                    pub fn diff(&self, other: Self) -> Self {
                        Self(self.0 ^ other.0)
                    }
                }

                $crate::__doc_comment! {
                    core::concat!(
                        "Returns a new [`", core::stringify!($name), "`]",
                        "with ones for flags that were set on either input. ",
                        "Does not consume `self`."
                    ),
                    pub fn combine(&self, other: Self) -> Self {
                        Self(self.0 | other.0)
                    }
                }

                $crate::__doc_comment! {
                    core::concat!(
                        "Returns a new [`", core::stringify!($name), "`]",
                        "with ones for flags that were set on both inputs. ",
                        "Does not consume `self`."
                    ),
                    pub fn intersect(&self, other: Self) -> Self {
                        Self(self.0 & other.0)
                    }
                }
            }
        }
    };
}

/// Generates conversion methods for `$name`
#[doc(hidden)]
#[macro_export]
macro_rules! __impl_field_flag_converters {
    ($name:ident) => {
        $crate::__doc_comment! {
            core::concat!(
                "Conversion methods."
            ),
            impl $name {
                $crate::__doc_comment! {
                    core::concat!(
                        "Returns a new [`", core::stringify!($name), "`]",
                        "with ones for flags that do not match. ",
                        "Consumes `self`."
                    ),
                    pub fn into_diff(self, other: Self) -> Self {
                        Self(self.0 ^ other.0)
                    }
                }

                $crate::__doc_comment! {
                    core::concat!(
                        "Returns a new [`", core::stringify!($name), "`]",
                        "with ones for flags that were set on either input. ",
                        "Consumes `self`."
                    ),
                    pub fn into_combined(self, other: Self) -> Self {
                        Self(self.0 | other.0)
                    }
                }

                $crate::__doc_comment! {
                    core::concat!(
                        "Returns a new [`", core::stringify!($name), "`]",
                        "with ones for flags that were set on both inputs. ",
                        "Consumes `self`."
                    ),
                    pub fn into_intersection(self, other: Self) -> Self {
                        Self(self.0 & other.0)
                    }
                }
            }
        }
    };
}

// -------------------------------------------------------------------------------------------------
// Trait Implementations

// Operators

/// Generates a set of bitwise operators
/// for a set of parameters
#[doc(hidden)]
#[macro_export(local_inner_macros)]
macro_rules! __impl_bitwise_operators {

    // Owned
    (
        [$flag:ty] for $dest:ident : ($self:ident $other:ident -> $output:ident)
        $( BitAnd => $bitand:block )?
        $( BitOr => $bitor:block )?
        $( BitXor => $bitxor:block )?
        $( BitAndAssign => $bitand_assign:block )?
        $( BitOrAssign => $bitor_assign:block )?
        $( BitXorAssign => $bitxor_assign:block )?
    ) => {
        $( __impl_operator! {BitAnd [$flag] bitand for $dest ($self $other -> $output) => $bitand} )?
        $( __impl_operator! {BitOr [$flag] bitor for $dest ($self $other -> $output) => $bitor} )?
        $( __impl_operator! {BitXor [$flag] bitxor for $dest ($self $other -> $output) => $bitxor} )?

        $( __impl_assign_operator! {BitAndAssign [$flag] bitand_assign for $dest ($self $other) => $bitand_assign} )?
        $( __impl_assign_operator! {BitOrAssign [$flag] bitor_assign for $dest ($self $other) => $bitor_assign} )?
        $( __impl_assign_operator! {BitXorAssign [$flag] bitxor_assign for $dest ($self $other) => $bitxor_assign} )?
    };

    // Reference
    (
        [$flag:ty] for & $dest:ident : ($self:ident $other:ident -> $output:ident)
        $( BitAnd => $bitand:block )?
        $( BitOr => $bitor:block )?
        $( BitXor => $bitxor:block )?
    ) => {
        $( __impl_operator! {BitAnd [$flag] bitand for & $dest ($self $other -> $output) => $bitand} )?
        $( __impl_operator! {BitOr [$flag] bitor for & $dest ($self $other -> $output) => $bitor} )?
        $( __impl_operator! {BitXor [$flag] bitxor for & $dest ($self $other -> $output) => $bitxor} )?
    };

    // Mutable Reference
    (
        [$flag:ty] for &mut $dest:ident : ($self:ident $other:ident -> $output:ident)
        $( BitAnd => $bitand:block )?
        $( BitOr => $bitor:block )?
        $( BitXor => $bitxor:block )?
    ) => {
        $( __impl_operator! {BitAnd [$flag] bitand for &mut $dest ($self $other -> $output) => $bitand} )?
        $( __impl_operator! {BitOr [$flag] bitor for &mut $dest ($self $other -> $output) => $bitor} )?
        $( __impl_operator! {BitXor [$flag] bitxor for &mut $dest ($self $other -> $output) => $bitxor} )?
    };
}

/// Generates a generic operator
#[doc(hidden)]
#[macro_export]
macro_rules! __impl_operator {

    // Owned
    ( $op_type:ident [$rhs:ty] $op_lower:ident for $dest:ident ($self:ident $other:ident -> $output:ident) => $block:block ) => {
        impl std::ops::$op_type<$rhs> for $dest {
            type Output = $output;
            fn $op_lower($self, $other: $rhs) -> Self::Output $block
        }
    };

    // Reference
    ( $op_type:ident [$rhs:ty] $op_lower:ident for & $dest:ident ($self:ident $other:ident -> $output:ident) => $block:block ) => {
        impl std::ops::$op_type<$rhs> for &$dest {
            type Output = $output;
            fn $op_lower($self, $other: $rhs) -> Self::Output $block
        }
    };

    // Mutable Reference
    ( $op_type:ident [$rhs:ty] $op_lower:ident for &mut $dest:ident ($self:ident $other:ident -> $output:ident) => $block:block ) => {
        impl std::ops::$op_type<$rhs> for &mut $dest {
            type Output = $output;
            fn $op_lower($self, $other: $rhs) -> Self::Output $block
        }
    };
}

/// Generates a generic assign operator
#[doc(hidden)]
#[macro_export]
macro_rules! __impl_assign_operator {
    ( $op_type:ident [$rhs:ty] $op_lower:ident for $dest:ident ($self:ident $other:ident) => $block:block ) => {
        impl std::ops::$op_type<$rhs> for $dest {
            fn $op_lower(&mut $self, $other: $rhs) $block
        }
    };
}

// Defaults

/// Generates a [`Default`] implementation.
#[doc(hidden)]
#[macro_export]
macro_rules! __impl_default {
    ($name:ident => $block:block) => {
        impl Default for $name {
            fn default() -> Self $block
        }
    };
}

// Converters

/// Generates a [`From`] implementation.
#[doc(hidden)]
#[macro_export]
macro_rules! __impl_from {
    ($(#[$doc:meta])* $from:ty as $to:ident ($arg:ident) => $block:block) => {
        $(#[$doc])*
        impl From<$from> for $to {
            fn from($arg: $from) -> $to $block
        }
    };
}

// Formatters

/// Generates formatter implementations
/// from a set of parameters.
#[doc(hidden)]
#[macro_export(local_inner_macros)]
macro_rules! __impl_formatters {
    ($name:ident ($self:ident $f:ident) { $( $formatter:ident => $block:block )+ } ) => {
        $( __impl_formatter!($formatter for $name ($self $f) => $block); )+
    };
}

/// Generates a generic formatter implementation.
#[doc(hidden)]
#[macro_export]
macro_rules! __impl_formatter {
    ($formatter:ident for $name:ident ($self:ident $f:ident) => $block:block) => {
        impl std::fmt::$formatter for $name {
            fn fmt(&$self, $f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result $block
        }
    };
}

// -------------------------------------------------------------------------------------------------
// Example

#[allow(dead_code)]
#[allow(unused_variables)]
#[cfg(doc)]
pub mod examples {
    //! Examples of generated output using different configurations.
    //!
    //! __This module is only compiled with documentation builds
    //! and will not be available for import.__
    //!
    //! use the [`bitfield`] macro to generate a field struct and flag enum.
    //!

    #[allow(dead_code)]
    #[allow(unused_variables)]
    #[cfg(doc)]
    pub mod full_example {
        //! Example of generated output, using both a bit field and a flag enum.
        //!
        //! __This module is only compiled with documentation builds
        //! and will not be available for import.__
        //!
        //! use the [`bitfield`] macro to generate a field struct and flag enum.
        //!
        use super::*;

        bitfield! {
            /// Optional docstring for my example bit field.
            pub u8 ExampleField
            /// Optional docstring for my flag enum.
            ExampleFlags {
                /// Optional docstring for Flag0
                0 : Flag0,
                /// Optional docstring for Flag1
                1 : Flag1,
                /// Optional docstring for Flag2
                2 : Flag2,
                /// Optional docstring for Flag3
                3 : Flag3,
                /// Optional docstring for Flag4
                4 : Flag4,
                /// Optional docstring for Flag5
                5 : Flag5,
                /// Optional docstring for Flag6
                6 : Flag6,
                /// Optional docstring for Flag7
                7 : Flag7,
            }
        }
    }

    #[allow(dead_code)]
    #[allow(unused_variables)]
    #[cfg(doc)]
    pub mod flagless_example {
        //! Example of generated output, using only a bit field without a flag enum.
        //!
        //! __This module is only compiled with documentation builds
        //! and will not be available for import.__
        //!
        //! use the [`bitfield`] macro to generate a field struct and flag enum.
        //!
        use super::*;

        bitfield! {
            /// Optional docstring for my example bit field.
            pub u8 ExampleField
        }
    }
}

// -------------------------------------------------------------------------------------------------
// Tests

#[allow(dead_code)]
#[allow(unused_variables)]
#[cfg(test)]
mod test {
    use super::*;
    use std::{u128, u16, u32, u64, usize};

    macro_rules! tests {
        ( $( $name:ident => $block:block )+ ) => {
            $(
                #[test]
                fn $name() $block
            )+
        };
    }

    bitfield! {
        /// My Test Thing
        u8 MyFieldU8
        MyFlagsU8 {
            0 : Flag0, 1 : Flag1, 2 : Flag2, 3 : Flag3, 4 : Flag4, 5 : Flag5, 6 : Flag6, 7 : Flag7,
        }
    }

    bitfield! {
        /// My Test Thing
        u16 MyFieldU16
        MyFlagsU16 {
            0 : Flag0, 1 : Flag1, 2 : Flag2, 3 : Flag3, 4 : Flag4, 5 : Flag5, 6 : Flag6, 7 : Flag7,
            8 : Flag8, 9 : Flag9, 10 : Flag10, 11 : Flag11, 12 : Flag12, 13 : Flag13, 14 : Flag14, 15 : Flag15,
        }
    }

    bitfield! {
        /// My Test Thing
        u32 MyFieldU32
        MyFlagsU32 {
            0 : Flag0, 1 : Flag1, 2 : Flag2, 3 : Flag3, 4 : Flag4, 5 : Flag5, 6 : Flag6, 7 : Flag7,
            8 : Flag8, 9 : Flag9, 10 : Flag10, 11 : Flag11, 12 : Flag12, 13 : Flag13, 14 : Flag14, 15 : Flag15,
            16 : Flag16, 17 : Flag17, 18 : Flag18, 19 : Flag19, 20 : Flag20, 21 : Flag21, 22 : Flag22, 23 : Flag23,
            24 : Flag24, 25 : Flag25, 26 : Flag26, 27 : Flag27, 28 : Flag28, 29 : Flag29, 30 : Flag30, 31 : Flag31,
        }
    }

    bitfield! {
        /// My Test Thing
        u64 MyFieldU64
        MyFlagsU64 {
            0 : Flag0, 1 : Flag1, 2 : Flag2, 3 : Flag3, 4 : Flag4, 5 : Flag5, 6 : Flag6, 7 : Flag7,
            8 : Flag8, 9 : Flag9, 10 : Flag10, 11 : Flag11, 12 : Flag12, 13 : Flag13, 14 : Flag14, 15 : Flag15,
            16 : Flag16, 17 : Flag17, 18 : Flag18, 19 : Flag19, 20 : Flag20, 21 : Flag21, 22 : Flag22, 23 : Flag23,
            24 : Flag24, 25 : Flag25, 26 : Flag26, 27 : Flag27, 28 : Flag28, 29 : Flag29, 30 : Flag30, 31 : Flag31,
            32 : Flag32, 33 : Flag33, 34 : Flag34, 35 : Flag35, 36 : Flag36, 37 : Flag37, 38 : Flag38, 39 : Flag39,
            40 : Flag40, 41 : Flag41, 42 : Flag42, 43 : Flag43, 44 : Flag44, 45 : Flag45, 46 : Flag46, 47 : Flag47,
            48 : Flag48, 49 : Flag49, 50 : Flag50, 51 : Flag51, 52 : Flag52, 53 : Flag53, 54 : Flag54, 55 : Flag55,
            56 : Flag56, 57 : Flag57, 58 : Flag58, 59 : Flag59, 60 : Flag60, 61 : Flag61, 62 : Flag62, 63 : Flag63,
        }
    }

    bitfield! {
        /// My Test Thing
        u128 MyFieldU128
        MyFlagsU128 {
            0 : Flag0, 1 : Flag1, 2 : Flag2, 3 : Flag3, 4 : Flag4, 5 : Flag5, 6 : Flag6, 7 : Flag7,
            8 : Flag8, 9 : Flag9, 10 : Flag10, 11 : Flag11, 12 : Flag12, 13 : Flag13, 14 : Flag14, 15 : Flag15,
            16 : Flag16, 17 : Flag17, 18 : Flag18, 19 : Flag19, 20 : Flag20, 21 : Flag21, 22 : Flag22, 23 : Flag23,
            24 : Flag24, 25 : Flag25, 26 : Flag26, 27 : Flag27, 28 : Flag28, 29 : Flag29, 30 : Flag30, 31 : Flag31,
            32 : Flag32, 33 : Flag33, 34 : Flag34, 35 : Flag35, 36 : Flag36, 37 : Flag37, 38 : Flag38, 39 : Flag39,
            40 : Flag40, 41 : Flag41, 42 : Flag42, 43 : Flag43, 44 : Flag44, 45 : Flag45, 46 : Flag46, 47 : Flag47,
            48 : Flag48, 49 : Flag49, 50 : Flag50, 51 : Flag51, 52 : Flag52, 53 : Flag53, 54 : Flag54, 55 : Flag55,
            56 : Flag56, 57 : Flag57, 58 : Flag58, 59 : Flag59, 60 : Flag60, 61 : Flag61, 62 : Flag62, 63 : Flag63,
            64 : Flag64, 65 : Flag65, 66 : Flag66, 67 : Flag67, 68 : Flag68, 69 : Flag69, 70 : Flag70, 71 : Flag71,
            72 : Flag72, 73 : Flag73, 74 : Flag74, 75 : Flag75, 76 : Flag76, 77 : Flag77, 78 : Flag78, 79 : Flag79,
            80 : Flag80, 81 : Flag81, 82 : Flag82, 83 : Flag83, 84 : Flag84, 85 : Flag85, 86 : Flag86, 87 : Flag87,
            88 : Flag88, 89 : Flag89, 90 : Flag90, 91 : Flag91, 92 : Flag92, 93 : Flag93, 94 : Flag94, 95 : Flag95,
            96 : Flag96, 97 : Flag97, 98 : Flag98, 99 : Flag99, 100 : Flag100, 101 : Flag101, 102 : Flag102, 103 : Flag103,
            104 : Flag104, 105 : Flag105, 106 : Flag106, 107 : Flag107, 108 : Flag108, 109 : Flag109, 110 : Flag110, 111 : Flag111,
            112 : Flag112, 113 : Flag113, 114 : Flag114, 115 : Flag115, 116 : Flag116, 117 : Flag117, 118 : Flag118, 119 : Flag119,
            120 : Flag120, 121 : Flag121, 122 : Flag122, 123 : Flag123, 124 : Flag124, 125 : Flag125, 126 : Flag126, 127 : Flag127,
        }
    }

    #[cfg(target_pointer_width = "64")]
    bitfield! {
        /// My Test Thing
        usize MyFieldUsize
        MyFlagsUsize {
            0 : Flag0, 1 : Flag1, 2 : Flag2, 3 : Flag3, 4 : Flag4, 5 : Flag5, 6 : Flag6, 7 : Flag7,
            8 : Flag8, 9 : Flag9, 10 : Flag10, 11 : Flag11, 12 : Flag12, 13 : Flag13, 14 : Flag14, 15 : Flag15,
            16 : Flag16, 17 : Flag17, 18 : Flag18, 19 : Flag19, 20 : Flag20, 21 : Flag21, 22 : Flag22, 23 : Flag23,
            24 : Flag24, 25 : Flag25, 26 : Flag26, 27 : Flag27, 28 : Flag28, 29 : Flag29, 30 : Flag30, 31 : Flag31,
            32 : Flag32, 33 : Flag33, 34 : Flag34, 35 : Flag35, 36 : Flag36, 37 : Flag37, 38 : Flag38, 39 : Flag39,
            40 : Flag40, 41 : Flag41, 42 : Flag42, 43 : Flag43, 44 : Flag44, 45 : Flag45, 46 : Flag46, 47 : Flag47,
            48 : Flag48, 49 : Flag49, 50 : Flag50, 51 : Flag51, 52 : Flag52, 53 : Flag53, 54 : Flag54, 55 : Flag55,
            56 : Flag56, 57 : Flag57, 58 : Flag58, 59 : Flag59, 60 : Flag60, 61 : Flag61, 62 : Flag62, 63 : Flag63,
        }
    }

    #[cfg(target_pointer_width = "32")]
    bitfield! {
        /// My Test Thing
        usize MyFieldUsize
        MyFlagsUsize {
            0 : Flag0, 1 : Flag1, 2 : Flag2, 3 : Flag3, 4 : Flag4, 5 : Flag5, 6 : Flag6, 7 : Flag7,
            8 : Flag8, 9 : Flag9, 10 : Flag10, 11 : Flag11, 12 : Flag12, 13 : Flag13, 14 : Flag14, 15 : Flag15,
            16 : Flag16, 17 : Flag17, 18 : Flag18, 19 : Flag19, 20 : Flag20, 21 : Flag21, 22 : Flag22, 23 : Flag23,
            24 : Flag24, 25 : Flag25, 26 : Flag26, 27 : Flag27, 28 : Flag28, 29 : Flag29, 30 : Flag30, 31 : Flag31,
        }
    }

    bitfield! {
        /// My Test Thing
        u8 MyFieldU8Nf
        MyFlagsU8Nf {
            0 : Flag0, 1 : Flag1, 2 : Flag2, 3 : Flag3, 4 : Flag4, 5 : Flag5, 6 : Flag6, 7 : Flag7,
        }
    }

    bitfield! {
        /// My Test Thing
        u16 MyFieldU16Nf
        MyFlagsU16Nf {
            0 : Flag0, 1 : Flag1, 2 : Flag2, 3 : Flag3, 4 : Flag4, 5 : Flag5, 6 : Flag6, 7 : Flag7,
            8 : Flag8, 9 : Flag9, 10 : Flag10, 11 : Flag11, 12 : Flag12, 13 : Flag13, 14 : Flag14, 15 : Flag15,
        }
    }

    bitfield! {
        /// My Test Thing
        u32 MyFieldU32Nf
        MyFlagsU32Nf {
            0 : Flag0, 1 : Flag1, 2 : Flag2, 3 : Flag3, 4 : Flag4, 5 : Flag5, 6 : Flag6, 7 : Flag7,
            8 : Flag8, 9 : Flag9, 10 : Flag10, 11 : Flag11, 12 : Flag12, 13 : Flag13, 14 : Flag14, 15 : Flag15,
            16 : Flag16, 17 : Flag17, 18 : Flag18, 19 : Flag19, 20 : Flag20, 21 : Flag21, 22 : Flag22, 23 : Flag23,
            24 : Flag24, 25 : Flag25, 26 : Flag26, 27 : Flag27, 28 : Flag28, 29 : Flag29, 30 : Flag30, 31 : Flag31,
        }
    }

    bitfield! {
        /// My Test Thing
        u64 MyFieldU64Nf
        MyFlagsU64Nf {
            0 : Flag0, 1 : Flag1, 2 : Flag2, 3 : Flag3, 4 : Flag4, 5 : Flag5, 6 : Flag6, 7 : Flag7,
            8 : Flag8, 9 : Flag9, 10 : Flag10, 11 : Flag11, 12 : Flag12, 13 : Flag13, 14 : Flag14, 15 : Flag15,
            16 : Flag16, 17 : Flag17, 18 : Flag18, 19 : Flag19, 20 : Flag20, 21 : Flag21, 22 : Flag22, 23 : Flag23,
            24 : Flag24, 25 : Flag25, 26 : Flag26, 27 : Flag27, 28 : Flag28, 29 : Flag29, 30 : Flag30, 31 : Flag31,
            32 : Flag32, 33 : Flag33, 34 : Flag34, 35 : Flag35, 36 : Flag36, 37 : Flag37, 38 : Flag38, 39 : Flag39,
            40 : Flag40, 41 : Flag41, 42 : Flag42, 43 : Flag43, 44 : Flag44, 45 : Flag45, 46 : Flag46, 47 : Flag47,
            48 : Flag48, 49 : Flag49, 50 : Flag50, 51 : Flag51, 52 : Flag52, 53 : Flag53, 54 : Flag54, 55 : Flag55,
            56 : Flag56, 57 : Flag57, 58 : Flag58, 59 : Flag59, 60 : Flag60, 61 : Flag61, 62 : Flag62, 63 : Flag63,
        }
    }

    bitfield! {
        /// My Test Thing
        u128 MyFieldU128Nf
        MyFlagsU128Nf {
            0 : Flag0, 1 : Flag1, 2 : Flag2, 3 : Flag3, 4 : Flag4, 5 : Flag5, 6 : Flag6, 7 : Flag7,
            8 : Flag8, 9 : Flag9, 10 : Flag10, 11 : Flag11, 12 : Flag12, 13 : Flag13, 14 : Flag14, 15 : Flag15,
            16 : Flag16, 17 : Flag17, 18 : Flag18, 19 : Flag19, 20 : Flag20, 21 : Flag21, 22 : Flag22, 23 : Flag23,
            24 : Flag24, 25 : Flag25, 26 : Flag26, 27 : Flag27, 28 : Flag28, 29 : Flag29, 30 : Flag30, 31 : Flag31,
            32 : Flag32, 33 : Flag33, 34 : Flag34, 35 : Flag35, 36 : Flag36, 37 : Flag37, 38 : Flag38, 39 : Flag39,
            40 : Flag40, 41 : Flag41, 42 : Flag42, 43 : Flag43, 44 : Flag44, 45 : Flag45, 46 : Flag46, 47 : Flag47,
            48 : Flag48, 49 : Flag49, 50 : Flag50, 51 : Flag51, 52 : Flag52, 53 : Flag53, 54 : Flag54, 55 : Flag55,
            56 : Flag56, 57 : Flag57, 58 : Flag58, 59 : Flag59, 60 : Flag60, 61 : Flag61, 62 : Flag62, 63 : Flag63,
            64 : Flag64, 65 : Flag65, 66 : Flag66, 67 : Flag67, 68 : Flag68, 69 : Flag69, 70 : Flag70, 71 : Flag71,
            72 : Flag72, 73 : Flag73, 74 : Flag74, 75 : Flag75, 76 : Flag76, 77 : Flag77, 78 : Flag78, 79 : Flag79,
            80 : Flag80, 81 : Flag81, 82 : Flag82, 83 : Flag83, 84 : Flag84, 85 : Flag85, 86 : Flag86, 87 : Flag87,
            88 : Flag88, 89 : Flag89, 90 : Flag90, 91 : Flag91, 92 : Flag92, 93 : Flag93, 94 : Flag94, 95 : Flag95,
            96 : Flag96, 97 : Flag97, 98 : Flag98, 99 : Flag99, 100 : Flag100, 101 : Flag101, 102 : Flag102, 103 : Flag103,
            104 : Flag104, 105 : Flag105, 106 : Flag106, 107 : Flag107, 108 : Flag108, 109 : Flag109, 110 : Flag110, 111 : Flag111,
            112 : Flag112, 113 : Flag113, 114 : Flag114, 115 : Flag115, 116 : Flag116, 117 : Flag117, 118 : Flag118, 119 : Flag119,
            120 : Flag120, 121 : Flag121, 122 : Flag122, 123 : Flag123, 124 : Flag124, 125 : Flag125, 126 : Flag126, 127 : Flag127,
        }
    }

    #[cfg(target_pointer_width = "64")]
    bitfield! {
        /// My Test Thing
        usize MyFieldUsizeNf
        MyFlagsUsizeNf {
            0 : Flag0, 1 : Flag1, 2 : Flag2, 3 : Flag3, 4 : Flag4, 5 : Flag5, 6 : Flag6, 7 : Flag7,
            8 : Flag8, 9 : Flag9, 10 : Flag10, 11 : Flag11, 12 : Flag12, 13 : Flag13, 14 : Flag14, 15 : Flag15,
            16 : Flag16, 17 : Flag17, 18 : Flag18, 19 : Flag19, 20 : Flag20, 21 : Flag21, 22 : Flag22, 23 : Flag23,
            24 : Flag24, 25 : Flag25, 26 : Flag26, 27 : Flag27, 28 : Flag28, 29 : Flag29, 30 : Flag30, 31 : Flag31,
            32 : Flag32, 33 : Flag33, 34 : Flag34, 35 : Flag35, 36 : Flag36, 37 : Flag37, 38 : Flag38, 39 : Flag39,
            40 : Flag40, 41 : Flag41, 42 : Flag42, 43 : Flag43, 44 : Flag44, 45 : Flag45, 46 : Flag46, 47 : Flag47,
            48 : Flag48, 49 : Flag49, 50 : Flag50, 51 : Flag51, 52 : Flag52, 53 : Flag53, 54 : Flag54, 55 : Flag55,
            56 : Flag56, 57 : Flag57, 58 : Flag58, 59 : Flag59, 60 : Flag60, 61 : Flag61, 62 : Flag62, 63 : Flag63,
        }
    }

    #[cfg(target_pointer_width = "32")]
    bitfield! {
        /// My Test Thing
        usize MyFieldUsizeNf
        MyFlagsUsizeNf {
            0 : Flag0, 1 : Flag1, 2 : Flag2, 3 : Flag3, 4 : Flag4, 5 : Flag5, 6 : Flag6, 7 : Flag7,
            8 : Flag8, 9 : Flag9, 10 : Flag10, 11 : Flag11, 12 : Flag12, 13 : Flag13, 14 : Flag14, 15 : Flag15,
            16 : Flag16, 17 : Flag17, 18 : Flag18, 19 : Flag19, 20 : Flag20, 21 : Flag21, 22 : Flag22, 23 : Flag23,
            24 : Flag24, 25 : Flag25, 26 : Flag26, 27 : Flag27, 28 : Flag28, 29 : Flag29, 30 : Flag30, 31 : Flag31,
        }
    }

    const U8: u8 = 123;
    const B8: &str = "01111011";
    const U16: u16 = 51526;
    const B16: &str = "1100100101000110";
    const U32: u32 = 2828091834;
    const B32: &str = "10101000100100010100000110111010";
    const U64: u64 = 18325744043706550615;
    const B64: &str = "1111111001010010000111110001110100100001110111000000100101010111";
    const U128: u128 = 160140183460469230739687303714882104727;
    const B128: &str = "01111000011110011110000010100001011000111110000100101011010111011111111110000110011101100010110100100000110001101010110110010111";

    #[cfg(target_pointer_width = "64")]
    const USIZE: usize = 18325744043706550615;
    #[cfg(target_pointer_width = "64")]
    const BSIZE: &str = "1111111001010010000111110001110100100001110111000000100101010111";

    #[cfg(target_pointer_width = "32")]
    const USIZE: usize = 2828091834;
    #[cfg(target_pointer_width = "32")]
    const BSIZE: &str = "10101000100100010100000110111010";

    // constructors
    tests! {
        new_u8 => {
            MyFieldU8(U8);
        }
        new_u16 => {
            MyFieldU16(U16);
        }
        new_u32 => {
            MyFieldU32(U32);
        }
        new_u64 => {
            MyFieldU64(U64);
        }
        new_u128 => {
            MyFieldU128(U128);
        }
        new_usize => {
            MyFieldUsize(USIZE);
        }
    }

    // from binary str
    tests! {
        from_binary_str_u8 => {
            assert_eq!(MyFieldU8::from_binary_str(B8), MyFieldU8(U8));
        }
        from_binary_str_u16 => {
            assert_eq!(MyFieldU16::from_binary_str(B16), MyFieldU16(U16));
        }
        from_binary_str_u32 => {
            assert_eq!(MyFieldU32::from_binary_str(B32), MyFieldU32(U32));
        }
        from_binary_str_u64 => {
            assert_eq!(MyFieldU64::from_binary_str(B64), MyFieldU64(U64));
        }
        from_binary_str_u128 => {
            assert_eq!(MyFieldU128::from_binary_str(B128), MyFieldU128(U128));
        }
        from_binary_str_usize => {
            assert_eq!(MyFieldUsize::from_binary_str(BSIZE), MyFieldUsize(USIZE));
        }
    }

    // from binary string
    tests! {
        from_binary_string_u8 => {
            assert_eq!(MyFieldU8::from_binary_string(String::from(B8)), MyFieldU8(U8));
        }
        from_binary_string_u16 => {
            assert_eq!(MyFieldU16::from_binary_string(String::from(B16)), MyFieldU16(U16));
        }
        from_binary_string_u32 => {
            assert_eq!(MyFieldU32::from_binary_string(String::from(B32)), MyFieldU32(U32));
        }
        from_binary_string_u64 => {
            assert_eq!(MyFieldU64::from_binary_string(String::from(B64)), MyFieldU64(U64));
        }
        from_binary_string_u128 => {
            assert_eq!(MyFieldU128::from_binary_string(String::from(B128)), MyFieldU128(U128));
        }
        from_binary_string_usize => {
            assert_eq!(MyFieldUsize::from_binary_string(String::from(BSIZE)), MyFieldUsize(USIZE));
        }
    }

    // ones
    tests! {
        ones_u8 => {
            assert_eq!(MyFieldU8::ones(), MyFieldU8(u8::MAX));
        }
        ones_u16 => {
            assert_eq!(MyFieldU16::ones(), MyFieldU16(u16::MAX));
        }
        ones_u32 => {
            assert_eq!(MyFieldU32::ones(), MyFieldU32(u32::MAX));
        }
        ones_u64 => {
            assert_eq!(MyFieldU64::ones(), MyFieldU64(u64::MAX));
        }
        ones_u128 => {
            assert_eq!(MyFieldU128::ones(), MyFieldU128(u128::MAX));
        }
        ones_usize => {
            assert_eq!(MyFieldUsize::ones(), MyFieldUsize(usize::MAX));
        }
    }

    // zeros
    tests! {
        zeros_u8 => {
            assert_eq!(MyFieldU8::zeros(), MyFieldU8(0));
        }
        zeros_u16 => {
            assert_eq!(MyFieldU16::zeros(), MyFieldU16(0));
        }
        zeros_u32 => {
            assert_eq!(MyFieldU32::zeros(), MyFieldU32(0));
        }
        zeros_u64 => {
            assert_eq!(MyFieldU64::zeros(), MyFieldU64(0));
        }
        zeros_u128 => {
            assert_eq!(MyFieldU128::zeros(), MyFieldU128(0));
        }
        zeros_usize => {
            assert_eq!(MyFieldUsize::zeros(), MyFieldUsize(0));
        }
    }

    // defaults
    tests! {
        default_u8 => {
            assert_eq!(0, MyFieldU8::default().as_integer())
        }
        default_u16 => {
            assert_eq!(0, MyFieldU16::default().as_integer())
        }
        default_u32 => {
            assert_eq!(0, MyFieldU32::default().as_integer())
        }
        default_u64 => {
            assert_eq!(0, MyFieldU64::default().as_integer())
        }
        default_u128 => {
            assert_eq!(0, MyFieldU128::default().as_integer())
        }
        default_usize => {
            assert_eq!(0, MyFieldUsize::default().as_integer())
        }
    }

    // bit count
    tests! {
        bit_count_u8 => {
            let field = MyFieldU8(U8);
            assert_eq!(u8::BITS as usize, MyFieldU8::BITS)
        }
        bit_count_u16 => {
            let field = MyFieldU16(U16);
            assert_eq!(u16::BITS as usize, MyFieldU16::BITS)
        }
        bit_count_u32 => {
            let field = MyFieldU32(U32);
            assert_eq!(u32::BITS as usize, MyFieldU32::BITS)
        }
        bit_count_u64 => {
            let field = MyFieldU64(U64);
            assert_eq!(u64::BITS as usize, MyFieldU64::BITS)
        }
        bit_count_u128 => {
            let field = MyFieldU128(U128);
            assert_eq!(u128::BITS as usize, MyFieldU128::BITS)
        }
        bit_count_usize => {
            let field = MyFieldUsize(USIZE);
            assert_eq!(usize::BITS as usize, MyFieldUsize::BITS)
        }
    }

    // byte count
    tests! {
        byte_count_u8 => {
            let field = MyFieldU8(U8);
            assert_eq!(1, MyFieldU8::BYTES)
        }
        byte_count_u16 => {
            let field = MyFieldU16(U16);
            assert_eq!(2, MyFieldU16::BYTES)
        }
        byte_count_u32 => {
            let field = MyFieldU32(U32);
            assert_eq!(4, MyFieldU32::BYTES)
        }
        byte_count_u64 => {
            let field = MyFieldU64(U64);
            assert_eq!(8, MyFieldU64::BYTES)
        }
        byte_count_u128 => {
            let field = MyFieldU128(U128);
            assert_eq!(16, MyFieldU128::BYTES)
        }
        byte_count_usize => {
            let field = MyFieldUsize(USIZE);
            assert_eq!((usize::BITS as usize) / 8, MyFieldUsize::BYTES)
        }
    }

    // integer value
    tests! {
        integer_u8 => {
            let field = MyFieldU8(U8);
            assert_eq!(U8, field.as_integer())
        }
        integer_u16 => {
            let field = MyFieldU16(U16);
            assert_eq!(U16, field.as_integer())
        }
        integer_u32 => {
            let field = MyFieldU32(U32);
            assert_eq!(U32, field.as_integer())
        }
        integer_u64 => {
            let field = MyFieldU64(U64);
            assert_eq!(U64, field.as_integer())
        }
        integer_u128 => {
            let field = MyFieldU128(U128);
            assert_eq!(U128, field.as_integer())
        }
        integer_usize => {
            let field = MyFieldUsize(USIZE);
            assert_eq!(USIZE, field.as_integer())
        }
    }

    // binary value
    tests! {
        binary_u8 => {
            let field = MyFieldU8(U8);
            assert_eq!(B8, field.as_binary())
        }
        binary_u16 => {
            let field = MyFieldU16(U16);
            assert_eq!(B16, field.as_binary())
        }
        binary_u32 => {
            let field = MyFieldU32(U32);
            assert_eq!(B32, field.as_binary())
        }
        binary_u64 => {
            let field = MyFieldU64(U64);
            assert_eq!(B64, field.as_binary())
        }
        binary_u128 => {
            let field = MyFieldU128(U128);
            assert_eq!(B128, field.as_binary())
        }
        binary_usize => {
            let field = MyFieldUsize(USIZE);
            assert_eq!(BSIZE, field.as_binary())
        }
    }

    // get index
    tests! {
        get_index_u8 => {
            let field = MyFieldU8(U8);
            for (index, char) in B8.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get_index(index as u8))
                } else if char == '0' {
                    assert!(!field.get_index(index as u8))
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
        get_index_u16 => {
            let field = MyFieldU16(U16);
            for (index, char) in B16.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get_index(index as u8))
                } else if char == '0' {
                    assert!(!field.get_index(index as u8))
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
        get_index_u32 => {
            let field = MyFieldU32(U32);
            for (index, char) in B32.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get_index(index as u8))
                } else if char == '0' {
                    assert!(!field.get_index(index as u8))
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
        get_index_u64 => {
            let field = MyFieldU64(U64);
            for (index, char) in B64.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get_index(index as u8))
                } else if char == '0' {
                    assert!(!field.get_index(index as u8))
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
        get_index_u128 => {
            let field = MyFieldU128(U128);
            for (index, char) in B128.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get_index(index as u8))
                } else if char == '0' {
                    assert!(!field.get_index(index as u8))
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
        get_index_usize => {
            let field = MyFieldUsize(USIZE);
            for (index, char) in BSIZE.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get_index(index as u8))
                } else if char == '0' {
                    assert!(!field.get_index(index as u8))
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
    }

    // set index
    tests! {
        set_index_u8 => {
            let mut field = MyFieldU8(0);
            for (index, char) in B8.chars().rev().enumerate() {
                field.set_index(index as u8);
                for (s_index, _char) in B8.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        set_index_u16 => {
            let mut field = MyFieldU16(0);
            for (index, char) in B16.chars().rev().enumerate() {
                field.set_index(index as u8);
                for (s_index, _char) in B16.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        set_index_u32 => {
            let mut field = MyFieldU32(0);
            for (index, char) in B32.chars().rev().enumerate() {
                field.set_index(index as u8);
                for (s_index, _char) in B32.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        set_index_u64 => {
            let mut field = MyFieldU64(0);
            for (index, char) in B64.chars().rev().enumerate() {
                field.set_index(index as u8);
                for (s_index, _char) in B64.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        set_index_u128 => {
            let mut field = MyFieldU128(0);
            for (index, char) in B128.chars().rev().enumerate() {
                field.set_index(index as u8);
                for (s_index, _char) in B128.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        set_index_usize => {
            let mut field = MyFieldUsize(0);
            for (index, char) in BSIZE.chars().rev().enumerate() {
                field.set_index(index as u8);
                for (s_index, _char) in BSIZE.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
    }

    // clear index
    tests! {
        clear_index_u8 => {
            let mut field = MyFieldU8(u8::MAX);
            for (index, char) in B8.chars().rev().enumerate() {
                assert!(field.get_index(index as u8), "{:#?}", field);
                field.clear_index(index as u8);
                for (s_index, _char) in B8.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        clear_index_u16 => {
            let mut field = MyFieldU16(u16::MAX);
            for (index, char) in B16.chars().rev().enumerate() {
                assert!(field.get_index(index as u8), "{:#?}", field);
                field.clear_index(index as u8);
                for (s_index, _char) in B16.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        clear_index_u32 => {
            let mut field = MyFieldU32(u32::MAX);
            for (index, char) in B32.chars().rev().enumerate() {
                assert!(field.get_index(index as u8), "{:#?}", field);
                field.clear_index(index as u8);
                for (s_index, _char) in B32.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        clear_index_u64 => {
            let mut field = MyFieldU64(u64::MAX);
            for (index, char) in B64.chars().rev().enumerate() {
                assert!(field.get_index(index as u8), "{:#?}", field);
                field.clear_index(index as u8);
                for (s_index, _char) in B64.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        clear_index_u128 => {
            let mut field = MyFieldU128(u128::MAX);
            for (index, char) in B128.chars().rev().enumerate() {
                assert!(field.get_index(index as u8), "{:#?}", field);
                field.clear_index(index as u8);
                for (s_index, _char) in B128.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        clear_index_usize => {
            let mut field = MyFieldUsize(usize::MAX);
            for (index, char) in BSIZE.chars().rev().enumerate() {
                assert!(field.get_index(index as u8), "{:#?}", field);
                field.clear_index(index as u8);
                for (s_index, _char) in BSIZE.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
    }

    // toggle index
    tests! {
        toggle_index_u8 => {
            let mut field = MyFieldU8(U8);
            for (index, char) in B8.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get_index(index as u8));
                    field.toggle_index(index as u8);
                    assert!(!field.get_index(index as u8));
                } else if char == '0' {
                    assert!(!field.get_index(index as u8));
                    field.toggle_index(index as u8);
                    assert!(field.get_index(index as u8));
                } else {
                    panic!("invalid char in binary string");
                }
            }
        }
        toggle_index_u16 => {
            let mut field = MyFieldU16(U16);
            for (index, char) in B16.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get_index(index as u8));
                    field.toggle_index(index as u8);
                    assert!(!field.get_index(index as u8));
                } else if char == '0' {
                    assert!(!field.get_index(index as u8));
                    field.toggle_index(index as u8);
                    assert!(field.get_index(index as u8));
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
        toggle_index_u32 => {
            let mut field = MyFieldU32(U32);
            for (index, char) in B32.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get_index(index as u8));
                    field.toggle_index(index as u8);
                    assert!(!field.get_index(index as u8));
                } else if char == '0' {
                    assert!(!field.get_index(index as u8));
                    field.toggle_index(index as u8);
                    assert!(field.get_index(index as u8));
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
        toggle_index_u64 => {
            let mut field = MyFieldU64(U64);
            for (index, char) in B64.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get_index(index as u8));
                    field.toggle_index(index as u8);
                    assert!(!field.get_index(index as u8));
                } else if char == '0' {
                    assert!(!field.get_index(index as u8));
                    field.toggle_index(index as u8);
                    assert!(field.get_index(index as u8));
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
        toggle_index_u128 => {
            let mut field = MyFieldU128(U128);
            for (index, char) in B128.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get_index(index as u8));
                    field.toggle_index(index as u8);
                    assert!(!field.get_index(index as u8));
                } else if char == '0' {
                    assert!(!field.get_index(index as u8));
                    field.toggle_index(index as u8);
                    assert!(field.get_index(index as u8));
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
        toggle_index_usize => {
            let mut field = MyFieldUsize(USIZE);
            for (index, char) in BSIZE.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get_index(index as u8));
                    field.toggle_index(index as u8);
                    assert!(!field.get_index(index as u8));
                } else if char == '0' {
                    assert!(!field.get_index(index as u8));
                    field.toggle_index(index as u8);
                    assert!(field.get_index(index as u8));
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
    }

    // get
    tests! {
        get_u8 => {
            let field = MyFieldU8(U8);
            for (index, char) in B8.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get(MyFlagsU8::from(index as u8)))
                } else if char == '0' {
                    assert!(!field.get(MyFlagsU8::from(index as u8)))
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
        get_u16 => {
            let field = MyFieldU16(U16);
            for (index, char) in B16.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get(MyFlagsU16::from(index as u8)))
                } else if char == '0' {
                    assert!(!field.get(MyFlagsU16::from(index as u8)))
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
        get_u32 => {
            let field = MyFieldU32(U32);
            for (index, char) in B32.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get(MyFlagsU32::from(index as u8)))
                } else if char == '0' {
                    assert!(!field.get(MyFlagsU32::from(index as u8)))
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
        get_u64 => {
            let field = MyFieldU64(U64);
            for (index, char) in B64.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get(MyFlagsU64::from(index as u8)))
                } else if char == '0' {
                    assert!(!field.get(MyFlagsU64::from(index as u8)))
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
        get_u128 => {
            let field = MyFieldU128(U128);
            for (index, char) in B128.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get(MyFlagsU128::from(index as u8)))
                } else if char == '0' {
                    assert!(!field.get(MyFlagsU128::from(index as u8)))
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
        get_usize => {
            let field = MyFieldUsize(USIZE);
            for (index, char) in BSIZE.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get(MyFlagsUsize::from(index as u8)))
                } else if char == '0' {
                    assert!(!field.get(MyFlagsUsize::from(index as u8)))
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
    }

    // set
    tests! {
        set_u8 => {
            let mut field = MyFieldU8(0);
            for (index, char) in B8.chars().rev().enumerate() {
                field.set(MyFlagsU8::from(index as u8));
                for (s_index, _char) in B8.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        set_u16 => {
            let mut field = MyFieldU16(0);
            for (index, char) in B16.chars().rev().enumerate() {
                field.set(MyFlagsU16::from(index as u8));
                for (s_index, _char) in B16.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        set_u32 => {
            let mut field = MyFieldU32(0);
            for (index, char) in B32.chars().rev().enumerate() {
                field.set(MyFlagsU32::from(index as u8));
                for (s_index, _char) in B32.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        set_u64 => {
            let mut field = MyFieldU64(0);
            for (index, char) in B64.chars().rev().enumerate() {
                field.set(MyFlagsU64::from(index as u8));
                for (s_index, _char) in B64.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        set_u128 => {
            let mut field = MyFieldU128(0);
            for (index, char) in B128.chars().rev().enumerate() {
                field.set(MyFlagsU128::from(index as u8));
                for (s_index, _char) in B128.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        set_usize => {
            let mut field = MyFieldUsize(0);
            for (index, char) in BSIZE.chars().rev().enumerate() {
                field.set(MyFlagsUsize::from(index as u8));
                for (s_index, _char) in BSIZE.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
    }

    // clear
    tests! {
        clear_u8 => {
            let mut field = MyFieldU8(u8::MAX);
            for (index, char) in B8.chars().rev().enumerate() {
                assert!(field.get_index(index as u8), "{:#?}", field);
                field.clear(MyFlagsU8::from(index as u8));
                for (s_index, _char) in B8.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        clear_u16 => {
            let mut field = MyFieldU16(u16::MAX);
            for (index, char) in B16.chars().rev().enumerate() {
                assert!(field.get_index(index as u8), "{:#?}", field);
                field.clear(MyFlagsU16::from(index as u8));
                for (s_index, _char) in B16.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        clear_u32 => {
            let mut field = MyFieldU32(u32::MAX);
            for (index, char) in B32.chars().rev().enumerate() {
                assert!(field.get_index(index as u8), "{:#?}", field);
                field.clear(MyFlagsU32::from(index as u8));
                for (s_index, _char) in B32.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        clear_u64 => {
            let mut field = MyFieldU64(u64::MAX);
            for (index, char) in B64.chars().rev().enumerate() {
                assert!(field.get_index(index as u8), "{:#?}", field);
                field.clear(MyFlagsU64::from(index as u8));
                for (s_index, _char) in B64.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        clear_u128 => {
            let mut field = MyFieldU128(u128::MAX);
            for (index, char) in B128.chars().rev().enumerate() {
                assert!(field.get_index(index as u8), "{:#?}", field);
                field.clear(MyFlagsU128::from(index as u8));
                for (s_index, _char) in B128.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
        clear_usize => {
            let mut field = MyFieldUsize(usize::MAX);
            for (index, char) in BSIZE.chars().rev().enumerate() {
                assert!(field.get_index(index as u8), "{:#?}", field);
                field.clear(MyFlagsUsize::from(index as u8));
                for (s_index, _char) in BSIZE.chars().rev().enumerate() {
                    if s_index <= index {
                        assert!(!field.get_index(s_index as u8), "{:#?}", field);
                    }  else {
                        assert!(field.get_index(s_index as u8), "{:#?}", field);
                    }
                }
            }
        }
    }

    // toggle
    tests! {
        toggle_u8 => {
            let mut field = MyFieldU8(U8);
            for (index, char) in B8.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get_index(index as u8));
                    field.toggle(MyFlagsU8::from(index as u8));
                    assert!(!field.get_index(index as u8));
                } else if char == '0' {
                    assert!(!field.get_index(index as u8));
                    field.toggle(MyFlagsU8::from(index as u8));
                    assert!(field.get_index(index as u8));
                } else {
                    panic!("invalid char in binary string");
                }
            }
        }
        toggle_u16 => {
            let mut field = MyFieldU16(U16);
            for (index, char) in B16.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get_index(index as u8));
                    field.toggle(MyFlagsU16::from(index as u8));
                    assert!(!field.get_index(index as u8));
                } else if char == '0' {
                    assert!(!field.get_index(index as u8));
                    field.toggle(MyFlagsU16::from(index as u8));
                    assert!(field.get_index(index as u8));
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
        toggle_u32 => {
            let mut field = MyFieldU32(U32);
            for (index, char) in B32.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get_index(index as u8));
                    field.toggle(MyFlagsU32::from(index as u8));
                    assert!(!field.get_index(index as u8));
                } else if char == '0' {
                    assert!(!field.get_index(index as u8));
                    field.toggle(MyFlagsU32::from(index as u8));
                    assert!(field.get_index(index as u8));
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
        toggle_u64 => {
            let mut field = MyFieldU64(U64);
            for (index, char) in B64.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get_index(index as u8));
                    field.toggle(MyFlagsU64::from(index as u8));
                    assert!(!field.get_index(index as u8));
                } else if char == '0' {
                    assert!(!field.get_index(index as u8));
                    field.toggle(MyFlagsU64::from(index as u8));
                    assert!(field.get_index(index as u8));
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
        toggle_u128 => {
            let mut field = MyFieldU128(U128);
            for (index, char) in B128.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get_index(index as u8));
                    field.toggle(MyFlagsU128::from(index as u8));
                    assert!(!field.get_index(index as u8));
                } else if char == '0' {
                    assert!(!field.get_index(index as u8));
                    field.toggle(MyFlagsU128::from(index as u8));
                    assert!(field.get_index(index as u8));
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
        toggle_usize => {
            let mut field = MyFieldUsize(USIZE);
            for (index, char) in BSIZE.chars().rev().enumerate() {
                if char == '1' {
                    assert!(field.get_index(index as u8));
                    field.toggle(MyFlagsUsize::from(index as u8));
                    assert!(!field.get_index(index as u8));
                } else if char == '0' {
                    assert!(!field.get_index(index as u8));
                    field.toggle(MyFlagsUsize::from(index as u8));
                    assert!(field.get_index(index as u8));
                } else {
                    panic!("invalid char in binary string")
                }
            }
        }
    }

    // diff
    tests! {
        diff_u8 => {
            let field = MyFieldU8(U8);
            let reversed = MyFieldU8(u8::from_str_radix(B8.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.diff(reversed);
            assert_eq!("10100101", out.as_binary());
            assert_eq!(165, out.as_integer());
        }
        diff_u16 => {
            let field = MyFieldU16(U16);
            let reversed = MyFieldU16(u16::from_str_radix(B16.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.diff(reversed);
            assert_eq!("1010101111010101", out.as_binary());
            assert_eq!(43989, out.as_integer());
        }
        diff_u32 => {
            let field = MyFieldU32(U32);
            let reversed = MyFieldU32(u32::from_str_radix(B32.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.diff(reversed);
            assert_eq!("11110101000100111100100010101111", out.as_binary());
            assert_eq!(4111714479, out.as_integer());
        }
        diff_u64 => {
            let field = MyFieldU64(U64);
            let reversed = MyFieldU64(u64::from_str_radix(B64.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.diff(reversed);
            assert_eq!("0001010011000010001001001001100110011001001001000100001100101000", out.as_binary());
            assert_eq!(1495798268358312744, out.as_integer());
        }
        diff_u128 => {
            let field = MyFieldU128(U128);
            let reversed = MyFieldU128(u128::from_str_radix(B128.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.diff(reversed);
            assert_eq!("10010001110011001000001110100101110101111000111101001010101000100100010101010010111100011110101110100101110000010011001110001001", out.as_binary());
            assert_eq!(193799958082971305581995159990093034377, out.as_integer());
        }
        // diff_usize => {
        //     let field = MyFieldUsize(USIZE);
        // }
    }

    // combine
    tests! {
        combine_u8 => {
            let field = MyFieldU8(U8);
            let reversed = MyFieldU8(u8::from_str_radix(B8.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.combine(reversed);
            assert_eq!("11111111", out.as_binary());
            assert_eq!(255, out.as_integer());
        }
        combine_u16 => {
            let field = MyFieldU16(U16);
            let reversed = MyFieldU16(u16::from_str_radix(B16.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.combine(reversed);
            assert_eq!("1110101111010111", out.as_binary());
            assert_eq!(60375, out.as_integer());
        }
        combine_u32 => {
            let field = MyFieldU32(U32);
            let reversed = MyFieldU32(u32::from_str_radix(B32.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.combine(reversed);
            assert_eq!("11111101100100111100100110111111", out.as_binary());
            assert_eq!(4254321087, out.as_integer());
        }
        combine_u64 => {
            let field = MyFieldU64(U64);
            let reversed = MyFieldU64(u64::from_str_radix(B64.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.combine(reversed);
            assert_eq!("1111111011010010001111111001110110111001111111000100101101111111", out.as_binary());
            assert_eq!(18361808577405668223, out.as_integer());
        }
        combine_u128 => {
            let field = MyFieldU128(U128);
            let reversed = MyFieldU128(u128::from_str_radix(B128.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.combine(reversed);
            assert_eq!("11111001111111011110001110100101111101111110111101101011111111111111111111010110111101111110111110100101110001111011111110011111", out.as_binary());
            assert_eq!(332296039312012448238196841470979719071, out.as_integer());
        }
        // combine_usize => {
        //     let field = MyFieldUsize(USIZE);
        // }
    }

    // intersect
    tests! {
        intersect_u8 => {
            let field = MyFieldU8(U8);
            let reversed = MyFieldU8(u8::from_str_radix(B8.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.intersect(reversed);
            assert_eq!("01011010", out.as_binary());
            assert_eq!(90, out.as_integer());
        }
        intersect_u16 => {
            let field = MyFieldU16(U16);
            let reversed = MyFieldU16(u16::from_str_radix(B16.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.intersect(reversed);
            assert_eq!("0100000000000010", out.as_binary());
            assert_eq!(16386, out.as_integer());
        }
        intersect_u32 => {
            let field = MyFieldU32(U32);
            let reversed = MyFieldU32(u32::from_str_radix(B32.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.intersect(reversed);
            assert_eq!("00001000100000000000000100010000", out.as_binary());
            assert_eq!(142606608, out.as_integer());
        }
        intersect_u64 => {
            let field = MyFieldU64(U64);
            let reversed = MyFieldU64(u64::from_str_radix(B64.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.intersect(reversed);
            assert_eq!("1110101000010000000110110000010000100000110110000000100001010111", out.as_binary());
            assert_eq!(16866010309047355479, out.as_integer());
        }
        intersect_u128 => {
            let field = MyFieldU128(U128);
            let reversed = MyFieldU128(u128::from_str_radix(B128.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.intersect(reversed);
            assert_eq!("01101000001100010110000000000000001000000110000000100001010111011011101010000100000001100000010000000000000001101000110000010110", out.as_binary());
            assert_eq!(138496081229041142656201681480886684694, out.as_integer());
        }
        // intersect_usize => {
        //     let field = MyFieldUsize(USIZE);
        // }
    }

    // into diff
    tests! {
        into_diff_u8 => {
            let field = MyFieldU8(U8);
            let reversed = MyFieldU8(u8::from_str_radix(B8.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.into_diff(reversed);
            assert_eq!("10100101", out.as_binary());
            assert_eq!(165, out.as_integer());
        }
        into_diff_u16 => {
            let field = MyFieldU16(U16);
            let reversed = MyFieldU16(u16::from_str_radix(B16.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.into_diff(reversed);
            assert_eq!("1010101111010101", out.as_binary());
            assert_eq!(43989, out.as_integer());
        }
        into_diff_u32 => {
            let field = MyFieldU32(U32);
            let reversed = MyFieldU32(u32::from_str_radix(B32.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.into_diff(reversed);
            assert_eq!("11110101000100111100100010101111", out.as_binary());
            assert_eq!(4111714479, out.as_integer());
        }
        into_diff_u64 => {
            let field = MyFieldU64(U64);
            let reversed = MyFieldU64(u64::from_str_radix(B64.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.into_diff(reversed);
            assert_eq!("0001010011000010001001001001100110011001001001000100001100101000", out.as_binary());
            assert_eq!(1495798268358312744, out.as_integer());
        }
        into_diff_u128 => {
            let field = MyFieldU128(U128);
            let reversed = MyFieldU128(u128::from_str_radix(B128.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.into_diff(reversed);
            assert_eq!("10010001110011001000001110100101110101111000111101001010101000100100010101010010111100011110101110100101110000010011001110001001", out.as_binary());
            assert_eq!(193799958082971305581995159990093034377, out.as_integer());
        }
        // into_diff_usize => {
        //     let field = MyFieldUsize(USIZE);
        // }
    }

    // into combined
    tests! {
        into_combined_u8 => {
            let field = MyFieldU8(U8);
            let reversed = MyFieldU8(u8::from_str_radix(B8.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.into_combined(reversed);
            assert_eq!("11111111", out.as_binary());
            assert_eq!(255, out.as_integer());
        }
        into_combined_u16 => {
            let field = MyFieldU16(U16);
            let reversed = MyFieldU16(u16::from_str_radix(B16.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.into_combined(reversed);
            assert_eq!("1110101111010111", out.as_binary());
            assert_eq!(60375, out.as_integer());
        }
        into_combined_u32 => {
            let field = MyFieldU32(U32);
            let reversed = MyFieldU32(u32::from_str_radix(B32.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.into_combined(reversed);
            assert_eq!("11111101100100111100100110111111", out.as_binary());
            assert_eq!(4254321087, out.as_integer());
        }
        into_combined_u64 => {
            let field = MyFieldU64(U64);
            let reversed = MyFieldU64(u64::from_str_radix(B64.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.into_combined(reversed);
            assert_eq!("1111111011010010001111111001110110111001111111000100101101111111", out.as_binary());
            assert_eq!(18361808577405668223, out.as_integer());
        }
        into_combined_u128 => {
            let field = MyFieldU128(U128);
            let reversed = MyFieldU128(u128::from_str_radix(B128.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.into_combined(reversed);
            assert_eq!("11111001111111011110001110100101111101111110111101101011111111111111111111010110111101111110111110100101110001111011111110011111", out.as_binary());
            assert_eq!(332296039312012448238196841470979719071, out.as_integer());
        }
        // into_combined_usize => {
        //     let field = MyFieldUsize(USIZE);
        // }
    }

    // into intersection
    tests! {
        into_intersection_u8 => {
            let field = MyFieldU8(U8);
            let reversed = MyFieldU8(u8::from_str_radix(B8.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.into_intersection(reversed);
            assert_eq!("01011010", out.as_binary());
            assert_eq!(90, out.as_integer());
        }
        into_intersection_u16 => {
            let field = MyFieldU16(U16);
            let reversed = MyFieldU16(u16::from_str_radix(B16.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.into_intersection(reversed);
            assert_eq!("0100000000000010", out.as_binary());
            assert_eq!(16386, out.as_integer());
        }
        into_intersection_u32 => {
            let field = MyFieldU32(U32);
            let reversed = MyFieldU32(u32::from_str_radix(B32.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.into_intersection(reversed);
            assert_eq!("00001000100000000000000100010000", out.as_binary());
            assert_eq!(142606608, out.as_integer());
        }
        into_intersection_u64 => {
            let field = MyFieldU64(U64);
            let reversed = MyFieldU64(u64::from_str_radix(B64.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.into_intersection(reversed);
            assert_eq!("1110101000010000000110110000010000100000110110000000100001010111", out.as_binary());
            assert_eq!(16866010309047355479, out.as_integer());
        }
        into_intersection_u128 => {
            let field = MyFieldU128(U128);
            let reversed = MyFieldU128(u128::from_str_radix(B128.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field.into_intersection(reversed);
            assert_eq!("01101000001100010110000000000000001000000110000000100001010111011011101010000100000001100000010000000000000001101000110000010110", out.as_binary());
            assert_eq!(138496081229041142656201681480886684694, out.as_integer());
        }
        // into_intersection_usize => {
        //     let field = MyFieldUsize(USIZE);
        // }
    }

    // field from flag
    tests! {
        field_from_flag_u8 => {
            for index in 0..u8::BITS {
                let flag = MyFlagsU8::from(index as u8);
                let field = MyFieldU8::from(flag);
                assert_eq!(MyFieldU8(0u8 | (1 << index)), field)
            }
        }
        field_from_flag_u16 => {
            for index in 0..u16::BITS {
                let flag = MyFlagsU16::from(index as u8);
                let field = MyFieldU16::from(flag);
                assert_eq!(MyFieldU16(0u16 | (1 << index)), field)
            }
        }
        field_from_flag_u32 => {
            for index in 0..u32::BITS {
                let flag = MyFlagsU32::from(index as u8);
                let field = MyFieldU32::from(flag);
                assert_eq!(MyFieldU32(0u32 | (1 << index)), field)
            }
        }
        field_from_flag_u64 => {
            for index in 0..u64::BITS {
                let flag = MyFlagsU64::from(index as u8);
                let field = MyFieldU64::from(flag);
                assert_eq!(MyFieldU64(0u64 | (1 << index)), field)
            }
        }
        field_from_flag_u128 => {
            for index in 0..u128::BITS {
                let flag = MyFlagsU128::from(index as u8);
                let field = MyFieldU128::from(flag);
                assert_eq!(MyFieldU128(0u128 | (1 << index)), field)
            }
        }
        field_from_flag_usize => {
            for index in 0..usize::BITS {
                let flag = MyFlagsUsize::from(index as u8);
                let field = MyFieldUsize::from(flag);
                assert_eq!(MyFieldUsize(0usize | (1 << index)), field)
            }
        }
    }

    // $name & $name
    tests! {
        bitwise_name_and_name_u8 => {
            let field = MyFieldU8(U8);
            let reversed = MyFieldU8(u8::from_str_radix(B8.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field & reversed;
            assert_eq!("01011010", out.as_binary());
            assert_eq!(90, out.as_integer());
        }
        bitwise_name_and_name_u16 => {
            let field = MyFieldU16(U16);
            let reversed = MyFieldU16(u16::from_str_radix(B16.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field & reversed;
            assert_eq!("0100000000000010", out.as_binary());
            assert_eq!(16386, out.as_integer());
        }
        bitwise_name_and_name_u32 => {
            let field = MyFieldU32(U32);
            let reversed = MyFieldU32(u32::from_str_radix(B32.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field & reversed;
            assert_eq!("00001000100000000000000100010000", out.as_binary());
            assert_eq!(142606608, out.as_integer());
        }
        bitwise_name_and_name_u64 => {
            let field = MyFieldU64(U64);
            let reversed = MyFieldU64(u64::from_str_radix(B64.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field & reversed;
            assert_eq!("1110101000010000000110110000010000100000110110000000100001010111", out.as_binary());
            assert_eq!(16866010309047355479, out.as_integer());
        }
        bitwise_name_and_name_u128 => {
            let field = MyFieldU128(U128);
            let reversed = MyFieldU128(u128::from_str_radix(B128.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field & reversed;
            assert_eq!("01101000001100010110000000000000001000000110000000100001010111011011101010000100000001100000010000000000000001101000110000010110", out.as_binary());
            assert_eq!(138496081229041142656201681480886684694, out.as_integer());
        }
        // bitwise_name_and_name_usize => {
        //
        // }
    }

    // $name | $name
    tests! {
        bitwise_name_or_name_u8 => {
            let field = MyFieldU8(U8);
            let reversed = MyFieldU8(u8::from_str_radix(B8.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field | reversed;
            assert_eq!("11111111", out.as_binary());
            assert_eq!(255, out.as_integer());
        }
        bitwise_name_or_name_u16 => {
            let field = MyFieldU16(U16);
            let reversed = MyFieldU16(u16::from_str_radix(B16.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field | reversed;
            assert_eq!("1110101111010111", out.as_binary());
            assert_eq!(60375, out.as_integer());
        }
        bitwise_name_or_name_u32 => {
            let field = MyFieldU32(U32);
            let reversed = MyFieldU32(u32::from_str_radix(B32.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field | reversed;
            assert_eq!("11111101100100111100100110111111", out.as_binary());
            assert_eq!(4254321087, out.as_integer());
        }
        bitwise_name_or_name_u64 => {
            let field = MyFieldU64(U64);
            let reversed = MyFieldU64(u64::from_str_radix(B64.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field | reversed;
            assert_eq!("1111111011010010001111111001110110111001111111000100101101111111", out.as_binary());
            assert_eq!(18361808577405668223, out.as_integer());
        }
        bitwise_name_or_name_u128 => {
            let field = MyFieldU128(U128);
            let reversed = MyFieldU128(u128::from_str_radix(B128.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field | reversed;
            assert_eq!("11111001111111011110001110100101111101111110111101101011111111111111111111010110111101111110111110100101110001111011111110011111", out.as_binary());
            assert_eq!(332296039312012448238196841470979719071, out.as_integer());
        }
        // bitwise_name_or_name_usize => {
        //
        // }
    }

    // $name ^ $name
    tests! {
        bitwise_name_xor_name_u8 => {
            let field = MyFieldU8(U8);
            let reversed = MyFieldU8(u8::from_str_radix(B8.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field ^ reversed;
            assert_eq!("10100101", out.as_binary());
            assert_eq!(165, out.as_integer());
        }
        bitwise_name_xor_name_u16 => {
            let field = MyFieldU16(U16);
            let reversed = MyFieldU16(u16::from_str_radix(B16.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field ^ reversed;
            assert_eq!("1010101111010101", out.as_binary());
            assert_eq!(43989, out.as_integer());
        }
        bitwise_name_xor_name_u32 => {
            let field = MyFieldU32(U32);
            let reversed = MyFieldU32(u32::from_str_radix(B32.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field ^ reversed;
            assert_eq!("11110101000100111100100010101111", out.as_binary());
            assert_eq!(4111714479, out.as_integer());
        }
        bitwise_name_xor_name_u64 => {
            let field = MyFieldU64(U64);
            let reversed = MyFieldU64(u64::from_str_radix(B64.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field ^ reversed;
            assert_eq!("0001010011000010001001001001100110011001001001000100001100101000", out.as_binary());
            assert_eq!(1495798268358312744, out.as_integer());
        }
        bitwise_name_xor_name_u128 => {
            let field = MyFieldU128(U128);
            let reversed = MyFieldU128(u128::from_str_radix(B128.chars().rev().collect::<String>().as_str(), 2).unwrap());
            let out = field ^ reversed;
            assert_eq!("10010001110011001000001110100101110101111000111101001010101000100100010101010010111100011110101110100101110000010011001110001001", out.as_binary());
            assert_eq!(193799958082971305581995159990093034377, out.as_integer());
        }
        // bitwise_name_xor_name_usize => {
        //
        // }
    }

    // $name &= $name
    tests! {
        bitwise_name_and_assign_name_u8 => {
            let mut field = MyFieldU8(U8);
            let reversed = MyFieldU8(u8::from_str_radix(B8.chars().rev().collect::<String>().as_str(), 2).unwrap());
            field &= reversed;
            assert_eq!("01011010", field.as_binary());
            assert_eq!(90, field.as_integer());
        }
        bitwise_name_and_assign_name_u16 => {
            let mut field = MyFieldU16(U16);
            let reversed = MyFieldU16(u16::from_str_radix(B16.chars().rev().collect::<String>().as_str(), 2).unwrap());
            field &= reversed;
            assert_eq!("0100000000000010", field.as_binary());
            assert_eq!(16386, field.as_integer());
        }
        bitwise_name_and_assign_name_u32 => {
            let mut field = MyFieldU32(U32);
            let reversed = MyFieldU32(u32::from_str_radix(B32.chars().rev().collect::<String>().as_str(), 2).unwrap());
            field &= reversed;
            assert_eq!("00001000100000000000000100010000", field.as_binary());
            assert_eq!(142606608, field.as_integer());
        }
        bitwise_name_and_assign_name_u64 => {
            let mut field = MyFieldU64(U64);
            let reversed = MyFieldU64(u64::from_str_radix(B64.chars().rev().collect::<String>().as_str(), 2).unwrap());
            field &= reversed;
            assert_eq!("1110101000010000000110110000010000100000110110000000100001010111", field.as_binary());
            assert_eq!(16866010309047355479, field.as_integer());
        }
        bitwise_name_and_assign_name_u128 => {
            let mut field = MyFieldU128(U128);
            let reversed = MyFieldU128(u128::from_str_radix(B128.chars().rev().collect::<String>().as_str(), 2).unwrap());
            field &= reversed;
            assert_eq!("01101000001100010110000000000000001000000110000000100001010111011011101010000100000001100000010000000000000001101000110000010110", field.as_binary());
            assert_eq!(138496081229041142656201681480886684694, field.as_integer());
        }
    //     bitwise_name_and_assign_name_usize => {
    //
    //     }
    }

    // $name |= $name
    tests! {
        bitwise_name_or_assign_name_u8 => {
            let mut field = MyFieldU8(U8);
            let reversed = MyFieldU8(u8::from_str_radix(B8.chars().rev().collect::<String>().as_str(), 2).unwrap());
            field |= reversed;
            assert_eq!("11111111", field.as_binary());
            assert_eq!(255, field.as_integer());
        }
        bitwise_name_or_assign_name_u16 => {
            let mut field = MyFieldU16(U16);
            let reversed = MyFieldU16(u16::from_str_radix(B16.chars().rev().collect::<String>().as_str(), 2).unwrap());
            field |= reversed;
            assert_eq!("1110101111010111", field.as_binary());
            assert_eq!(60375, field.as_integer());
        }
        bitwise_name_or_assign_name_u32 => {
            let mut field = MyFieldU32(U32);
            let reversed = MyFieldU32(u32::from_str_radix(B32.chars().rev().collect::<String>().as_str(), 2).unwrap());
            field |= reversed;
            assert_eq!("11111101100100111100100110111111", field.as_binary());
            assert_eq!(4254321087, field.as_integer());
        }
        bitwise_name_or_assign_name_u64 => {
            let mut field = MyFieldU64(U64);
            let reversed = MyFieldU64(u64::from_str_radix(B64.chars().rev().collect::<String>().as_str(), 2).unwrap());
            field |= reversed;
            assert_eq!("1111111011010010001111111001110110111001111111000100101101111111", field.as_binary());
            assert_eq!(18361808577405668223, field.as_integer());
        }
        bitwise_name_or_assign_name_u128 => {
            let mut field = MyFieldU128(U128);
            let reversed = MyFieldU128(u128::from_str_radix(B128.chars().rev().collect::<String>().as_str(), 2).unwrap());
            field |= reversed;
            assert_eq!("11111001111111011110001110100101111101111110111101101011111111111111111111010110111101111110111110100101110001111011111110011111", field.as_binary());
            assert_eq!(332296039312012448238196841470979719071, field.as_integer());
        }
        // bitwise_name_or_assign_name_usize => {
        //
        // }
    }

    // $name ^= $name
    tests! {
        bitwise_name_xor_assign_name_u8 => {
            let mut field = MyFieldU8(U8);
            let reversed = MyFieldU8(u8::from_str_radix(B8.chars().rev().collect::<String>().as_str(), 2).unwrap());
            field ^= reversed;
            assert_eq!("10100101", field.as_binary());
            assert_eq!(165, field.as_integer());
        }
        bitwise_name_xor_assign_name_u16 => {
            let mut field = MyFieldU16(U16);
            let reversed = MyFieldU16(u16::from_str_radix(B16.chars().rev().collect::<String>().as_str(), 2).unwrap());
            field ^= reversed;
            assert_eq!("1010101111010101", field.as_binary());
            assert_eq!(43989, field.as_integer());
        }
        bitwise_name_xor_assign_name_u32 => {
            let mut field = MyFieldU32(U32);
            let reversed = MyFieldU32(u32::from_str_radix(B32.chars().rev().collect::<String>().as_str(), 2).unwrap());
            field ^= reversed;
            assert_eq!("11110101000100111100100010101111", field.as_binary());
            assert_eq!(4111714479, field.as_integer());
        }
        bitwise_name_xor_assign_name_u64 => {
            let mut field = MyFieldU64(U64);
            let reversed = MyFieldU64(u64::from_str_radix(B64.chars().rev().collect::<String>().as_str(), 2).unwrap());
            field ^= reversed;
            assert_eq!("0001010011000010001001001001100110011001001001000100001100101000", field.as_binary());
            assert_eq!(1495798268358312744, field.as_integer());
        }
        bitwise_name_xor_assign_name_u128 => {
            let mut field = MyFieldU128(U128);
            let reversed = MyFieldU128(u128::from_str_radix(B128.chars().rev().collect::<String>().as_str(), 2).unwrap());
            field ^= reversed;
            assert_eq!("10010001110011001000001110100101110101111000111101001010101000100100010101010010111100011110101110100101110000010011001110001001", field.as_binary());
            assert_eq!(193799958082971305581995159990093034377, field.as_integer());
        }
        // bitwise_name_xor_assign_name_usize => {
        //
        // }
    }

    // $name & $flag
    tests! {
        bitwise_name_and_flag_u8 => {
            for index in 0..u8::BITS {
                let mask = MyFieldU8(u8::MAX) & MyFlagsU8::from(index as u8);
                assert_eq!(0 | !(1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_and_flag_u16 => {
            for index in 0..u16::BITS {
                let mask = MyFieldU16(u16::MAX) & MyFlagsU16::from(index as u8);
                assert_eq!(0 | !(1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_and_flag_u32 => {
            for index in 0..u32::BITS {
                let mask = MyFieldU32(u32::MAX) & MyFlagsU32::from(index as u8);
                assert_eq!(0 | !(1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_and_flag_u64 => {
            for index in 0..u64::BITS {
                let mask = MyFieldU64(u64::MAX) & MyFlagsU64::from(index as u8);
                assert_eq!(0 | !(1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_and_flag_u128 => {
            for index in 0..u128::BITS {
                let mask = MyFieldU128(u128::MAX) & MyFlagsU128::from(index as u8);
                assert_eq!(0 | !(1 << (index as u8) ), mask.as_integer());
            }
        }
        // bitwise_name_and_flag_usize => {
        //
        // }
    }

    // $name | $flag
    tests! {
        bitwise_name_or_flag_u8 => {
            for index in 0..u8::BITS {
                let mask = MyFieldU8(0) | MyFlagsU8::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_or_flag_u16 => {
            for index in 0..u16::BITS {
                let mask = MyFieldU16(0) | MyFlagsU16::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_or_flag_u32 => {
            for index in 0..u32::BITS {
                let mask = MyFieldU32(0) | MyFlagsU32::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_or_flag_u64 => {
            for index in 0..u64::BITS {
                let mask = MyFieldU64(0) | MyFlagsU64::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_or_flag_u128 => {
            for index in 0..u128::BITS {
                let mask = MyFieldU128(0) | MyFlagsU128::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        // bitwise_name_or_flag_usize => {
        //
        // }
    }

    // $name ^ $flag
    tests! {
        bitwise_name_xor_flag_u8 => {
            for index in 0..u8::BITS {
                let mask = MyFieldU8(0) ^ MyFlagsU8::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_xor_flag_u16 => {
            for index in 0..u16::BITS {
                let mask = MyFieldU16(0) ^ MyFlagsU16::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_xor_flag_u32 => {
            for index in 0..u32::BITS {
                let mask = MyFieldU32(0) ^ MyFlagsU32::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_xor_flag_u64 => {
            for index in 0..u64::BITS {
                let mask = MyFieldU64(0) ^ MyFlagsU64::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_xor_flag_u128 => {
            for index in 0..u128::BITS {
                let mask = MyFieldU128(0) ^ MyFlagsU128::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        // bitwise_name_xor_flag_usize => {
        //
        // }
    }

    // $name &= $flag
    tests! {
        bitwise_name_and_assign_flag_u8 => {
            for index in 0..u8::BITS {
                let mut mask = MyFieldU8(u8::MAX);
                mask &= MyFlagsU8::from(index as u8);
                assert_eq!(0 | !(1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_and_assign_flag_u16 => {
            for index in 0..u16::BITS {
                let mut mask = MyFieldU16(u16::MAX);
                mask &= MyFlagsU16::from(index as u8);
                assert_eq!(0 | !(1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_and_assign_flag_u32 => {
            for index in 0..u32::BITS {
                let mut mask = MyFieldU32(u32::MAX);
                mask &= MyFlagsU32::from(index as u8);
                assert_eq!(0 | !(1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_and_assign_flag_u64 => {
            for index in 0..u64::BITS {
                let mut mask = MyFieldU64(u64::MAX);
                mask &= MyFlagsU64::from(index as u8);
                assert_eq!(0 | !(1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_and_assign_flag_u128 => {
            for index in 0..u128::BITS {
                let mut mask = MyFieldU128(u128::MAX);
                mask &= MyFlagsU128::from(index as u8);
                assert_eq!(0 | !(1 << (index as u8) ), mask.as_integer());
            }
        }
        // bitwise_name_and_assign_flag_usize => {
        //
        // }
    }

    // $name |= $flag
    tests! {
        bitwise_name_or_assign_flag_u8 => {
            for index in 0..u8::BITS {
                let mut mask = MyFieldU8(0);
                mask |= MyFlagsU8::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_or_assign_flag_u16 => {
            for index in 0..u16::BITS {
                let mut mask = MyFieldU16(0);
                mask |= MyFlagsU16::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_or_assign_flag_u32 => {
            for index in 0..u32::BITS {
                let mut mask = MyFieldU32(0);
                mask |= MyFlagsU32::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_or_assign_flag_u64 => {
            for index in 0..u64::BITS {
                let mut mask = MyFieldU64(0);
                mask |= MyFlagsU64::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_or_assign_flag_u128 => {
            for index in 0..u128::BITS {
                let mut mask = MyFieldU128(0);
                mask |= MyFlagsU128::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        // bitwise_name_or_assign_flag_usize => {
        //
        // }
    }

    // $name ^= $flag
    tests! {
        bitwise_name_xor_assign_flag_u8 => {
            for index in 0..u8::BITS {
                let mut mask = MyFieldU8(0);
                mask ^= MyFlagsU8::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_xor_assign_flag_u16 => {
            for index in 0..u16::BITS {
                let mut mask = MyFieldU16(0);
                mask ^= MyFlagsU16::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_xor_assign_flag_u32 => {
            for index in 0..u32::BITS {
                let mut mask = MyFieldU32(0);
                mask ^= MyFlagsU32::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_xor_assign_flag_u64 => {
            for index in 0..u64::BITS {
                let mut mask = MyFieldU64(0);
                mask ^= MyFlagsU64::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        bitwise_name_xor_assign_flag_u128 => {
            for index in 0..u128::BITS {
                let mut mask = MyFieldU128(0);
                mask ^= MyFlagsU128::from(index as u8);
                assert_eq!(0 | (1 << (index as u8) ), mask.as_integer());
            }
        }
        // bitwise_name_xor_assign_flag_usize => {
        //
        // }
    }

    // $flag & $flag
    tests! {
        bitwise_flag_and_flag_u8 => {
            for index_a in 0..u8::BITS {
                let a = MyFlagsU8::from(index_a as u8);
                for index_b in 0..u8::BITS {
                    let b = MyFlagsU8::from(index_b as u8);
                    assert_eq!(MyFieldU8(0 & 1 << index_a as u8 & 1 << index_b as u8), a & b);
                }
            }
        }
        bitwise_flag_and_flag_u16 => {
            for index_a in 0..u16::BITS {
                let a = MyFlagsU16::from(index_a as u8);
                for index_b in 0..u16::BITS {
                    let b = MyFlagsU16::from(index_b as u8);
                    assert_eq!(MyFieldU16(0 & 1 << index_a as u8 & 1 << index_b as u8), a & b);
                }
            }
        }
        bitwise_flag_and_flag_u32 => {
            for index_a in 0..u32::BITS {
                let a = MyFlagsU32::from(index_a as u8);
                for index_b in 0..u32::BITS {
                    let b = MyFlagsU32::from(index_b as u8);
                    assert_eq!(MyFieldU32(0 & 1 << index_a as u8 & 1 << index_b as u8), a & b);
                }
            }
        }
        bitwise_flag_and_flag_u64 => {
            for index_a in 0..u64::BITS {
                let a = MyFlagsU64::from(index_a as u8);
                for index_b in 0..u64::BITS {
                    let b = MyFlagsU64::from(index_b as u8);
                    assert_eq!(MyFieldU64(0 & 1 << index_a as u8 & 1 << index_b as u8), a & b);
                }
            }
        }
        bitwise_flag_and_flag_u128 => {
            for index_a in 0..u128::BITS {
                let a = MyFlagsU128::from(index_a as u8);
                for index_b in 0..u128::BITS {
                    let b = MyFlagsU128::from(index_b as u8);
                    assert_eq!(MyFieldU128(0 & 1 << index_a as u8 & 1 << index_b as u8), a & b);
                }
            }
        }
        // bitwise_flag_and_flag_usize => {
        //
        // }
    }

    // $flag | $flag
    tests! {
        bitwise_flag_or_flag_u8 => {
            for index_a in 0..u8::BITS {
                let a = MyFlagsU8::from(index_a as u8);
                for index_b in 0..u8::BITS {
                    let b = MyFlagsU8::from(index_b as u8);
                    assert_eq!(MyFieldU8(0 | 1 << index_a as u8 | 1 << index_b as u8), a | b);
                }
            }
        }
        bitwise_flag_or_flag_u16 => {
            for index_a in 0..u16::BITS {
                let a = MyFlagsU16::from(index_a as u8);
                for index_b in 0..u16::BITS {
                    let b = MyFlagsU16::from(index_b as u8);
                    assert_eq!(MyFieldU16(0 | 1 << index_a as u8 | 1 << index_b as u8), a | b);
                }
            }
        }
        bitwise_flag_or_flag_u32 => {
            for index_a in 0..u32::BITS {
                let a = MyFlagsU32::from(index_a as u8);
                for index_b in 0..u32::BITS {
                    let b = MyFlagsU32::from(index_b as u8);
                    assert_eq!(MyFieldU32(0 | 1 << index_a as u8 | 1 << index_b as u8), a | b);
                }
            }
        }
        bitwise_flag_or_flag_u64 => {
            for index_a in 0..u64::BITS {
                let a = MyFlagsU64::from(index_a as u8);
                for index_b in 0..u64::BITS {
                    let b = MyFlagsU64::from(index_b as u8);
                    assert_eq!(MyFieldU64(0 | 1 << index_a as u8 | 1 << index_b as u8), a | b);
                }
            }
        }
        bitwise_flag_or_flag_u128 => {
            for index_a in 0..u128::BITS {
                let a = MyFlagsU128::from(index_a as u8);
                for index_b in 0..u128::BITS {
                    let b = MyFlagsU128::from(index_b as u8);
                    assert_eq!(MyFieldU128(0 | 1 << index_a as u8 | 1 << index_b as u8), a | b);
                }
            }
        }
        // bitwise_flag_or_flag_usize => {
        //
        // }
    }

    // $flag ^ $flag
    tests! {
        bitwise_flag_xor_flag_u8 => {
            for index_a in 0..u8::BITS {
                let a = MyFlagsU8::from(index_a as u8);
                for index_b in 0..u8::BITS {
                    let b = MyFlagsU8::from(index_b as u8);
                    assert_eq!(MyFieldU8(0 ^ 1 << index_a as u8 ^ 1 << index_b as u8), a ^ b);
                }
            }
        }
        bitwise_flag_xor_flag_u16 => {
            for index_a in 0..u16::BITS {
                let a = MyFlagsU16::from(index_a as u8);
                for index_b in 0..u16::BITS {
                    let b = MyFlagsU16::from(index_b as u8);
                    assert_eq!(MyFieldU16(0 ^ 1 << index_a as u8 ^ 1 << index_b as u8), a ^ b);
                }
            }
        }
        bitwise_flag_xor_flag_u32 => {
            for index_a in 0..u32::BITS {
                let a = MyFlagsU32::from(index_a as u8);
                for index_b in 0..u32::BITS {
                    let b = MyFlagsU32::from(index_b as u8);
                    assert_eq!(MyFieldU32(0 ^ 1 << index_a as u8 ^ 1 << index_b as u8), a ^ b);
                }
            }
        }
        bitwise_flag_xor_flag_u64 => {
            for index_a in 0..u64::BITS {
                let a = MyFlagsU64::from(index_a as u8);
                for index_b in 0..u64::BITS {
                    let b = MyFlagsU64::from(index_b as u8);
                    assert_eq!(MyFieldU64(0 ^ 1 << index_a as u8 ^ 1 << index_b as u8), a ^ b);
                }
            }
        }
        bitwise_flag_xor_flag_u128 => {
            for index_a in 0..u128::BITS {
                let a = MyFlagsU128::from(index_a as u8);
                for index_b in 0..u128::BITS {
                    let b = MyFlagsU128::from(index_b as u8);
                    assert_eq!(MyFieldU128(0 ^ 1 << index_a as u8 ^ 1 << index_b as u8), a ^ b);
                }
            }
        }
        // bitwise_flag_xor_flag_usize => {
        //
        // }
    }

    // flag from u8
    tests! {
        flag_from_u8_u8 => {
            for index in 0..u8::BITS {
                let _ = MyFlagsU8::from(index as u8);
            }
        }
        flag_from_u8_u16 => {
            for index in 0..u16::BITS {
                let _ = MyFlagsU16::from(index as u8);
            }
        }
        flag_from_u8_u32 => {
            for index in 0..u32::BITS {
                let _ = MyFlagsU32::from(index as u8);
            }
        }
        flag_from_u8_u64 => {
            for index in 0..u64::BITS {
                let _ = MyFlagsU64::from(index as u8);
            }
        }
        flag_from_u8_u128 => {
            for index in 0..u128::BITS {
                let _ = MyFlagsU128::from(index as u8);
            }
        }
        flag_from_u8_usize => {
            for index in 0..usize::BITS {
                let _ = MyFlagsUsize::from(index as u8);
            }
        }
    }

    // u8 from flag
    tests! {
        u8_from_flag_u8 => {
            for index in 0..u8::BITS {
                assert_eq!(MyFlagsU8::from(index as u8) as u8, index as u8)
            }
        }
        u8_from_flag_u16 => {
            for index in 0..u16::BITS {
                assert_eq!(MyFlagsU16::from(index as u8) as u8, index as u8)
            }
        }
        u8_from_flag_u32 => {
            for index in 0..u32::BITS {
                assert_eq!(MyFlagsU32::from(index as u8) as u8, index as u8)
            }
        }
        u8_from_flag_u64 => {
            for index in 0..u64::BITS {
                assert_eq!(MyFlagsU64::from(index as u8) as u8, index as u8)
            }
        }
        u8_from_flag_u128 => {
            for index in 0..u128::BITS {
                assert_eq!(MyFlagsU128::from(index as u8) as u8, index as u8)
            }
        }
        u8_from_flag_usize => {
            for index in 0..usize::BITS {
                assert_eq!(MyFlagsUsize::from(index as u8) as u8, index as u8)
            }
        }
    }
}

// -------------------------------------------------------------------------------------------------
