/**
Copyright (C) 2021 Kunal Mehta <legoktm@debian.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
*/
use anyhow::Result;
use matrix_sdk::{
    room::Room,
    ruma::{
        events::{
            room::{
                member::{MemberEventContent, MembershipChange},
                message::MessageEventContent,
            },
            AnyMessageEventContent, StateEvent, SyncStateEvent,
        },
        RoomId, UserId,
    },
    Client, ClientConfig, SyncSettings,
};
use serde::Deserialize;
use std::convert::TryFrom;
use tokio::fs;
use tracing::{debug, error, info};
use tracing_subscriber::EnvFilter;

#[derive(Clone, Deserialize)]
struct Config {
    username: String,
    password: String,
    report_room: String,
}

async fn handle_event(
    ev: SyncStateEvent<MemberEventContent>,
    client: Client,
    room: Room,
    config: Config,
) {
    let event: StateEvent<_> = ev.into_full_event(room.room_id().clone());
    debug!(event = ?event);
    let room_str = match room.canonical_alias() {
        Some(alias) => alias.to_string(),
        None => room.room_id().to_string(),
    };
    let reason = event
        .content
        .reason
        .clone()
        .unwrap_or_else(|| "no reason provided".to_string());
    if let MembershipChange::Kicked = event.membership_change() {
        let content = AnyMessageEventContent::RoomMessage(
            MessageEventContent::text_plain(format!(
                "{} was kicked by {} from {} ({})",
                event.state_key, event.sender, room_str, reason
            )),
        );
        let send_room = client.get_joined_room(
            &RoomId::try_from(config.report_room.as_str()).unwrap(),
        );
        if let Some(room) = send_room {
            info!(
                room = %room.room_id(),
                content = ?content,
                "Sending notification"
            );
            room.send(content, None).await.unwrap();
        } else {
            error!(
                report_room = %config.report_room,
                "Not joined to report_room"
            );
        }
    }
}

#[tokio::main]
async fn main() -> Result<()> {
    tracing_subscriber::FmtSubscriber::builder()
        .with_env_filter(EnvFilter::new(
            // Default to RUST_LOG=info if not explicitly set
            std::env::var("RUST_LOG").unwrap_or_else(|_| "info".to_string()),
        ))
        .init();
    let home = dirs::home_dir().expect("unable to find home directory");
    let config: Config =
        toml::from_str(&fs::read_to_string(home.join("uatu.toml")).await?)?;
    let me = UserId::try_from(config.username.as_str())?;
    let client = Client::new_from_user_id_with_config(
        me.clone(),
        ClientConfig::new().store_path(home.join("store")),
    )
    .await?;

    client
        .login(me.localpart(), &config.password, None, Some("uatu"))
        .await?;

    debug!("Logged in");
    client.sync_once(SyncSettings::default()).await?;
    client
        .register_event_handler({
            let config = config.clone();
            move |ev: SyncStateEvent<MemberEventContent>,
                  client: Client,
                  room: Room| {
                let config = config.clone();
                async move {
                    handle_event(ev, client, room, config).await;
                }
            }
        })
        .await;
    debug!("Finished sync_once, registered handlers");
    let settings =
        SyncSettings::default().token(client.sync_token().await.unwrap());
    client.sync(settings).await;
    Ok(())
}
