use {
    crate::{
        identifier::Identifier,
        schema::{self, relativize_namespace},
    },
    std::{
        collections::BTreeMap,
        fmt::{self, Write},
        path::PathBuf,
    },
};

// The string to be used for each indentation level.
const INDENTATION: &str = "    ";

// The generated types will derive these traits.
const TRAITS_TO_DERIVE: &[&str] = &["Clone", "Debug"];

// This is the full list of Rust 2018 keywords, both in use and reserved.
const RUST_KEYWORDS: &[&str] = &[
    "Self", "abstract", "as", "async", "await", "become", "box", "break", "const", "continue",
    "crate", "do", "dyn", "else", "enum", "extern", "false", "final", "fn", "for", "if", "impl",
    "in", "let", "loop", "macro", "match", "mod", "move", "mut", "override", "priv", "pub", "ref",
    "return", "self", "static", "struct", "super", "trait", "true", "try", "type", "typeof",
    "unsafe", "unsized", "use", "virtual", "where", "while", "yield",
];

// This struct represents a tree of schemas organized in a module hierarchy.
#[derive(Clone, Debug)]
struct Module {
    children: BTreeMap<Identifier, Module>,
    schema: schema::Schema,
}

// This enum represents a case convention for the `write_identifier` function below.
#[derive(Clone, Copy, Debug, Eq, PartialEq)]
enum CaseConvention {
    Pascal,
    Snake,
}

use CaseConvention::{Pascal, Snake};

// This enum is used to distinguish between the ingress and egress versions of a type.
#[derive(Clone, Copy, Debug, Eq, PartialEq)]
enum Direction {
    Out,
    In,
}

use Direction::{In, Out};

// Generate Rust code from a schema and its transitive dependencies.
#[allow(clippy::too_many_lines)]
pub fn generate(
    typical_version: &str,
    schemas: &BTreeMap<schema::Namespace, (schema::Schema, PathBuf, String)>,
) -> String {
    // Construct a tree of modules and schemas. We start with an empty tree.
    let mut tree = Module {
        children: BTreeMap::new(),
        schema: schema::Schema {
            comment: vec![],
            imports: BTreeMap::new(),
            declarations: vec![],
        },
    };

    // Populate the tree with all the schemas.
    for (namespace, (schema, _, _)) in schemas {
        insert_schema(&mut tree, namespace, schema);
    }

    // Write the code.
    let mut buffer = String::new();

    if !tree.children.is_empty() || !tree.schema.declarations.is_empty() {
        // The `unwrap` is safe because the `std::fmt::Write` impl for `String` is infallible.
        // For functions that take abstract parameters which implement `std::io::Read` or
        // `std::io::Write`, it's idiomatic to consume the reader or writer rather than borrowing
        // them (https://rust-lang.github.io/api-guidelines/interoperability.html
        // #generic-readerwriter-functions-take-r-read-and-w-write-by-value-c-rw-value). However,
        // we borrow them anyway since that allows us to pass the reference (after reborrowing)
        // to recursive calls, rather than building up and following chains of references.
        writeln!(
            &mut buffer,
            "\
// This file was automatically generated by Typical {}.
// Visit https://github.com/stepchowfun/typical for more information.

use std::{{
    cmp::min,
    io::{{self, BufRead, Error, ErrorKind, Write}},
    mem::transmute,
}};

fn zigzag_encode(value: i64) -> u64 {{
    unsafe {{ transmute::<i64, u64>(value >> 63) ^ transmute::<i64, u64>(value << 1) }}
}}

fn zigzag_decode(value: u64) -> i64 {{
    unsafe {{ transmute::<u64, i64>(value >> 1) ^ -transmute::<u64, i64>(value & 1) }}
}}

fn varint_size_from_value(value: u64) -> usize {{
    match value {{
        0_u64..=127_u64 => 1,
        128_u64..=16_511_u64 => 2,
        16_512_u64..=2_113_663_u64 => 3,
        2_113_664_u64..=270_549_119_u64 => 4,
        270_549_120_u64..=34_630_287_487_u64 => 5,
        34_630_287_488_u64..=4_432_676_798_591_u64 => 6,
        4_432_676_798_592_u64..=567_382_630_219_903_u64 => 7,
        567_382_630_219_904_u64..=72_624_976_668_147_839_u64 => 8,
        72_624_976_668_147_840_u64..=18_446_744_073_709_551_615_u64 => 9,
    }}
}}

fn varint_size_from_first_byte(first_byte: u8) -> u32 {{
    first_byte.trailing_zeros() + 1
}}

fn serialize_varint<T: Write>(mut value: u64, writer: &mut T) -> io::Result<()> {{
    match value {{
        0_u64..=127_u64 => writer.write_all(&[((value << 1) as u8) | 0b0000_0001]),
        128_u64..=16_511_u64 => {{
            value -= 128_u64;
            writer.write_all(&[((value << 2) as u8) | 0b0000_0010, (value >> 6) as u8])
        }}
        16_512_u64..=2_113_663_u64 => {{
            value -= 16_512_u64;
            writer.write_all(&[
                ((value << 3) as u8) | 0b0000_0100,
                (value >> 5) as u8,
                (value >> 13) as u8,
            ])
        }}
        2_113_664_u64..=270_549_119_u64 => {{
            value -= 2_113_664_u64;
            writer.write_all(&[
                ((value << 4) as u8) | 0b0000_1000,
                (value >> 4) as u8,
                (value >> 12) as u8,
                (value >> 20) as u8,
            ])
        }}
        270_549_120_u64..=34_630_287_487_u64 => {{
            value -= 270_549_120_u64;
            writer.write_all(&[
                ((value << 5) as u8) | 0b0001_0000,
                (value >> 3) as u8,
                (value >> 11) as u8,
                (value >> 19) as u8,
                (value >> 27) as u8,
            ])
        }}
        34_630_287_488_u64..=4_432_676_798_591_u64 => {{
            value -= 34_630_287_488_u64;
            writer.write_all(&[
                ((value << 6) as u8) | 0b0010_0000,
                (value >> 2) as u8,
                (value >> 10) as u8,
                (value >> 18) as u8,
                (value >> 26) as u8,
                (value >> 34) as u8,
            ])
        }}
        4_432_676_798_592_u64..=567_382_630_219_903_u64 => {{
            value -= 4_432_676_798_592_u64;
            writer.write_all(&[
                ((value << 7) as u8) | 0b0100_0000,
                (value >> 1) as u8,
                (value >> 9) as u8,
                (value >> 17) as u8,
                (value >> 25) as u8,
                (value >> 33) as u8,
                (value >> 41) as u8,
            ])
        }}
        567_382_630_219_904_u64..=72_624_976_668_147_839_u64 => {{
            value -= 567_382_630_219_904_u64;
            writer.write_all(&[
                0b1000_0000,
                value as u8,
                (value >> 8) as u8,
                (value >> 16) as u8,
                (value >> 24) as u8,
                (value >> 32) as u8,
                (value >> 40) as u8,
                (value >> 48) as u8,
            ])
        }}
        72_624_976_668_147_840_u64..=18_446_744_073_709_551_615_u64 => {{
            value -= 72_624_976_668_147_840_u64;
            writer.write_all(&[
                0b0000_0000,
                value as u8,
                (value >> 8) as u8,
                (value >> 16) as u8,
                (value >> 24) as u8,
                (value >> 32) as u8,
                (value >> 40) as u8,
                (value >> 48) as u8,
                (value >> 56) as u8,
            ])
        }}
    }}
}}

fn deserialize_varint<T: BufRead>(reader: &mut T) -> io::Result<u64> {{
    let mut first_byte_buffer = [0; 1];
    reader.read_exact(&mut first_byte_buffer[..])?;
    let first_byte = first_byte_buffer[0];
    let size_minus_one = first_byte.trailing_zeros();
    let mut remaining_bytes_buffer = [0; 8];
    reader.read_exact(&mut remaining_bytes_buffer[0..size_minus_one as usize])?;
    let remaining_bytes_value = u64::from_le_bytes(remaining_bytes_buffer);
    match size_minus_one {{
        0 => Ok(u64::from(first_byte >> 1)),
        1 => Ok(128_u64 + u64::from(first_byte >> 2) + (remaining_bytes_value << 6)),
        2 => Ok(16_512_u64 + u64::from(first_byte >> 3) + (remaining_bytes_value << 5)),
        3 => Ok(2_113_664_u64 + u64::from(first_byte >> 4) + (remaining_bytes_value << 4)),
        4 => Ok(270_549_120_u64 + u64::from(first_byte >> 5) + (remaining_bytes_value << 3)),
        5 => Ok(34_630_287_488_u64 + u64::from(first_byte >> 6) + (remaining_bytes_value << 2)),
        6 => Ok(4_432_676_798_592_u64 + u64::from(first_byte >> 7) + (remaining_bytes_value << 1)),
        7 => Ok(567_382_630_219_904_u64 + remaining_bytes_value),
        _ => Ok(72_624_976_668_147_840_u64.wrapping_add(remaining_bytes_value)),
    }}
}}

fn field_header_size(index: u64, payload_size: usize, integer_encoded: bool) -> usize {{
    match payload_size {{
        0 => varint_size_from_value((index << 2) | 0b00),
        8 => varint_size_from_value((index << 2) | 0b01),
        size => {{
            if integer_encoded {{
                varint_size_from_value((index << 2) | 0b11)
            }} else {{
                varint_size_from_value((index << 2) | 0b10) + varint_size_from_value(size as u64)
            }}
        }}
    }}
}}

fn serialize_field_header<T: Write>(
    writer: &mut T,
    index: u64,
    payload_size: usize,
    integer_encoded: bool,
) -> io::Result<()> {{
    match payload_size {{
        0 => serialize_varint((index << 2) | 0b00, writer),
        8 => serialize_varint((index << 2) | 0b01, writer),
        size => {{
            if integer_encoded {{
                serialize_varint((index << 2) | 0b11, writer)
            }} else {{
                serialize_varint((index << 2) | 0b10, writer)?;
                serialize_varint(size as u64, writer)
            }}
        }}
    }}
}}

fn deserialize_field_header<T: BufRead>(reader: &mut T) -> io::Result<(u64, usize)> {{
    let tag = deserialize_varint(&mut *reader)?;

    let index = tag >> 2;

    let size = match tag & 0b11 {{
        0b00 => 0,
        0b01 => 8,
        0b10 => deserialize_varint(&mut *reader)? as usize,
        _ => {{
            let buffer = (&mut *reader).fill_buf()?;

            if buffer.is_empty() {{
                return Err(Error::new(
                    ErrorKind::UnexpectedEof,
                    \"Error decoding field.\",
                ));
            }}

            varint_size_from_first_byte(buffer[0]) as usize
        }}
    }};

    Ok((index, size))
}}

fn skip<T: BufRead>(reader: &mut T, mut amount: usize) -> io::Result<()> {{
    while amount > 0 {{
        let buffer = reader.fill_buf()?;
        let num_bytes_to_consume = min(buffer.len(), amount);
        reader.consume(num_bytes_to_consume);
        amount -= num_bytes_to_consume;
    }}

    Ok(())
}}

pub trait Serialize {{
    fn size(&self) -> usize;

    fn serialize<T: Write>(&self, writer: &mut T) -> io::Result<()>;
}}

pub trait Deserialize {{
    fn deserialize<T>(reader: &mut T) -> io::Result<Self>
    where
        Self: Sized,
        T: BufRead;
}}",
            typical_version,
        )
        .unwrap();

        // The `unwrap` is safe because the `std::fmt::Write` impl for `String` is infallible.
        writeln!(&mut buffer).unwrap();

        // The `unwrap` is safe because the `std::fmt::Write` impl for `String` is infallible.
        write_module_contents(
            &mut buffer,
            0,
            &schema::Namespace { components: vec![] },
            &tree.children,
            &tree.schema,
        )
        .unwrap();
    }

    buffer
}

// Insert a schema into a module.
fn insert_schema(module: &mut Module, namespace: &schema::Namespace, schema: &schema::Schema) {
    let mut iter = namespace.components.iter();

    if let Some(head) = iter.next() {
        if let Some(child) = module.children.get_mut(head) {
            insert_schema(
                child,
                &schema::Namespace {
                    components: iter.cloned().collect(),
                },
                schema,
            );
        } else {
            let mut child = Module {
                children: BTreeMap::new(),
                schema: schema::Schema {
                    comment: vec![],
                    imports: BTreeMap::new(),
                    declarations: vec![],
                },
            };

            insert_schema(
                &mut child,
                &schema::Namespace {
                    components: iter.cloned().collect(),
                },
                schema,
            );

            module.children.insert(head.clone(), child);
        }
    } else {
        module.schema = schema.clone();
    }
}

// Write a module, including a trailing line break.
fn write_module<T: Write>(
    buffer: &mut T,
    indentation: usize,
    namespace: &schema::Namespace,
    name: &Identifier,
    module: &Module,
) -> Result<(), fmt::Error> {
    write_indentation(buffer, indentation)?;
    write!(buffer, "pub mod ")?;
    write_identifier(buffer, name, Snake, None)?;
    writeln!(buffer, " {{")?;

    let mut new_namespace = namespace.clone();
    new_namespace.components.push(name.clone());

    write_module_contents(
        buffer,
        indentation + 1,
        &new_namespace,
        &module.children,
        &module.schema,
    )?;

    write_indentation(buffer, indentation)?;
    writeln!(buffer, "}}")?;

    Ok(())
}

// Write the contents of a module, including a trailing line break if there was anything to render.
fn write_module_contents<T: Write>(
    buffer: &mut T,
    indentation: usize,
    namespace: &schema::Namespace,
    children: &BTreeMap<Identifier, Module>,
    schema: &schema::Schema,
) -> Result<(), fmt::Error> {
    let schema_empty = schema.declarations.is_empty();

    for (i, (child_name, child)) in children.iter().enumerate() {
        write_module(buffer, indentation, namespace, child_name, child)?;

        if i < children.len() - 1 || !schema_empty {
            writeln!(buffer)?;
        }
    }

    write_schema(buffer, indentation, namespace, schema)?;

    Ok(())
}

// Write a schema, including a trailing line break if there was anything to render.
#[allow(clippy::too_many_lines)]
fn write_schema<T: Write>(
    buffer: &mut T,
    indentation: usize,
    namespace: &schema::Namespace,
    schema: &schema::Schema,
) -> Result<(), fmt::Error> {
    // Construct a map from import name to namespace.
    let mut imports = BTreeMap::new();
    for (name, import) in &schema.imports {
        // The unwrap is safe due to [ref:namespace_populated].
        imports.insert(name.clone(), import.namespace.clone().unwrap());
    }

    // Write the declarations.
    let mut iter = schema.declarations.iter().peekable();
    while let Some(declaration) = iter.next() {
        match &declaration.variant {
            schema::DeclarationVariant::Struct => {
                write_struct(
                    buffer,
                    indentation,
                    &imports,
                    namespace,
                    &declaration.name,
                    &declaration.fields,
                    Out,
                )?;

                writeln!(buffer)?;

                write_struct(
                    buffer,
                    indentation,
                    &imports,
                    namespace,
                    &declaration.name,
                    &declaration.fields,
                    In,
                )?;

                writeln!(buffer)?;

                write_indentation(buffer, indentation)?;
                write!(buffer, "impl ")?;
                write_supers(buffer, indentation)?;
                write!(buffer, "Serialize for ")?;
                write_identifier(buffer, &declaration.name, Pascal, Some(Out))?;
                writeln!(buffer, " {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "fn size(&self) -> usize {{")?;
                if declaration.fields.is_empty() {
                    write_indentation(buffer, indentation + 2)?;
                    writeln!(buffer, "0")?;
                }
                for (i, field) in declaration.fields.iter().enumerate() {
                    let is_first = i == 0;
                    let is_last = i == declaration.fields.len() - 1;
                    if is_first {
                        write_indentation(buffer, indentation + 2)?;
                    }
                    match field.rule {
                        schema::Rule::Asymmetric | schema::Rule::Required => {
                            writeln!(buffer, "({{")?;
                            write_indentation(buffer, indentation + 3)?;
                            write!(buffer, "let payload = &self.")?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            writeln!(buffer, ";")?;
                        }
                        schema::Rule::Optional => {
                            write!(buffer, "self.")?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            writeln!(buffer, ".as_ref().map_or(0, |payload| {{")?;
                        }
                    }
                    write_indentation(buffer, indentation + 3)?;
                    write!(buffer, "let payload_size = ")?;
                    write_size_calculation_invocation(
                        buffer,
                        indentation,
                        &field.r#type.variant,
                        true,
                    )?;
                    writeln!(buffer, ";")?;
                    write_indentation(buffer, indentation + 3)?;
                    write_supers(buffer, indentation)?;
                    writeln!(
                        buffer,
                        "field_header_size({}, payload_size, {}) + payload_size",
                        field.index,
                        integer_encoded(&field.r#type),
                    )?;
                    write_indentation(buffer, indentation + 2)?;
                    write!(buffer, "}})")?;
                    if is_last {
                        writeln!(buffer)?;
                    } else {
                        write!(buffer, " + ")?;
                    }
                }
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}}")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(
                    buffer,
                    "fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> \
                        ::std::io::Result<()> {{",
                )?;
                for field in &declaration.fields {
                    match field.rule {
                        schema::Rule::Asymmetric | schema::Rule::Required => {
                            write_indentation(buffer, indentation + 2)?;
                            writeln!(buffer, "{{")?;
                            write_indentation(buffer, indentation + 3)?;
                            write!(buffer, "let payload = &self.")?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            writeln!(buffer, ";")?;
                        }
                        schema::Rule::Optional => {
                            write_indentation(buffer, indentation + 2)?;
                            write!(buffer, "if let Some(payload) = &self.")?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            writeln!(buffer, " {{")?;
                        }
                    }
                    write_indentation(buffer, indentation + 3)?;
                    write!(buffer, "let payload_size = ")?;
                    write_size_calculation_invocation(
                        buffer,
                        indentation,
                        &field.r#type.variant,
                        true,
                    )?;
                    writeln!(buffer, ";")?;
                    write_indentation(buffer, indentation + 3)?;
                    write_supers(buffer, indentation)?;
                    writeln!(
                        buffer,
                        "serialize_field_header(writer, {}, payload_size, {})?;",
                        field.index,
                        integer_encoded(&field.r#type),
                    )?;
                    write_serialization_invocation(
                        buffer,
                        indentation + 3,
                        indentation,
                        &field.r#type.variant,
                        true,
                    )?;
                    write_indentation(buffer, indentation + 2)?;
                    writeln!(buffer, "}}")?;
                    writeln!(buffer)?;
                }
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "Ok(())")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")?;

                writeln!(buffer)?;

                write_indentation(buffer, indentation)?;
                write!(buffer, "impl ")?;
                write_supers(buffer, indentation)?;
                write!(buffer, "Deserialize for ")?;
                write_identifier(buffer, &declaration.name, Pascal, Some(In))?;
                writeln!(buffer, " {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(
                    buffer,
                    "fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>",
                )?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "where")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "Self: Sized,")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "T: ::std::io::BufRead,")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "{{")?;
                if !&declaration.fields.is_empty() {
                    for field in &declaration.fields {
                        write_indentation(buffer, indentation + 2)?;
                        write!(buffer, "let mut ")?;
                        write_identifier(buffer, &field.name, Snake, None)?;
                        write!(buffer, ": Option<")?;
                        write_type(buffer, &imports, namespace, &field.r#type.variant, In)?;
                        writeln!(buffer, "> = None;")?;
                    }
                    writeln!(buffer)?;
                }
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "loop {{")?;
                write_indentation(buffer, indentation + 3)?;
                write!(buffer, "let (index, size) = match ")?;
                write_supers(buffer, indentation)?;
                writeln!(buffer, "deserialize_field_header(&mut *reader) {{")?;
                write_indentation(buffer, indentation + 4)?;
                writeln!(buffer, "Ok(header) => header,")?;
                write_indentation(buffer, indentation + 4)?;
                writeln!(buffer, "Err(err) => {{")?;
                write_indentation(buffer, indentation + 5)?;
                writeln!(
                    buffer,
                    "if let std::io::ErrorKind::UnexpectedEof = err.kind() {{",
                )?;
                write_indentation(buffer, indentation + 6)?;
                writeln!(buffer, "break;")?;
                write_indentation(buffer, indentation + 5)?;
                writeln!(buffer, "}}")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 5)?;
                writeln!(buffer, "return Err(err);")?;
                write_indentation(buffer, indentation + 4)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(buffer, "}};")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(
                    buffer,
                    "let mut sub_reader = ::std::io::Read::take(&mut *reader, size as u64);",
                )?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(buffer, "match index {{")?;
                for field in &declaration.fields {
                    write_indentation(buffer, indentation + 4)?;
                    writeln!(buffer, "{} => {{", field.index)?;
                    write_deserialization_invocation(
                        buffer,
                        indentation + 5,
                        indentation,
                        &imports,
                        namespace,
                        &field.r#type.variant,
                        true,
                    )?;
                    writeln!(buffer)?;
                    write_indentation(buffer, indentation + 5)?;
                    write_identifier(buffer, &field.name, Snake, None)?;
                    writeln!(buffer, ".get_or_insert(payload);")?;
                    write_indentation(buffer, indentation + 4)?;
                    writeln!(buffer, "}}")?;
                }
                write_indentation(buffer, indentation + 4)?;
                writeln!(buffer, "_ => {{")?;
                write_indentation(buffer, indentation + 5)?;
                write_supers(buffer, indentation)?;
                writeln!(buffer, "skip(&mut sub_reader, size as usize)?;")?;
                write_indentation(buffer, indentation + 4)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "}}")?;
                writeln!(buffer)?;
                if declaration.fields.iter().any(|field| match field.rule {
                    schema::Rule::Asymmetric | schema::Rule::Optional => false,
                    schema::Rule::Required => true,
                }) {
                    write_indentation(buffer, indentation + 2)?;
                    write!(buffer, "if ")?;
                    let mut first = true;
                    for field in &declaration.fields {
                        match field.rule {
                            schema::Rule::Asymmetric | schema::Rule::Optional => {}
                            schema::Rule::Required => {
                                if first {
                                    first = false;
                                } else {
                                    write!(buffer, " || ")?;
                                }
                                write_identifier(buffer, &field.name, Snake, None)?;
                                write!(buffer, ".is_none()")?;
                            }
                        }
                    }
                    writeln!(buffer, " {{")?;
                    write_indentation(buffer, indentation + 3)?;
                    writeln!(buffer, "return Err(::std::io::Error::new(")?;
                    write_indentation(buffer, indentation + 4)?;
                    writeln!(buffer, "::std::io::ErrorKind::InvalidData,")?;
                    write_indentation(buffer, indentation + 4)?;
                    writeln!(buffer, "\"Struct missing one or more field(s).\",")?;
                    write_indentation(buffer, indentation + 3)?;
                    writeln!(buffer, "));")?;
                    write_indentation(buffer, indentation + 2)?;
                    writeln!(buffer, "}}")?;
                    writeln!(buffer)?;
                }
                write_indentation(buffer, indentation + 2)?;
                write!(buffer, "Ok(")?;
                write_identifier(buffer, &declaration.name, Pascal, Some(In))?;
                writeln!(buffer, " {{")?;
                for field in &declaration.fields {
                    write_indentation(buffer, indentation + 3)?;
                    write_identifier(buffer, &field.name, Snake, None)?;
                    match field.rule {
                        schema::Rule::Asymmetric | schema::Rule::Optional => {}
                        schema::Rule::Required => {
                            write!(buffer, ": ")?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            write!(buffer, ".unwrap()")?;
                        }
                    }
                    writeln!(buffer, ",")?;
                }
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "}})")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")?;

                writeln!(buffer)?;

                write_indentation(buffer, indentation)?;
                write!(buffer, "impl From<")?;
                write_identifier(buffer, &declaration.name, Pascal, Some(Out))?;
                write!(buffer, "> for ")?;
                write_identifier(buffer, &declaration.name, Pascal, Some(In))?;
                writeln!(buffer, " {{")?;
                write_indentation(buffer, indentation + 1)?;
                write!(buffer, "fn from(message: ")?;
                write_identifier(buffer, &declaration.name, Pascal, Some(Out))?;
                writeln!(buffer, ") -> Self {{")?;
                write_indentation(buffer, indentation + 2)?;
                write_identifier(buffer, &declaration.name, Pascal, Some(In))?;
                writeln!(buffer, " {{")?;
                for field in &declaration.fields {
                    match field.rule {
                        schema::Rule::Asymmetric => {
                            write_indentation(buffer, indentation + 3)?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            write!(buffer, ": Some(message.")?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            writeln!(buffer, ".into()),")?;
                        }
                        schema::Rule::Optional => {
                            write_indentation(buffer, indentation + 3)?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            write!(buffer, ": message.")?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            writeln!(buffer, ".map(|payload| payload.into()),")?;
                        }
                        schema::Rule::Required => {
                            write_indentation(buffer, indentation + 3)?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            write!(buffer, ": message.")?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            writeln!(buffer, ".into(),")?;
                        }
                    }
                }
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")?;
            }
            schema::DeclarationVariant::Choice => {
                write_choice(
                    buffer,
                    indentation,
                    &imports,
                    namespace,
                    &declaration.name,
                    &declaration.fields,
                    Out,
                )?;

                writeln!(buffer)?;

                write_choice(
                    buffer,
                    indentation,
                    &imports,
                    namespace,
                    &declaration.name,
                    &declaration.fields,
                    In,
                )?;

                writeln!(buffer)?;

                write_indentation(buffer, indentation)?;
                write!(buffer, "impl ")?;
                write_supers(buffer, indentation)?;
                write!(buffer, "Serialize for ")?;
                write_identifier(buffer, &declaration.name, Pascal, Some(Out))?;
                writeln!(buffer, " {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "fn size(&self) -> usize {{")?;
                write_indentation(buffer, indentation + 2)?;
                // [tag:empty_enum_ref_match] We match on `self*` instead of `self` due to
                // https://github.com/rust-lang/rust/issues/78123.
                writeln!(buffer, "match *self {{")?;
                for field in &declaration.fields {
                    write_indentation(buffer, indentation + 3)?;
                    write_identifier(buffer, &declaration.name, Pascal, Some(Out))?;
                    write!(buffer, "::")?;
                    write_identifier(buffer, &field.name, Pascal, None)?;
                    match field.rule {
                        schema::Rule::Asymmetric | schema::Rule::Optional => {
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                writeln!(buffer, "(ref fallback) => {{")?;
                            } else {
                                writeln!(buffer, "(ref payload, ref fallback) => {{")?;
                            }
                        }
                        schema::Rule::Required => {
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                writeln!(buffer, " => {{")?;
                            } else {
                                writeln!(buffer, "(ref payload) => {{")?;
                            }
                        }
                    }
                    write_indentation(buffer, indentation + 4)?;
                    write!(buffer, "let payload_size = ")?;
                    write_size_calculation_invocation(
                        buffer,
                        indentation,
                        &field.r#type.variant,
                        true,
                    )?;
                    writeln!(buffer, ";")?;
                    write_indentation(buffer, indentation + 4)?;
                    write_supers(buffer, indentation)?;
                    writeln!(
                        buffer,
                        "field_header_size({}, payload_size, {}) +",
                        field.index,
                        integer_encoded(&field.r#type),
                    )?;
                    write_indentation(buffer, indentation + 5)?;
                    write!(buffer, "payload_size")?;
                    match field.rule {
                        schema::Rule::Asymmetric | schema::Rule::Optional => {
                            writeln!(buffer, " +")?;
                            write_indentation(buffer, indentation + 5)?;
                            writeln!(buffer, "fallback.size()")?;
                        }
                        schema::Rule::Required => {
                            writeln!(buffer)?;
                        }
                    }
                    write_indentation(buffer, indentation + 3)?;
                    writeln!(buffer, "}}")?;
                }
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}}")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(
                    buffer,
                    "fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> \
                        ::std::io::Result<()> {{",
                )?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "match *self {{")?; // [ref:empty_enum_ref_match]
                for field in &declaration.fields {
                    write_indentation(buffer, indentation + 3)?;
                    write_identifier(buffer, &declaration.name, Pascal, Some(Out))?;
                    write!(buffer, "::")?;
                    write_identifier(buffer, &field.name, Pascal, None)?;
                    match field.rule {
                        schema::Rule::Asymmetric | schema::Rule::Optional => {
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                writeln!(buffer, "(ref fallback) => {{")?;
                            } else {
                                writeln!(buffer, "(ref payload, ref fallback) => {{")?;
                            }
                        }
                        schema::Rule::Required => {
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                writeln!(buffer, " => {{")?;
                            } else {
                                writeln!(buffer, "(ref payload) => {{")?;
                            }
                        }
                    }
                    write_indentation(buffer, indentation + 4)?;
                    write!(buffer, "let payload_size = ")?;
                    write_size_calculation_invocation(
                        buffer,
                        indentation,
                        &field.r#type.variant,
                        true,
                    )?;
                    writeln!(buffer, ";")?;
                    write_indentation(buffer, indentation + 4)?;
                    write_supers(buffer, indentation)?;
                    writeln!(
                        buffer,
                        "serialize_field_header(writer, {}, payload_size, {})?;",
                        field.index,
                        integer_encoded(&field.r#type),
                    )?;
                    write_serialization_invocation(
                        buffer,
                        indentation + 4,
                        indentation,
                        &field.r#type.variant,
                        true,
                    )?;
                    match field.rule {
                        schema::Rule::Asymmetric | schema::Rule::Optional => {
                            write_indentation(buffer, indentation + 4)?;
                            writeln!(buffer, "fallback.serialize(writer)")?;
                        }
                        schema::Rule::Required => {
                            write_indentation(buffer, indentation + 4)?;
                            writeln!(buffer, "Ok(())")?;
                        }
                    }
                    write_indentation(buffer, indentation + 3)?;
                    writeln!(buffer, "}}")?;
                }
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")?;

                writeln!(buffer)?;

                write_indentation(buffer, indentation)?;
                write!(buffer, "impl ")?;
                write_supers(buffer, indentation)?;
                write!(buffer, "Deserialize for ")?;
                write_identifier(buffer, &declaration.name, Pascal, Some(In))?;
                writeln!(buffer, " {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(
                    buffer,
                    "fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>",
                )?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "where")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "Self: Sized,")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "T: ::std::io::BufRead,")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "{{")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "loop {{")?;
                write_indentation(buffer, indentation + 3)?;
                write!(buffer, "let (index, size) = ")?;
                write_supers(buffer, indentation)?;
                writeln!(buffer, "deserialize_field_header(&mut *reader)?;")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(
                    buffer,
                    "let mut sub_reader = ::std::io::Read::take(&mut *reader, size as u64);",
                )?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(buffer, "match index {{")?;
                for field in &declaration.fields {
                    write_indentation(buffer, indentation + 4)?;
                    writeln!(buffer, "{} => {{", field.index)?;
                    write_deserialization_invocation(
                        buffer,
                        indentation + 5,
                        indentation,
                        &imports,
                        namespace,
                        &field.r#type.variant,
                        true,
                    )?;
                    match field.rule {
                        schema::Rule::Optional => {
                            write_indentation(buffer, indentation + 5)?;
                            write!(buffer, "let fallback = Box::new(<")?;
                            write_identifier(buffer, &declaration.name, Pascal, Some(In))?;
                            write!(buffer, " as ")?;
                            write_supers(buffer, indentation)?;
                            writeln!(buffer, "Deserialize>::deserialize(&mut *reader)?);")?;
                            write_indentation(buffer, indentation + 5)?;
                            write!(buffer, "return Ok(")?;
                            write_identifier(buffer, &declaration.name, Pascal, Some(In))?;
                            write!(buffer, "::")?;
                            write_identifier(buffer, &field.name, Pascal, None)?;
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                writeln!(buffer, "(fallback));")?;
                            } else {
                                writeln!(buffer, "(payload, fallback));")?;
                            }
                        }
                        schema::Rule::Asymmetric | schema::Rule::Required => {
                            write_indentation(buffer, indentation + 5)?;
                            write!(buffer, "return Ok(")?;
                            write_identifier(buffer, &declaration.name, Pascal, Some(In))?;
                            write!(buffer, "::")?;
                            write_identifier(buffer, &field.name, Pascal, None)?;
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                writeln!(buffer, ");")?;
                            } else {
                                writeln!(buffer, "(payload));")?;
                            }
                        }
                    }
                    write_indentation(buffer, indentation + 4)?;
                    writeln!(buffer, "}}")?;
                }
                write_indentation(buffer, indentation + 4)?;
                writeln!(buffer, "_ => {{")?;
                write_indentation(buffer, indentation + 5)?;
                write_supers(buffer, indentation)?;
                writeln!(buffer, "skip(&mut sub_reader, size as usize)?;")?;
                write_indentation(buffer, indentation + 4)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")?;

                writeln!(buffer)?;

                write_indentation(buffer, indentation)?;
                write!(buffer, "impl From<")?;
                write_identifier(buffer, &declaration.name, Pascal, Some(Out))?;
                write!(buffer, "> for ")?;
                write_identifier(buffer, &declaration.name, Pascal, Some(In))?;
                writeln!(buffer, " {{")?;
                write_indentation(buffer, indentation + 1)?;
                write!(buffer, "fn from(message: ")?;
                write_identifier(buffer, &declaration.name, Pascal, Some(Out))?;
                writeln!(buffer, ") -> Self {{")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "match message {{")?;
                for field in &declaration.fields {
                    write_indentation(buffer, indentation + 3)?;
                    write_identifier(buffer, &declaration.name, Pascal, Some(Out))?;
                    write!(buffer, "::")?;
                    write_identifier(buffer, &field.name, Pascal, None)?;
                    match field.rule {
                        schema::Rule::Asymmetric | schema::Rule::Optional => {
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                write!(buffer, "(fallback) => ")?;
                            } else {
                                write!(buffer, "(payload, fallback) => ")?;
                            }
                        }
                        schema::Rule::Required => {
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                write!(buffer, " => ")?;
                            } else {
                                write!(buffer, "(payload) => ")?;
                            }
                        }
                    }
                    write_identifier(buffer, &declaration.name, Pascal, Some(In))?;
                    write!(buffer, "::")?;
                    write_identifier(buffer, &field.name, Pascal, None)?;
                    match field.rule {
                        schema::Rule::Optional => {
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                writeln!(buffer, "(Box::new((*fallback).into())),")?;
                            } else {
                                writeln!(
                                    buffer,
                                    "(payload.into(), Box::new((*fallback).into())),",
                                )?;
                            }
                        }
                        schema::Rule::Asymmetric | schema::Rule::Required => {
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                writeln!(buffer, ",")?;
                            } else {
                                writeln!(buffer, "(payload.into()),")?;
                            }
                        }
                    }
                }
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")?;
            }
        }

        if iter.peek().is_some() {
            writeln!(buffer)?;
        }
    }

    Ok(())
}

// Write a struct, including a trailing line break.
fn write_struct<T: Write>(
    buffer: &mut T,
    indentation: usize,
    imports: &BTreeMap<Identifier, schema::Namespace>,
    namespace: &schema::Namespace,
    name: &Identifier,
    fields: &[schema::Field],
    direction: Direction,
) -> Result<(), fmt::Error> {
    write_indentation(buffer, indentation)?;
    writeln!(buffer, "#[derive({})]", TRAITS_TO_DERIVE.join(", "))?;
    write_indentation(buffer, indentation)?;
    write!(buffer, "pub struct ")?;
    write_identifier(buffer, name, Pascal, Some(direction))?;
    writeln!(buffer, " {{")?;

    for field in fields {
        write_indentation(buffer, indentation + 1)?;
        write!(buffer, "pub ")?;
        write_identifier(buffer, &field.name, Snake, None)?;
        write!(buffer, ": ")?;
        match field.rule {
            schema::Rule::Asymmetric => match direction {
                Direction::Out => {}
                Direction::In => {
                    write!(buffer, "Option<")?;
                }
            },
            schema::Rule::Optional => {
                write!(buffer, "Option<")?;
            }
            schema::Rule::Required => {}
        }
        write_type(buffer, imports, namespace, &field.r#type.variant, direction)?;
        match field.rule {
            schema::Rule::Asymmetric => match direction {
                Direction::Out => {}
                Direction::In => {
                    write!(buffer, ">")?;
                }
            },
            schema::Rule::Optional => {
                write!(buffer, ">")?;
            }
            schema::Rule::Required => {}
        }
        writeln!(buffer, ",")?;
    }

    write_indentation(buffer, indentation)?;
    writeln!(buffer, "}}")?;

    Ok(())
}

// Write a choice, including a trailing line break.
fn write_choice<T: Write>(
    buffer: &mut T,
    indentation: usize,
    imports: &BTreeMap<Identifier, schema::Namespace>,
    namespace: &schema::Namespace,
    name: &Identifier,
    fields: &[schema::Field],
    direction: Direction,
) -> Result<(), fmt::Error> {
    write_indentation(buffer, indentation)?;
    writeln!(buffer, "#[derive({})]", TRAITS_TO_DERIVE.join(", "))?;
    write_indentation(buffer, indentation)?;
    write!(buffer, "pub enum ")?;
    write_identifier(buffer, name, Pascal, Some(direction))?;
    writeln!(buffer, " {{")?;

    for field in fields {
        write_indentation(buffer, indentation + 1)?;
        write_identifier(buffer, &field.name, Pascal, None)?;
        let fallback = match field.rule {
            schema::Rule::Asymmetric => match direction {
                Direction::Out => true,
                Direction::In => false,
            },
            schema::Rule::Optional => true,
            schema::Rule::Required => false,
        };
        if fallback {
            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                write!(buffer, "(Box<")?;
            } else {
                write!(buffer, "(")?;
                write_type(buffer, imports, namespace, &field.r#type.variant, direction)?;
                write!(buffer, ", Box<")?;
            }
            write_identifier(buffer, name, Pascal, Some(direction))?;
            writeln!(buffer, ">),")?;
        } else if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
            writeln!(buffer, ",")?;
        } else {
            write!(buffer, "(")?;
            write_type(buffer, imports, namespace, &field.r#type.variant, direction)?;
            writeln!(buffer, "),")?;
        }
    }

    write_indentation(buffer, indentation)?;
    writeln!(buffer, "}}")?;

    Ok(())
}

// Write a type.
fn write_type<T: Write>(
    buffer: &mut T,
    imports: &BTreeMap<Identifier, schema::Namespace>,
    namespace: &schema::Namespace,
    type_variant: &schema::TypeVariant,
    direction: Direction,
) -> Result<(), fmt::Error> {
    match type_variant {
        schema::TypeVariant::Array(inner_type) => {
            write!(buffer, "Vec<")?;
            write_type(buffer, imports, namespace, &inner_type.variant, direction)?;
            write!(buffer, ">")?;
        }
        schema::TypeVariant::Bool => {
            write!(buffer, "bool")?;
        }
        schema::TypeVariant::Bytes => {
            write!(buffer, "Vec<u8>")?;
        }
        schema::TypeVariant::Custom(import, name) => {
            let type_namespace = schema::Namespace {
                components: import.as_ref().map_or_else(
                    || namespace.components.clone(),
                    |import| imports[import].components.clone(),
                ),
            };

            let (relative_type_namespace, ancestors) =
                relativize_namespace(&type_namespace, namespace);

            write_supers(buffer, ancestors)?;

            for component in relative_type_namespace.components {
                write_identifier(buffer, &component, Snake, None)?;
                write!(buffer, "::")?;
            }

            write_identifier(buffer, name, Pascal, Some(direction))?;
        }
        schema::TypeVariant::F64 => {
            write!(buffer, "f64")?;
        }
        schema::TypeVariant::S64 => {
            write!(buffer, "i64")?;
        }
        schema::TypeVariant::String => {
            write!(buffer, "String")?;
        }
        schema::TypeVariant::U64 => {
            write!(buffer, "u64")?;
        }
        schema::TypeVariant::Unit => {
            write!(buffer, "()")?;
        }
    }

    Ok(())
}

// Write an identifier with an optional direction suffix in a way that Rust will be happy with.
fn write_identifier<T: Write>(
    buffer: &mut T,
    identifier: &Identifier,
    case: CaseConvention,
    suffix: Option<Direction>,
) -> Result<(), fmt::Error> {
    let identifier_with_suffix = suffix.map_or_else(
        || identifier.clone(),
        |suffix| {
            identifier.join(
                &match suffix {
                    Direction::In => "In",
                    Direction::Out => "Out",
                }
                .into(),
            )
        },
    );

    let converted_identifier = match case {
        CaseConvention::Pascal => identifier_with_suffix.pascal_case(),
        CaseConvention::Snake => identifier_with_suffix.snake_case(),
    };

    if RUST_KEYWORDS
        .iter()
        .any(|keyword| converted_identifier == *keyword)
    {
        write!(buffer, "r#")?;
    }

    write!(buffer, "{}", converted_identifier)?;

    Ok(())
}

// Write the given level of indentation.
fn write_indentation<T: Write>(buffer: &mut T, indentation: usize) -> Result<(), fmt::Error> {
    for _ in 0..indentation {
        write!(buffer, "{}", INDENTATION)?;
    }

    Ok(())
}

// Write a series of `super::super::...`.
fn write_supers<T: Write>(buffer: &mut T, count: usize) -> Result<(), fmt::Error> {
    for _ in 0..count {
        write!(buffer, "super::")?;
    }

    Ok(())
}

// Write the logic to invoke the size calculation logic for a value.
fn write_size_calculation_invocation<T: Write>(
    buffer: &mut T,
    supers: usize,
    type_variant: &schema::TypeVariant,
    is_field: bool,
) -> Result<(), fmt::Error> {
    match type_variant {
        schema::TypeVariant::Array(inner_type) => match &inner_type.variant {
            schema::TypeVariant::Array(_)
            | schema::TypeVariant::Bytes
            | schema::TypeVariant::Custom(_, _)
            | schema::TypeVariant::String => {
                write!(
                    buffer,
                    "payload.iter().fold(0_usize, |x, payload| {{ let payload_size = ",
                )?;
                write_size_calculation_invocation(buffer, supers, &inner_type.variant, false)?;
                write!(buffer, "; x + ")?;
                write_supers(buffer, supers)?;
                write!(
                    buffer,
                    "varint_size_from_value(payload_size as u64) + payload_size }})",
                )
            }
            schema::TypeVariant::Bool | schema::TypeVariant::S64 | schema::TypeVariant::U64 => {
                write!(buffer, "payload.iter().fold(0_usize, |x, payload| x + ")?;
                write_size_calculation_invocation(buffer, supers, &inner_type.variant, false)?;
                write!(buffer, ")")
            }
            schema::TypeVariant::F64 => write!(buffer, "8_usize * payload.len()"),
            schema::TypeVariant::Unit => {
                write!(buffer, "{{ let payload = &(payload.len() as u64); ")?;
                write_size_calculation_invocation(
                    buffer,
                    supers,
                    &schema::TypeVariant::U64,
                    is_field,
                )?;
                write!(buffer, " }}")
            }
        },
        schema::TypeVariant::Bool => {
            if is_field {
                write!(buffer, "if *payload {{ 1_usize }} else {{ 0_usize }}")
            } else {
                write!(buffer, "1_usize")
            }
        }
        schema::TypeVariant::Bytes | schema::TypeVariant::String => write!(buffer, "payload.len()"),
        schema::TypeVariant::Custom(_, _) => write!(buffer, "payload.size()"),
        schema::TypeVariant::F64 => {
            if is_field {
                write!(
                    buffer,
                    "if payload.to_bits() == 0_u64 {{ 0_usize }} else {{ 8_usize }}",
                )
            } else {
                write!(buffer, "8_usize")
            }
        }
        schema::TypeVariant::S64 => {
            if is_field {
                write!(buffer, "{{ let zigzag = ")?;
                write_supers(buffer, supers)?;
                write!(
                    buffer,
                    "zigzag_encode(*payload); match zigzag {{ 0_u64 => {{ 0_usize }}, \
                        1_u64..=567_382_630_219_903_u64 => {{ ",
                )?;
                write_supers(buffer, supers)?;
                write!(
                    buffer,
                    "varint_size_from_value(zigzag) }}, 567_382_630_219_904_u64..=\
                        18_446_744_073_709_551_615_u64 => {{ 8_usize }} }} }}",
                )
            } else {
                write_supers(buffer, supers)?;
                write!(buffer, "varint_size_from_value(")?;
                write_supers(buffer, supers)?;
                write!(buffer, "zigzag_encode(*payload))")
            }
        }
        schema::TypeVariant::U64 => {
            if is_field {
                write!(
                    buffer,
                    "match *payload {{ 0_u64 => {{ 0_usize }}, 1_u64..=567_382_630_219_903_u64 => \
                        {{ ",
                )?;
                write_supers(buffer, supers)?;
                write!(
                    buffer,
                    "varint_size_from_value(*payload) }}, 567_382_630_219_904_u64..=\
                        18_446_744_073_709_551_615_u64 => {{ 8_usize }} }}",
                )
            } else {
                write_supers(buffer, supers)?;
                write!(buffer, "varint_size_from_value(*payload)")
            }
        }
        schema::TypeVariant::Unit => write!(buffer, "0_usize"),
    }
}

// Write the logic to invoke the serialization logic for a value, including a trailing line break.
#[allow(clippy::too_many_lines)]
fn write_serialization_invocation<T: Write>(
    buffer: &mut T,
    indentation: usize,
    supers: usize,
    type_variant: &schema::TypeVariant,
    is_field: bool,
) -> Result<(), fmt::Error> {
    write_indentation(buffer, indentation)?;

    match type_variant {
        schema::TypeVariant::Array(inner_type) => match &inner_type.variant {
            schema::TypeVariant::Array(_)
            | schema::TypeVariant::Bytes
            | schema::TypeVariant::Custom(_, _)
            | schema::TypeVariant::String => {
                writeln!(buffer, "for payload in payload {{")?;
                write_indentation(buffer, indentation + 1)?;
                write_supers(buffer, supers)?;
                write!(buffer, "serialize_varint(")?;
                write_size_calculation_invocation(buffer, supers, &inner_type.variant, false)?;
                writeln!(buffer, " as u64, writer)?;")?;
                write_serialization_invocation(
                    buffer,
                    indentation + 1,
                    supers,
                    &inner_type.variant,
                    false,
                )?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")
            }
            schema::TypeVariant::Bool
            | schema::TypeVariant::S64
            | schema::TypeVariant::U64
            | schema::TypeVariant::F64 => {
                writeln!(buffer, "for payload in payload {{")?;
                write_serialization_invocation(
                    buffer,
                    indentation + 1,
                    supers,
                    &inner_type.variant,
                    false,
                )?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")
            }
            schema::TypeVariant::Unit => {
                writeln!(buffer, "{{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "let payload = &(payload.len() as u64);")?;
                write_serialization_invocation(
                    buffer,
                    indentation + 1,
                    supers,
                    &schema::TypeVariant::U64,
                    is_field,
                )?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")
            }
        },
        schema::TypeVariant::Bool => {
            if is_field {
                writeln!(buffer, "if payload_size != 0_usize {{")?;
                write_indentation(buffer, indentation + 1)?;
                write_supers(buffer, supers)?;
                writeln!(buffer, "serialize_varint(*payload as u64, writer)?;")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")
            } else {
                write_supers(buffer, supers)?;
                writeln!(buffer, "serialize_varint(*payload as u64, writer)?;")
            }
        }
        schema::TypeVariant::Bytes => writeln!(buffer, "writer.write_all(payload)?;"),
        schema::TypeVariant::Custom(_, _) => writeln!(buffer, "payload.serialize(writer)?;"),
        schema::TypeVariant::F64 => {
            if is_field {
                writeln!(buffer, "if payload_size != 0_usize {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "writer.write_all(&payload.to_le_bytes())?;")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")
            } else {
                writeln!(buffer, "writer.write_all(&payload.to_le_bytes())?;")
            }
        }
        schema::TypeVariant::S64 => {
            if is_field {
                writeln!(buffer, "match payload_size {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "0_usize => {{}}")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(
                    buffer,
                    "8_usize => writer.write_all(&payload.to_le_bytes())?,",
                )?;
                write_indentation(buffer, indentation + 1)?;
                write!(buffer, "_ => ")?;
                write_supers(buffer, supers)?;
                writeln!(buffer, "serialize_varint(")?;
                write_indentation(buffer, indentation + 2)?;
                write_supers(buffer, supers)?;
                writeln!(buffer, "zigzag_encode(*payload),")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "writer,")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, ")?,")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")
            } else {
                write_supers(buffer, supers)?;
                write!(buffer, "serialize_varint(")?;
                write_supers(buffer, supers)?;
                writeln!(buffer, "zigzag_encode(*payload), writer)?;")
            }
        }
        schema::TypeVariant::String => writeln!(buffer, "writer.write_all(payload.as_bytes())?;"),
        schema::TypeVariant::U64 => {
            if is_field {
                writeln!(buffer, "match payload_size {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "0_usize => {{}}")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(
                    buffer,
                    "8_usize => writer.write_all(&payload.to_le_bytes())?,",
                )?;
                write_indentation(buffer, indentation + 1)?;
                write!(buffer, "_ => ")?;
                write_supers(buffer, supers)?;
                writeln!(buffer, "serialize_varint(*payload, writer)?,")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")
            } else {
                write_supers(buffer, supers)?;
                writeln!(buffer, "serialize_varint(*payload, writer)?;")
            }
        }
        schema::TypeVariant::Unit => writeln!(buffer, "();"),
    }
}

// Write the logic to invoke the deserialization logic for a value, including a trailing line break.
#[allow(clippy::too_many_lines)]
fn write_deserialization_invocation<T: Write>(
    buffer: &mut T,
    indentation: usize,
    supers: usize,
    imports: &BTreeMap<Identifier, schema::Namespace>,
    namespace: &schema::Namespace,
    type_variant: &schema::TypeVariant,
    is_field: bool,
) -> Result<(), fmt::Error> {
    match type_variant {
        schema::TypeVariant::Array(inner_type) => match &inner_type.variant {
            schema::TypeVariant::Array(_)
            | schema::TypeVariant::Bytes
            | schema::TypeVariant::Custom(_, _)
            | schema::TypeVariant::String => {
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "let mut payload = Vec::new();")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "loop {{")?;
                write_indentation(buffer, indentation + 1)?;
                write!(buffer, "let element_size = match ")?;
                write_supers(buffer, supers)?;
                writeln!(buffer, "deserialize_varint(&mut sub_reader) {{")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "Ok(element_size) => element_size,")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "Err(err) => {{")?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(
                    buffer,
                    "if let std::io::ErrorKind::UnexpectedEof = err.kind() {{",
                )?;
                write_indentation(buffer, indentation + 4)?;
                writeln!(buffer, "break;")?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(buffer, "}}")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(buffer, "return Err(err);")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}};")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(
                    buffer,
                    "let mut sub_reader = ::std::io::Read::take(\
                            &mut sub_reader, element_size as u64);",
                )?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "payload.push({{")?;
                write_deserialization_invocation(
                    buffer,
                    indentation + 2,
                    supers,
                    imports,
                    namespace,
                    &inner_type.variant,
                    false,
                )?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "payload")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}});")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")
            }
            schema::TypeVariant::Bool
            | schema::TypeVariant::S64
            | schema::TypeVariant::U64
            | schema::TypeVariant::F64 => {
                write_indentation(buffer, indentation)?;
                write!(
                    buffer,
                    "fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut T) -> \
                            ::std::io::Result<",
                )?;
                write_type(buffer, imports, namespace, &inner_type.variant, In)?;
                writeln!(buffer, "> {{")?;
                write_deserialization_invocation(
                    buffer,
                    indentation + 1,
                    supers,
                    imports,
                    namespace,
                    &inner_type.variant,
                    false,
                )?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "Ok(payload)")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "let mut payload = Vec::new();")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "loop {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(
                    buffer,
                    "payload.push(match deserialize_element(&mut sub_reader) {{",
                )?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "Ok(element) => element,")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "Err(err) => {{")?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(
                    buffer,
                    "if let std::io::ErrorKind::UnexpectedEof = err.kind() {{",
                )?;
                write_indentation(buffer, indentation + 4)?;
                writeln!(buffer, "break;")?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(buffer, "}}")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(buffer, "return Err(err);")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}});")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")
            }
            schema::TypeVariant::Unit => {
                write_deserialization_invocation(
                    buffer,
                    indentation,
                    supers,
                    imports,
                    namespace,
                    &schema::TypeVariant::U64,
                    is_field,
                )?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "let payload = vec![(); payload as usize];")
            }
        },
        schema::TypeVariant::Bool => {
            write_indentation(buffer, indentation)?;
            if is_field {
                writeln!(buffer, "let payload = if size == 0_usize {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "false")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}} else {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "let mut buffer = [0_u8];")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(
                    buffer,
                    "::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;",
                )?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "buffer[0] != 0b0000_0001")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}};")
            } else {
                writeln!(buffer, "let mut buffer = [0_u8];")?;
                write_indentation(buffer, indentation)?;
                writeln!(
                    buffer,
                    "::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;",
                )?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "let payload = buffer[0] != 0b0000_0001;")
            }
        }
        schema::TypeVariant::Bytes => {
            write_indentation(buffer, indentation)?;
            writeln!(buffer, "let mut payload = vec![];")?;
            write_indentation(buffer, indentation)?;
            writeln!(
                buffer,
                "::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;",
            )
        }
        schema::TypeVariant::Custom(_, _) => {
            write_indentation(buffer, indentation)?;
            write!(buffer, "let payload = <")?;
            write_type(buffer, imports, namespace, type_variant, In)?;
            write!(buffer, " as ")?;
            write_supers(buffer, supers)?;
            writeln!(buffer, "Deserialize>::deserialize(&mut sub_reader)?;")
        }
        schema::TypeVariant::F64 => {
            write_indentation(buffer, indentation)?;
            if is_field {
                writeln!(buffer, "let payload = if size == 0_usize {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "0.0_f64")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}} else {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "let mut buffer = [0; 8];")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(
                    buffer,
                    "::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;",
                )?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "f64::from_le_bytes(buffer)")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}};")
            } else {
                writeln!(buffer, "let mut buffer = [0; 8];")?;
                write_indentation(buffer, indentation)?;
                writeln!(
                    buffer,
                    "::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;",
                )?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "let payload = f64::from_le_bytes(buffer);")
            }
        }
        schema::TypeVariant::S64 => {
            write_indentation(buffer, indentation)?;
            if is_field {
                writeln!(buffer, "let payload = match size {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "0_usize => 0_i64,")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "8_usize => {{")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "let mut buffer = [0; 8];")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(
                    buffer,
                    "::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;",
                )?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "i64::from_le_bytes(buffer)")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 1)?;
                write!(buffer, "_ => ")?;
                write_supers(buffer, supers)?;
                write!(buffer, "zigzag_decode(")?;
                write_supers(buffer, supers)?;
                writeln!(buffer, "deserialize_varint(&mut sub_reader)?),")?;
                write_indentation(buffer, indentation)?;
                write!(buffer, "}};")
            } else {
                write!(buffer, "let payload = ")?;
                write_supers(buffer, supers)?;
                write!(buffer, "zigzag_decode(")?;
                write_supers(buffer, supers)?;
                writeln!(buffer, "deserialize_varint(&mut sub_reader)?);")
            }
        }
        schema::TypeVariant::String => {
            write_indentation(buffer, indentation)?;
            writeln!(buffer, "let mut buffer = vec![];")?;
            write_indentation(buffer, indentation)?;
            writeln!(
                buffer,
                "::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;",
            )?;
            write_indentation(buffer, indentation)?;
            writeln!(
                buffer,
                "let payload = std::str::from_utf8(&buffer).map_or_else(",
            )?;
            write_indentation(buffer, indentation + 1)?;
            writeln!(
                buffer,
                "|err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),",
            )?;
            write_indentation(buffer, indentation + 1)?;
            writeln!(buffer, "|result| Ok(result.to_owned()),")?;
            write_indentation(buffer, indentation)?;
            writeln!(buffer, ")?;")
        }
        schema::TypeVariant::U64 => {
            write_indentation(buffer, indentation)?;
            if is_field {
                writeln!(buffer, "let payload = match size {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "0_usize => 0_u64,")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "8_usize => {{")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "let mut buffer = [0; 8];")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(
                    buffer,
                    "::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;",
                )?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "u64::from_le_bytes(buffer)")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(
                    buffer,
                    "_ => super::super::deserialize_varint(&mut sub_reader)?,",
                )?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}};")
            } else {
                write!(buffer, "let payload = ")?;
                write_supers(buffer, supers)?;
                writeln!(buffer, "deserialize_varint(&mut sub_reader)?;")
            }
        }
        schema::TypeVariant::Unit => {
            write_indentation(buffer, indentation)?;
            writeln!(buffer, "let payload = ();")
        }
    }
}

// Determine whether a type is encoded as a varint.
fn integer_encoded(r#type: &schema::Type) -> bool {
    match &r#type.variant {
        schema::TypeVariant::Bool | schema::TypeVariant::S64 | schema::TypeVariant::U64 => true,
        schema::TypeVariant::Array(_)
        | schema::TypeVariant::Bytes
        | schema::TypeVariant::Custom(_, _)
        | schema::TypeVariant::F64
        | schema::TypeVariant::String
        | schema::TypeVariant::Unit => false,
    }
}

#[cfg(test)]
mod tests {
    use {
        crate::{generate_rust::generate, schema_loader::load_schemas, validator::validate},
        std::path::Path,
    };

    // This test doesn't work on Windows, for some reason.
    #[allow(clippy::too_many_lines)]
    #[test]
    fn generate_example() {
        let schemas = load_schemas(Path::new("integration_tests/types/main.t")).unwrap();
        validate(&schemas).unwrap();

        assert_eq!(
            generate("0.0.0", &schemas),
            "\
// This file was automatically generated by Typical 0.0.0.
// Visit https://github.com/stepchowfun/typical for more information.

use std::{
    cmp::min,
    io::{self, BufRead, Error, ErrorKind, Write},
    mem::transmute,
};

fn zigzag_encode(value: i64) -> u64 {
    unsafe { transmute::<i64, u64>(value >> 63) ^ transmute::<i64, u64>(value << 1) }
}

fn zigzag_decode(value: u64) -> i64 {
    unsafe { transmute::<u64, i64>(value >> 1) ^ -transmute::<u64, i64>(value & 1) }
}

fn varint_size_from_value(value: u64) -> usize {
    match value {
        0_u64..=127_u64 => 1,
        128_u64..=16_511_u64 => 2,
        16_512_u64..=2_113_663_u64 => 3,
        2_113_664_u64..=270_549_119_u64 => 4,
        270_549_120_u64..=34_630_287_487_u64 => 5,
        34_630_287_488_u64..=4_432_676_798_591_u64 => 6,
        4_432_676_798_592_u64..=567_382_630_219_903_u64 => 7,
        567_382_630_219_904_u64..=72_624_976_668_147_839_u64 => 8,
        72_624_976_668_147_840_u64..=18_446_744_073_709_551_615_u64 => 9,
    }
}

fn varint_size_from_first_byte(first_byte: u8) -> u32 {
    first_byte.trailing_zeros() + 1
}

fn serialize_varint<T: Write>(mut value: u64, writer: &mut T) -> io::Result<()> {
    match value {
        0_u64..=127_u64 => writer.write_all(&[((value << 1) as u8) | 0b0000_0001]),
        128_u64..=16_511_u64 => {
            value -= 128_u64;
            writer.write_all(&[((value << 2) as u8) | 0b0000_0010, (value >> 6) as u8])
        }
        16_512_u64..=2_113_663_u64 => {
            value -= 16_512_u64;
            writer.write_all(&[
                ((value << 3) as u8) | 0b0000_0100,
                (value >> 5) as u8,
                (value >> 13) as u8,
            ])
        }
        2_113_664_u64..=270_549_119_u64 => {
            value -= 2_113_664_u64;
            writer.write_all(&[
                ((value << 4) as u8) | 0b0000_1000,
                (value >> 4) as u8,
                (value >> 12) as u8,
                (value >> 20) as u8,
            ])
        }
        270_549_120_u64..=34_630_287_487_u64 => {
            value -= 270_549_120_u64;
            writer.write_all(&[
                ((value << 5) as u8) | 0b0001_0000,
                (value >> 3) as u8,
                (value >> 11) as u8,
                (value >> 19) as u8,
                (value >> 27) as u8,
            ])
        }
        34_630_287_488_u64..=4_432_676_798_591_u64 => {
            value -= 34_630_287_488_u64;
            writer.write_all(&[
                ((value << 6) as u8) | 0b0010_0000,
                (value >> 2) as u8,
                (value >> 10) as u8,
                (value >> 18) as u8,
                (value >> 26) as u8,
                (value >> 34) as u8,
            ])
        }
        4_432_676_798_592_u64..=567_382_630_219_903_u64 => {
            value -= 4_432_676_798_592_u64;
            writer.write_all(&[
                ((value << 7) as u8) | 0b0100_0000,
                (value >> 1) as u8,
                (value >> 9) as u8,
                (value >> 17) as u8,
                (value >> 25) as u8,
                (value >> 33) as u8,
                (value >> 41) as u8,
            ])
        }
        567_382_630_219_904_u64..=72_624_976_668_147_839_u64 => {
            value -= 567_382_630_219_904_u64;
            writer.write_all(&[
                0b1000_0000,
                value as u8,
                (value >> 8) as u8,
                (value >> 16) as u8,
                (value >> 24) as u8,
                (value >> 32) as u8,
                (value >> 40) as u8,
                (value >> 48) as u8,
            ])
        }
        72_624_976_668_147_840_u64..=18_446_744_073_709_551_615_u64 => {
            value -= 72_624_976_668_147_840_u64;
            writer.write_all(&[
                0b0000_0000,
                value as u8,
                (value >> 8) as u8,
                (value >> 16) as u8,
                (value >> 24) as u8,
                (value >> 32) as u8,
                (value >> 40) as u8,
                (value >> 48) as u8,
                (value >> 56) as u8,
            ])
        }
    }
}

fn deserialize_varint<T: BufRead>(reader: &mut T) -> io::Result<u64> {
    let mut first_byte_buffer = [0; 1];
    reader.read_exact(&mut first_byte_buffer[..])?;
    let first_byte = first_byte_buffer[0];
    let size_minus_one = first_byte.trailing_zeros();
    let mut remaining_bytes_buffer = [0; 8];
    reader.read_exact(&mut remaining_bytes_buffer[0..size_minus_one as usize])?;
    let remaining_bytes_value = u64::from_le_bytes(remaining_bytes_buffer);
    match size_minus_one {
        0 => Ok(u64::from(first_byte >> 1)),
        1 => Ok(128_u64 + u64::from(first_byte >> 2) + (remaining_bytes_value << 6)),
        2 => Ok(16_512_u64 + u64::from(first_byte >> 3) + (remaining_bytes_value << 5)),
        3 => Ok(2_113_664_u64 + u64::from(first_byte >> 4) + (remaining_bytes_value << 4)),
        4 => Ok(270_549_120_u64 + u64::from(first_byte >> 5) + (remaining_bytes_value << 3)),
        5 => Ok(34_630_287_488_u64 + u64::from(first_byte >> 6) + (remaining_bytes_value << 2)),
        6 => Ok(4_432_676_798_592_u64 + u64::from(first_byte >> 7) + (remaining_bytes_value << 1)),
        7 => Ok(567_382_630_219_904_u64 + remaining_bytes_value),
        _ => Ok(72_624_976_668_147_840_u64.wrapping_add(remaining_bytes_value)),
    }
}

fn field_header_size(index: u64, payload_size: usize, integer_encoded: bool) -> usize {
    match payload_size {
        0 => varint_size_from_value((index << 2) | 0b00),
        8 => varint_size_from_value((index << 2) | 0b01),
        size => {
            if integer_encoded {
                varint_size_from_value((index << 2) | 0b11)
            } else {
                varint_size_from_value((index << 2) | 0b10) + varint_size_from_value(size as u64)
            }
        }
    }
}

fn serialize_field_header<T: Write>(
    writer: &mut T,
    index: u64,
    payload_size: usize,
    integer_encoded: bool,
) -> io::Result<()> {
    match payload_size {
        0 => serialize_varint((index << 2) | 0b00, writer),
        8 => serialize_varint((index << 2) | 0b01, writer),
        size => {
            if integer_encoded {
                serialize_varint((index << 2) | 0b11, writer)
            } else {
                serialize_varint((index << 2) | 0b10, writer)?;
                serialize_varint(size as u64, writer)
            }
        }
    }
}

fn deserialize_field_header<T: BufRead>(reader: &mut T) -> io::Result<(u64, usize)> {
    let tag = deserialize_varint(&mut *reader)?;

    let index = tag >> 2;

    let size = match tag & 0b11 {
        0b00 => 0,
        0b01 => 8,
        0b10 => deserialize_varint(&mut *reader)? as usize,
        _ => {
            let buffer = (&mut *reader).fill_buf()?;

            if buffer.is_empty() {
                return Err(Error::new(
                    ErrorKind::UnexpectedEof,
                    \"Error decoding field.\",
                ));
            }

            varint_size_from_first_byte(buffer[0]) as usize
        }
    };

    Ok((index, size))
}

fn skip<T: BufRead>(reader: &mut T, mut amount: usize) -> io::Result<()> {
    while amount > 0 {
        let buffer = reader.fill_buf()?;
        let num_bytes_to_consume = min(buffer.len(), amount);
        reader.consume(num_bytes_to_consume);
        amount -= num_bytes_to_consume;
    }

    Ok(())
}

pub trait Serialize {
    fn size(&self) -> usize;

    fn serialize<T: Write>(&self, writer: &mut T) -> io::Result<()>;
}

pub trait Deserialize {
    fn deserialize<T>(reader: &mut T) -> io::Result<Self>
    where
        Self: Sized,
        T: BufRead;
}

pub mod circular_dependency {
    pub mod dependency {
        pub mod main {
            #[derive(Clone, Debug)]
            pub struct StructFromBelowOut {
                pub x: super::super::main::StructFromAboveOut,
            }

            #[derive(Clone, Debug)]
            pub struct StructFromBelowIn {
                pub x: super::super::main::StructFromAboveIn,
            }

            impl super::super::super::Serialize for StructFromBelowOut {
                fn size(&self) -> usize {
                    ({
                        let payload = &self.x;
                        let payload_size = payload.size();
                        super::super::super::field_header_size(0, payload_size, false) + \
                            payload_size
                    })
                }

                fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                    {
                        let payload = &self.x;
                        let payload_size = payload.size();
                        super::super::super::serialize_field_header(writer, 0, payload_size, \
                            false)?;
                        payload.serialize(writer)?;
                    }

                    Ok(())
                }
            }

            impl super::super::super::Deserialize for StructFromBelowIn {
                fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
                where
                    Self: Sized,
                    T: ::std::io::BufRead,
                {
                    let mut x: Option<super::super::main::StructFromAboveIn> = None;

                    loop {
                        let (index, size) = match \
                            super::super::super::deserialize_field_header(&mut *reader) {
                            Ok(header) => header,
                            Err(err) => {
                                if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                    break;
                                }

                                return Err(err);
                            }
                        };

                        let mut sub_reader = ::std::io::Read::take(&mut *reader, size as u64);

                        match index {
                            0 => {
                                let payload = <super::super::main::StructFromAboveIn as \
                                    super::super::super::Deserialize>::deserialize(&mut \
                                    sub_reader)?;

                                x.get_or_insert(payload);
                            }
                            _ => {
                                super::super::super::skip(&mut sub_reader, size as usize)?;
                            }
                        }
                    }

                    if x.is_none() {
                        return Err(::std::io::Error::new(
                            ::std::io::ErrorKind::InvalidData,
                            \"Struct missing one or more field(s).\",
                        ));
                    }

                    Ok(StructFromBelowIn {
                        x: x.unwrap(),
                    })
                }
            }

            impl From<StructFromBelowOut> for StructFromBelowIn {
                fn from(message: StructFromBelowOut) -> Self {
                    StructFromBelowIn {
                        x: message.x.into(),
                    }
                }
            }
        }
    }

    pub mod main {
        #[derive(Clone, Debug)]
        pub struct StructFromAboveOut {
        }

        #[derive(Clone, Debug)]
        pub struct StructFromAboveIn {
        }

        impl super::super::Serialize for StructFromAboveOut {
            fn size(&self) -> usize {
                0
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                Ok(())
            }
        }

        impl super::super::Deserialize for StructFromAboveIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                loop {
                    let (index, size) = match super::super::deserialize_field_header(&mut \
                        *reader) {
                        Ok(header) => header,
                        Err(err) => {
                            if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                break;
                            }

                            return Err(err);
                        }
                    };

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size as u64);

                    match index {
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }

                Ok(StructFromAboveIn {
                })
            }
        }

        impl From<StructFromAboveOut> for StructFromAboveIn {
            fn from(message: StructFromAboveOut) -> Self {
                StructFromAboveIn {
                }
            }
        }
    }
}

pub mod comprehensive {
    pub mod bar {
        #[derive(Clone, Debug)]
        pub enum BarOut {
            ARequired,
            BRequired(f64),
            CRequired(u64),
            DRequired(i64),
            ERequired(bool),
            FRequired(Vec<u8>),
            GRequired(String),
            HRequired(Vec<()>),
            IRequired(Vec<f64>),
            JRequired(Vec<u64>),
            KRequired(Vec<i64>),
            LRequired(Vec<bool>),
            MRequired(Vec<Vec<u8>>),
            NRequired(Vec<String>),
            ORequired(Vec<Vec<String>>),
            AAsymmetric(Box<BarOut>),
            BAsymmetric(f64, Box<BarOut>),
            CAsymmetric(u64, Box<BarOut>),
            DAsymmetric(i64, Box<BarOut>),
            EAsymmetric(bool, Box<BarOut>),
            FAsymmetric(Vec<u8>, Box<BarOut>),
            GAsymmetric(String, Box<BarOut>),
            HAsymmetric(Vec<()>, Box<BarOut>),
            IAsymmetric(Vec<f64>, Box<BarOut>),
            JAsymmetric(Vec<u64>, Box<BarOut>),
            KAsymmetric(Vec<i64>, Box<BarOut>),
            LAsymmetric(Vec<bool>, Box<BarOut>),
            MAsymmetric(Vec<Vec<u8>>, Box<BarOut>),
            NAsymmetric(Vec<String>, Box<BarOut>),
            OAsymmetric(Vec<Vec<String>>, Box<BarOut>),
            AOptional(Box<BarOut>),
            BOptional(f64, Box<BarOut>),
            COptional(u64, Box<BarOut>),
            DOptional(i64, Box<BarOut>),
            EOptional(bool, Box<BarOut>),
            FOptional(Vec<u8>, Box<BarOut>),
            GOptional(String, Box<BarOut>),
            HOptional(Vec<()>, Box<BarOut>),
            IOptional(Vec<f64>, Box<BarOut>),
            JOptional(Vec<u64>, Box<BarOut>),
            KOptional(Vec<i64>, Box<BarOut>),
            LOptional(Vec<bool>, Box<BarOut>),
            MOptional(Vec<Vec<u8>>, Box<BarOut>),
            NOptional(Vec<String>, Box<BarOut>),
            OOptional(Vec<Vec<String>>, Box<BarOut>),
        }

        #[derive(Clone, Debug)]
        pub enum BarIn {
            ARequired,
            BRequired(f64),
            CRequired(u64),
            DRequired(i64),
            ERequired(bool),
            FRequired(Vec<u8>),
            GRequired(String),
            HRequired(Vec<()>),
            IRequired(Vec<f64>),
            JRequired(Vec<u64>),
            KRequired(Vec<i64>),
            LRequired(Vec<bool>),
            MRequired(Vec<Vec<u8>>),
            NRequired(Vec<String>),
            ORequired(Vec<Vec<String>>),
            AAsymmetric,
            BAsymmetric(f64),
            CAsymmetric(u64),
            DAsymmetric(i64),
            EAsymmetric(bool),
            FAsymmetric(Vec<u8>),
            GAsymmetric(String),
            HAsymmetric(Vec<()>),
            IAsymmetric(Vec<f64>),
            JAsymmetric(Vec<u64>),
            KAsymmetric(Vec<i64>),
            LAsymmetric(Vec<bool>),
            MAsymmetric(Vec<Vec<u8>>),
            NAsymmetric(Vec<String>),
            OAsymmetric(Vec<Vec<String>>),
            AOptional(Box<BarIn>),
            BOptional(f64, Box<BarIn>),
            COptional(u64, Box<BarIn>),
            DOptional(i64, Box<BarIn>),
            EOptional(bool, Box<BarIn>),
            FOptional(Vec<u8>, Box<BarIn>),
            GOptional(String, Box<BarIn>),
            HOptional(Vec<()>, Box<BarIn>),
            IOptional(Vec<f64>, Box<BarIn>),
            JOptional(Vec<u64>, Box<BarIn>),
            KOptional(Vec<i64>, Box<BarIn>),
            LOptional(Vec<bool>, Box<BarIn>),
            MOptional(Vec<Vec<u8>>, Box<BarIn>),
            NOptional(Vec<String>, Box<BarIn>),
            OOptional(Vec<Vec<String>>, Box<BarIn>),
        }

        impl super::super::Serialize for BarOut {
            fn size(&self) -> usize {
                match *self {
                    BarOut::ARequired => {
                        let payload_size = 0_usize;
                        super::super::field_header_size(0, payload_size, false) +
                            payload_size
                    }
                    BarOut::BRequired(ref payload) => {
                        let payload_size = if payload.to_bits() == 0_u64 { 0_usize } else { \
                            8_usize };
                        super::super::field_header_size(1, payload_size, false) +
                            payload_size
                    }
                    BarOut::CRequired(ref payload) => {
                        let payload_size = match *payload { 0_u64 => { 0_usize }, \
                            1_u64..=567_382_630_219_903_u64 => { \
                            super::super::varint_size_from_value(*payload) }, \
                            567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                            8_usize } };
                        super::super::field_header_size(2, payload_size, true) +
                            payload_size
                    }
                    BarOut::DRequired(ref payload) => {
                        let payload_size = { let zigzag = super::super::zigzag_encode(*payload); \
                            match zigzag { 0_u64 => { 0_usize }, \
                            1_u64..=567_382_630_219_903_u64 => { \
                            super::super::varint_size_from_value(zigzag) }, \
                            567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                            8_usize } } };
                        super::super::field_header_size(3, payload_size, true) +
                            payload_size
                    }
                    BarOut::ERequired(ref payload) => {
                        let payload_size = if *payload { 1_usize } else { 0_usize };
                        super::super::field_header_size(4, payload_size, true) +
                            payload_size
                    }
                    BarOut::FRequired(ref payload) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(5, payload_size, false) +
                            payload_size
                    }
                    BarOut::GRequired(ref payload) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(6, payload_size, false) +
                            payload_size
                    }
                    BarOut::HRequired(ref payload) => {
                        let payload_size = { let payload = &(payload.len() as u64); match \
                            *payload { 0_u64 => { 0_usize }, 1_u64..=567_382_630_219_903_u64 => { \
                            super::super::varint_size_from_value(*payload) }, \
                            567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                            8_usize } } };
                        super::super::field_header_size(7, payload_size, false) +
                            payload_size
                    }
                    BarOut::IRequired(ref payload) => {
                        let payload_size = 8_usize * payload.len();
                        super::super::field_header_size(8, payload_size, false) +
                            payload_size
                    }
                    BarOut::JRequired(ref payload) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                            super::super::varint_size_from_value(*payload));
                        super::super::field_header_size(9, payload_size, false) +
                            payload_size
                    }
                    BarOut::KRequired(ref payload) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                            super::super::varint_size_from_value(super::super::zigzag_encode(*\
                            payload)));
                        super::super::field_header_size(10, payload_size, false) +
                            payload_size
                    }
                    BarOut::LRequired(ref payload) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| x + 1_usize);
                        super::super::field_header_size(11, payload_size, false) +
                            payload_size
                    }
                    BarOut::MRequired(ref payload) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size });
                        super::super::field_header_size(12, payload_size, false) +
                            payload_size
                    }
                    BarOut::NRequired(ref payload) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size });
                        super::super::field_header_size(13, payload_size, false) +
                            payload_size
                    }
                    BarOut::ORequired(ref payload) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size }); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size });
                        super::super::field_header_size(14, payload_size, false) +
                            payload_size
                    }
                    BarOut::AAsymmetric(ref fallback) => {
                        let payload_size = 0_usize;
                        super::super::field_header_size(16, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::BAsymmetric(ref payload, ref fallback) => {
                        let payload_size = if payload.to_bits() == 0_u64 { 0_usize } else { \
                            8_usize };
                        super::super::field_header_size(17, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::CAsymmetric(ref payload, ref fallback) => {
                        let payload_size = match *payload { 0_u64 => { 0_usize }, \
                            1_u64..=567_382_630_219_903_u64 => { \
                            super::super::varint_size_from_value(*payload) }, \
                            567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                            8_usize } };
                        super::super::field_header_size(18, payload_size, true) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::DAsymmetric(ref payload, ref fallback) => {
                        let payload_size = { let zigzag = super::super::zigzag_encode(*payload); \
                            match zigzag { 0_u64 => { 0_usize }, \
                            1_u64..=567_382_630_219_903_u64 => { \
                            super::super::varint_size_from_value(zigzag) }, \
                            567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                            8_usize } } };
                        super::super::field_header_size(19, payload_size, true) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::EAsymmetric(ref payload, ref fallback) => {
                        let payload_size = if *payload { 1_usize } else { 0_usize };
                        super::super::field_header_size(20, payload_size, true) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::FAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(21, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::GAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(22, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::HAsymmetric(ref payload, ref fallback) => {
                        let payload_size = { let payload = &(payload.len() as u64); match \
                            *payload { 0_u64 => { 0_usize }, 1_u64..=567_382_630_219_903_u64 => { \
                            super::super::varint_size_from_value(*payload) }, \
                            567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                            8_usize } } };
                        super::super::field_header_size(23, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::IAsymmetric(ref payload, ref fallback) => {
                        let payload_size = 8_usize * payload.len();
                        super::super::field_header_size(24, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::JAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                            super::super::varint_size_from_value(*payload));
                        super::super::field_header_size(25, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::KAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                            super::super::varint_size_from_value(super::super::zigzag_encode(*\
                            payload)));
                        super::super::field_header_size(26, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::LAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| x + 1_usize);
                        super::super::field_header_size(27, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::MAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size });
                        super::super::field_header_size(28, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::NAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size });
                        super::super::field_header_size(29, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::OAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size }); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size });
                        super::super::field_header_size(30, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::AOptional(ref fallback) => {
                        let payload_size = 0_usize;
                        super::super::field_header_size(32, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::BOptional(ref payload, ref fallback) => {
                        let payload_size = if payload.to_bits() == 0_u64 { 0_usize } else { \
                            8_usize };
                        super::super::field_header_size(33, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::COptional(ref payload, ref fallback) => {
                        let payload_size = match *payload { 0_u64 => { 0_usize }, \
                            1_u64..=567_382_630_219_903_u64 => { \
                            super::super::varint_size_from_value(*payload) }, \
                            567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                            8_usize } };
                        super::super::field_header_size(34, payload_size, true) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::DOptional(ref payload, ref fallback) => {
                        let payload_size = { let zigzag = super::super::zigzag_encode(*payload); \
                            match zigzag { 0_u64 => { 0_usize }, \
                            1_u64..=567_382_630_219_903_u64 => { \
                            super::super::varint_size_from_value(zigzag) }, \
                            567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                            8_usize } } };
                        super::super::field_header_size(35, payload_size, true) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::EOptional(ref payload, ref fallback) => {
                        let payload_size = if *payload { 1_usize } else { 0_usize };
                        super::super::field_header_size(36, payload_size, true) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::FOptional(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(37, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::GOptional(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(38, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::HOptional(ref payload, ref fallback) => {
                        let payload_size = { let payload = &(payload.len() as u64); match \
                            *payload { 0_u64 => { 0_usize }, 1_u64..=567_382_630_219_903_u64 => { \
                            super::super::varint_size_from_value(*payload) }, \
                            567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                            8_usize } } };
                        super::super::field_header_size(39, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::IOptional(ref payload, ref fallback) => {
                        let payload_size = 8_usize * payload.len();
                        super::super::field_header_size(40, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::JOptional(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                            super::super::varint_size_from_value(*payload));
                        super::super::field_header_size(41, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::KOptional(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                            super::super::varint_size_from_value(super::super::zigzag_encode(*\
                            payload)));
                        super::super::field_header_size(42, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::LOptional(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| x + 1_usize);
                        super::super::field_header_size(43, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::MOptional(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size });
                        super::super::field_header_size(44, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::NOptional(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size });
                        super::super::field_header_size(45, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::OOptional(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size }); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size });
                        super::super::field_header_size(46, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                }
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                match *self {
                    BarOut::ARequired => {
                        let payload_size = 0_usize;
                        super::super::serialize_field_header(writer, 0, payload_size, false)?;
                        ();
                        Ok(())
                    }
                    BarOut::BRequired(ref payload) => {
                        let payload_size = if payload.to_bits() == 0_u64 { 0_usize } else { \
                            8_usize };
                        super::super::serialize_field_header(writer, 1, payload_size, false)?;
                        if payload_size != 0_usize {
                            writer.write_all(&payload.to_le_bytes())?;
                        }
                        Ok(())
                    }
                    BarOut::CRequired(ref payload) => {
                        let payload_size = match *payload { 0_u64 => { 0_usize }, \
                            1_u64..=567_382_630_219_903_u64 => { \
                            super::super::varint_size_from_value(*payload) }, \
                            567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                            8_usize } };
                        super::super::serialize_field_header(writer, 2, payload_size, true)?;
                        match payload_size {
                            0_usize => {}
                            8_usize => writer.write_all(&payload.to_le_bytes())?,
                            _ => super::super::serialize_varint(*payload, writer)?,
                        }
                        Ok(())
                    }
                    BarOut::DRequired(ref payload) => {
                        let payload_size = { let zigzag = super::super::zigzag_encode(*payload); \
                            match zigzag { 0_u64 => { 0_usize }, \
                            1_u64..=567_382_630_219_903_u64 => { \
                            super::super::varint_size_from_value(zigzag) }, \
                            567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                            8_usize } } };
                        super::super::serialize_field_header(writer, 3, payload_size, true)?;
                        match payload_size {
                            0_usize => {}
                            8_usize => writer.write_all(&payload.to_le_bytes())?,
                            _ => super::super::serialize_varint(
                                super::super::zigzag_encode(*payload),
                                writer,
                            )?,
                        }
                        Ok(())
                    }
                    BarOut::ERequired(ref payload) => {
                        let payload_size = if *payload { 1_usize } else { 0_usize };
                        super::super::serialize_field_header(writer, 4, payload_size, true)?;
                        if payload_size != 0_usize {
                            super::super::serialize_varint(*payload as u64, writer)?;
                        }
                        Ok(())
                    }
                    BarOut::FRequired(ref payload) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 5, payload_size, false)?;
                        writer.write_all(payload)?;
                        Ok(())
                    }
                    BarOut::GRequired(ref payload) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 6, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        Ok(())
                    }
                    BarOut::HRequired(ref payload) => {
                        let payload_size = { let payload = &(payload.len() as u64); match \
                            *payload { 0_u64 => { 0_usize }, 1_u64..=567_382_630_219_903_u64 => { \
                            super::super::varint_size_from_value(*payload) }, \
                            567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                            8_usize } } };
                        super::super::serialize_field_header(writer, 7, payload_size, false)?;
                        {
                            let payload = &(payload.len() as u64);
                            match payload_size {
                                0_usize => {}
                                8_usize => writer.write_all(&payload.to_le_bytes())?,
                                _ => super::super::serialize_varint(*payload, writer)?,
                            }
                        }
                        Ok(())
                    }
                    BarOut::IRequired(ref payload) => {
                        let payload_size = 8_usize * payload.len();
                        super::super::serialize_field_header(writer, 8, payload_size, false)?;
                        for payload in payload {
                            writer.write_all(&payload.to_le_bytes())?;
                        }
                        Ok(())
                    }
                    BarOut::JRequired(ref payload) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                            super::super::varint_size_from_value(*payload));
                        super::super::serialize_field_header(writer, 9, payload_size, false)?;
                        for payload in payload {
                            super::super::serialize_varint(*payload, writer)?;
                        }
                        Ok(())
                    }
                    BarOut::KRequired(ref payload) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                            super::super::varint_size_from_value(super::super::zigzag_encode(*\
                            payload)));
                        super::super::serialize_field_header(writer, 10, payload_size, false)?;
                        for payload in payload {
                            super::super::serialize_varint(super::super::zigzag_encode(*payload), \
                                writer)?;
                        }
                        Ok(())
                    }
                    BarOut::LRequired(ref payload) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| x + 1_usize);
                        super::super::serialize_field_header(writer, 11, payload_size, false)?;
                        for payload in payload {
                            super::super::serialize_varint(*payload as u64, writer)?;
                        }
                        Ok(())
                    }
                    BarOut::MRequired(ref payload) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size });
                        super::super::serialize_field_header(writer, 12, payload_size, false)?;
                        for payload in payload {
                            super::super::serialize_varint(payload.len() as u64, writer)?;
                            writer.write_all(payload)?;
                        }
                        Ok(())
                    }
                    BarOut::NRequired(ref payload) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size });
                        super::super::serialize_field_header(writer, 13, payload_size, false)?;
                        for payload in payload {
                            super::super::serialize_varint(payload.len() as u64, writer)?;
                            writer.write_all(payload.as_bytes())?;
                        }
                        Ok(())
                    }
                    BarOut::ORequired(ref payload) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size }); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size });
                        super::super::serialize_field_header(writer, 14, payload_size, false)?;
                        for payload in payload {
                            super::super::serialize_varint(payload.iter().fold(0_usize, |x, \
                                payload| { let payload_size = payload.len(); x + \
                                super::super::varint_size_from_value(payload_size as u64) + \
                                payload_size }) as u64, writer)?;
                            for payload in payload {
                                super::super::serialize_varint(payload.len() as u64, writer)?;
                                writer.write_all(payload.as_bytes())?;
                            }
                        }
                        Ok(())
                    }
                    BarOut::AAsymmetric(ref fallback) => {
                        let payload_size = 0_usize;
                        super::super::serialize_field_header(writer, 16, payload_size, false)?;
                        ();
                        fallback.serialize(writer)
                    }
                    BarOut::BAsymmetric(ref payload, ref fallback) => {
                        let payload_size = if payload.to_bits() == 0_u64 { 0_usize } else { \
                            8_usize };
                        super::super::serialize_field_header(writer, 17, payload_size, false)?;
                        if payload_size != 0_usize {
                            writer.write_all(&payload.to_le_bytes())?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::CAsymmetric(ref payload, ref fallback) => {
                        let payload_size = match *payload { 0_u64 => { 0_usize }, \
                            1_u64..=567_382_630_219_903_u64 => { \
                            super::super::varint_size_from_value(*payload) }, \
                            567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                            8_usize } };
                        super::super::serialize_field_header(writer, 18, payload_size, true)?;
                        match payload_size {
                            0_usize => {}
                            8_usize => writer.write_all(&payload.to_le_bytes())?,
                            _ => super::super::serialize_varint(*payload, writer)?,
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::DAsymmetric(ref payload, ref fallback) => {
                        let payload_size = { let zigzag = super::super::zigzag_encode(*payload); \
                            match zigzag { 0_u64 => { 0_usize }, \
                            1_u64..=567_382_630_219_903_u64 => { \
                            super::super::varint_size_from_value(zigzag) }, \
                            567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                            8_usize } } };
                        super::super::serialize_field_header(writer, 19, payload_size, true)?;
                        match payload_size {
                            0_usize => {}
                            8_usize => writer.write_all(&payload.to_le_bytes())?,
                            _ => super::super::serialize_varint(
                                super::super::zigzag_encode(*payload),
                                writer,
                            )?,
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::EAsymmetric(ref payload, ref fallback) => {
                        let payload_size = if *payload { 1_usize } else { 0_usize };
                        super::super::serialize_field_header(writer, 20, payload_size, true)?;
                        if payload_size != 0_usize {
                            super::super::serialize_varint(*payload as u64, writer)?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::FAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 21, payload_size, false)?;
                        writer.write_all(payload)?;
                        fallback.serialize(writer)
                    }
                    BarOut::GAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 22, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    BarOut::HAsymmetric(ref payload, ref fallback) => {
                        let payload_size = { let payload = &(payload.len() as u64); match \
                            *payload { 0_u64 => { 0_usize }, 1_u64..=567_382_630_219_903_u64 => { \
                            super::super::varint_size_from_value(*payload) }, \
                            567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                            8_usize } } };
                        super::super::serialize_field_header(writer, 23, payload_size, false)?;
                        {
                            let payload = &(payload.len() as u64);
                            match payload_size {
                                0_usize => {}
                                8_usize => writer.write_all(&payload.to_le_bytes())?,
                                _ => super::super::serialize_varint(*payload, writer)?,
                            }
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::IAsymmetric(ref payload, ref fallback) => {
                        let payload_size = 8_usize * payload.len();
                        super::super::serialize_field_header(writer, 24, payload_size, false)?;
                        for payload in payload {
                            writer.write_all(&payload.to_le_bytes())?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::JAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                            super::super::varint_size_from_value(*payload));
                        super::super::serialize_field_header(writer, 25, payload_size, false)?;
                        for payload in payload {
                            super::super::serialize_varint(*payload, writer)?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::KAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                            super::super::varint_size_from_value(super::super::zigzag_encode(*\
                            payload)));
                        super::super::serialize_field_header(writer, 26, payload_size, false)?;
                        for payload in payload {
                            super::super::serialize_varint(super::super::zigzag_encode(*payload), \
                                writer)?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::LAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| x + 1_usize);
                        super::super::serialize_field_header(writer, 27, payload_size, false)?;
                        for payload in payload {
                            super::super::serialize_varint(*payload as u64, writer)?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::MAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size });
                        super::super::serialize_field_header(writer, 28, payload_size, false)?;
                        for payload in payload {
                            super::super::serialize_varint(payload.len() as u64, writer)?;
                            writer.write_all(payload)?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::NAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size });
                        super::super::serialize_field_header(writer, 29, payload_size, false)?;
                        for payload in payload {
                            super::super::serialize_varint(payload.len() as u64, writer)?;
                            writer.write_all(payload.as_bytes())?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::OAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size }); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size });
                        super::super::serialize_field_header(writer, 30, payload_size, false)?;
                        for payload in payload {
                            super::super::serialize_varint(payload.iter().fold(0_usize, |x, \
                                payload| { let payload_size = payload.len(); x + \
                                super::super::varint_size_from_value(payload_size as u64) + \
                                payload_size }) as u64, writer)?;
                            for payload in payload {
                                super::super::serialize_varint(payload.len() as u64, writer)?;
                                writer.write_all(payload.as_bytes())?;
                            }
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::AOptional(ref fallback) => {
                        let payload_size = 0_usize;
                        super::super::serialize_field_header(writer, 32, payload_size, false)?;
                        ();
                        fallback.serialize(writer)
                    }
                    BarOut::BOptional(ref payload, ref fallback) => {
                        let payload_size = if payload.to_bits() == 0_u64 { 0_usize } else { \
                            8_usize };
                        super::super::serialize_field_header(writer, 33, payload_size, false)?;
                        if payload_size != 0_usize {
                            writer.write_all(&payload.to_le_bytes())?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::COptional(ref payload, ref fallback) => {
                        let payload_size = match *payload { 0_u64 => { 0_usize }, \
                            1_u64..=567_382_630_219_903_u64 => { \
                            super::super::varint_size_from_value(*payload) }, \
                            567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                            8_usize } };
                        super::super::serialize_field_header(writer, 34, payload_size, true)?;
                        match payload_size {
                            0_usize => {}
                            8_usize => writer.write_all(&payload.to_le_bytes())?,
                            _ => super::super::serialize_varint(*payload, writer)?,
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::DOptional(ref payload, ref fallback) => {
                        let payload_size = { let zigzag = super::super::zigzag_encode(*payload); \
                            match zigzag { 0_u64 => { 0_usize }, \
                            1_u64..=567_382_630_219_903_u64 => { \
                            super::super::varint_size_from_value(zigzag) }, \
                            567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                            8_usize } } };
                        super::super::serialize_field_header(writer, 35, payload_size, true)?;
                        match payload_size {
                            0_usize => {}
                            8_usize => writer.write_all(&payload.to_le_bytes())?,
                            _ => super::super::serialize_varint(
                                super::super::zigzag_encode(*payload),
                                writer,
                            )?,
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::EOptional(ref payload, ref fallback) => {
                        let payload_size = if *payload { 1_usize } else { 0_usize };
                        super::super::serialize_field_header(writer, 36, payload_size, true)?;
                        if payload_size != 0_usize {
                            super::super::serialize_varint(*payload as u64, writer)?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::FOptional(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 37, payload_size, false)?;
                        writer.write_all(payload)?;
                        fallback.serialize(writer)
                    }
                    BarOut::GOptional(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 38, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    BarOut::HOptional(ref payload, ref fallback) => {
                        let payload_size = { let payload = &(payload.len() as u64); match \
                            *payload { 0_u64 => { 0_usize }, 1_u64..=567_382_630_219_903_u64 => { \
                            super::super::varint_size_from_value(*payload) }, \
                            567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                            8_usize } } };
                        super::super::serialize_field_header(writer, 39, payload_size, false)?;
                        {
                            let payload = &(payload.len() as u64);
                            match payload_size {
                                0_usize => {}
                                8_usize => writer.write_all(&payload.to_le_bytes())?,
                                _ => super::super::serialize_varint(*payload, writer)?,
                            }
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::IOptional(ref payload, ref fallback) => {
                        let payload_size = 8_usize * payload.len();
                        super::super::serialize_field_header(writer, 40, payload_size, false)?;
                        for payload in payload {
                            writer.write_all(&payload.to_le_bytes())?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::JOptional(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                            super::super::varint_size_from_value(*payload));
                        super::super::serialize_field_header(writer, 41, payload_size, false)?;
                        for payload in payload {
                            super::super::serialize_varint(*payload, writer)?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::KOptional(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                            super::super::varint_size_from_value(super::super::zigzag_encode(*\
                            payload)));
                        super::super::serialize_field_header(writer, 42, payload_size, false)?;
                        for payload in payload {
                            super::super::serialize_varint(super::super::zigzag_encode(*payload), \
                                writer)?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::LOptional(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| x + 1_usize);
                        super::super::serialize_field_header(writer, 43, payload_size, false)?;
                        for payload in payload {
                            super::super::serialize_varint(*payload as u64, writer)?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::MOptional(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size });
                        super::super::serialize_field_header(writer, 44, payload_size, false)?;
                        for payload in payload {
                            super::super::serialize_varint(payload.len() as u64, writer)?;
                            writer.write_all(payload)?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::NOptional(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size });
                        super::super::serialize_field_header(writer, 45, payload_size, false)?;
                        for payload in payload {
                            super::super::serialize_varint(payload.len() as u64, writer)?;
                            writer.write_all(payload.as_bytes())?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::OOptional(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                            payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size }); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size });
                        super::super::serialize_field_header(writer, 46, payload_size, false)?;
                        for payload in payload {
                            super::super::serialize_varint(payload.iter().fold(0_usize, |x, \
                                payload| { let payload_size = payload.len(); x + \
                                super::super::varint_size_from_value(payload_size as u64) + \
                                payload_size }) as u64, writer)?;
                            for payload in payload {
                                super::super::serialize_varint(payload.len() as u64, writer)?;
                                writer.write_all(payload.as_bytes())?;
                            }
                        }
                        fallback.serialize(writer)
                    }
                }
            }
        }

        impl super::super::Deserialize for BarIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                loop {
                    let (index, size) = super::super::deserialize_field_header(&mut *reader)?;

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size as u64);

                    match index {
                        0 => {
                            let payload = ();
                            return Ok(BarIn::ARequired);
                        }
                        1 => {
                            let payload = if size == 0_usize {
                                0.0_f64
                            } else {
                                let mut buffer = [0; 8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                                f64::from_le_bytes(buffer)
                            };
                            return Ok(BarIn::BRequired(payload));
                        }
                        2 => {
                            let payload = match size {
                                0_usize => 0_u64,
                                8_usize => {
                                    let mut buffer = [0; 8];
                                    ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                    u64::from_le_bytes(buffer)
                                }
                                _ => super::super::deserialize_varint(&mut sub_reader)?,
                            };
                            return Ok(BarIn::CRequired(payload));
                        }
                        3 => {
                            let payload = match size {
                                0_usize => 0_i64,
                                8_usize => {
                                    let mut buffer = [0; 8];
                                    ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                    i64::from_le_bytes(buffer)
                                }
                                _ => super::super::zigzag_decode(super::super::deserialize_varint\
                                    (&mut sub_reader)?),
                            };                            return Ok(BarIn::DRequired(payload));
                        }
                        4 => {
                            let payload = if size == 0_usize {
                                false
                            } else {
                                let mut buffer = [0_u8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                buffer[0] != 0b0000_0001
                            };
                            return Ok(BarIn::ERequired(payload));
                        }
                        5 => {
                            let mut payload = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;
                            return Ok(BarIn::FRequired(payload));
                        }
                        6 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(BarIn::GRequired(payload));
                        }
                        7 => {
                            let payload = match size {
                                0_usize => 0_u64,
                                8_usize => {
                                    let mut buffer = [0; 8];
                                    ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                    u64::from_le_bytes(buffer)
                                }
                                _ => super::super::deserialize_varint(&mut sub_reader)?,
                            };
                            let payload = vec![(); payload as usize];
                            return Ok(BarIn::HRequired(payload));
                        }
                        8 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<f64> {
                                let mut buffer = [0; 8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                                let payload = f64::from_le_bytes(buffer);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }
                            return Ok(BarIn::IRequired(payload));
                        }
                        9 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<u64> {
                                let payload = super::super::deserialize_varint(&mut sub_reader)?;

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }
                            return Ok(BarIn::JRequired(payload));
                        }
                        10 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<i64> {
                                let payload = \
                                    super::super::zigzag_decode(super::super::deserialize_varint(&\
                                    mut sub_reader)?);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }
                            return Ok(BarIn::KRequired(payload));
                        }
                        11 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<bool> {
                                let mut buffer = [0_u8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                let payload = buffer[0] != 0b0000_0001;

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }
                            return Ok(BarIn::LRequired(payload));
                        }
                        12 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size as u64);
                                payload.push({
                                    let mut payload = vec![];
                                    ::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;
                                    payload
                                });
                            }
                            return Ok(BarIn::MRequired(payload));
                        }
                        13 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size as u64);
                                payload.push({
                                    let mut buffer = vec![];
                                    ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                                    let payload = std::str::from_utf8(&buffer).map_or_else(
                                        |err| Err(::std::io::Error::new(::std::io::ErrorKind::\
                                            Other, err)),
                                        |result| Ok(result.to_owned()),
                                    )?;
                                    payload
                                });
                            }
                            return Ok(BarIn::NRequired(payload));
                        }
                        14 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size as u64);
                                payload.push({
                                    let mut payload = Vec::new();

                                    loop {
                                        let element_size = match \
                                            super::super::deserialize_varint(&mut sub_reader) {
                                            Ok(element_size) => element_size,
                                            Err(err) => {
                                                if let std::io::ErrorKind::UnexpectedEof = \
                                                    err.kind() {
                                                    break;
                                                }

                                                return Err(err);
                                            }
                                        };

                                        let mut sub_reader = ::std::io::Read::take(&mut \
                                            sub_reader, element_size as u64);
                                        payload.push({
                                            let mut buffer = vec![];
                                            ::std::io::Read::read_to_end(&mut sub_reader, &mut \
                                                buffer)?;
                                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                                |err| Err(::std::io::Error::new(::std::io::\
                                                    ErrorKind::Other, err)),
                                                |result| Ok(result.to_owned()),
                                            )?;
                                            payload
                                        });
                                    }
                                    payload
                                });
                            }
                            return Ok(BarIn::ORequired(payload));
                        }
                        16 => {
                            let payload = ();
                            return Ok(BarIn::AAsymmetric);
                        }
                        17 => {
                            let payload = if size == 0_usize {
                                0.0_f64
                            } else {
                                let mut buffer = [0; 8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                                f64::from_le_bytes(buffer)
                            };
                            return Ok(BarIn::BAsymmetric(payload));
                        }
                        18 => {
                            let payload = match size {
                                0_usize => 0_u64,
                                8_usize => {
                                    let mut buffer = [0; 8];
                                    ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                    u64::from_le_bytes(buffer)
                                }
                                _ => super::super::deserialize_varint(&mut sub_reader)?,
                            };
                            return Ok(BarIn::CAsymmetric(payload));
                        }
                        19 => {
                            let payload = match size {
                                0_usize => 0_i64,
                                8_usize => {
                                    let mut buffer = [0; 8];
                                    ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                    i64::from_le_bytes(buffer)
                                }
                                _ => super::super::zigzag_decode(super::super::deserialize_varint\
                                    (&mut sub_reader)?),
                            };                            return Ok(BarIn::DAsymmetric(payload));
                        }
                        20 => {
                            let payload = if size == 0_usize {
                                false
                            } else {
                                let mut buffer = [0_u8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                buffer[0] != 0b0000_0001
                            };
                            return Ok(BarIn::EAsymmetric(payload));
                        }
                        21 => {
                            let mut payload = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;
                            return Ok(BarIn::FAsymmetric(payload));
                        }
                        22 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(BarIn::GAsymmetric(payload));
                        }
                        23 => {
                            let payload = match size {
                                0_usize => 0_u64,
                                8_usize => {
                                    let mut buffer = [0; 8];
                                    ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                    u64::from_le_bytes(buffer)
                                }
                                _ => super::super::deserialize_varint(&mut sub_reader)?,
                            };
                            let payload = vec![(); payload as usize];
                            return Ok(BarIn::HAsymmetric(payload));
                        }
                        24 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<f64> {
                                let mut buffer = [0; 8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                                let payload = f64::from_le_bytes(buffer);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }
                            return Ok(BarIn::IAsymmetric(payload));
                        }
                        25 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<u64> {
                                let payload = super::super::deserialize_varint(&mut sub_reader)?;

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }
                            return Ok(BarIn::JAsymmetric(payload));
                        }
                        26 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<i64> {
                                let payload = \
                                    super::super::zigzag_decode(super::super::deserialize_varint(&\
                                    mut sub_reader)?);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }
                            return Ok(BarIn::KAsymmetric(payload));
                        }
                        27 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<bool> {
                                let mut buffer = [0_u8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                let payload = buffer[0] != 0b0000_0001;

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }
                            return Ok(BarIn::LAsymmetric(payload));
                        }
                        28 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size as u64);
                                payload.push({
                                    let mut payload = vec![];
                                    ::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;
                                    payload
                                });
                            }
                            return Ok(BarIn::MAsymmetric(payload));
                        }
                        29 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size as u64);
                                payload.push({
                                    let mut buffer = vec![];
                                    ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                                    let payload = std::str::from_utf8(&buffer).map_or_else(
                                        |err| Err(::std::io::Error::new(::std::io::ErrorKind::\
                                            Other, err)),
                                        |result| Ok(result.to_owned()),
                                    )?;
                                    payload
                                });
                            }
                            return Ok(BarIn::NAsymmetric(payload));
                        }
                        30 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size as u64);
                                payload.push({
                                    let mut payload = Vec::new();

                                    loop {
                                        let element_size = match \
                                            super::super::deserialize_varint(&mut sub_reader) {
                                            Ok(element_size) => element_size,
                                            Err(err) => {
                                                if let std::io::ErrorKind::UnexpectedEof = \
                                                    err.kind() {
                                                    break;
                                                }

                                                return Err(err);
                                            }
                                        };

                                        let mut sub_reader = ::std::io::Read::take(&mut \
                                            sub_reader, element_size as u64);
                                        payload.push({
                                            let mut buffer = vec![];
                                            ::std::io::Read::read_to_end(&mut sub_reader, &mut \
                                                buffer)?;
                                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                                |err| Err(::std::io::Error::new(::std::io::\
                                                    ErrorKind::Other, err)),
                                                |result| Ok(result.to_owned()),
                                            )?;
                                            payload
                                        });
                                    }
                                    payload
                                });
                            }
                            return Ok(BarIn::OAsymmetric(payload));
                        }
                        32 => {
                            let payload = ();
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::AOptional(fallback));
                        }
                        33 => {
                            let payload = if size == 0_usize {
                                0.0_f64
                            } else {
                                let mut buffer = [0; 8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                                f64::from_le_bytes(buffer)
                            };
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::BOptional(payload, fallback));
                        }
                        34 => {
                            let payload = match size {
                                0_usize => 0_u64,
                                8_usize => {
                                    let mut buffer = [0; 8];
                                    ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                    u64::from_le_bytes(buffer)
                                }
                                _ => super::super::deserialize_varint(&mut sub_reader)?,
                            };
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::COptional(payload, fallback));
                        }
                        35 => {
                            let payload = match size {
                                0_usize => 0_i64,
                                8_usize => {
                                    let mut buffer = [0; 8];
                                    ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                    i64::from_le_bytes(buffer)
                                }
                                _ => super::super::zigzag_decode(super::super::deserialize_varint\
                                    (&mut sub_reader)?),
                            };                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::DOptional(payload, fallback));
                        }
                        36 => {
                            let payload = if size == 0_usize {
                                false
                            } else {
                                let mut buffer = [0_u8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                buffer[0] != 0b0000_0001
                            };
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::EOptional(payload, fallback));
                        }
                        37 => {
                            let mut payload = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::FOptional(payload, fallback));
                        }
                        38 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::GOptional(payload, fallback));
                        }
                        39 => {
                            let payload = match size {
                                0_usize => 0_u64,
                                8_usize => {
                                    let mut buffer = [0; 8];
                                    ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                    u64::from_le_bytes(buffer)
                                }
                                _ => super::super::deserialize_varint(&mut sub_reader)?,
                            };
                            let payload = vec![(); payload as usize];
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::HOptional(payload, fallback));
                        }
                        40 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<f64> {
                                let mut buffer = [0; 8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                                let payload = f64::from_le_bytes(buffer);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::IOptional(payload, fallback));
                        }
                        41 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<u64> {
                                let payload = super::super::deserialize_varint(&mut sub_reader)?;

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::JOptional(payload, fallback));
                        }
                        42 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<i64> {
                                let payload = \
                                    super::super::zigzag_decode(super::super::deserialize_varint(&\
                                    mut sub_reader)?);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::KOptional(payload, fallback));
                        }
                        43 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<bool> {
                                let mut buffer = [0_u8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                let payload = buffer[0] != 0b0000_0001;

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::LOptional(payload, fallback));
                        }
                        44 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size as u64);
                                payload.push({
                                    let mut payload = vec![];
                                    ::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;
                                    payload
                                });
                            }
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::MOptional(payload, fallback));
                        }
                        45 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size as u64);
                                payload.push({
                                    let mut buffer = vec![];
                                    ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                                    let payload = std::str::from_utf8(&buffer).map_or_else(
                                        |err| Err(::std::io::Error::new(::std::io::ErrorKind::\
                                            Other, err)),
                                        |result| Ok(result.to_owned()),
                                    )?;
                                    payload
                                });
                            }
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::NOptional(payload, fallback));
                        }
                        46 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size as u64);
                                payload.push({
                                    let mut payload = Vec::new();

                                    loop {
                                        let element_size = match \
                                            super::super::deserialize_varint(&mut sub_reader) {
                                            Ok(element_size) => element_size,
                                            Err(err) => {
                                                if let std::io::ErrorKind::UnexpectedEof = \
                                                    err.kind() {
                                                    break;
                                                }

                                                return Err(err);
                                            }
                                        };

                                        let mut sub_reader = ::std::io::Read::take(&mut \
                                            sub_reader, element_size as u64);
                                        payload.push({
                                            let mut buffer = vec![];
                                            ::std::io::Read::read_to_end(&mut sub_reader, &mut \
                                                buffer)?;
                                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                                |err| Err(::std::io::Error::new(::std::io::\
                                                    ErrorKind::Other, err)),
                                                |result| Ok(result.to_owned()),
                                            )?;
                                            payload
                                        });
                                    }
                                    payload
                                });
                            }
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::OOptional(payload, fallback));
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }
            }
        }

        impl From<BarOut> for BarIn {
            fn from(message: BarOut) -> Self {
                match message {
                    BarOut::ARequired => BarIn::ARequired,
                    BarOut::BRequired(payload) => BarIn::BRequired(payload.into()),
                    BarOut::CRequired(payload) => BarIn::CRequired(payload.into()),
                    BarOut::DRequired(payload) => BarIn::DRequired(payload.into()),
                    BarOut::ERequired(payload) => BarIn::ERequired(payload.into()),
                    BarOut::FRequired(payload) => BarIn::FRequired(payload.into()),
                    BarOut::GRequired(payload) => BarIn::GRequired(payload.into()),
                    BarOut::HRequired(payload) => BarIn::HRequired(payload.into()),
                    BarOut::IRequired(payload) => BarIn::IRequired(payload.into()),
                    BarOut::JRequired(payload) => BarIn::JRequired(payload.into()),
                    BarOut::KRequired(payload) => BarIn::KRequired(payload.into()),
                    BarOut::LRequired(payload) => BarIn::LRequired(payload.into()),
                    BarOut::MRequired(payload) => BarIn::MRequired(payload.into()),
                    BarOut::NRequired(payload) => BarIn::NRequired(payload.into()),
                    BarOut::ORequired(payload) => BarIn::ORequired(payload.into()),
                    BarOut::AAsymmetric(fallback) => BarIn::AAsymmetric,
                    BarOut::BAsymmetric(payload, fallback) => BarIn::BAsymmetric(payload.into()),
                    BarOut::CAsymmetric(payload, fallback) => BarIn::CAsymmetric(payload.into()),
                    BarOut::DAsymmetric(payload, fallback) => BarIn::DAsymmetric(payload.into()),
                    BarOut::EAsymmetric(payload, fallback) => BarIn::EAsymmetric(payload.into()),
                    BarOut::FAsymmetric(payload, fallback) => BarIn::FAsymmetric(payload.into()),
                    BarOut::GAsymmetric(payload, fallback) => BarIn::GAsymmetric(payload.into()),
                    BarOut::HAsymmetric(payload, fallback) => BarIn::HAsymmetric(payload.into()),
                    BarOut::IAsymmetric(payload, fallback) => BarIn::IAsymmetric(payload.into()),
                    BarOut::JAsymmetric(payload, fallback) => BarIn::JAsymmetric(payload.into()),
                    BarOut::KAsymmetric(payload, fallback) => BarIn::KAsymmetric(payload.into()),
                    BarOut::LAsymmetric(payload, fallback) => BarIn::LAsymmetric(payload.into()),
                    BarOut::MAsymmetric(payload, fallback) => BarIn::MAsymmetric(payload.into()),
                    BarOut::NAsymmetric(payload, fallback) => BarIn::NAsymmetric(payload.into()),
                    BarOut::OAsymmetric(payload, fallback) => BarIn::OAsymmetric(payload.into()),
                    BarOut::AOptional(fallback) => BarIn::AOptional(Box::new((*fallback).into())),
                    BarOut::BOptional(payload, fallback) => BarIn::BOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::COptional(payload, fallback) => BarIn::COptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::DOptional(payload, fallback) => BarIn::DOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::EOptional(payload, fallback) => BarIn::EOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::FOptional(payload, fallback) => BarIn::FOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::GOptional(payload, fallback) => BarIn::GOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::HOptional(payload, fallback) => BarIn::HOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::IOptional(payload, fallback) => BarIn::IOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::JOptional(payload, fallback) => BarIn::JOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::KOptional(payload, fallback) => BarIn::KOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::LOptional(payload, fallback) => BarIn::LOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::MOptional(payload, fallback) => BarIn::MOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::NOptional(payload, fallback) => BarIn::NOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::OOptional(payload, fallback) => BarIn::OOptional(payload.into(), \
                        Box::new((*fallback).into())),
                }
            }
        }
    }

    pub mod foo {
        #[derive(Clone, Debug)]
        pub struct FooOut {
            pub a_required: (),
            pub b_required: f64,
            pub c_required: u64,
            pub d_required: i64,
            pub e_required: bool,
            pub f_required: Vec<u8>,
            pub g_required: String,
            pub h_required: Vec<()>,
            pub i_required: Vec<f64>,
            pub j_required: Vec<u64>,
            pub k_required: Vec<i64>,
            pub l_required: Vec<bool>,
            pub m_required: Vec<Vec<u8>>,
            pub n_required: Vec<String>,
            pub o_required: Vec<Vec<String>>,
            pub a_asymmetric: (),
            pub b_asymmetric: f64,
            pub c_asymmetric: u64,
            pub d_asymmetric: i64,
            pub e_asymmetric: bool,
            pub f_asymmetric: Vec<u8>,
            pub g_asymmetric: String,
            pub h_asymmetric: Vec<()>,
            pub i_asymmetric: Vec<f64>,
            pub j_asymmetric: Vec<u64>,
            pub k_asymmetric: Vec<i64>,
            pub l_asymmetric: Vec<bool>,
            pub m_asymmetric: Vec<Vec<u8>>,
            pub n_asymmetric: Vec<String>,
            pub o_asymmetric: Vec<Vec<String>>,
            pub a_optional: Option<()>,
            pub b_optional: Option<f64>,
            pub c_optional: Option<u64>,
            pub d_optional: Option<i64>,
            pub e_optional: Option<bool>,
            pub f_optional: Option<Vec<u8>>,
            pub g_optional: Option<String>,
            pub h_optional: Option<Vec<()>>,
            pub i_optional: Option<Vec<f64>>,
            pub j_optional: Option<Vec<u64>>,
            pub k_optional: Option<Vec<i64>>,
            pub l_optional: Option<Vec<bool>>,
            pub m_optional: Option<Vec<Vec<u8>>>,
            pub n_optional: Option<Vec<String>>,
            pub o_optional: Option<Vec<Vec<String>>>,
        }

        #[derive(Clone, Debug)]
        pub struct FooIn {
            pub a_required: (),
            pub b_required: f64,
            pub c_required: u64,
            pub d_required: i64,
            pub e_required: bool,
            pub f_required: Vec<u8>,
            pub g_required: String,
            pub h_required: Vec<()>,
            pub i_required: Vec<f64>,
            pub j_required: Vec<u64>,
            pub k_required: Vec<i64>,
            pub l_required: Vec<bool>,
            pub m_required: Vec<Vec<u8>>,
            pub n_required: Vec<String>,
            pub o_required: Vec<Vec<String>>,
            pub a_asymmetric: Option<()>,
            pub b_asymmetric: Option<f64>,
            pub c_asymmetric: Option<u64>,
            pub d_asymmetric: Option<i64>,
            pub e_asymmetric: Option<bool>,
            pub f_asymmetric: Option<Vec<u8>>,
            pub g_asymmetric: Option<String>,
            pub h_asymmetric: Option<Vec<()>>,
            pub i_asymmetric: Option<Vec<f64>>,
            pub j_asymmetric: Option<Vec<u64>>,
            pub k_asymmetric: Option<Vec<i64>>,
            pub l_asymmetric: Option<Vec<bool>>,
            pub m_asymmetric: Option<Vec<Vec<u8>>>,
            pub n_asymmetric: Option<Vec<String>>,
            pub o_asymmetric: Option<Vec<Vec<String>>>,
            pub a_optional: Option<()>,
            pub b_optional: Option<f64>,
            pub c_optional: Option<u64>,
            pub d_optional: Option<i64>,
            pub e_optional: Option<bool>,
            pub f_optional: Option<Vec<u8>>,
            pub g_optional: Option<String>,
            pub h_optional: Option<Vec<()>>,
            pub i_optional: Option<Vec<f64>>,
            pub j_optional: Option<Vec<u64>>,
            pub k_optional: Option<Vec<i64>>,
            pub l_optional: Option<Vec<bool>>,
            pub m_optional: Option<Vec<Vec<u8>>>,
            pub n_optional: Option<Vec<String>>,
            pub o_optional: Option<Vec<Vec<String>>>,
        }

        impl super::super::Serialize for FooOut {
            fn size(&self) -> usize {
                ({
                    let payload = &self.a_required;
                    let payload_size = 0_usize;
                    super::super::field_header_size(0, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.b_required;
                    let payload_size = if payload.to_bits() == 0_u64 { 0_usize } else { 8_usize };
                    super::super::field_header_size(1, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.c_required;
                    let payload_size = match *payload { 0_u64 => { 0_usize }, \
                        1_u64..=567_382_630_219_903_u64 => { \
                        super::super::varint_size_from_value(*payload) }, \
                        567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { 8_usize } };
                    super::super::field_header_size(2, payload_size, true) + payload_size
                }) + ({
                    let payload = &self.d_required;
                    let payload_size = { let zigzag = super::super::zigzag_encode(*payload); \
                        match zigzag { 0_u64 => { 0_usize }, 1_u64..=567_382_630_219_903_u64 => { \
                        super::super::varint_size_from_value(zigzag) }, \
                        567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                        8_usize } } };
                    super::super::field_header_size(3, payload_size, true) + payload_size
                }) + ({
                    let payload = &self.e_required;
                    let payload_size = if *payload { 1_usize } else { 0_usize };
                    super::super::field_header_size(4, payload_size, true) + payload_size
                }) + ({
                    let payload = &self.f_required;
                    let payload_size = payload.len();
                    super::super::field_header_size(5, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.g_required;
                    let payload_size = payload.len();
                    super::super::field_header_size(6, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.h_required;
                    let payload_size = { let payload = &(payload.len() as u64); match *payload { \
                        0_u64 => { 0_usize }, 1_u64..=567_382_630_219_903_u64 => { \
                        super::super::varint_size_from_value(*payload) }, \
                        567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                        8_usize } } };
                    super::super::field_header_size(7, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.i_required;
                    let payload_size = 8_usize * payload.len();
                    super::super::field_header_size(8, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.j_required;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                        super::super::varint_size_from_value(*payload));
                    super::super::field_header_size(9, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.k_required;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                        super::super::varint_size_from_value(super::super::zigzag_encode(*payload\
                        )));
                    super::super::field_header_size(10, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.l_required;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| x + 1_usize);
                    super::super::field_header_size(11, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.m_required;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.len(); x + \
                        super::super::varint_size_from_value(payload_size as u64) + \
                        payload_size });
                    super::super::field_header_size(12, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.n_required;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.len(); x + \
                        super::super::varint_size_from_value(payload_size as u64) + \
                        payload_size });
                    super::super::field_header_size(13, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.o_required;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.len(); x + \
                        super::super::varint_size_from_value(payload_size as u64) + \
                        payload_size }); x + super::super::varint_size_from_value(payload_size as \
                        u64) + payload_size });
                    super::super::field_header_size(14, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.a_asymmetric;
                    let payload_size = 0_usize;
                    super::super::field_header_size(16, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.b_asymmetric;
                    let payload_size = if payload.to_bits() == 0_u64 { 0_usize } else { 8_usize };
                    super::super::field_header_size(17, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.c_asymmetric;
                    let payload_size = match *payload { 0_u64 => { 0_usize }, \
                        1_u64..=567_382_630_219_903_u64 => { \
                        super::super::varint_size_from_value(*payload) }, \
                        567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { 8_usize } };
                    super::super::field_header_size(18, payload_size, true) + payload_size
                }) + ({
                    let payload = &self.d_asymmetric;
                    let payload_size = { let zigzag = super::super::zigzag_encode(*payload); \
                        match zigzag { 0_u64 => { 0_usize }, 1_u64..=567_382_630_219_903_u64 => { \
                        super::super::varint_size_from_value(zigzag) }, \
                        567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                        8_usize } } };
                    super::super::field_header_size(19, payload_size, true) + payload_size
                }) + ({
                    let payload = &self.e_asymmetric;
                    let payload_size = if *payload { 1_usize } else { 0_usize };
                    super::super::field_header_size(20, payload_size, true) + payload_size
                }) + ({
                    let payload = &self.f_asymmetric;
                    let payload_size = payload.len();
                    super::super::field_header_size(21, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.g_asymmetric;
                    let payload_size = payload.len();
                    super::super::field_header_size(22, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.h_asymmetric;
                    let payload_size = { let payload = &(payload.len() as u64); match *payload { \
                        0_u64 => { 0_usize }, 1_u64..=567_382_630_219_903_u64 => { \
                        super::super::varint_size_from_value(*payload) }, \
                        567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                        8_usize } } };
                    super::super::field_header_size(23, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.i_asymmetric;
                    let payload_size = 8_usize * payload.len();
                    super::super::field_header_size(24, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.j_asymmetric;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                        super::super::varint_size_from_value(*payload));
                    super::super::field_header_size(25, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.k_asymmetric;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                        super::super::varint_size_from_value(super::super::zigzag_encode(*payload\
                        )));
                    super::super::field_header_size(26, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.l_asymmetric;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| x + 1_usize);
                    super::super::field_header_size(27, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.m_asymmetric;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.len(); x + \
                        super::super::varint_size_from_value(payload_size as u64) + \
                        payload_size });
                    super::super::field_header_size(28, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.n_asymmetric;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.len(); x + \
                        super::super::varint_size_from_value(payload_size as u64) + \
                        payload_size });
                    super::super::field_header_size(29, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.o_asymmetric;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.len(); x + \
                        super::super::varint_size_from_value(payload_size as u64) + \
                        payload_size }); x + super::super::varint_size_from_value(payload_size as \
                        u64) + payload_size });
                    super::super::field_header_size(30, payload_size, false) + payload_size
                }) + self.a_optional.as_ref().map_or(0, |payload| {
                    let payload_size = 0_usize;
                    super::super::field_header_size(32, payload_size, false) + payload_size
                }) + self.b_optional.as_ref().map_or(0, |payload| {
                    let payload_size = if payload.to_bits() == 0_u64 { 0_usize } else { 8_usize };
                    super::super::field_header_size(33, payload_size, false) + payload_size
                }) + self.c_optional.as_ref().map_or(0, |payload| {
                    let payload_size = match *payload { 0_u64 => { 0_usize }, \
                        1_u64..=567_382_630_219_903_u64 => { \
                        super::super::varint_size_from_value(*payload) }, \
                        567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { 8_usize } };
                    super::super::field_header_size(34, payload_size, true) + payload_size
                }) + self.d_optional.as_ref().map_or(0, |payload| {
                    let payload_size = { let zigzag = super::super::zigzag_encode(*payload); \
                        match zigzag { 0_u64 => { 0_usize }, 1_u64..=567_382_630_219_903_u64 => { \
                        super::super::varint_size_from_value(zigzag) }, \
                        567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                        8_usize } } };
                    super::super::field_header_size(35, payload_size, true) + payload_size
                }) + self.e_optional.as_ref().map_or(0, |payload| {
                    let payload_size = if *payload { 1_usize } else { 0_usize };
                    super::super::field_header_size(36, payload_size, true) + payload_size
                }) + self.f_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len();
                    super::super::field_header_size(37, payload_size, false) + payload_size
                }) + self.g_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len();
                    super::super::field_header_size(38, payload_size, false) + payload_size
                }) + self.h_optional.as_ref().map_or(0, |payload| {
                    let payload_size = { let payload = &(payload.len() as u64); match *payload { \
                        0_u64 => { 0_usize }, 1_u64..=567_382_630_219_903_u64 => { \
                        super::super::varint_size_from_value(*payload) }, \
                        567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                        8_usize } } };
                    super::super::field_header_size(39, payload_size, false) + payload_size
                }) + self.i_optional.as_ref().map_or(0, |payload| {
                    let payload_size = 8_usize * payload.len();
                    super::super::field_header_size(40, payload_size, false) + payload_size
                }) + self.j_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                        super::super::varint_size_from_value(*payload));
                    super::super::field_header_size(41, payload_size, false) + payload_size
                }) + self.k_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                        super::super::varint_size_from_value(super::super::zigzag_encode(*payload\
                        )));
                    super::super::field_header_size(42, payload_size, false) + payload_size
                }) + self.l_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.iter().fold(0_usize, |x, payload| x + 1_usize);
                    super::super::field_header_size(43, payload_size, false) + payload_size
                }) + self.m_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.len(); x + \
                        super::super::varint_size_from_value(payload_size as u64) + \
                        payload_size });
                    super::super::field_header_size(44, payload_size, false) + payload_size
                }) + self.n_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.len(); x + \
                        super::super::varint_size_from_value(payload_size as u64) + \
                        payload_size });
                    super::super::field_header_size(45, payload_size, false) + payload_size
                }) + self.o_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.len(); x + \
                        super::super::varint_size_from_value(payload_size as u64) + \
                        payload_size }); x + super::super::varint_size_from_value(payload_size as \
                        u64) + payload_size });
                    super::super::field_header_size(46, payload_size, false) + payload_size
                })
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                {
                    let payload = &self.a_required;
                    let payload_size = 0_usize;
                    super::super::serialize_field_header(writer, 0, payload_size, false)?;
                    ();
                }

                {
                    let payload = &self.b_required;
                    let payload_size = if payload.to_bits() == 0_u64 { 0_usize } else { 8_usize };
                    super::super::serialize_field_header(writer, 1, payload_size, false)?;
                    if payload_size != 0_usize {
                        writer.write_all(&payload.to_le_bytes())?;
                    }
                }

                {
                    let payload = &self.c_required;
                    let payload_size = match *payload { 0_u64 => { 0_usize }, \
                        1_u64..=567_382_630_219_903_u64 => { \
                        super::super::varint_size_from_value(*payload) }, \
                        567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { 8_usize } };
                    super::super::serialize_field_header(writer, 2, payload_size, true)?;
                    match payload_size {
                        0_usize => {}
                        8_usize => writer.write_all(&payload.to_le_bytes())?,
                        _ => super::super::serialize_varint(*payload, writer)?,
                    }
                }

                {
                    let payload = &self.d_required;
                    let payload_size = { let zigzag = super::super::zigzag_encode(*payload); \
                        match zigzag { 0_u64 => { 0_usize }, 1_u64..=567_382_630_219_903_u64 => { \
                        super::super::varint_size_from_value(zigzag) }, \
                        567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                        8_usize } } };
                    super::super::serialize_field_header(writer, 3, payload_size, true)?;
                    match payload_size {
                        0_usize => {}
                        8_usize => writer.write_all(&payload.to_le_bytes())?,
                        _ => super::super::serialize_varint(
                            super::super::zigzag_encode(*payload),
                            writer,
                        )?,
                    }
                }

                {
                    let payload = &self.e_required;
                    let payload_size = if *payload { 1_usize } else { 0_usize };
                    super::super::serialize_field_header(writer, 4, payload_size, true)?;
                    if payload_size != 0_usize {
                        super::super::serialize_varint(*payload as u64, writer)?;
                    }
                }

                {
                    let payload = &self.f_required;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 5, payload_size, false)?;
                    writer.write_all(payload)?;
                }

                {
                    let payload = &self.g_required;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 6, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.h_required;
                    let payload_size = { let payload = &(payload.len() as u64); match *payload { \
                        0_u64 => { 0_usize }, 1_u64..=567_382_630_219_903_u64 => { \
                        super::super::varint_size_from_value(*payload) }, \
                        567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                        8_usize } } };
                    super::super::serialize_field_header(writer, 7, payload_size, false)?;
                    {
                        let payload = &(payload.len() as u64);
                        match payload_size {
                            0_usize => {}
                            8_usize => writer.write_all(&payload.to_le_bytes())?,
                            _ => super::super::serialize_varint(*payload, writer)?,
                        }
                    }
                }

                {
                    let payload = &self.i_required;
                    let payload_size = 8_usize * payload.len();
                    super::super::serialize_field_header(writer, 8, payload_size, false)?;
                    for payload in payload {
                        writer.write_all(&payload.to_le_bytes())?;
                    }
                }

                {
                    let payload = &self.j_required;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                        super::super::varint_size_from_value(*payload));
                    super::super::serialize_field_header(writer, 9, payload_size, false)?;
                    for payload in payload {
                        super::super::serialize_varint(*payload, writer)?;
                    }
                }

                {
                    let payload = &self.k_required;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                        super::super::varint_size_from_value(super::super::zigzag_encode(*payload\
                        )));
                    super::super::serialize_field_header(writer, 10, payload_size, false)?;
                    for payload in payload {
                        super::super::serialize_varint(super::super::zigzag_encode(*payload), \
                            writer)?;
                    }
                }

                {
                    let payload = &self.l_required;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| x + 1_usize);
                    super::super::serialize_field_header(writer, 11, payload_size, false)?;
                    for payload in payload {
                        super::super::serialize_varint(*payload as u64, writer)?;
                    }
                }

                {
                    let payload = &self.m_required;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.len(); x + \
                        super::super::varint_size_from_value(payload_size as u64) + \
                        payload_size });
                    super::super::serialize_field_header(writer, 12, payload_size, false)?;
                    for payload in payload {
                        super::super::serialize_varint(payload.len() as u64, writer)?;
                        writer.write_all(payload)?;
                    }
                }

                {
                    let payload = &self.n_required;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.len(); x + \
                        super::super::varint_size_from_value(payload_size as u64) + \
                        payload_size });
                    super::super::serialize_field_header(writer, 13, payload_size, false)?;
                    for payload in payload {
                        super::super::serialize_varint(payload.len() as u64, writer)?;
                        writer.write_all(payload.as_bytes())?;
                    }
                }

                {
                    let payload = &self.o_required;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.len(); x + \
                        super::super::varint_size_from_value(payload_size as u64) + \
                        payload_size }); x + super::super::varint_size_from_value(payload_size as \
                        u64) + payload_size });
                    super::super::serialize_field_header(writer, 14, payload_size, false)?;
                    for payload in payload {
                        super::super::serialize_varint(payload.iter().fold(0_usize, |x, \
                            payload| { let payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size }) as u64, writer)?;
                        for payload in payload {
                            super::super::serialize_varint(payload.len() as u64, writer)?;
                            writer.write_all(payload.as_bytes())?;
                        }
                    }
                }

                {
                    let payload = &self.a_asymmetric;
                    let payload_size = 0_usize;
                    super::super::serialize_field_header(writer, 16, payload_size, false)?;
                    ();
                }

                {
                    let payload = &self.b_asymmetric;
                    let payload_size = if payload.to_bits() == 0_u64 { 0_usize } else { 8_usize };
                    super::super::serialize_field_header(writer, 17, payload_size, false)?;
                    if payload_size != 0_usize {
                        writer.write_all(&payload.to_le_bytes())?;
                    }
                }

                {
                    let payload = &self.c_asymmetric;
                    let payload_size = match *payload { 0_u64 => { 0_usize }, \
                        1_u64..=567_382_630_219_903_u64 => { \
                        super::super::varint_size_from_value(*payload) }, \
                        567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { 8_usize } };
                    super::super::serialize_field_header(writer, 18, payload_size, true)?;
                    match payload_size {
                        0_usize => {}
                        8_usize => writer.write_all(&payload.to_le_bytes())?,
                        _ => super::super::serialize_varint(*payload, writer)?,
                    }
                }

                {
                    let payload = &self.d_asymmetric;
                    let payload_size = { let zigzag = super::super::zigzag_encode(*payload); \
                        match zigzag { 0_u64 => { 0_usize }, 1_u64..=567_382_630_219_903_u64 => { \
                        super::super::varint_size_from_value(zigzag) }, \
                        567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                        8_usize } } };
                    super::super::serialize_field_header(writer, 19, payload_size, true)?;
                    match payload_size {
                        0_usize => {}
                        8_usize => writer.write_all(&payload.to_le_bytes())?,
                        _ => super::super::serialize_varint(
                            super::super::zigzag_encode(*payload),
                            writer,
                        )?,
                    }
                }

                {
                    let payload = &self.e_asymmetric;
                    let payload_size = if *payload { 1_usize } else { 0_usize };
                    super::super::serialize_field_header(writer, 20, payload_size, true)?;
                    if payload_size != 0_usize {
                        super::super::serialize_varint(*payload as u64, writer)?;
                    }
                }

                {
                    let payload = &self.f_asymmetric;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 21, payload_size, false)?;
                    writer.write_all(payload)?;
                }

                {
                    let payload = &self.g_asymmetric;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 22, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.h_asymmetric;
                    let payload_size = { let payload = &(payload.len() as u64); match *payload { \
                        0_u64 => { 0_usize }, 1_u64..=567_382_630_219_903_u64 => { \
                        super::super::varint_size_from_value(*payload) }, \
                        567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                        8_usize } } };
                    super::super::serialize_field_header(writer, 23, payload_size, false)?;
                    {
                        let payload = &(payload.len() as u64);
                        match payload_size {
                            0_usize => {}
                            8_usize => writer.write_all(&payload.to_le_bytes())?,
                            _ => super::super::serialize_varint(*payload, writer)?,
                        }
                    }
                }

                {
                    let payload = &self.i_asymmetric;
                    let payload_size = 8_usize * payload.len();
                    super::super::serialize_field_header(writer, 24, payload_size, false)?;
                    for payload in payload {
                        writer.write_all(&payload.to_le_bytes())?;
                    }
                }

                {
                    let payload = &self.j_asymmetric;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                        super::super::varint_size_from_value(*payload));
                    super::super::serialize_field_header(writer, 25, payload_size, false)?;
                    for payload in payload {
                        super::super::serialize_varint(*payload, writer)?;
                    }
                }

                {
                    let payload = &self.k_asymmetric;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                        super::super::varint_size_from_value(super::super::zigzag_encode(*payload\
                        )));
                    super::super::serialize_field_header(writer, 26, payload_size, false)?;
                    for payload in payload {
                        super::super::serialize_varint(super::super::zigzag_encode(*payload), \
                            writer)?;
                    }
                }

                {
                    let payload = &self.l_asymmetric;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| x + 1_usize);
                    super::super::serialize_field_header(writer, 27, payload_size, false)?;
                    for payload in payload {
                        super::super::serialize_varint(*payload as u64, writer)?;
                    }
                }

                {
                    let payload = &self.m_asymmetric;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.len(); x + \
                        super::super::varint_size_from_value(payload_size as u64) + \
                        payload_size });
                    super::super::serialize_field_header(writer, 28, payload_size, false)?;
                    for payload in payload {
                        super::super::serialize_varint(payload.len() as u64, writer)?;
                        writer.write_all(payload)?;
                    }
                }

                {
                    let payload = &self.n_asymmetric;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.len(); x + \
                        super::super::varint_size_from_value(payload_size as u64) + \
                        payload_size });
                    super::super::serialize_field_header(writer, 29, payload_size, false)?;
                    for payload in payload {
                        super::super::serialize_varint(payload.len() as u64, writer)?;
                        writer.write_all(payload.as_bytes())?;
                    }
                }

                {
                    let payload = &self.o_asymmetric;
                    let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.len(); x + \
                        super::super::varint_size_from_value(payload_size as u64) + \
                        payload_size }); x + super::super::varint_size_from_value(payload_size as \
                        u64) + payload_size });
                    super::super::serialize_field_header(writer, 30, payload_size, false)?;
                    for payload in payload {
                        super::super::serialize_varint(payload.iter().fold(0_usize, |x, \
                            payload| { let payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size }) as u64, writer)?;
                        for payload in payload {
                            super::super::serialize_varint(payload.len() as u64, writer)?;
                            writer.write_all(payload.as_bytes())?;
                        }
                    }
                }

                if let Some(payload) = &self.a_optional {
                    let payload_size = 0_usize;
                    super::super::serialize_field_header(writer, 32, payload_size, false)?;
                    ();
                }

                if let Some(payload) = &self.b_optional {
                    let payload_size = if payload.to_bits() == 0_u64 { 0_usize } else { 8_usize };
                    super::super::serialize_field_header(writer, 33, payload_size, false)?;
                    if payload_size != 0_usize {
                        writer.write_all(&payload.to_le_bytes())?;
                    }
                }

                if let Some(payload) = &self.c_optional {
                    let payload_size = match *payload { 0_u64 => { 0_usize }, \
                        1_u64..=567_382_630_219_903_u64 => { \
                        super::super::varint_size_from_value(*payload) }, \
                        567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { 8_usize } };
                    super::super::serialize_field_header(writer, 34, payload_size, true)?;
                    match payload_size {
                        0_usize => {}
                        8_usize => writer.write_all(&payload.to_le_bytes())?,
                        _ => super::super::serialize_varint(*payload, writer)?,
                    }
                }

                if let Some(payload) = &self.d_optional {
                    let payload_size = { let zigzag = super::super::zigzag_encode(*payload); \
                        match zigzag { 0_u64 => { 0_usize }, 1_u64..=567_382_630_219_903_u64 => { \
                        super::super::varint_size_from_value(zigzag) }, \
                        567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                        8_usize } } };
                    super::super::serialize_field_header(writer, 35, payload_size, true)?;
                    match payload_size {
                        0_usize => {}
                        8_usize => writer.write_all(&payload.to_le_bytes())?,
                        _ => super::super::serialize_varint(
                            super::super::zigzag_encode(*payload),
                            writer,
                        )?,
                    }
                }

                if let Some(payload) = &self.e_optional {
                    let payload_size = if *payload { 1_usize } else { 0_usize };
                    super::super::serialize_field_header(writer, 36, payload_size, true)?;
                    if payload_size != 0_usize {
                        super::super::serialize_varint(*payload as u64, writer)?;
                    }
                }

                if let Some(payload) = &self.f_optional {
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 37, payload_size, false)?;
                    writer.write_all(payload)?;
                }

                if let Some(payload) = &self.g_optional {
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 38, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.h_optional {
                    let payload_size = { let payload = &(payload.len() as u64); match *payload { \
                        0_u64 => { 0_usize }, 1_u64..=567_382_630_219_903_u64 => { \
                        super::super::varint_size_from_value(*payload) }, \
                        567_382_630_219_904_u64..=18_446_744_073_709_551_615_u64 => { \
                        8_usize } } };
                    super::super::serialize_field_header(writer, 39, payload_size, false)?;
                    {
                        let payload = &(payload.len() as u64);
                        match payload_size {
                            0_usize => {}
                            8_usize => writer.write_all(&payload.to_le_bytes())?,
                            _ => super::super::serialize_varint(*payload, writer)?,
                        }
                    }
                }

                if let Some(payload) = &self.i_optional {
                    let payload_size = 8_usize * payload.len();
                    super::super::serialize_field_header(writer, 40, payload_size, false)?;
                    for payload in payload {
                        writer.write_all(&payload.to_le_bytes())?;
                    }
                }

                if let Some(payload) = &self.j_optional {
                    let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                        super::super::varint_size_from_value(*payload));
                    super::super::serialize_field_header(writer, 41, payload_size, false)?;
                    for payload in payload {
                        super::super::serialize_varint(*payload, writer)?;
                    }
                }

                if let Some(payload) = &self.k_optional {
                    let payload_size = payload.iter().fold(0_usize, |x, payload| x + \
                        super::super::varint_size_from_value(super::super::zigzag_encode(*payload\
                        )));
                    super::super::serialize_field_header(writer, 42, payload_size, false)?;
                    for payload in payload {
                        super::super::serialize_varint(super::super::zigzag_encode(*payload), \
                            writer)?;
                    }
                }

                if let Some(payload) = &self.l_optional {
                    let payload_size = payload.iter().fold(0_usize, |x, payload| x + 1_usize);
                    super::super::serialize_field_header(writer, 43, payload_size, false)?;
                    for payload in payload {
                        super::super::serialize_varint(*payload as u64, writer)?;
                    }
                }

                if let Some(payload) = &self.m_optional {
                    let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.len(); x + \
                        super::super::varint_size_from_value(payload_size as u64) + \
                        payload_size });
                    super::super::serialize_field_header(writer, 44, payload_size, false)?;
                    for payload in payload {
                        super::super::serialize_varint(payload.len() as u64, writer)?;
                        writer.write_all(payload)?;
                    }
                }

                if let Some(payload) = &self.n_optional {
                    let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.len(); x + \
                        super::super::varint_size_from_value(payload_size as u64) + \
                        payload_size });
                    super::super::serialize_field_header(writer, 45, payload_size, false)?;
                    for payload in payload {
                        super::super::serialize_varint(payload.len() as u64, writer)?;
                        writer.write_all(payload.as_bytes())?;
                    }
                }

                if let Some(payload) = &self.o_optional {
                    let payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.iter().fold(0_usize, |x, payload| { let \
                        payload_size = payload.len(); x + \
                        super::super::varint_size_from_value(payload_size as u64) + \
                        payload_size }); x + super::super::varint_size_from_value(payload_size as \
                        u64) + payload_size });
                    super::super::serialize_field_header(writer, 46, payload_size, false)?;
                    for payload in payload {
                        super::super::serialize_varint(payload.iter().fold(0_usize, |x, \
                            payload| { let payload_size = payload.len(); x + \
                            super::super::varint_size_from_value(payload_size as u64) + \
                            payload_size }) as u64, writer)?;
                        for payload in payload {
                            super::super::serialize_varint(payload.len() as u64, writer)?;
                            writer.write_all(payload.as_bytes())?;
                        }
                    }
                }

                Ok(())
            }
        }

        impl super::super::Deserialize for FooIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                let mut a_required: Option<()> = None;
                let mut b_required: Option<f64> = None;
                let mut c_required: Option<u64> = None;
                let mut d_required: Option<i64> = None;
                let mut e_required: Option<bool> = None;
                let mut f_required: Option<Vec<u8>> = None;
                let mut g_required: Option<String> = None;
                let mut h_required: Option<Vec<()>> = None;
                let mut i_required: Option<Vec<f64>> = None;
                let mut j_required: Option<Vec<u64>> = None;
                let mut k_required: Option<Vec<i64>> = None;
                let mut l_required: Option<Vec<bool>> = None;
                let mut m_required: Option<Vec<Vec<u8>>> = None;
                let mut n_required: Option<Vec<String>> = None;
                let mut o_required: Option<Vec<Vec<String>>> = None;
                let mut a_asymmetric: Option<()> = None;
                let mut b_asymmetric: Option<f64> = None;
                let mut c_asymmetric: Option<u64> = None;
                let mut d_asymmetric: Option<i64> = None;
                let mut e_asymmetric: Option<bool> = None;
                let mut f_asymmetric: Option<Vec<u8>> = None;
                let mut g_asymmetric: Option<String> = None;
                let mut h_asymmetric: Option<Vec<()>> = None;
                let mut i_asymmetric: Option<Vec<f64>> = None;
                let mut j_asymmetric: Option<Vec<u64>> = None;
                let mut k_asymmetric: Option<Vec<i64>> = None;
                let mut l_asymmetric: Option<Vec<bool>> = None;
                let mut m_asymmetric: Option<Vec<Vec<u8>>> = None;
                let mut n_asymmetric: Option<Vec<String>> = None;
                let mut o_asymmetric: Option<Vec<Vec<String>>> = None;
                let mut a_optional: Option<()> = None;
                let mut b_optional: Option<f64> = None;
                let mut c_optional: Option<u64> = None;
                let mut d_optional: Option<i64> = None;
                let mut e_optional: Option<bool> = None;
                let mut f_optional: Option<Vec<u8>> = None;
                let mut g_optional: Option<String> = None;
                let mut h_optional: Option<Vec<()>> = None;
                let mut i_optional: Option<Vec<f64>> = None;
                let mut j_optional: Option<Vec<u64>> = None;
                let mut k_optional: Option<Vec<i64>> = None;
                let mut l_optional: Option<Vec<bool>> = None;
                let mut m_optional: Option<Vec<Vec<u8>>> = None;
                let mut n_optional: Option<Vec<String>> = None;
                let mut o_optional: Option<Vec<Vec<String>>> = None;

                loop {
                    let (index, size) = match super::super::deserialize_field_header(&mut \
                        *reader) {
                        Ok(header) => header,
                        Err(err) => {
                            if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                break;
                            }

                            return Err(err);
                        }
                    };

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size as u64);

                    match index {
                        0 => {
                            let payload = ();

                            a_required.get_or_insert(payload);
                        }
                        1 => {
                            let payload = if size == 0_usize {
                                0.0_f64
                            } else {
                                let mut buffer = [0; 8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                                f64::from_le_bytes(buffer)
                            };

                            b_required.get_or_insert(payload);
                        }
                        2 => {
                            let payload = match size {
                                0_usize => 0_u64,
                                8_usize => {
                                    let mut buffer = [0; 8];
                                    ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                    u64::from_le_bytes(buffer)
                                }
                                _ => super::super::deserialize_varint(&mut sub_reader)?,
                            };

                            c_required.get_or_insert(payload);
                        }
                        3 => {
                            let payload = match size {
                                0_usize => 0_i64,
                                8_usize => {
                                    let mut buffer = [0; 8];
                                    ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                    i64::from_le_bytes(buffer)
                                }
                                _ => super::super::zigzag_decode(super::super::deserialize_varint\
                                    (&mut sub_reader)?),
                            };
                            d_required.get_or_insert(payload);
                        }
                        4 => {
                            let payload = if size == 0_usize {
                                false
                            } else {
                                let mut buffer = [0_u8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                buffer[0] != 0b0000_0001
                            };

                            e_required.get_or_insert(payload);
                        }
                        5 => {
                            let mut payload = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;

                            f_required.get_or_insert(payload);
                        }
                        6 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            g_required.get_or_insert(payload);
                        }
                        7 => {
                            let payload = match size {
                                0_usize => 0_u64,
                                8_usize => {
                                    let mut buffer = [0; 8];
                                    ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                    u64::from_le_bytes(buffer)
                                }
                                _ => super::super::deserialize_varint(&mut sub_reader)?,
                            };
                            let payload = vec![(); payload as usize];

                            h_required.get_or_insert(payload);
                        }
                        8 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<f64> {
                                let mut buffer = [0; 8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                                let payload = f64::from_le_bytes(buffer);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }

                            i_required.get_or_insert(payload);
                        }
                        9 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<u64> {
                                let payload = super::super::deserialize_varint(&mut sub_reader)?;

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }

                            j_required.get_or_insert(payload);
                        }
                        10 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<i64> {
                                let payload = \
                                    super::super::zigzag_decode(super::super::deserialize_varint(&\
                                    mut sub_reader)?);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }

                            k_required.get_or_insert(payload);
                        }
                        11 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<bool> {
                                let mut buffer = [0_u8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                let payload = buffer[0] != 0b0000_0001;

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }

                            l_required.get_or_insert(payload);
                        }
                        12 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size as u64);
                                payload.push({
                                    let mut payload = vec![];
                                    ::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;
                                    payload
                                });
                            }

                            m_required.get_or_insert(payload);
                        }
                        13 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size as u64);
                                payload.push({
                                    let mut buffer = vec![];
                                    ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                                    let payload = std::str::from_utf8(&buffer).map_or_else(
                                        |err| Err(::std::io::Error::new(::std::io::ErrorKind::\
                                            Other, err)),
                                        |result| Ok(result.to_owned()),
                                    )?;
                                    payload
                                });
                            }

                            n_required.get_or_insert(payload);
                        }
                        14 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size as u64);
                                payload.push({
                                    let mut payload = Vec::new();

                                    loop {
                                        let element_size = match \
                                            super::super::deserialize_varint(&mut sub_reader) {
                                            Ok(element_size) => element_size,
                                            Err(err) => {
                                                if let std::io::ErrorKind::UnexpectedEof = \
                                                    err.kind() {
                                                    break;
                                                }

                                                return Err(err);
                                            }
                                        };

                                        let mut sub_reader = ::std::io::Read::take(&mut \
                                            sub_reader, element_size as u64);
                                        payload.push({
                                            let mut buffer = vec![];
                                            ::std::io::Read::read_to_end(&mut sub_reader, &mut \
                                                buffer)?;
                                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                                |err| Err(::std::io::Error::new(::std::io::\
                                                    ErrorKind::Other, err)),
                                                |result| Ok(result.to_owned()),
                                            )?;
                                            payload
                                        });
                                    }
                                    payload
                                });
                            }

                            o_required.get_or_insert(payload);
                        }
                        16 => {
                            let payload = ();

                            a_asymmetric.get_or_insert(payload);
                        }
                        17 => {
                            let payload = if size == 0_usize {
                                0.0_f64
                            } else {
                                let mut buffer = [0; 8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                                f64::from_le_bytes(buffer)
                            };

                            b_asymmetric.get_or_insert(payload);
                        }
                        18 => {
                            let payload = match size {
                                0_usize => 0_u64,
                                8_usize => {
                                    let mut buffer = [0; 8];
                                    ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                    u64::from_le_bytes(buffer)
                                }
                                _ => super::super::deserialize_varint(&mut sub_reader)?,
                            };

                            c_asymmetric.get_or_insert(payload);
                        }
                        19 => {
                            let payload = match size {
                                0_usize => 0_i64,
                                8_usize => {
                                    let mut buffer = [0; 8];
                                    ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                    i64::from_le_bytes(buffer)
                                }
                                _ => super::super::zigzag_decode(super::super::deserialize_varint\
                                    (&mut sub_reader)?),
                            };
                            d_asymmetric.get_or_insert(payload);
                        }
                        20 => {
                            let payload = if size == 0_usize {
                                false
                            } else {
                                let mut buffer = [0_u8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                buffer[0] != 0b0000_0001
                            };

                            e_asymmetric.get_or_insert(payload);
                        }
                        21 => {
                            let mut payload = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;

                            f_asymmetric.get_or_insert(payload);
                        }
                        22 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            g_asymmetric.get_or_insert(payload);
                        }
                        23 => {
                            let payload = match size {
                                0_usize => 0_u64,
                                8_usize => {
                                    let mut buffer = [0; 8];
                                    ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                    u64::from_le_bytes(buffer)
                                }
                                _ => super::super::deserialize_varint(&mut sub_reader)?,
                            };
                            let payload = vec![(); payload as usize];

                            h_asymmetric.get_or_insert(payload);
                        }
                        24 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<f64> {
                                let mut buffer = [0; 8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                                let payload = f64::from_le_bytes(buffer);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }

                            i_asymmetric.get_or_insert(payload);
                        }
                        25 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<u64> {
                                let payload = super::super::deserialize_varint(&mut sub_reader)?;

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }

                            j_asymmetric.get_or_insert(payload);
                        }
                        26 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<i64> {
                                let payload = \
                                    super::super::zigzag_decode(super::super::deserialize_varint(&\
                                    mut sub_reader)?);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }

                            k_asymmetric.get_or_insert(payload);
                        }
                        27 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<bool> {
                                let mut buffer = [0_u8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                let payload = buffer[0] != 0b0000_0001;

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }

                            l_asymmetric.get_or_insert(payload);
                        }
                        28 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size as u64);
                                payload.push({
                                    let mut payload = vec![];
                                    ::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;
                                    payload
                                });
                            }

                            m_asymmetric.get_or_insert(payload);
                        }
                        29 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size as u64);
                                payload.push({
                                    let mut buffer = vec![];
                                    ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                                    let payload = std::str::from_utf8(&buffer).map_or_else(
                                        |err| Err(::std::io::Error::new(::std::io::ErrorKind::\
                                            Other, err)),
                                        |result| Ok(result.to_owned()),
                                    )?;
                                    payload
                                });
                            }

                            n_asymmetric.get_or_insert(payload);
                        }
                        30 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size as u64);
                                payload.push({
                                    let mut payload = Vec::new();

                                    loop {
                                        let element_size = match \
                                            super::super::deserialize_varint(&mut sub_reader) {
                                            Ok(element_size) => element_size,
                                            Err(err) => {
                                                if let std::io::ErrorKind::UnexpectedEof = \
                                                    err.kind() {
                                                    break;
                                                }

                                                return Err(err);
                                            }
                                        };

                                        let mut sub_reader = ::std::io::Read::take(&mut \
                                            sub_reader, element_size as u64);
                                        payload.push({
                                            let mut buffer = vec![];
                                            ::std::io::Read::read_to_end(&mut sub_reader, &mut \
                                                buffer)?;
                                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                                |err| Err(::std::io::Error::new(::std::io::\
                                                    ErrorKind::Other, err)),
                                                |result| Ok(result.to_owned()),
                                            )?;
                                            payload
                                        });
                                    }
                                    payload
                                });
                            }

                            o_asymmetric.get_or_insert(payload);
                        }
                        32 => {
                            let payload = ();

                            a_optional.get_or_insert(payload);
                        }
                        33 => {
                            let payload = if size == 0_usize {
                                0.0_f64
                            } else {
                                let mut buffer = [0; 8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                                f64::from_le_bytes(buffer)
                            };

                            b_optional.get_or_insert(payload);
                        }
                        34 => {
                            let payload = match size {
                                0_usize => 0_u64,
                                8_usize => {
                                    let mut buffer = [0; 8];
                                    ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                    u64::from_le_bytes(buffer)
                                }
                                _ => super::super::deserialize_varint(&mut sub_reader)?,
                            };

                            c_optional.get_or_insert(payload);
                        }
                        35 => {
                            let payload = match size {
                                0_usize => 0_i64,
                                8_usize => {
                                    let mut buffer = [0; 8];
                                    ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                    i64::from_le_bytes(buffer)
                                }
                                _ => super::super::zigzag_decode(super::super::deserialize_varint\
                                    (&mut sub_reader)?),
                            };
                            d_optional.get_or_insert(payload);
                        }
                        36 => {
                            let payload = if size == 0_usize {
                                false
                            } else {
                                let mut buffer = [0_u8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                buffer[0] != 0b0000_0001
                            };

                            e_optional.get_or_insert(payload);
                        }
                        37 => {
                            let mut payload = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;

                            f_optional.get_or_insert(payload);
                        }
                        38 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            g_optional.get_or_insert(payload);
                        }
                        39 => {
                            let payload = match size {
                                0_usize => 0_u64,
                                8_usize => {
                                    let mut buffer = [0; 8];
                                    ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                    u64::from_le_bytes(buffer)
                                }
                                _ => super::super::deserialize_varint(&mut sub_reader)?,
                            };
                            let payload = vec![(); payload as usize];

                            h_optional.get_or_insert(payload);
                        }
                        40 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<f64> {
                                let mut buffer = [0; 8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                                let payload = f64::from_le_bytes(buffer);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }

                            i_optional.get_or_insert(payload);
                        }
                        41 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<u64> {
                                let payload = super::super::deserialize_varint(&mut sub_reader)?;

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }

                            j_optional.get_or_insert(payload);
                        }
                        42 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<i64> {
                                let payload = \
                                    super::super::zigzag_decode(super::super::deserialize_varint(&\
                                    mut sub_reader)?);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }

                            k_optional.get_or_insert(payload);
                        }
                        43 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<bool> {
                                let mut buffer = [0_u8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                                let payload = buffer[0] != 0b0000_0001;

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }

                            l_optional.get_or_insert(payload);
                        }
                        44 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size as u64);
                                payload.push({
                                    let mut payload = vec![];
                                    ::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;
                                    payload
                                });
                            }

                            m_optional.get_or_insert(payload);
                        }
                        45 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size as u64);
                                payload.push({
                                    let mut buffer = vec![];
                                    ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                                    let payload = std::str::from_utf8(&buffer).map_or_else(
                                        |err| Err(::std::io::Error::new(::std::io::ErrorKind::\
                                            Other, err)),
                                        |result| Ok(result.to_owned()),
                                    )?;
                                    payload
                                });
                            }

                            n_optional.get_or_insert(payload);
                        }
                        46 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size as u64);
                                payload.push({
                                    let mut payload = Vec::new();

                                    loop {
                                        let element_size = match \
                                            super::super::deserialize_varint(&mut sub_reader) {
                                            Ok(element_size) => element_size,
                                            Err(err) => {
                                                if let std::io::ErrorKind::UnexpectedEof = \
                                                    err.kind() {
                                                    break;
                                                }

                                                return Err(err);
                                            }
                                        };

                                        let mut sub_reader = ::std::io::Read::take(&mut \
                                            sub_reader, element_size as u64);
                                        payload.push({
                                            let mut buffer = vec![];
                                            ::std::io::Read::read_to_end(&mut sub_reader, &mut \
                                                buffer)?;
                                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                                |err| Err(::std::io::Error::new(::std::io::\
                                                    ErrorKind::Other, err)),
                                                |result| Ok(result.to_owned()),
                                            )?;
                                            payload
                                        });
                                    }
                                    payload
                                });
                            }

                            o_optional.get_or_insert(payload);
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }

                if a_required.is_none() || b_required.is_none() || c_required.is_none() || \
                    d_required.is_none() || e_required.is_none() || f_required.is_none() || \
                    g_required.is_none() || h_required.is_none() || i_required.is_none() || \
                    j_required.is_none() || k_required.is_none() || l_required.is_none() || \
                    m_required.is_none() || n_required.is_none() || o_required.is_none() {
                    return Err(::std::io::Error::new(
                        ::std::io::ErrorKind::InvalidData,
                        \"Struct missing one or more field(s).\",
                    ));
                }

                Ok(FooIn {
                    a_required: a_required.unwrap(),
                    b_required: b_required.unwrap(),
                    c_required: c_required.unwrap(),
                    d_required: d_required.unwrap(),
                    e_required: e_required.unwrap(),
                    f_required: f_required.unwrap(),
                    g_required: g_required.unwrap(),
                    h_required: h_required.unwrap(),
                    i_required: i_required.unwrap(),
                    j_required: j_required.unwrap(),
                    k_required: k_required.unwrap(),
                    l_required: l_required.unwrap(),
                    m_required: m_required.unwrap(),
                    n_required: n_required.unwrap(),
                    o_required: o_required.unwrap(),
                    a_asymmetric,
                    b_asymmetric,
                    c_asymmetric,
                    d_asymmetric,
                    e_asymmetric,
                    f_asymmetric,
                    g_asymmetric,
                    h_asymmetric,
                    i_asymmetric,
                    j_asymmetric,
                    k_asymmetric,
                    l_asymmetric,
                    m_asymmetric,
                    n_asymmetric,
                    o_asymmetric,
                    a_optional,
                    b_optional,
                    c_optional,
                    d_optional,
                    e_optional,
                    f_optional,
                    g_optional,
                    h_optional,
                    i_optional,
                    j_optional,
                    k_optional,
                    l_optional,
                    m_optional,
                    n_optional,
                    o_optional,
                })
            }
        }

        impl From<FooOut> for FooIn {
            fn from(message: FooOut) -> Self {
                FooIn {
                    a_required: message.a_required.into(),
                    b_required: message.b_required.into(),
                    c_required: message.c_required.into(),
                    d_required: message.d_required.into(),
                    e_required: message.e_required.into(),
                    f_required: message.f_required.into(),
                    g_required: message.g_required.into(),
                    h_required: message.h_required.into(),
                    i_required: message.i_required.into(),
                    j_required: message.j_required.into(),
                    k_required: message.k_required.into(),
                    l_required: message.l_required.into(),
                    m_required: message.m_required.into(),
                    n_required: message.n_required.into(),
                    o_required: message.o_required.into(),
                    a_asymmetric: Some(message.a_asymmetric.into()),
                    b_asymmetric: Some(message.b_asymmetric.into()),
                    c_asymmetric: Some(message.c_asymmetric.into()),
                    d_asymmetric: Some(message.d_asymmetric.into()),
                    e_asymmetric: Some(message.e_asymmetric.into()),
                    f_asymmetric: Some(message.f_asymmetric.into()),
                    g_asymmetric: Some(message.g_asymmetric.into()),
                    h_asymmetric: Some(message.h_asymmetric.into()),
                    i_asymmetric: Some(message.i_asymmetric.into()),
                    j_asymmetric: Some(message.j_asymmetric.into()),
                    k_asymmetric: Some(message.k_asymmetric.into()),
                    l_asymmetric: Some(message.l_asymmetric.into()),
                    m_asymmetric: Some(message.m_asymmetric.into()),
                    n_asymmetric: Some(message.n_asymmetric.into()),
                    o_asymmetric: Some(message.o_asymmetric.into()),
                    a_optional: message.a_optional.map(|payload| payload.into()),
                    b_optional: message.b_optional.map(|payload| payload.into()),
                    c_optional: message.c_optional.map(|payload| payload.into()),
                    d_optional: message.d_optional.map(|payload| payload.into()),
                    e_optional: message.e_optional.map(|payload| payload.into()),
                    f_optional: message.f_optional.map(|payload| payload.into()),
                    g_optional: message.g_optional.map(|payload| payload.into()),
                    h_optional: message.h_optional.map(|payload| payload.into()),
                    i_optional: message.i_optional.map(|payload| payload.into()),
                    j_optional: message.j_optional.map(|payload| payload.into()),
                    k_optional: message.k_optional.map(|payload| payload.into()),
                    l_optional: message.l_optional.map(|payload| payload.into()),
                    m_optional: message.m_optional.map(|payload| payload.into()),
                    n_optional: message.n_optional.map(|payload| payload.into()),
                    o_optional: message.o_optional.map(|payload| payload.into()),
                }
            }
        }
    }

    pub mod main {
        #[derive(Clone, Debug)]
        pub struct EmptyStructOut {
        }

        #[derive(Clone, Debug)]
        pub struct EmptyStructIn {
        }

        impl super::super::Serialize for EmptyStructOut {
            fn size(&self) -> usize {
                0
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                Ok(())
            }
        }

        impl super::super::Deserialize for EmptyStructIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                loop {
                    let (index, size) = match super::super::deserialize_field_header(&mut \
                        *reader) {
                        Ok(header) => header,
                        Err(err) => {
                            if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                break;
                            }

                            return Err(err);
                        }
                    };

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size as u64);

                    match index {
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }

                Ok(EmptyStructIn {
                })
            }
        }

        impl From<EmptyStructOut> for EmptyStructIn {
            fn from(message: EmptyStructOut) -> Self {
                EmptyStructIn {
                }
            }
        }

        #[derive(Clone, Debug)]
        pub enum EmptyChoiceOut {
        }

        #[derive(Clone, Debug)]
        pub enum EmptyChoiceIn {
        }

        impl super::super::Serialize for EmptyChoiceOut {
            fn size(&self) -> usize {
                match *self {
                }
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                match *self {
                }
            }
        }

        impl super::super::Deserialize for EmptyChoiceIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                loop {
                    let (index, size) = super::super::deserialize_field_header(&mut *reader)?;

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size as u64);

                    match index {
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }
            }
        }

        impl From<EmptyChoiceOut> for EmptyChoiceIn {
            fn from(message: EmptyChoiceOut) -> Self {
                match message {
                }
            }
        }

        #[derive(Clone, Debug)]
        pub struct FooAndBarOut {
            pub x: super::foo::FooOut,
            pub y: super::bar::BarOut,
        }

        #[derive(Clone, Debug)]
        pub struct FooAndBarIn {
            pub x: super::foo::FooIn,
            pub y: super::bar::BarIn,
        }

        impl super::super::Serialize for FooAndBarOut {
            fn size(&self) -> usize {
                ({
                    let payload = &self.x;
                    let payload_size = payload.size();
                    super::super::field_header_size(0, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.y;
                    let payload_size = payload.size();
                    super::super::field_header_size(1, payload_size, false) + payload_size
                })
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                {
                    let payload = &self.x;
                    let payload_size = payload.size();
                    super::super::serialize_field_header(writer, 0, payload_size, false)?;
                    payload.serialize(writer)?;
                }

                {
                    let payload = &self.y;
                    let payload_size = payload.size();
                    super::super::serialize_field_header(writer, 1, payload_size, false)?;
                    payload.serialize(writer)?;
                }

                Ok(())
            }
        }

        impl super::super::Deserialize for FooAndBarIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                let mut x: Option<super::foo::FooIn> = None;
                let mut y: Option<super::bar::BarIn> = None;

                loop {
                    let (index, size) = match super::super::deserialize_field_header(&mut \
                        *reader) {
                        Ok(header) => header,
                        Err(err) => {
                            if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                break;
                            }

                            return Err(err);
                        }
                    };

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size as u64);

                    match index {
                        0 => {
                            let payload = <super::foo::FooIn as \
                                super::super::Deserialize>::deserialize(&mut sub_reader)?;

                            x.get_or_insert(payload);
                        }
                        1 => {
                            let payload = <super::bar::BarIn as \
                                super::super::Deserialize>::deserialize(&mut sub_reader)?;

                            y.get_or_insert(payload);
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }

                if x.is_none() || y.is_none() {
                    return Err(::std::io::Error::new(
                        ::std::io::ErrorKind::InvalidData,
                        \"Struct missing one or more field(s).\",
                    ));
                }

                Ok(FooAndBarIn {
                    x: x.unwrap(),
                    y: y.unwrap(),
                })
            }
        }

        impl From<FooAndBarOut> for FooAndBarIn {
            fn from(message: FooAndBarOut) -> Self {
                FooAndBarIn {
                    x: message.x.into(),
                    y: message.y.into(),
                }
            }
        }

        #[derive(Clone, Debug)]
        pub enum FooOrBarOut {
            X(super::foo::FooOut),
            Y(super::bar::BarOut),
        }

        #[derive(Clone, Debug)]
        pub enum FooOrBarIn {
            X(super::foo::FooIn),
            Y(super::bar::BarIn),
        }

        impl super::super::Serialize for FooOrBarOut {
            fn size(&self) -> usize {
                match *self {
                    FooOrBarOut::X(ref payload) => {
                        let payload_size = payload.size();
                        super::super::field_header_size(0, payload_size, false) +
                            payload_size
                    }
                    FooOrBarOut::Y(ref payload) => {
                        let payload_size = payload.size();
                        super::super::field_header_size(1, payload_size, false) +
                            payload_size
                    }
                }
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                match *self {
                    FooOrBarOut::X(ref payload) => {
                        let payload_size = payload.size();
                        super::super::serialize_field_header(writer, 0, payload_size, false)?;
                        payload.serialize(writer)?;
                        Ok(())
                    }
                    FooOrBarOut::Y(ref payload) => {
                        let payload_size = payload.size();
                        super::super::serialize_field_header(writer, 1, payload_size, false)?;
                        payload.serialize(writer)?;
                        Ok(())
                    }
                }
            }
        }

        impl super::super::Deserialize for FooOrBarIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                loop {
                    let (index, size) = super::super::deserialize_field_header(&mut *reader)?;

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size as u64);

                    match index {
                        0 => {
                            let payload = <super::foo::FooIn as \
                                super::super::Deserialize>::deserialize(&mut sub_reader)?;
                            return Ok(FooOrBarIn::X(payload));
                        }
                        1 => {
                            let payload = <super::bar::BarIn as \
                                super::super::Deserialize>::deserialize(&mut sub_reader)?;
                            return Ok(FooOrBarIn::Y(payload));
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }
            }
        }

        impl From<FooOrBarOut> for FooOrBarIn {
            fn from(message: FooOrBarOut) -> Self {
                match message {
                    FooOrBarOut::X(payload) => FooOrBarIn::X(payload.into()),
                    FooOrBarOut::Y(payload) => FooOrBarIn::Y(payload.into()),
                }
            }
        }
    }
}

pub mod main {
}

pub mod schema_evolution {
    pub mod after {
        #[derive(Clone, Debug)]
        pub struct ExampleStructOut {
            pub required_to_required: String,
            pub required_to_asymmetric: String,
            pub required_to_optional: Option<String>,
            pub asymmetric_to_required: String,
            pub asymmetric_to_asymmetric: String,
            pub asymmetric_to_optional: Option<String>,
            pub optional_none_to_asymmetric: String,
            pub optional_none_to_optional: Option<String>,
            pub optional_some_to_required: String,
            pub optional_some_to_asymmetric: String,
            pub optional_some_to_optional: Option<String>,
            pub nonexistent_to_asymmetric: String,
            pub nonexistent_to_optional: Option<String>,
        }

        #[derive(Clone, Debug)]
        pub struct ExampleStructIn {
            pub required_to_required: String,
            pub required_to_asymmetric: Option<String>,
            pub required_to_optional: Option<String>,
            pub asymmetric_to_required: String,
            pub asymmetric_to_asymmetric: Option<String>,
            pub asymmetric_to_optional: Option<String>,
            pub optional_none_to_asymmetric: Option<String>,
            pub optional_none_to_optional: Option<String>,
            pub optional_some_to_required: String,
            pub optional_some_to_asymmetric: Option<String>,
            pub optional_some_to_optional: Option<String>,
            pub nonexistent_to_asymmetric: Option<String>,
            pub nonexistent_to_optional: Option<String>,
        }

        impl super::super::Serialize for ExampleStructOut {
            fn size(&self) -> usize {
                ({
                    let payload = &self.required_to_required;
                    let payload_size = payload.len();
                    super::super::field_header_size(0, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.required_to_asymmetric;
                    let payload_size = payload.len();
                    super::super::field_header_size(1, payload_size, false) + payload_size
                }) + self.required_to_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len();
                    super::super::field_header_size(2, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.asymmetric_to_required;
                    let payload_size = payload.len();
                    super::super::field_header_size(4, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.asymmetric_to_asymmetric;
                    let payload_size = payload.len();
                    super::super::field_header_size(5, payload_size, false) + payload_size
                }) + self.asymmetric_to_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len();
                    super::super::field_header_size(6, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.optional_none_to_asymmetric;
                    let payload_size = payload.len();
                    super::super::field_header_size(9, payload_size, false) + payload_size
                }) + self.optional_none_to_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len();
                    super::super::field_header_size(10, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.optional_some_to_required;
                    let payload_size = payload.len();
                    super::super::field_header_size(12, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.optional_some_to_asymmetric;
                    let payload_size = payload.len();
                    super::super::field_header_size(13, payload_size, false) + payload_size
                }) + self.optional_some_to_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len();
                    super::super::field_header_size(14, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.nonexistent_to_asymmetric;
                    let payload_size = payload.len();
                    super::super::field_header_size(17, payload_size, false) + payload_size
                }) + self.nonexistent_to_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len();
                    super::super::field_header_size(18, payload_size, false) + payload_size
                })
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                {
                    let payload = &self.required_to_required;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 0, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.required_to_asymmetric;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 1, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.required_to_optional {
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 2, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.asymmetric_to_required;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 4, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.asymmetric_to_asymmetric;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 5, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.asymmetric_to_optional {
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 6, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.optional_none_to_asymmetric;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 9, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.optional_none_to_optional {
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 10, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.optional_some_to_required;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 12, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.optional_some_to_asymmetric;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 13, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.optional_some_to_optional {
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 14, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.nonexistent_to_asymmetric;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 17, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.nonexistent_to_optional {
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 18, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                Ok(())
            }
        }

        impl super::super::Deserialize for ExampleStructIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                let mut required_to_required: Option<String> = None;
                let mut required_to_asymmetric: Option<String> = None;
                let mut required_to_optional: Option<String> = None;
                let mut asymmetric_to_required: Option<String> = None;
                let mut asymmetric_to_asymmetric: Option<String> = None;
                let mut asymmetric_to_optional: Option<String> = None;
                let mut optional_none_to_asymmetric: Option<String> = None;
                let mut optional_none_to_optional: Option<String> = None;
                let mut optional_some_to_required: Option<String> = None;
                let mut optional_some_to_asymmetric: Option<String> = None;
                let mut optional_some_to_optional: Option<String> = None;
                let mut nonexistent_to_asymmetric: Option<String> = None;
                let mut nonexistent_to_optional: Option<String> = None;

                loop {
                    let (index, size) = match super::super::deserialize_field_header(&mut \
                        *reader) {
                        Ok(header) => header,
                        Err(err) => {
                            if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                break;
                            }

                            return Err(err);
                        }
                    };

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size as u64);

                    match index {
                        0 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            required_to_required.get_or_insert(payload);
                        }
                        1 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            required_to_asymmetric.get_or_insert(payload);
                        }
                        2 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            required_to_optional.get_or_insert(payload);
                        }
                        4 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            asymmetric_to_required.get_or_insert(payload);
                        }
                        5 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            asymmetric_to_asymmetric.get_or_insert(payload);
                        }
                        6 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            asymmetric_to_optional.get_or_insert(payload);
                        }
                        9 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_none_to_asymmetric.get_or_insert(payload);
                        }
                        10 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_none_to_optional.get_or_insert(payload);
                        }
                        12 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_some_to_required.get_or_insert(payload);
                        }
                        13 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_some_to_asymmetric.get_or_insert(payload);
                        }
                        14 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_some_to_optional.get_or_insert(payload);
                        }
                        17 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            nonexistent_to_asymmetric.get_or_insert(payload);
                        }
                        18 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            nonexistent_to_optional.get_or_insert(payload);
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }

                if required_to_required.is_none() || asymmetric_to_required.is_none() || \
                    optional_some_to_required.is_none() {
                    return Err(::std::io::Error::new(
                        ::std::io::ErrorKind::InvalidData,
                        \"Struct missing one or more field(s).\",
                    ));
                }

                Ok(ExampleStructIn {
                    required_to_required: required_to_required.unwrap(),
                    required_to_asymmetric,
                    required_to_optional,
                    asymmetric_to_required: asymmetric_to_required.unwrap(),
                    asymmetric_to_asymmetric,
                    asymmetric_to_optional,
                    optional_none_to_asymmetric,
                    optional_none_to_optional,
                    optional_some_to_required: optional_some_to_required.unwrap(),
                    optional_some_to_asymmetric,
                    optional_some_to_optional,
                    nonexistent_to_asymmetric,
                    nonexistent_to_optional,
                })
            }
        }

        impl From<ExampleStructOut> for ExampleStructIn {
            fn from(message: ExampleStructOut) -> Self {
                ExampleStructIn {
                    required_to_required: message.required_to_required.into(),
                    required_to_asymmetric: Some(message.required_to_asymmetric.into()),
                    required_to_optional: message.required_to_optional.map(|payload| \
                        payload.into()),
                    asymmetric_to_required: message.asymmetric_to_required.into(),
                    asymmetric_to_asymmetric: Some(message.asymmetric_to_asymmetric.into()),
                    asymmetric_to_optional: message.asymmetric_to_optional.map(|payload| \
                        payload.into()),
                    optional_none_to_asymmetric: Some(message.optional_none_to_asymmetric.into()),
                    optional_none_to_optional: message.optional_none_to_optional.map(|payload| \
                        payload.into()),
                    optional_some_to_required: message.optional_some_to_required.into(),
                    optional_some_to_asymmetric: Some(message.optional_some_to_asymmetric.into()),
                    optional_some_to_optional: message.optional_some_to_optional.map(|payload| \
                        payload.into()),
                    nonexistent_to_asymmetric: Some(message.nonexistent_to_asymmetric.into()),
                    nonexistent_to_optional: message.nonexistent_to_optional.map(|payload| \
                        payload.into()),
                }
            }
        }

        #[derive(Clone, Debug)]
        pub enum ExampleChoiceOut {
            RequiredToRequired(String),
            RequiredToAsymmetric(String, Box<ExampleChoiceOut>),
            AsymmetricToRequired(String),
            AsymmetricToAsymmetric(String, Box<ExampleChoiceOut>),
            AsymmetricToOptionalHandled(String, Box<ExampleChoiceOut>),
            AsymmetricToOptionalFallback(String, Box<ExampleChoiceOut>),
            OptionalToRequired(String),
            OptionalToAsymmetric(String, Box<ExampleChoiceOut>),
            OptionalToOptionalHandled(String, Box<ExampleChoiceOut>),
            OptionalToOptionalFallback(String, Box<ExampleChoiceOut>),
            NonexistentToRequired(String),
            NonexistentToAsymmetric(String, Box<ExampleChoiceOut>),
            NonexistentToOptionalHandled(String, Box<ExampleChoiceOut>),
            NonexistentToOptionalFallback(String, Box<ExampleChoiceOut>),
        }

        #[derive(Clone, Debug)]
        pub enum ExampleChoiceIn {
            RequiredToRequired(String),
            RequiredToAsymmetric(String),
            AsymmetricToRequired(String),
            AsymmetricToAsymmetric(String),
            AsymmetricToOptionalHandled(String, Box<ExampleChoiceIn>),
            AsymmetricToOptionalFallback(String, Box<ExampleChoiceIn>),
            OptionalToRequired(String),
            OptionalToAsymmetric(String),
            OptionalToOptionalHandled(String, Box<ExampleChoiceIn>),
            OptionalToOptionalFallback(String, Box<ExampleChoiceIn>),
            NonexistentToRequired(String),
            NonexistentToAsymmetric(String),
            NonexistentToOptionalHandled(String, Box<ExampleChoiceIn>),
            NonexistentToOptionalFallback(String, Box<ExampleChoiceIn>),
        }

        impl super::super::Serialize for ExampleChoiceOut {
            fn size(&self) -> usize {
                match *self {
                    ExampleChoiceOut::RequiredToRequired(ref payload) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(0, payload_size, false) +
                            payload_size
                    }
                    ExampleChoiceOut::RequiredToAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(1, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::AsymmetricToRequired(ref payload) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(5, payload_size, false) +
                            payload_size
                    }
                    ExampleChoiceOut::AsymmetricToAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(6, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::AsymmetricToOptionalHandled(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(7, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::AsymmetricToOptionalFallback(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(8, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::OptionalToRequired(ref payload) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(10, payload_size, false) +
                            payload_size
                    }
                    ExampleChoiceOut::OptionalToAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(11, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::OptionalToOptionalHandled(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(12, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::OptionalToOptionalFallback(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(13, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::NonexistentToRequired(ref payload) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(15, payload_size, false) +
                            payload_size
                    }
                    ExampleChoiceOut::NonexistentToAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(16, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::NonexistentToOptionalHandled(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(17, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::NonexistentToOptionalFallback(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(18, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                }
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                match *self {
                    ExampleChoiceOut::RequiredToRequired(ref payload) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 0, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        Ok(())
                    }
                    ExampleChoiceOut::RequiredToAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 1, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::AsymmetricToRequired(ref payload) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 5, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        Ok(())
                    }
                    ExampleChoiceOut::AsymmetricToAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 6, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::AsymmetricToOptionalHandled(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 7, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::AsymmetricToOptionalFallback(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 8, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::OptionalToRequired(ref payload) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 10, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        Ok(())
                    }
                    ExampleChoiceOut::OptionalToAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 11, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::OptionalToOptionalHandled(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 12, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::OptionalToOptionalFallback(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 13, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::NonexistentToRequired(ref payload) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 15, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        Ok(())
                    }
                    ExampleChoiceOut::NonexistentToAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 16, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::NonexistentToOptionalHandled(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 17, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::NonexistentToOptionalFallback(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 18, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                }
            }
        }

        impl super::super::Deserialize for ExampleChoiceIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                loop {
                    let (index, size) = super::super::deserialize_field_header(&mut *reader)?;

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size as u64);

                    match index {
                        0 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::RequiredToRequired(payload));
                        }
                        1 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::RequiredToAsymmetric(payload));
                        }
                        5 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::AsymmetricToRequired(payload));
                        }
                        6 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::AsymmetricToAsymmetric(payload));
                        }
                        7 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::AsymmetricToOptionalHandled(payload, \
                                fallback));
                        }
                        8 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::AsymmetricToOptionalFallback(payload, \
                                fallback));
                        }
                        10 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::OptionalToRequired(payload));
                        }
                        11 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::OptionalToAsymmetric(payload));
                        }
                        12 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::OptionalToOptionalHandled(payload, \
                                fallback));
                        }
                        13 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::OptionalToOptionalFallback(payload, \
                                fallback));
                        }
                        15 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::NonexistentToRequired(payload));
                        }
                        16 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::NonexistentToAsymmetric(payload));
                        }
                        17 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::NonexistentToOptionalHandled(payload, \
                                fallback));
                        }
                        18 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::NonexistentToOptionalFallback(payload, \
                                fallback));
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }
            }
        }

        impl From<ExampleChoiceOut> for ExampleChoiceIn {
            fn from(message: ExampleChoiceOut) -> Self {
                match message {
                    ExampleChoiceOut::RequiredToRequired(payload) => \
                        ExampleChoiceIn::RequiredToRequired(payload.into()),
                    ExampleChoiceOut::RequiredToAsymmetric(payload, fallback) => \
                        ExampleChoiceIn::RequiredToAsymmetric(payload.into()),
                    ExampleChoiceOut::AsymmetricToRequired(payload) => \
                        ExampleChoiceIn::AsymmetricToRequired(payload.into()),
                    ExampleChoiceOut::AsymmetricToAsymmetric(payload, fallback) => \
                        ExampleChoiceIn::AsymmetricToAsymmetric(payload.into()),
                    ExampleChoiceOut::AsymmetricToOptionalHandled(payload, fallback) => \
                        ExampleChoiceIn::AsymmetricToOptionalHandled(payload.into(), \
                        Box::new((*fallback).into())),
                    ExampleChoiceOut::AsymmetricToOptionalFallback(payload, fallback) => \
                        ExampleChoiceIn::AsymmetricToOptionalFallback(payload.into(), \
                        Box::new((*fallback).into())),
                    ExampleChoiceOut::OptionalToRequired(payload) => \
                        ExampleChoiceIn::OptionalToRequired(payload.into()),
                    ExampleChoiceOut::OptionalToAsymmetric(payload, fallback) => \
                        ExampleChoiceIn::OptionalToAsymmetric(payload.into()),
                    ExampleChoiceOut::OptionalToOptionalHandled(payload, fallback) => \
                        ExampleChoiceIn::OptionalToOptionalHandled(payload.into(), \
                        Box::new((*fallback).into())),
                    ExampleChoiceOut::OptionalToOptionalFallback(payload, fallback) => \
                        ExampleChoiceIn::OptionalToOptionalFallback(payload.into(), \
                        Box::new((*fallback).into())),
                    ExampleChoiceOut::NonexistentToRequired(payload) => \
                        ExampleChoiceIn::NonexistentToRequired(payload.into()),
                    ExampleChoiceOut::NonexistentToAsymmetric(payload, fallback) => \
                        ExampleChoiceIn::NonexistentToAsymmetric(payload.into()),
                    ExampleChoiceOut::NonexistentToOptionalHandled(payload, fallback) => \
                        ExampleChoiceIn::NonexistentToOptionalHandled(payload.into(), \
                        Box::new((*fallback).into())),
                    ExampleChoiceOut::NonexistentToOptionalFallback(payload, fallback) => \
                        ExampleChoiceIn::NonexistentToOptionalFallback(payload.into(), \
                        Box::new((*fallback).into())),
                }
            }
        }
    }

    pub mod before {
        #[derive(Clone, Debug)]
        pub struct ExampleStructOut {
            pub required_to_required: String,
            pub required_to_asymmetric: String,
            pub required_to_optional: String,
            pub required_to_nonexistent: String,
            pub asymmetric_to_required: String,
            pub asymmetric_to_asymmetric: String,
            pub asymmetric_to_optional: String,
            pub asymmetric_to_nonexistent: String,
            pub optional_none_to_asymmetric: Option<String>,
            pub optional_none_to_optional: Option<String>,
            pub optional_none_to_nonexistent: Option<String>,
            pub optional_some_to_required: Option<String>,
            pub optional_some_to_asymmetric: Option<String>,
            pub optional_some_to_optional: Option<String>,
            pub optional_some_to_nonexistent: Option<String>,
        }

        #[derive(Clone, Debug)]
        pub struct ExampleStructIn {
            pub required_to_required: String,
            pub required_to_asymmetric: String,
            pub required_to_optional: String,
            pub required_to_nonexistent: String,
            pub asymmetric_to_required: Option<String>,
            pub asymmetric_to_asymmetric: Option<String>,
            pub asymmetric_to_optional: Option<String>,
            pub asymmetric_to_nonexistent: Option<String>,
            pub optional_none_to_asymmetric: Option<String>,
            pub optional_none_to_optional: Option<String>,
            pub optional_none_to_nonexistent: Option<String>,
            pub optional_some_to_required: Option<String>,
            pub optional_some_to_asymmetric: Option<String>,
            pub optional_some_to_optional: Option<String>,
            pub optional_some_to_nonexistent: Option<String>,
        }

        impl super::super::Serialize for ExampleStructOut {
            fn size(&self) -> usize {
                ({
                    let payload = &self.required_to_required;
                    let payload_size = payload.len();
                    super::super::field_header_size(0, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.required_to_asymmetric;
                    let payload_size = payload.len();
                    super::super::field_header_size(1, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.required_to_optional;
                    let payload_size = payload.len();
                    super::super::field_header_size(2, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.required_to_nonexistent;
                    let payload_size = payload.len();
                    super::super::field_header_size(3, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.asymmetric_to_required;
                    let payload_size = payload.len();
                    super::super::field_header_size(4, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.asymmetric_to_asymmetric;
                    let payload_size = payload.len();
                    super::super::field_header_size(5, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.asymmetric_to_optional;
                    let payload_size = payload.len();
                    super::super::field_header_size(6, payload_size, false) + payload_size
                }) + ({
                    let payload = &self.asymmetric_to_nonexistent;
                    let payload_size = payload.len();
                    super::super::field_header_size(7, payload_size, false) + payload_size
                }) + self.optional_none_to_asymmetric.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len();
                    super::super::field_header_size(9, payload_size, false) + payload_size
                }) + self.optional_none_to_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len();
                    super::super::field_header_size(10, payload_size, false) + payload_size
                }) + self.optional_none_to_nonexistent.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len();
                    super::super::field_header_size(11, payload_size, false) + payload_size
                }) + self.optional_some_to_required.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len();
                    super::super::field_header_size(12, payload_size, false) + payload_size
                }) + self.optional_some_to_asymmetric.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len();
                    super::super::field_header_size(13, payload_size, false) + payload_size
                }) + self.optional_some_to_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len();
                    super::super::field_header_size(14, payload_size, false) + payload_size
                }) + self.optional_some_to_nonexistent.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len();
                    super::super::field_header_size(15, payload_size, false) + payload_size
                })
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                {
                    let payload = &self.required_to_required;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 0, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.required_to_asymmetric;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 1, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.required_to_optional;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 2, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.required_to_nonexistent;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 3, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.asymmetric_to_required;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 4, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.asymmetric_to_asymmetric;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 5, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.asymmetric_to_optional;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 6, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.asymmetric_to_nonexistent;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 7, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.optional_none_to_asymmetric {
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 9, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.optional_none_to_optional {
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 10, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.optional_none_to_nonexistent {
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 11, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.optional_some_to_required {
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 12, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.optional_some_to_asymmetric {
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 13, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.optional_some_to_optional {
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 14, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.optional_some_to_nonexistent {
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 15, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                Ok(())
            }
        }

        impl super::super::Deserialize for ExampleStructIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                let mut required_to_required: Option<String> = None;
                let mut required_to_asymmetric: Option<String> = None;
                let mut required_to_optional: Option<String> = None;
                let mut required_to_nonexistent: Option<String> = None;
                let mut asymmetric_to_required: Option<String> = None;
                let mut asymmetric_to_asymmetric: Option<String> = None;
                let mut asymmetric_to_optional: Option<String> = None;
                let mut asymmetric_to_nonexistent: Option<String> = None;
                let mut optional_none_to_asymmetric: Option<String> = None;
                let mut optional_none_to_optional: Option<String> = None;
                let mut optional_none_to_nonexistent: Option<String> = None;
                let mut optional_some_to_required: Option<String> = None;
                let mut optional_some_to_asymmetric: Option<String> = None;
                let mut optional_some_to_optional: Option<String> = None;
                let mut optional_some_to_nonexistent: Option<String> = None;

                loop {
                    let (index, size) = match super::super::deserialize_field_header(&mut \
                        *reader) {
                        Ok(header) => header,
                        Err(err) => {
                            if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                break;
                            }

                            return Err(err);
                        }
                    };

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size as u64);

                    match index {
                        0 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            required_to_required.get_or_insert(payload);
                        }
                        1 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            required_to_asymmetric.get_or_insert(payload);
                        }
                        2 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            required_to_optional.get_or_insert(payload);
                        }
                        3 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            required_to_nonexistent.get_or_insert(payload);
                        }
                        4 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            asymmetric_to_required.get_or_insert(payload);
                        }
                        5 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            asymmetric_to_asymmetric.get_or_insert(payload);
                        }
                        6 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            asymmetric_to_optional.get_or_insert(payload);
                        }
                        7 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            asymmetric_to_nonexistent.get_or_insert(payload);
                        }
                        9 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_none_to_asymmetric.get_or_insert(payload);
                        }
                        10 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_none_to_optional.get_or_insert(payload);
                        }
                        11 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_none_to_nonexistent.get_or_insert(payload);
                        }
                        12 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_some_to_required.get_or_insert(payload);
                        }
                        13 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_some_to_asymmetric.get_or_insert(payload);
                        }
                        14 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_some_to_optional.get_or_insert(payload);
                        }
                        15 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_some_to_nonexistent.get_or_insert(payload);
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }

                if required_to_required.is_none() || required_to_asymmetric.is_none() || \
                    required_to_optional.is_none() || required_to_nonexistent.is_none() {
                    return Err(::std::io::Error::new(
                        ::std::io::ErrorKind::InvalidData,
                        \"Struct missing one or more field(s).\",
                    ));
                }

                Ok(ExampleStructIn {
                    required_to_required: required_to_required.unwrap(),
                    required_to_asymmetric: required_to_asymmetric.unwrap(),
                    required_to_optional: required_to_optional.unwrap(),
                    required_to_nonexistent: required_to_nonexistent.unwrap(),
                    asymmetric_to_required,
                    asymmetric_to_asymmetric,
                    asymmetric_to_optional,
                    asymmetric_to_nonexistent,
                    optional_none_to_asymmetric,
                    optional_none_to_optional,
                    optional_none_to_nonexistent,
                    optional_some_to_required,
                    optional_some_to_asymmetric,
                    optional_some_to_optional,
                    optional_some_to_nonexistent,
                })
            }
        }

        impl From<ExampleStructOut> for ExampleStructIn {
            fn from(message: ExampleStructOut) -> Self {
                ExampleStructIn {
                    required_to_required: message.required_to_required.into(),
                    required_to_asymmetric: message.required_to_asymmetric.into(),
                    required_to_optional: message.required_to_optional.into(),
                    required_to_nonexistent: message.required_to_nonexistent.into(),
                    asymmetric_to_required: Some(message.asymmetric_to_required.into()),
                    asymmetric_to_asymmetric: Some(message.asymmetric_to_asymmetric.into()),
                    asymmetric_to_optional: Some(message.asymmetric_to_optional.into()),
                    asymmetric_to_nonexistent: Some(message.asymmetric_to_nonexistent.into()),
                    optional_none_to_asymmetric: \
                        message.optional_none_to_asymmetric.map(|payload| payload.into()),
                    optional_none_to_optional: message.optional_none_to_optional.map(|payload| \
                        payload.into()),
                    optional_none_to_nonexistent: \
                        message.optional_none_to_nonexistent.map(|payload| payload.into()),
                    optional_some_to_required: message.optional_some_to_required.map(|payload| \
                        payload.into()),
                    optional_some_to_asymmetric: \
                        message.optional_some_to_asymmetric.map(|payload| payload.into()),
                    optional_some_to_optional: message.optional_some_to_optional.map(|payload| \
                        payload.into()),
                    optional_some_to_nonexistent: \
                        message.optional_some_to_nonexistent.map(|payload| payload.into()),
                }
            }
        }

        #[derive(Clone, Debug)]
        pub enum ExampleChoiceOut {
            RequiredToRequired(String),
            RequiredToAsymmetric(String),
            AsymmetricToRequired(String, Box<ExampleChoiceOut>),
            AsymmetricToAsymmetric(String, Box<ExampleChoiceOut>),
            AsymmetricToOptionalHandled(String, Box<ExampleChoiceOut>),
            AsymmetricToOptionalFallback(String, Box<ExampleChoiceOut>),
            AsymmetricToNonexistent(String, Box<ExampleChoiceOut>),
            OptionalToRequired(String, Box<ExampleChoiceOut>),
            OptionalToAsymmetric(String, Box<ExampleChoiceOut>),
            OptionalToOptionalHandled(String, Box<ExampleChoiceOut>),
            OptionalToOptionalFallback(String, Box<ExampleChoiceOut>),
            OptionalToNonexistent(String, Box<ExampleChoiceOut>),
        }

        #[derive(Clone, Debug)]
        pub enum ExampleChoiceIn {
            RequiredToRequired(String),
            RequiredToAsymmetric(String),
            AsymmetricToRequired(String),
            AsymmetricToAsymmetric(String),
            AsymmetricToOptionalHandled(String),
            AsymmetricToOptionalFallback(String),
            AsymmetricToNonexistent(String),
            OptionalToRequired(String, Box<ExampleChoiceIn>),
            OptionalToAsymmetric(String, Box<ExampleChoiceIn>),
            OptionalToOptionalHandled(String, Box<ExampleChoiceIn>),
            OptionalToOptionalFallback(String, Box<ExampleChoiceIn>),
            OptionalToNonexistent(String, Box<ExampleChoiceIn>),
        }

        impl super::super::Serialize for ExampleChoiceOut {
            fn size(&self) -> usize {
                match *self {
                    ExampleChoiceOut::RequiredToRequired(ref payload) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(0, payload_size, false) +
                            payload_size
                    }
                    ExampleChoiceOut::RequiredToAsymmetric(ref payload) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(1, payload_size, false) +
                            payload_size
                    }
                    ExampleChoiceOut::AsymmetricToRequired(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(5, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::AsymmetricToAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(6, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::AsymmetricToOptionalHandled(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(7, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::AsymmetricToOptionalFallback(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(8, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::AsymmetricToNonexistent(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(9, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::OptionalToRequired(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(10, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::OptionalToAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(11, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::OptionalToOptionalHandled(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(12, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::OptionalToOptionalFallback(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(13, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::OptionalToNonexistent(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(14, payload_size, false) +
                            payload_size +
                            fallback.size()
                    }
                }
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                match *self {
                    ExampleChoiceOut::RequiredToRequired(ref payload) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 0, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        Ok(())
                    }
                    ExampleChoiceOut::RequiredToAsymmetric(ref payload) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 1, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        Ok(())
                    }
                    ExampleChoiceOut::AsymmetricToRequired(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 5, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::AsymmetricToAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 6, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::AsymmetricToOptionalHandled(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 7, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::AsymmetricToOptionalFallback(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 8, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::AsymmetricToNonexistent(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 9, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::OptionalToRequired(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 10, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::OptionalToAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 11, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::OptionalToOptionalHandled(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 12, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::OptionalToOptionalFallback(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 13, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::OptionalToNonexistent(ref payload, ref fallback) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 14, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                }
            }
        }

        impl super::super::Deserialize for ExampleChoiceIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                loop {
                    let (index, size) = super::super::deserialize_field_header(&mut *reader)?;

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size as u64);

                    match index {
                        0 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::RequiredToRequired(payload));
                        }
                        1 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::RequiredToAsymmetric(payload));
                        }
                        5 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::AsymmetricToRequired(payload));
                        }
                        6 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::AsymmetricToAsymmetric(payload));
                        }
                        7 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::AsymmetricToOptionalHandled(payload));
                        }
                        8 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::AsymmetricToOptionalFallback(payload));
                        }
                        9 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::AsymmetricToNonexistent(payload));
                        }
                        10 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::OptionalToRequired(payload, fallback));
                        }
                        11 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::OptionalToAsymmetric(payload, fallback));
                        }
                        12 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::OptionalToOptionalHandled(payload, \
                                fallback));
                        }
                        13 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::OptionalToOptionalFallback(payload, \
                                fallback));
                        }
                        14 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::OptionalToNonexistent(payload, fallback));
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }
            }
        }

        impl From<ExampleChoiceOut> for ExampleChoiceIn {
            fn from(message: ExampleChoiceOut) -> Self {
                match message {
                    ExampleChoiceOut::RequiredToRequired(payload) => \
                        ExampleChoiceIn::RequiredToRequired(payload.into()),
                    ExampleChoiceOut::RequiredToAsymmetric(payload) => \
                        ExampleChoiceIn::RequiredToAsymmetric(payload.into()),
                    ExampleChoiceOut::AsymmetricToRequired(payload, fallback) => \
                        ExampleChoiceIn::AsymmetricToRequired(payload.into()),
                    ExampleChoiceOut::AsymmetricToAsymmetric(payload, fallback) => \
                        ExampleChoiceIn::AsymmetricToAsymmetric(payload.into()),
                    ExampleChoiceOut::AsymmetricToOptionalHandled(payload, fallback) => \
                        ExampleChoiceIn::AsymmetricToOptionalHandled(payload.into()),
                    ExampleChoiceOut::AsymmetricToOptionalFallback(payload, fallback) => \
                        ExampleChoiceIn::AsymmetricToOptionalFallback(payload.into()),
                    ExampleChoiceOut::AsymmetricToNonexistent(payload, fallback) => \
                        ExampleChoiceIn::AsymmetricToNonexistent(payload.into()),
                    ExampleChoiceOut::OptionalToRequired(payload, fallback) => \
                        ExampleChoiceIn::OptionalToRequired(payload.into(), \
                        Box::new((*fallback).into())),
                    ExampleChoiceOut::OptionalToAsymmetric(payload, fallback) => \
                        ExampleChoiceIn::OptionalToAsymmetric(payload.into(), \
                        Box::new((*fallback).into())),
                    ExampleChoiceOut::OptionalToOptionalHandled(payload, fallback) => \
                        ExampleChoiceIn::OptionalToOptionalHandled(payload.into(), \
                        Box::new((*fallback).into())),
                    ExampleChoiceOut::OptionalToOptionalFallback(payload, fallback) => \
                        ExampleChoiceIn::OptionalToOptionalFallback(payload.into(), \
                        Box::new((*fallback).into())),
                    ExampleChoiceOut::OptionalToNonexistent(payload, fallback) => \
                        ExampleChoiceIn::OptionalToNonexistent(payload.into(), \
                        Box::new((*fallback).into())),
                }
            }
        }
    }

    pub mod main {
        #[derive(Clone, Debug)]
        pub struct SingletonStructOut {
            pub x: String,
        }

        #[derive(Clone, Debug)]
        pub struct SingletonStructIn {
            pub x: String,
        }

        impl super::super::Serialize for SingletonStructOut {
            fn size(&self) -> usize {
                ({
                    let payload = &self.x;
                    let payload_size = payload.len();
                    super::super::field_header_size(0, payload_size, false) + payload_size
                })
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                {
                    let payload = &self.x;
                    let payload_size = payload.len();
                    super::super::serialize_field_header(writer, 0, payload_size, false)?;
                    writer.write_all(payload.as_bytes())?;
                }

                Ok(())
            }
        }

        impl super::super::Deserialize for SingletonStructIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                let mut x: Option<String> = None;

                loop {
                    let (index, size) = match super::super::deserialize_field_header(&mut \
                        *reader) {
                        Ok(header) => header,
                        Err(err) => {
                            if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                break;
                            }

                            return Err(err);
                        }
                    };

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size as u64);

                    match index {
                        0 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            x.get_or_insert(payload);
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }

                if x.is_none() {
                    return Err(::std::io::Error::new(
                        ::std::io::ErrorKind::InvalidData,
                        \"Struct missing one or more field(s).\",
                    ));
                }

                Ok(SingletonStructIn {
                    x: x.unwrap(),
                })
            }
        }

        impl From<SingletonStructOut> for SingletonStructIn {
            fn from(message: SingletonStructOut) -> Self {
                SingletonStructIn {
                    x: message.x.into(),
                }
            }
        }

        #[derive(Clone, Debug)]
        pub enum SingletonChoiceOut {
            X(String),
        }

        #[derive(Clone, Debug)]
        pub enum SingletonChoiceIn {
            X(String),
        }

        impl super::super::Serialize for SingletonChoiceOut {
            fn size(&self) -> usize {
                match *self {
                    SingletonChoiceOut::X(ref payload) => {
                        let payload_size = payload.len();
                        super::super::field_header_size(0, payload_size, false) +
                            payload_size
                    }
                }
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                match *self {
                    SingletonChoiceOut::X(ref payload) => {
                        let payload_size = payload.len();
                        super::super::serialize_field_header(writer, 0, payload_size, false)?;
                        writer.write_all(payload.as_bytes())?;
                        Ok(())
                    }
                }
            }
        }

        impl super::super::Deserialize for SingletonChoiceIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                loop {
                    let (index, size) = super::super::deserialize_field_header(&mut *reader)?;

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size as u64);

                    match index {
                        0 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(SingletonChoiceIn::X(payload));
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }
            }
        }

        impl From<SingletonChoiceOut> for SingletonChoiceIn {
            fn from(message: SingletonChoiceOut) -> Self {
                match message {
                    SingletonChoiceOut::X(payload) => SingletonChoiceIn::X(payload.into()),
                }
            }
        }
    }
}
",
        );
    }
}
