use {
    crate::{
        identifier::Identifier,
        schema::{self, relativize_namespace},
    },
    std::{
        collections::BTreeMap,
        fmt::{self, Write},
        path::PathBuf,
    },
};

// The string to be used for each indentation level.
const INDENTATION: &str = "    ";

// The generated types will derive these traits.
const TRAITS_TO_DERIVE: &[&str] = &["Clone", "Debug"];

// This is the full list of Rust 2018 keywords, both in use and reserved.
const RUST_KEYWORDS: &[&str] = &[
    "Self", "abstract", "as", "async", "await", "become", "box", "break", "const", "continue",
    "crate", "do", "dyn", "else", "enum", "extern", "false", "final", "fn", "for", "if", "impl",
    "in", "let", "loop", "macro", "match", "mod", "move", "mut", "override", "priv", "pub", "ref",
    "return", "self", "static", "struct", "super", "trait", "true", "try", "type", "typeof",
    "unsafe", "unsized", "use", "virtual", "where", "while", "yield",
];

// This struct represents a tree of schemas organized in a module hierarchy.
#[derive(Clone, Debug)]
struct Module {
    children: BTreeMap<Identifier, Module>,
    schema: schema::Schema,
}

// This enum represents a case convention for the `write_identifier` function below.
#[derive(Clone, Copy, Debug, Eq, PartialEq)]
enum CaseConvention {
    Pascal,
    Snake,
}

use CaseConvention::{Pascal, Snake};

// This enum is used to distinguish between the ingress and egress versions of a type.
#[derive(Clone, Copy, Debug, Eq, PartialEq)]
enum Direction {
    Out,
    In,
}

use Direction::{In, Out};

// Generate Rust code from a schema and its transitive dependencies.
#[allow(clippy::too_many_lines)]
pub fn generate(
    typical_version: &str,
    schemas: &BTreeMap<schema::Namespace, (schema::Schema, PathBuf, String)>,
) -> String {
    // Construct a tree of modules and schemas. We start with an empty tree.
    let mut tree = Module {
        children: BTreeMap::new(),
        schema: schema::Schema {
            imports: BTreeMap::new(),
            declarations: BTreeMap::new(),
        },
    };

    // Populate the tree with all the schemas.
    for (namespace, (schema, _, _)) in schemas {
        insert_schema(&mut tree, namespace, schema);
    }

    // Write the code.
    let mut buffer = String::new();

    if !tree.children.is_empty() || !tree.schema.declarations.is_empty() {
        // The `unwrap` is safe because the `std::fmt::Write` impl for `String` is infallible.
        // For functions that take abstract parameters which implement `std::io::Read` or
        // `std::io::Write`, it's idiomatic to consume the reader or writer rather than borrowing
        // them (https://rust-lang.github.io/api-guidelines/interoperability.html
        // #generic-readerwriter-functions-take-r-read-and-w-write-by-value-c-rw-value). However,
        // we borrow them anyway since that allows us to pass the reference (after reborrowing)
        // to recursive calls, rather than building up and following chains of references.
        writeln!(
            &mut buffer,
            "\
// This file was automatically generated by Typical {}.
// Visit https://github.com/stepchowfun/typical for more information.

#![allow(clippy::all, clippy::pedantic, clippy::nursery, warnings)]

#[rustfmt::skip]
use std::{{
    cmp::min,
    io::{{self, BufRead, Error, ErrorKind, Write}},
    mem::transmute,
}};

#[rustfmt::skip]
fn zigzag_encode(value: i64) -> u64 {{
    unsafe {{
        transmute::<i64, u64>(value >> 63) ^ transmute::<i64, u64>(value << 1)
    }}
}}

#[rustfmt::skip]
fn zigzag_decode(value: u64) -> i64 {{
    unsafe {{
        transmute::<u64, i64>(value >> 1) ^ -transmute::<u64, i64>(value & 1)
    }}
}}

#[rustfmt::skip]
fn varint_size_from_value(value: u64) -> u64 {{
    let mut size = 1_u64;
    let mut upper_bound_exclusive = 0_u64;

    while size < 9 {{
        upper_bound_exclusive += 1_u64 << (size * 7);

        if value < upper_bound_exclusive {{
            break;
        }}

        size += 1;
    }}

    size
}}

#[rustfmt::skip]
fn varint_size_from_first_byte(first_byte: u8) -> u32 {{
    first_byte.trailing_zeros() + 1
}}

#[rustfmt::skip]
fn serialize_varint<T: Write>(value: u64, writer: &mut T) -> io::Result<()> {{
    let size = varint_size_from_value(value);
    let size_minus_one = size - 1;

    let mut x = value;
    for i in 1..size {{
        x -= 1_u64 << (i * 7);
    }}

    writer.write_all(&[((x << size) | (1_u64 << size_minus_one)) as u8])?;
    x >>= 8_u64.saturating_sub(size);

    for _ in 0..size_minus_one {{
        writer.write_all(&[x as u8])?;
        x >>= 8;
    }}

    Ok(())
}}

#[rustfmt::skip]
fn deserialize_varint<T: BufRead>(reader: &mut T) -> io::Result<u64> {{
    let mut buffer = [0; 9];
    reader.read_exact(&mut buffer[0..1])?;
    let first_byte = buffer[0];
    let size = varint_size_from_first_byte(first_byte) as usize;

    reader.read_exact(&mut buffer[1..size])?;

    let mut x = u64::from(first_byte) >> size;
    let mut bits_read = 8_usize.saturating_sub(size);

    for byte in buffer.iter().skip(1) {{
        x |= u64::from(*byte) << bits_read;
        bits_read += 8;
    }}

    for i in 1..size {{
        x = x
            .checked_add(1_u64 << (i * 7))
            .ok_or_else(|| Error::new(ErrorKind::InvalidData, \"Error decoding varint.\"))?;
    }}

    Ok(x)
}}

#[rustfmt::skip]
fn non_varint_field_header_size(index: u64, value_size: u64) -> u64 {{
    match value_size {{
        0 => varint_size_from_value((index << 2) | 0b00),
        8 => varint_size_from_value((index << 2) | 0b01),
        size => varint_size_from_value((index << 2) | 0b10) + varint_size_from_value(size),
    }}
}}

#[rustfmt::skip]
fn varint_field_header_size(index: u64) -> u64 {{
    varint_size_from_value((index << 2) | 0b11)
}}

#[rustfmt::skip]
fn serialize_non_varint_field_header<T: Write>(
    writer: &mut T,
    index: u64,
    payload_size: u64,
) -> io::Result<()> {{
    match payload_size {{
        0 => serialize_varint((index << 2) | 0b00, writer),
        8 => serialize_varint((index << 2) | 0b01, writer),
        size => {{
            serialize_varint((index << 2) | 0b10, writer)?;
            serialize_varint(size, writer)
        }}
    }}
}}

#[rustfmt::skip]
fn serialize_varint_field_header<T: Write>(writer: &mut T, index: u64) -> io::Result<()> {{
    serialize_varint((index << 2) | 0b11, writer)
}}

#[rustfmt::skip]
fn deserialize_field_header<T: BufRead>(reader: &mut T) -> io::Result<(u64, u64)> {{
    let tag = deserialize_varint(&mut *reader)?;

    let index = tag >> 2;

    let size = match tag & 0b11 {{
        0b00 => 0,
        0b01 => 8,
        0b10 => deserialize_varint(&mut *reader)?,
        0b11 => {{
            let buffer = (&mut *reader).fill_buf()?;

            if buffer.is_empty() {{
                return Err(Error::new(ErrorKind::UnexpectedEof, \"Error decoding field.\"));
            }}

            u64::from(varint_size_from_first_byte(buffer[0]))
        }}
        _ => panic!(),
    }};

    Ok((index, size))
}}

#[rustfmt::skip]
fn skip<T: BufRead>(reader: &mut T, mut amount: usize) -> io::Result<()> {{
    while amount > 0 {{
        let buffer = reader.fill_buf()?;
        let num_bytes_to_consume = min(buffer.len(), amount);
        reader.consume(num_bytes_to_consume);
        amount -= num_bytes_to_consume;
    }}

    Ok(())
}}

#[rustfmt::skip]
pub trait Serialize {{
    fn size(&self) -> u64;

    fn serialize<T: Write>(&self, writer: &mut T) -> io::Result<()>;
}}

#[rustfmt::skip]
pub trait Deserialize {{
    fn deserialize<T>(reader: &mut T) -> io::Result<Self>
    where
        Self: Sized,
        T: BufRead;
}}",
            typical_version,
        )
        .unwrap();

        // The `unwrap` is safe because the `std::fmt::Write` impl for `String` is infallible.
        writeln!(&mut buffer).unwrap();

        // The `unwrap` is safe because the `std::fmt::Write` impl for `String` is infallible.
        write_module_contents(
            &mut buffer,
            0,
            &schema::Namespace { components: vec![] },
            &tree.children,
            &tree.schema,
        )
        .unwrap();
    }

    buffer
}

// Insert a schema into a module.
fn insert_schema(module: &mut Module, namespace: &schema::Namespace, schema: &schema::Schema) {
    let mut iter = namespace.components.iter();

    if let Some(head) = iter.next() {
        if let Some(child) = module.children.get_mut(head) {
            insert_schema(
                child,
                &schema::Namespace {
                    components: iter.cloned().collect(),
                },
                schema,
            );
        } else {
            let mut child = Module {
                children: BTreeMap::new(),
                schema: schema::Schema {
                    imports: BTreeMap::new(),
                    declarations: BTreeMap::new(),
                },
            };

            insert_schema(
                &mut child,
                &schema::Namespace {
                    components: iter.cloned().collect(),
                },
                schema,
            );

            module.children.insert(head.clone(), child);
        }
    } else {
        module.schema = schema.clone();
    }
}

// Write a module, including a trailing line break.
fn write_module<T: Write>(
    buffer: &mut T,
    indentation: usize,
    namespace: &schema::Namespace,
    name: &Identifier,
    module: &Module,
) -> Result<(), fmt::Error> {
    if indentation == 0 {
        write_indentation(buffer, indentation)?;
        writeln!(buffer, "#[rustfmt::skip]")?;
    }

    write_indentation(buffer, indentation)?;
    write!(buffer, "pub mod ")?;
    write_identifier(buffer, name, Snake, None)?;
    writeln!(buffer, " {{")?;

    let mut new_namespace = namespace.clone();
    new_namespace.components.push(name.clone());

    write_module_contents(
        buffer,
        indentation + 1,
        &new_namespace,
        &module.children,
        &module.schema,
    )?;

    write_indentation(buffer, indentation)?;
    writeln!(buffer, "}}")?;

    Ok(())
}

// Write the contents of a module, including a trailing line break if there was anything to render.
fn write_module_contents<T: Write>(
    buffer: &mut T,
    indentation: usize,
    namespace: &schema::Namespace,
    children: &BTreeMap<Identifier, Module>,
    schema: &schema::Schema,
) -> Result<(), fmt::Error> {
    let schema_empty = schema.declarations.is_empty();

    for (i, (child_name, child)) in children.iter().enumerate() {
        write_module(buffer, indentation, namespace, child_name, child)?;

        if i < children.len() - 1 || !schema_empty {
            writeln!(buffer)?;
        }
    }

    write_schema(buffer, indentation, namespace, schema)?;

    Ok(())
}

// Write a schema, including a trailing line break if there was anything to render.
#[allow(clippy::too_many_lines)]
fn write_schema<T: Write>(
    buffer: &mut T,
    indentation: usize,
    namespace: &schema::Namespace,
    schema: &schema::Schema,
) -> Result<(), fmt::Error> {
    // Construct a map from import name to namespace.
    let mut imports = BTreeMap::new();
    for (name, import) in &schema.imports {
        // The unwrap is safe due to [ref:namespace_populated].
        imports.insert(name.clone(), import.namespace.clone().unwrap());
    }

    // Write the declarations.
    let mut iter = schema.declarations.iter().peekable();
    while let Some((name, declaration)) = iter.next() {
        match &declaration.variant {
            schema::DeclarationVariant::Struct(fields) => {
                write_struct(buffer, indentation, &imports, namespace, name, fields, Out)?;

                writeln!(buffer)?;

                write_struct(buffer, indentation, &imports, namespace, name, fields, In)?;

                writeln!(buffer)?;

                write_indentation(buffer, indentation)?;
                write!(buffer, "impl ")?;
                write_supers(buffer, indentation)?;
                write!(buffer, "Serialize for ")?;
                write_identifier(buffer, name, Pascal, Some(Out))?;
                writeln!(buffer, " {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "fn size(&self) -> u64 {{")?;
                if fields.is_empty() {
                    write_indentation(buffer, indentation + 2)?;
                    writeln!(buffer, "0")?;
                }
                for (i, field) in fields.iter().enumerate() {
                    let is_first = i == 0;
                    let is_last = i == fields.len() - 1;
                    if is_first {
                        write_indentation(buffer, indentation + 2)?;
                    }
                    match field.rule {
                        schema::Rule::Optional => {
                            write!(buffer, "self.")?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            writeln!(buffer, ".as_ref().map_or(0, |payload| {{")?;
                        }
                        schema::Rule::Required | schema::Rule::Asymmetric => {
                            writeln!(buffer, "({{")?;
                            write_indentation(buffer, indentation + 3)?;
                            write!(buffer, "let payload = &self.")?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            writeln!(buffer, ";")?;
                        }
                    }
                    write_indentation(buffer, indentation + 3)?;
                    write!(buffer, "let payload_size = ")?;
                    write_size_calculation_invocation(buffer, indentation, &field.r#type)?;
                    writeln!(buffer, ";")?;
                    write_indentation(buffer, indentation + 3)?;
                    write_supers(buffer, indentation)?;
                    if varint_encoded(&field.r#type) {
                        writeln!(
                            buffer,
                            "varint_field_header_size({}) + payload_size",
                            field.index,
                        )?;
                    } else {
                        writeln!(
                            buffer,
                            "non_varint_field_header_size({}, payload_size) + payload_size",
                            field.index,
                        )?;
                    }
                    write_indentation(buffer, indentation + 2)?;
                    write!(buffer, "}})")?;
                    if is_last {
                        writeln!(buffer)?;
                    } else {
                        write!(buffer, " + ")?;
                    }
                }
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}}")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(
                    buffer,
                    "fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> \
                        ::std::io::Result<()> {{",
                )?;
                for field in fields {
                    match field.rule {
                        schema::Rule::Optional => {
                            write_indentation(buffer, indentation + 2)?;
                            write!(buffer, "if let Some(payload) = &self.")?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            writeln!(buffer, " {{")?;
                        }
                        schema::Rule::Required | schema::Rule::Asymmetric => {
                            write_indentation(buffer, indentation + 2)?;
                            writeln!(buffer, "{{")?;
                            write_indentation(buffer, indentation + 3)?;
                            write!(buffer, "let payload = &self.")?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            writeln!(buffer, ";")?;
                        }
                    }
                    write_indentation(buffer, indentation + 3)?;
                    write!(buffer, "let payload_size = ")?;
                    write_size_calculation_invocation(buffer, indentation, &field.r#type)?;
                    writeln!(buffer, ";")?;
                    write_indentation(buffer, indentation + 3)?;
                    write_supers(buffer, indentation)?;
                    if varint_encoded(&field.r#type) {
                        write!(buffer, "serialize_varint_field_header")?;
                        writeln!(buffer, "(writer, {})?;", field.index)?;
                    } else {
                        write!(buffer, "serialize_non_varint_field_header")?;
                        writeln!(buffer, "(writer, {}, payload_size)?;", field.index)?;
                    }
                    write_serialization_invocation(
                        buffer,
                        indentation + 3,
                        indentation,
                        &field.r#type,
                    )?;
                    write_indentation(buffer, indentation + 2)?;
                    writeln!(buffer, "}}")?;
                    writeln!(buffer)?;
                }
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "Ok(())")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")?;

                writeln!(buffer)?;

                write_indentation(buffer, indentation)?;
                write!(buffer, "impl ")?;
                write_supers(buffer, indentation)?;
                write!(buffer, "Deserialize for ")?;
                write_identifier(buffer, name, Pascal, Some(In))?;
                writeln!(buffer, " {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(
                    buffer,
                    "fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>",
                )?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "where")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "Self: Sized,")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "T: ::std::io::BufRead,")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "{{")?;
                if !fields.is_empty() {
                    for field in fields {
                        write_indentation(buffer, indentation + 2)?;
                        write!(buffer, "let mut ")?;
                        write_identifier(buffer, &field.name, Snake, None)?;
                        write!(buffer, ": Option<")?;
                        write_type(buffer, &imports, namespace, &field.r#type, In)?;
                        writeln!(buffer, "> = None;")?;
                    }
                    writeln!(buffer)?;
                }
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "loop {{")?;
                write_indentation(buffer, indentation + 3)?;
                write!(buffer, "let (index, size) = match ")?;
                write_supers(buffer, indentation)?;
                writeln!(buffer, "deserialize_field_header(&mut *reader) {{")?;
                write_indentation(buffer, indentation + 4)?;
                writeln!(buffer, "Ok(header) => header,")?;
                write_indentation(buffer, indentation + 4)?;
                writeln!(buffer, "Err(err) => {{")?;
                write_indentation(buffer, indentation + 5)?;
                writeln!(
                    buffer,
                    "if let std::io::ErrorKind::UnexpectedEof = err.kind() {{",
                )?;
                write_indentation(buffer, indentation + 6)?;
                writeln!(buffer, "break;")?;
                write_indentation(buffer, indentation + 5)?;
                writeln!(buffer, "}}")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 5)?;
                writeln!(buffer, "return Err(err);")?;
                write_indentation(buffer, indentation + 4)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(buffer, "}};")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(
                    buffer,
                    "let mut sub_reader = ::std::io::Read::take(&mut *reader, size);",
                )?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(buffer, "match index {{")?;
                for field in fields {
                    write_indentation(buffer, indentation + 4)?;
                    writeln!(buffer, "{} => {{", field.index)?;
                    write_deserialization_invocation(
                        buffer,
                        indentation + 5,
                        indentation,
                        &imports,
                        namespace,
                        &field.r#type,
                    )?;
                    writeln!(buffer)?;
                    write_indentation(buffer, indentation + 5)?;
                    write_identifier(buffer, &field.name, Snake, None)?;
                    writeln!(buffer, ".get_or_insert(payload);")?;
                    write_indentation(buffer, indentation + 4)?;
                    writeln!(buffer, "}}")?;
                }
                write_indentation(buffer, indentation + 4)?;
                writeln!(buffer, "_ => {{")?;
                write_indentation(buffer, indentation + 5)?;
                write_supers(buffer, indentation)?;
                writeln!(buffer, "skip(&mut sub_reader, size as usize)?;")?;
                write_indentation(buffer, indentation + 4)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "}}")?;
                writeln!(buffer)?;
                if fields.iter().any(|field| match field.rule {
                    schema::Rule::Optional | schema::Rule::Asymmetric => false,
                    schema::Rule::Required => true,
                }) {
                    write_indentation(buffer, indentation + 2)?;
                    write!(buffer, "if ")?;
                    let mut first = true;
                    for field in fields {
                        match field.rule {
                            schema::Rule::Optional | schema::Rule::Asymmetric => {}
                            schema::Rule::Required => {
                                if first {
                                    first = false;
                                } else {
                                    write!(buffer, " || ")?;
                                }
                                write_identifier(buffer, &field.name, Snake, None)?;
                                write!(buffer, ".is_none()")?;
                            }
                        }
                    }
                    writeln!(buffer, " {{")?;
                    write_indentation(buffer, indentation + 3)?;
                    writeln!(buffer, "return Err(::std::io::Error::new(")?;
                    write_indentation(buffer, indentation + 4)?;
                    writeln!(buffer, "::std::io::ErrorKind::InvalidData,")?;
                    write_indentation(buffer, indentation + 4)?;
                    writeln!(buffer, "\"Struct missing one or more field(s).\",")?;
                    write_indentation(buffer, indentation + 3)?;
                    writeln!(buffer, "));")?;
                    write_indentation(buffer, indentation + 2)?;
                    writeln!(buffer, "}}")?;
                    writeln!(buffer)?;
                }
                write_indentation(buffer, indentation + 2)?;
                write!(buffer, "Ok(")?;
                write_identifier(buffer, name, Pascal, Some(In))?;
                writeln!(buffer, " {{")?;
                for field in fields {
                    write_indentation(buffer, indentation + 3)?;
                    write_identifier(buffer, &field.name, Snake, None)?;
                    match field.rule {
                        schema::Rule::Optional | schema::Rule::Asymmetric => {}
                        schema::Rule::Required => {
                            write!(buffer, ": ")?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            write!(buffer, ".unwrap()")?;
                        }
                    }
                    writeln!(buffer, ",")?;
                }
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "}})")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")?;

                writeln!(buffer)?;

                write_indentation(buffer, indentation)?;
                write!(buffer, "impl From<")?;
                write_identifier(buffer, name, Pascal, Some(Out))?;
                write!(buffer, "> for ")?;
                write_identifier(buffer, name, Pascal, Some(In))?;
                writeln!(buffer, " {{")?;
                write_indentation(buffer, indentation + 1)?;
                write!(buffer, "fn from(message: ")?;
                write_identifier(buffer, name, Pascal, Some(Out))?;
                writeln!(buffer, ") -> Self {{")?;
                write_indentation(buffer, indentation + 2)?;
                write_identifier(buffer, name, Pascal, Some(In))?;
                writeln!(buffer, " {{")?;
                for field in fields {
                    match field.rule {
                        schema::Rule::Optional => {
                            write_indentation(buffer, indentation + 3)?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            write!(buffer, ": message.")?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            writeln!(buffer, ".map(|payload| payload.into()),")?;
                        }
                        schema::Rule::Required => {
                            write_indentation(buffer, indentation + 3)?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            write!(buffer, ": message.")?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            writeln!(buffer, ".into(),")?;
                        }
                        schema::Rule::Asymmetric => {
                            write_indentation(buffer, indentation + 3)?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            write!(buffer, ": Some(message.")?;
                            write_identifier(buffer, &field.name, Snake, None)?;
                            writeln!(buffer, ".into()),")?;
                        }
                    }
                }
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")?;
            }
            schema::DeclarationVariant::Choice(fields) => {
                write_choice(buffer, indentation, &imports, namespace, name, fields, Out)?;

                writeln!(buffer)?;

                write_choice(buffer, indentation, &imports, namespace, name, fields, In)?;

                writeln!(buffer)?;

                write_indentation(buffer, indentation)?;
                write!(buffer, "impl ")?;
                write_supers(buffer, indentation)?;
                write!(buffer, "Serialize for ")?;
                write_identifier(buffer, name, Pascal, Some(Out))?;
                writeln!(buffer, " {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "fn size(&self) -> u64 {{")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "match *self {{")?;
                for field in fields {
                    write_indentation(buffer, indentation + 3)?;
                    write_identifier(buffer, name, Pascal, Some(Out))?;
                    write!(buffer, "::")?;
                    write_identifier(buffer, &field.name, Pascal, None)?;
                    match field.rule {
                        schema::Rule::Optional | schema::Rule::Asymmetric => {
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                writeln!(buffer, "(ref fallback) => {{")?;
                            } else {
                                writeln!(buffer, "(ref payload, ref fallback) => {{")?;
                            }
                        }
                        schema::Rule::Required => {
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                writeln!(buffer, " => {{")?;
                            } else {
                                writeln!(buffer, "(ref payload) => {{")?;
                            }
                        }
                    }
                    write_indentation(buffer, indentation + 4)?;
                    write!(buffer, "let payload_size = ")?;
                    write_size_calculation_invocation(buffer, indentation, &field.r#type)?;
                    writeln!(buffer, ";")?;
                    write_indentation(buffer, indentation + 4)?;
                    write_supers(buffer, indentation)?;
                    if varint_encoded(&field.r#type) {
                        writeln!(buffer, "varint_field_header_size({}) +", field.index)?;
                    } else {
                        writeln!(
                            buffer,
                            "non_varint_field_header_size({}, payload_size) +",
                            field.index,
                        )?;
                    }
                    write_indentation(buffer, indentation + 5)?;
                    write!(buffer, "payload_size")?;
                    match field.rule {
                        schema::Rule::Optional | schema::Rule::Asymmetric => {
                            writeln!(buffer, " +")?;
                            write_indentation(buffer, indentation + 5)?;
                            writeln!(buffer, "fallback.size()")?;
                        }
                        schema::Rule::Required => {
                            writeln!(buffer)?;
                        }
                    }
                    write_indentation(buffer, indentation + 3)?;
                    writeln!(buffer, "}}")?;
                }
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}}")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(
                    buffer,
                    "fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> \
                        ::std::io::Result<()> {{",
                )?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "match *self {{")?; // [ref:empty_enum_ref_match]
                for field in fields {
                    write_indentation(buffer, indentation + 3)?;
                    write_identifier(buffer, name, Pascal, Some(Out))?;
                    write!(buffer, "::")?;
                    write_identifier(buffer, &field.name, Pascal, None)?;
                    match field.rule {
                        schema::Rule::Optional | schema::Rule::Asymmetric => {
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                writeln!(buffer, "(ref fallback) => {{")?;
                            } else {
                                writeln!(buffer, "(ref payload, ref fallback) => {{")?;
                            }
                        }
                        schema::Rule::Required => {
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                writeln!(buffer, " => {{")?;
                            } else {
                                writeln!(buffer, "(ref payload) => {{")?;
                            }
                        }
                    }
                    write_indentation(buffer, indentation + 4)?;
                    write_supers(buffer, indentation)?;
                    if varint_encoded(&field.r#type) {
                        writeln!(
                            buffer,
                            "serialize_varint_field_header(writer, {})?;",
                            field.index,
                        )?;
                    } else {
                        write!(
                            buffer,
                            "serialize_non_varint_field_header(writer, {}, ",
                            field.index,
                        )?;
                        write_size_calculation_invocation(buffer, indentation, &field.r#type)?;
                        writeln!(buffer, ")?;")?;
                    }
                    write_serialization_invocation(
                        buffer,
                        indentation + 4,
                        indentation,
                        &field.r#type,
                    )?;
                    match field.rule {
                        schema::Rule::Optional | schema::Rule::Asymmetric => {
                            write_indentation(buffer, indentation + 4)?;
                            writeln!(buffer, "fallback.serialize(writer)")?;
                        }
                        schema::Rule::Required => {
                            write_indentation(buffer, indentation + 4)?;
                            writeln!(buffer, "Ok(())")?;
                        }
                    }
                    write_indentation(buffer, indentation + 3)?;
                    writeln!(buffer, "}}")?;
                }
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")?;

                writeln!(buffer)?;

                write_indentation(buffer, indentation)?;
                write!(buffer, "impl ")?;
                write_supers(buffer, indentation)?;
                write!(buffer, "Deserialize for ")?;
                write_identifier(buffer, name, Pascal, Some(In))?;
                writeln!(buffer, " {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(
                    buffer,
                    "fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>",
                )?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "where")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "Self: Sized,")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "T: ::std::io::BufRead,")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "{{")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "loop {{")?;
                write_indentation(buffer, indentation + 3)?;
                write!(buffer, "let (index, size) = ")?;
                write_supers(buffer, indentation)?;
                writeln!(buffer, "deserialize_field_header(&mut *reader)?;")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(
                    buffer,
                    "let mut sub_reader = ::std::io::Read::take(&mut *reader, size);",
                )?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(buffer, "match index {{")?;
                for field in fields {
                    write_indentation(buffer, indentation + 4)?;
                    writeln!(buffer, "{} => {{", field.index)?;
                    write_deserialization_invocation(
                        buffer,
                        indentation + 5,
                        indentation,
                        &imports,
                        namespace,
                        &field.r#type,
                    )?;
                    match field.rule {
                        schema::Rule::Optional => {
                            write_indentation(buffer, indentation + 5)?;
                            write!(buffer, "let fallback = Box::new(<")?;
                            write_identifier(buffer, name, Pascal, Some(In))?;
                            write!(buffer, " as ")?;
                            write_supers(buffer, indentation)?;
                            writeln!(buffer, "Deserialize>::deserialize(&mut *reader)?);")?;
                            write_indentation(buffer, indentation + 5)?;
                            write!(buffer, "return Ok(")?;
                            write_identifier(buffer, name, Pascal, Some(In))?;
                            write!(buffer, "::")?;
                            write_identifier(buffer, &field.name, Pascal, None)?;
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                writeln!(buffer, "(fallback));")?;
                            } else {
                                writeln!(buffer, "(payload, fallback));")?;
                            }
                        }
                        schema::Rule::Required | schema::Rule::Asymmetric => {
                            write_indentation(buffer, indentation + 5)?;
                            write!(buffer, "return Ok(")?;
                            write_identifier(buffer, name, Pascal, Some(In))?;
                            write!(buffer, "::")?;
                            write_identifier(buffer, &field.name, Pascal, None)?;
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                writeln!(buffer, ");")?;
                            } else {
                                writeln!(buffer, "(payload));")?;
                            }
                        }
                    }
                    write_indentation(buffer, indentation + 4)?;
                    writeln!(buffer, "}}")?;
                }
                write_indentation(buffer, indentation + 4)?;
                writeln!(buffer, "_ => {{")?;
                write_indentation(buffer, indentation + 5)?;
                write_supers(buffer, indentation)?;
                writeln!(buffer, "skip(&mut sub_reader, size as usize)?;")?;
                write_indentation(buffer, indentation + 4)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")?;

                writeln!(buffer)?;

                write_indentation(buffer, indentation)?;
                write!(buffer, "impl From<")?;
                write_identifier(buffer, name, Pascal, Some(Out))?;
                write!(buffer, "> for ")?;
                write_identifier(buffer, name, Pascal, Some(In))?;
                writeln!(buffer, " {{")?;
                write_indentation(buffer, indentation + 1)?;
                write!(buffer, "fn from(message: ")?;
                write_identifier(buffer, name, Pascal, Some(Out))?;
                writeln!(buffer, ") -> Self {{")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "match message {{")?;
                for field in fields {
                    write_indentation(buffer, indentation + 3)?;
                    write_identifier(buffer, name, Pascal, Some(Out))?;
                    write!(buffer, "::")?;
                    write_identifier(buffer, &field.name, Pascal, None)?;
                    match field.rule {
                        schema::Rule::Optional | schema::Rule::Asymmetric => {
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                write!(buffer, "(fallback) => ")?;
                            } else {
                                write!(buffer, "(payload, fallback) => ")?;
                            }
                        }
                        schema::Rule::Required => {
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                write!(buffer, " => ")?;
                            } else {
                                write!(buffer, "(payload) => ")?;
                            }
                        }
                    }
                    write_identifier(buffer, name, Pascal, Some(In))?;
                    write!(buffer, "::")?;
                    write_identifier(buffer, &field.name, Pascal, None)?;
                    match field.rule {
                        schema::Rule::Optional => {
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                writeln!(buffer, "(Box::new((*fallback).into())),")?;
                            } else {
                                writeln!(
                                    buffer,
                                    "(payload.into(), Box::new((*fallback).into())),",
                                )?;
                            }
                        }
                        schema::Rule::Required | schema::Rule::Asymmetric => {
                            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                                writeln!(buffer, ",")?;
                            } else {
                                writeln!(buffer, "(payload.into()),")?;
                            }
                        }
                    }
                }
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")?;
            }
        }

        if iter.peek().is_some() {
            writeln!(buffer)?;
        }
    }

    Ok(())
}

// Write a struct, including a trailing line break.
fn write_struct<T: Write>(
    buffer: &mut T,
    indentation: usize,
    imports: &BTreeMap<Identifier, schema::Namespace>,
    namespace: &schema::Namespace,
    name: &Identifier,
    fields: &[schema::Field],
    direction: Direction,
) -> Result<(), fmt::Error> {
    write_indentation(buffer, indentation)?;
    writeln!(buffer, "#[derive({})]", TRAITS_TO_DERIVE.join(", "))?;
    write_indentation(buffer, indentation)?;
    write!(buffer, "pub struct ")?;
    write_identifier(buffer, name, Pascal, Some(direction))?;
    writeln!(buffer, " {{")?;

    for field in fields {
        write_indentation(buffer, indentation + 1)?;
        write!(buffer, "pub ")?;
        write_identifier(buffer, &field.name, Snake, None)?;
        write!(buffer, ": ")?;
        match field.rule {
            schema::Rule::Optional => {
                write!(buffer, "Option<")?;
            }
            schema::Rule::Required => {}
            schema::Rule::Asymmetric => match direction {
                Direction::Out => {}
                Direction::In => {
                    write!(buffer, "Option<")?;
                }
            },
        }
        write_type(buffer, imports, namespace, &field.r#type, direction)?;
        match field.rule {
            schema::Rule::Optional => {
                write!(buffer, ">")?;
            }
            schema::Rule::Required => {}
            schema::Rule::Asymmetric => match direction {
                Direction::Out => {}
                Direction::In => {
                    write!(buffer, ">")?;
                }
            },
        }
        writeln!(buffer, ",")?;
    }

    write_indentation(buffer, indentation)?;
    writeln!(buffer, "}}")?;

    Ok(())
}

// Write a choice, including a trailing line break.
fn write_choice<T: Write>(
    buffer: &mut T,
    indentation: usize,
    imports: &BTreeMap<Identifier, schema::Namespace>,
    namespace: &schema::Namespace,
    name: &Identifier,
    fields: &[schema::Field],
    direction: Direction,
) -> Result<(), fmt::Error> {
    write_indentation(buffer, indentation)?;
    writeln!(buffer, "#[derive({})]", TRAITS_TO_DERIVE.join(", "))?;
    write_indentation(buffer, indentation)?;
    write!(buffer, "pub enum ")?;
    write_identifier(buffer, name, Pascal, Some(direction))?;
    writeln!(buffer, " {{")?;

    for field in fields {
        write_indentation(buffer, indentation + 1)?;
        write_identifier(buffer, &field.name, Pascal, None)?;
        let fallback = match field.rule {
            schema::Rule::Optional => true,
            schema::Rule::Required => false,
            schema::Rule::Asymmetric => match direction {
                Direction::Out => true,
                Direction::In => false,
            },
        };
        if fallback {
            if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
                write!(buffer, "(Box<")?;
            } else {
                write!(buffer, "(")?;
                write_type(buffer, imports, namespace, &field.r#type, direction)?;
                write!(buffer, ", Box<")?;
            }
            write_identifier(buffer, name, Pascal, Some(direction))?;
            writeln!(buffer, ">),")?;
        } else if matches!(field.r#type.variant, schema::TypeVariant::Unit) {
            writeln!(buffer, ",")?;
        } else {
            write!(buffer, "(")?;
            write_type(buffer, imports, namespace, &field.r#type, direction)?;
            writeln!(buffer, "),")?;
        }
    }

    write_indentation(buffer, indentation)?;
    writeln!(buffer, "}}")?;

    Ok(())
}

// Write a type.
fn write_type<T: Write>(
    buffer: &mut T,
    imports: &BTreeMap<Identifier, schema::Namespace>,
    namespace: &schema::Namespace,
    r#type: &schema::Type,
    direction: Direction,
) -> Result<(), fmt::Error> {
    match &r#type.variant {
        schema::TypeVariant::Array(inner_type) => {
            write!(buffer, "Vec<")?;
            write_type(buffer, imports, namespace, inner_type, direction)?;
            write!(buffer, ">")?;
        }
        schema::TypeVariant::Bool => {
            write!(buffer, "bool")?;
        }
        schema::TypeVariant::Bytes => {
            write!(buffer, "Vec<u8>")?;
        }
        schema::TypeVariant::Custom(import, name) => {
            let type_namespace = schema::Namespace {
                components: import.as_ref().map_or_else(
                    || namespace.components.clone(),
                    |import| imports[import].components.clone(),
                ),
            };

            let (relative_type_namespace, ancestors) =
                relativize_namespace(&type_namespace, namespace);

            write_supers(buffer, ancestors)?;

            for component in relative_type_namespace.components {
                write_identifier(buffer, &component, Snake, None)?;
                write!(buffer, "::")?;
            }

            write_identifier(buffer, name, Pascal, Some(direction))?;
        }
        schema::TypeVariant::F64 => {
            write!(buffer, "f64")?;
        }
        schema::TypeVariant::S64 => {
            write!(buffer, "i64")?;
        }
        schema::TypeVariant::String => {
            write!(buffer, "String")?;
        }
        schema::TypeVariant::U64 => {
            write!(buffer, "u64")?;
        }
        schema::TypeVariant::Unit => {
            write!(buffer, "()")?;
        }
    }

    Ok(())
}

// Write an identifier with an optional direction suffix in a way that Rust will be happy with.
fn write_identifier<T: Write>(
    buffer: &mut T,
    identifier: &Identifier,
    case: CaseConvention,
    suffix: Option<Direction>,
) -> Result<(), fmt::Error> {
    let identifier_with_suffix = suffix.map_or_else(
        || identifier.clone(),
        |suffix| {
            identifier.join(
                &match suffix {
                    Direction::In => "In",
                    Direction::Out => "Out",
                }
                .into(),
            )
        },
    );

    let converted_identifier = match case {
        CaseConvention::Pascal => identifier_with_suffix.pascal_case(),
        CaseConvention::Snake => identifier_with_suffix.snake_case(),
    };

    if !converted_identifier.starts_with("r#")
        && RUST_KEYWORDS
            .iter()
            .any(|keyword| converted_identifier == *keyword)
    {
        write!(buffer, "r#")?;
    }

    write!(buffer, "{}", converted_identifier)?;

    Ok(())
}

// Write the given level of indentation.
fn write_indentation<T: Write>(buffer: &mut T, indentation: usize) -> Result<(), fmt::Error> {
    for _ in 0..indentation {
        write!(buffer, "{}", INDENTATION)?;
    }

    Ok(())
}

// Write a series of `super::super::...`.
fn write_supers<T: Write>(buffer: &mut T, count: usize) -> Result<(), fmt::Error> {
    for _ in 0..count {
        write!(buffer, "super::")?;
    }

    Ok(())
}

// Write the logic to invoke the size calculation logic for a value.
fn write_size_calculation_invocation<T: Write>(
    buffer: &mut T,
    supers: usize,
    r#type: &schema::Type,
) -> Result<(), fmt::Error> {
    match &r#type.variant {
        schema::TypeVariant::Array(inner_type) => match &inner_type.variant {
            schema::TypeVariant::Array(_)
            | schema::TypeVariant::Bytes
            | schema::TypeVariant::Custom(_, _)
            | schema::TypeVariant::String => {
                write!(
                    buffer,
                    "payload.iter().fold(0_u64, |x, payload| {{ let payload_size = ",
                )?;
                write_size_calculation_invocation(buffer, supers, inner_type)?;
                write!(buffer, "; x + ")?;
                write_supers(buffer, supers)?;
                write!(
                    buffer,
                    "varint_size_from_value(payload_size) + payload_size }})",
                )
            }
            schema::TypeVariant::Bool | schema::TypeVariant::S64 | schema::TypeVariant::U64 => {
                write!(buffer, "payload.iter().fold(0_u64, |x, payload| x + ")?;
                write_size_calculation_invocation(buffer, supers, inner_type)?;
                write!(buffer, ")")
            }
            schema::TypeVariant::F64 => write!(buffer, "8_u64 * (payload.len() as u64)"),
            schema::TypeVariant::Unit => {
                write_supers(buffer, supers)?;
                write!(buffer, "varint_size_from_value(payload.len() as u64)")
            }
        },
        schema::TypeVariant::Bool => write!(buffer, "1_u64"),
        schema::TypeVariant::Bytes | schema::TypeVariant::String => {
            write!(buffer, "payload.len() as u64")
        }
        schema::TypeVariant::Custom(_, _) => write!(buffer, "payload.size()"),
        schema::TypeVariant::F64 => write!(buffer, "8_u64"),
        schema::TypeVariant::S64 => {
            write_supers(buffer, supers)?;
            write!(buffer, "varint_size_from_value(")?;
            write_supers(buffer, supers)?;
            write!(buffer, "zigzag_encode(*payload))")
        }
        schema::TypeVariant::U64 => {
            write_supers(buffer, supers)?;
            write!(buffer, "varint_size_from_value(*payload)")
        }
        schema::TypeVariant::Unit => write!(buffer, "0_u64"),
    }
}

// Write the logic to invoke the serialization logic for a value, including a trailing line break.
fn write_serialization_invocation<T: Write>(
    buffer: &mut T,
    indentation: usize,
    supers: usize,
    r#type: &schema::Type,
) -> Result<(), fmt::Error> {
    write_indentation(buffer, indentation)?;

    match &r#type.variant {
        schema::TypeVariant::Array(inner_type) => match &inner_type.variant {
            schema::TypeVariant::Array(_)
            | schema::TypeVariant::Bytes
            | schema::TypeVariant::Custom(_, _)
            | schema::TypeVariant::String => {
                writeln!(buffer, "for payload in payload {{")?;
                write_indentation(buffer, indentation + 1)?;
                write_supers(buffer, supers)?;
                write!(buffer, "serialize_varint(")?;
                write_size_calculation_invocation(buffer, supers, inner_type)?;
                writeln!(buffer, ", writer)?;")?;
                write_serialization_invocation(buffer, indentation + 1, supers, inner_type)?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")
            }
            schema::TypeVariant::Bool
            | schema::TypeVariant::S64
            | schema::TypeVariant::U64
            | schema::TypeVariant::F64 => {
                writeln!(buffer, "for payload in payload {{")?;
                write_serialization_invocation(buffer, indentation + 1, supers, inner_type)?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")
            }
            schema::TypeVariant::Unit => {
                write_supers(buffer, supers)?;
                writeln!(buffer, "serialize_varint(payload.len() as u64, writer)?;")
            }
        },
        schema::TypeVariant::Bool => {
            write_supers(buffer, supers)?;
            writeln!(buffer, "serialize_varint(*payload as u64, writer)?;")
        }
        schema::TypeVariant::Bytes => writeln!(buffer, "writer.write_all(payload)?;"),
        schema::TypeVariant::Custom(_, _) => writeln!(buffer, "payload.serialize(writer)?;"),
        schema::TypeVariant::F64 => writeln!(buffer, "writer.write_all(&payload.to_le_bytes())?;"),
        schema::TypeVariant::S64 => {
            write_supers(buffer, supers)?;
            write!(buffer, "serialize_varint(")?;
            write_supers(buffer, supers)?;
            writeln!(buffer, "zigzag_encode(*payload), writer)?;")
        }
        schema::TypeVariant::String => writeln!(buffer, "writer.write_all(payload.as_bytes())?;"),
        schema::TypeVariant::U64 => {
            write_supers(buffer, supers)?;
            writeln!(buffer, "serialize_varint(*payload, writer)?;")
        }
        schema::TypeVariant::Unit => writeln!(buffer, "();"),
    }
}

// Write the logic to invoke the deserialization logic for a value, including a trailing line break.
#[allow(clippy::too_many_lines)]
fn write_deserialization_invocation<T: Write>(
    buffer: &mut T,
    indentation: usize,
    supers: usize,
    imports: &BTreeMap<Identifier, schema::Namespace>,
    namespace: &schema::Namespace,
    r#type: &schema::Type,
) -> Result<(), fmt::Error> {
    write_indentation(buffer, indentation)?;

    match &r#type.variant {
        schema::TypeVariant::Array(inner_type) => match &inner_type.variant {
            schema::TypeVariant::Array(_)
            | schema::TypeVariant::Bytes
            | schema::TypeVariant::Custom(_, _)
            | schema::TypeVariant::String => {
                writeln!(buffer, "let mut payload = Vec::new();")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "loop {{")?;
                write_indentation(buffer, indentation + 1)?;
                write!(buffer, "let element_size = match ")?;
                write_supers(buffer, supers)?;
                writeln!(buffer, "deserialize_varint(&mut sub_reader) {{")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "Ok(element_size) => element_size,")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "Err(err) => {{")?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(
                    buffer,
                    "if let std::io::ErrorKind::UnexpectedEof = err.kind() {{",
                )?;
                write_indentation(buffer, indentation + 4)?;
                writeln!(buffer, "break;")?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(buffer, "}}")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(buffer, "return Err(err);")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}};")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(
                    buffer,
                    "let mut sub_reader = ::std::io::Read::take(\
                            &mut sub_reader, element_size);",
                )?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "payload.push({{")?;
                write_deserialization_invocation(
                    buffer,
                    indentation + 2,
                    supers,
                    imports,
                    namespace,
                    inner_type,
                )?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "payload")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}});")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")
            }
            schema::TypeVariant::Bool
            | schema::TypeVariant::S64
            | schema::TypeVariant::U64
            | schema::TypeVariant::F64 => {
                write!(
                    buffer,
                    "fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut T) -> \
                            ::std::io::Result<",
                )?;
                write_type(buffer, imports, namespace, inner_type, In)?;
                writeln!(buffer, "> {{")?;
                write_deserialization_invocation(
                    buffer,
                    indentation + 1,
                    supers,
                    imports,
                    namespace,
                    inner_type,
                )?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "Ok(payload)")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "let mut payload = Vec::new();")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "loop {{")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(
                    buffer,
                    "payload.push(match deserialize_element(&mut sub_reader) {{",
                )?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "Ok(element) => element,")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "Err(err) => {{")?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(
                    buffer,
                    "if let std::io::ErrorKind::UnexpectedEof = err.kind() {{",
                )?;
                write_indentation(buffer, indentation + 4)?;
                writeln!(buffer, "break;")?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(buffer, "}}")?;
                writeln!(buffer)?;
                write_indentation(buffer, indentation + 3)?;
                writeln!(buffer, "return Err(err);")?;
                write_indentation(buffer, indentation + 2)?;
                writeln!(buffer, "}}")?;
                write_indentation(buffer, indentation + 1)?;
                writeln!(buffer, "}});")?;
                write_indentation(buffer, indentation)?;
                writeln!(buffer, "}}")
            }
            schema::TypeVariant::Unit => {
                write!(buffer, "let payload = vec![(); ")?;
                write_supers(buffer, supers)?;
                writeln!(buffer, "deserialize_varint(&mut sub_reader)? as usize];")
            }
        },
        schema::TypeVariant::Bool => {
            writeln!(buffer, "let mut buffer = [0_u8];")?;
            write_indentation(buffer, indentation)?;
            writeln!(
                buffer,
                "::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;",
            )?;
            write_indentation(buffer, indentation)?;
            writeln!(buffer, "let payload = buffer[0] != 0b0000_0001;")
        }
        schema::TypeVariant::Bytes => {
            writeln!(buffer, "let mut payload = vec![];")?;
            write_indentation(buffer, indentation)?;
            writeln!(
                buffer,
                "::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;",
            )
        }
        schema::TypeVariant::Custom(_, _) => {
            write!(buffer, "let payload = <")?;
            write_type(buffer, imports, namespace, r#type, In)?;
            write!(buffer, " as ")?;
            write_supers(buffer, supers)?;
            writeln!(buffer, "Deserialize>::deserialize(&mut sub_reader)?;")
        }
        schema::TypeVariant::F64 => {
            writeln!(buffer, "let mut buffer = [0; 8];")?;
            write_indentation(buffer, indentation)?;
            writeln!(
                buffer,
                "::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;",
            )?;
            write_indentation(buffer, indentation)?;
            writeln!(buffer, "let payload = f64::from_le_bytes(buffer);")
        }
        schema::TypeVariant::S64 => {
            write!(buffer, "let payload = ")?;
            write_supers(buffer, supers)?;
            write!(buffer, "zigzag_decode(")?;
            write_supers(buffer, supers)?;
            writeln!(buffer, "deserialize_varint(&mut sub_reader)?);")
        }
        schema::TypeVariant::String => {
            writeln!(buffer, "let mut buffer = vec![];")?;
            write_indentation(buffer, indentation)?;
            writeln!(
                buffer,
                "::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;",
            )?;
            write_indentation(buffer, indentation)?;
            writeln!(
                buffer,
                "let payload = std::str::from_utf8(&buffer).map_or_else(",
            )?;
            write_indentation(buffer, indentation + 1)?;
            writeln!(
                buffer,
                "|err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),",
            )?;
            write_indentation(buffer, indentation + 1)?;
            writeln!(buffer, "|result| Ok(result.to_owned()),")?;
            write_indentation(buffer, indentation)?;
            writeln!(buffer, ")?;")
        }
        schema::TypeVariant::U64 => {
            write!(buffer, "let payload = ")?;
            write_supers(buffer, supers)?;
            writeln!(buffer, "deserialize_varint(&mut sub_reader)?;")
        }
        schema::TypeVariant::Unit => writeln!(buffer, "let payload = ();"),
    }
}

// Determine whether a type is encoded as a varint.
fn varint_encoded(r#type: &schema::Type) -> bool {
    match &r#type.variant {
        schema::TypeVariant::Bool | schema::TypeVariant::S64 | schema::TypeVariant::U64 => true,
        schema::TypeVariant::Array(_)
        | schema::TypeVariant::Bytes
        | schema::TypeVariant::Custom(_, _)
        | schema::TypeVariant::F64
        | schema::TypeVariant::String
        | schema::TypeVariant::Unit => false,
    }
}

#[cfg(test)]
mod tests {
    use {
        crate::{generate_rust::generate, schema_loader::load_schemas, validator::validate},
        std::path::Path,
    };

    // This test doesn't work on Windows, for some reason.
    #[allow(clippy::too_many_lines)]
    #[test]
    #[cfg_attr(target_os = "windows", ignore)]
    fn generate_example() {
        let schemas = load_schemas(Path::new("integration_tests/types/main.t")).unwrap();
        validate(&schemas).unwrap();

        assert_eq!(
            generate("0.0.0", &schemas),
            "\
// This file was automatically generated by Typical 0.0.0.
// Visit https://github.com/stepchowfun/typical for more information.

#![allow(clippy::all, clippy::pedantic, clippy::nursery, warnings)]

#[rustfmt::skip]
use std::{
    cmp::min,
    io::{self, BufRead, Error, ErrorKind, Write},
    mem::transmute,
};

#[rustfmt::skip]
fn zigzag_encode(value: i64) -> u64 {
    unsafe {
        transmute::<i64, u64>(value >> 63) ^ transmute::<i64, u64>(value << 1)
    }
}

#[rustfmt::skip]
fn zigzag_decode(value: u64) -> i64 {
    unsafe {
        transmute::<u64, i64>(value >> 1) ^ -transmute::<u64, i64>(value & 1)
    }
}

#[rustfmt::skip]
fn varint_size_from_value(value: u64) -> u64 {
    let mut size = 1_u64;
    let mut upper_bound_exclusive = 0_u64;

    while size < 9 {
        upper_bound_exclusive += 1_u64 << (size * 7);

        if value < upper_bound_exclusive {
            break;
        }

        size += 1;
    }

    size
}

#[rustfmt::skip]
fn varint_size_from_first_byte(first_byte: u8) -> u32 {
    first_byte.trailing_zeros() + 1
}

#[rustfmt::skip]
fn serialize_varint<T: Write>(value: u64, writer: &mut T) -> io::Result<()> {
    let size = varint_size_from_value(value);
    let size_minus_one = size - 1;

    let mut x = value;
    for i in 1..size {
        x -= 1_u64 << (i * 7);
    }

    writer.write_all(&[((x << size) | (1_u64 << size_minus_one)) as u8])?;
    x >>= 8_u64.saturating_sub(size);

    for _ in 0..size_minus_one {
        writer.write_all(&[x as u8])?;
        x >>= 8;
    }

    Ok(())
}

#[rustfmt::skip]
fn deserialize_varint<T: BufRead>(reader: &mut T) -> io::Result<u64> {
    let mut buffer = [0; 9];
    reader.read_exact(&mut buffer[0..1])?;
    let first_byte = buffer[0];
    let size = varint_size_from_first_byte(first_byte) as usize;

    reader.read_exact(&mut buffer[1..size])?;

    let mut x = u64::from(first_byte) >> size;
    let mut bits_read = 8_usize.saturating_sub(size);

    for byte in buffer.iter().skip(1) {
        x |= u64::from(*byte) << bits_read;
        bits_read += 8;
    }

    for i in 1..size {
        x = x
            .checked_add(1_u64 << (i * 7))
            .ok_or_else(|| Error::new(ErrorKind::InvalidData, \"Error decoding varint.\"))?;
    }

    Ok(x)
}

#[rustfmt::skip]
fn non_varint_field_header_size(index: u64, value_size: u64) -> u64 {
    match value_size {
        0 => varint_size_from_value((index << 2) | 0b00),
        8 => varint_size_from_value((index << 2) | 0b01),
        size => varint_size_from_value((index << 2) | 0b10) + varint_size_from_value(size),
    }
}

#[rustfmt::skip]
fn varint_field_header_size(index: u64) -> u64 {
    varint_size_from_value((index << 2) | 0b11)
}

#[rustfmt::skip]
fn serialize_non_varint_field_header<T: Write>(
    writer: &mut T,
    index: u64,
    payload_size: u64,
) -> io::Result<()> {
    match payload_size {
        0 => serialize_varint((index << 2) | 0b00, writer),
        8 => serialize_varint((index << 2) | 0b01, writer),
        size => {
            serialize_varint((index << 2) | 0b10, writer)?;
            serialize_varint(size, writer)
        }
    }
}

#[rustfmt::skip]
fn serialize_varint_field_header<T: Write>(writer: &mut T, index: u64) -> io::Result<()> {
    serialize_varint((index << 2) | 0b11, writer)
}

#[rustfmt::skip]
fn deserialize_field_header<T: BufRead>(reader: &mut T) -> io::Result<(u64, u64)> {
    let tag = deserialize_varint(&mut *reader)?;

    let index = tag >> 2;

    let size = match tag & 0b11 {
        0b00 => 0,
        0b01 => 8,
        0b10 => deserialize_varint(&mut *reader)?,
        0b11 => {
            let buffer = (&mut *reader).fill_buf()?;

            if buffer.is_empty() {
                return Err(Error::new(ErrorKind::UnexpectedEof, \"Error decoding field.\"));
            }

            u64::from(varint_size_from_first_byte(buffer[0]))
        }
        _ => panic!(),
    };

    Ok((index, size))
}

#[rustfmt::skip]
fn skip<T: BufRead>(reader: &mut T, mut amount: usize) -> io::Result<()> {
    while amount > 0 {
        let buffer = reader.fill_buf()?;
        let num_bytes_to_consume = min(buffer.len(), amount);
        reader.consume(num_bytes_to_consume);
        amount -= num_bytes_to_consume;
    }

    Ok(())
}

#[rustfmt::skip]
pub trait Serialize {
    fn size(&self) -> u64;

    fn serialize<T: Write>(&self, writer: &mut T) -> io::Result<()>;
}

#[rustfmt::skip]
pub trait Deserialize {
    fn deserialize<T>(reader: &mut T) -> io::Result<Self>
    where
        Self: Sized,
        T: BufRead;
}

#[rustfmt::skip]
pub mod circular_dependency {
    pub mod dependency {
        pub mod main {
            #[derive(Clone, Debug)]
            pub struct StructFromBelowOut {
                pub x: super::super::main::StructFromAboveOut,
            }

            #[derive(Clone, Debug)]
            pub struct StructFromBelowIn {
                pub x: super::super::main::StructFromAboveIn,
            }

            impl super::super::super::Serialize for StructFromBelowOut {
                fn size(&self) -> u64 {
                    ({
                        let payload = &self.x;
                        let payload_size = payload.size();
                        super::super::super::non_varint_field_header_size(0, payload_size) + \
                            payload_size
                    })
                }

                fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                    {
                        let payload = &self.x;
                        let payload_size = payload.size();
                        super::super::super::serialize_non_varint_field_header(writer, 0, \
                            payload_size)?;
                        payload.serialize(writer)?;
                    }

                    Ok(())
                }
            }

            impl super::super::super::Deserialize for StructFromBelowIn {
                fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
                where
                    Self: Sized,
                    T: ::std::io::BufRead,
                {
                    let mut x: Option<super::super::main::StructFromAboveIn> = None;

                    loop {
                        let (index, size) = match \
                            super::super::super::deserialize_field_header(&mut *reader) {
                            Ok(header) => header,
                            Err(err) => {
                                if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                    break;
                                }

                                return Err(err);
                            }
                        };

                        let mut sub_reader = ::std::io::Read::take(&mut *reader, size);

                        match index {
                            0 => {
                                let payload = <super::super::main::StructFromAboveIn as \
                                    super::super::super::Deserialize>::deserialize(&mut \
                                    sub_reader)?;

                                x.get_or_insert(payload);
                            }
                            _ => {
                                super::super::super::skip(&mut sub_reader, size as usize)?;
                            }
                        }
                    }

                    if x.is_none() {
                        return Err(::std::io::Error::new(
                            ::std::io::ErrorKind::InvalidData,
                            \"Struct missing one or more field(s).\",
                        ));
                    }

                    Ok(StructFromBelowIn {
                        x: x.unwrap(),
                    })
                }
            }

            impl From<StructFromBelowOut> for StructFromBelowIn {
                fn from(message: StructFromBelowOut) -> Self {
                    StructFromBelowIn {
                        x: message.x.into(),
                    }
                }
            }
        }
    }

    pub mod main {
        #[derive(Clone, Debug)]
        pub struct StructFromAboveOut {
        }

        #[derive(Clone, Debug)]
        pub struct StructFromAboveIn {
        }

        impl super::super::Serialize for StructFromAboveOut {
            fn size(&self) -> u64 {
                0
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                Ok(())
            }
        }

        impl super::super::Deserialize for StructFromAboveIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                loop {
                    let (index, size) = match super::super::deserialize_field_header(&mut \
                        *reader) {
                        Ok(header) => header,
                        Err(err) => {
                            if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                break;
                            }

                            return Err(err);
                        }
                    };

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size);

                    match index {
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }

                Ok(StructFromAboveIn {
                })
            }
        }

        impl From<StructFromAboveOut> for StructFromAboveIn {
            fn from(message: StructFromAboveOut) -> Self {
                StructFromAboveIn {
                }
            }
        }
    }
}

#[rustfmt::skip]
pub mod comprehensive {
    pub mod bar {
        #[derive(Clone, Debug)]
        pub enum BarOut {
            PRequired(Vec<()>),
            QRequired(Vec<f64>),
            RRequired(Vec<i64>),
            SRequired(Vec<Vec<String>>),
            TRequired(bool),
            URequired(Vec<u8>),
            VRequired(f64),
            WRequired(i64),
            XRequired(String),
            YRequired(u64),
            ZRequired,
            PAsymmetric(Vec<()>, Box<BarOut>),
            QAsymmetric(Vec<f64>, Box<BarOut>),
            RAsymmetric(Vec<i64>, Box<BarOut>),
            SAsymmetric(Vec<Vec<String>>, Box<BarOut>),
            TAsymmetric(bool, Box<BarOut>),
            UAsymmetric(Vec<u8>, Box<BarOut>),
            VAsymmetric(f64, Box<BarOut>),
            WAsymmetric(i64, Box<BarOut>),
            XAsymmetric(String, Box<BarOut>),
            YAsymmetric(u64, Box<BarOut>),
            ZAsymmetric(Box<BarOut>),
            POptional(Vec<()>, Box<BarOut>),
            QOptional(Vec<f64>, Box<BarOut>),
            ROptional(Vec<i64>, Box<BarOut>),
            SOptional(Vec<Vec<String>>, Box<BarOut>),
            TOptional(bool, Box<BarOut>),
            UOptional(Vec<u8>, Box<BarOut>),
            VOptional(f64, Box<BarOut>),
            WOptional(i64, Box<BarOut>),
            XOptional(String, Box<BarOut>),
            YOptional(u64, Box<BarOut>),
            ZOptional(Box<BarOut>),
        }

        #[derive(Clone, Debug)]
        pub enum BarIn {
            PRequired(Vec<()>),
            QRequired(Vec<f64>),
            RRequired(Vec<i64>),
            SRequired(Vec<Vec<String>>),
            TRequired(bool),
            URequired(Vec<u8>),
            VRequired(f64),
            WRequired(i64),
            XRequired(String),
            YRequired(u64),
            ZRequired,
            PAsymmetric(Vec<()>),
            QAsymmetric(Vec<f64>),
            RAsymmetric(Vec<i64>),
            SAsymmetric(Vec<Vec<String>>),
            TAsymmetric(bool),
            UAsymmetric(Vec<u8>),
            VAsymmetric(f64),
            WAsymmetric(i64),
            XAsymmetric(String),
            YAsymmetric(u64),
            ZAsymmetric,
            POptional(Vec<()>, Box<BarIn>),
            QOptional(Vec<f64>, Box<BarIn>),
            ROptional(Vec<i64>, Box<BarIn>),
            SOptional(Vec<Vec<String>>, Box<BarIn>),
            TOptional(bool, Box<BarIn>),
            UOptional(Vec<u8>, Box<BarIn>),
            VOptional(f64, Box<BarIn>),
            WOptional(i64, Box<BarIn>),
            XOptional(String, Box<BarIn>),
            YOptional(u64, Box<BarIn>),
            ZOptional(Box<BarIn>),
        }

        impl super::super::Serialize for BarOut {
            fn size(&self) -> u64 {
                match *self {
                    BarOut::PRequired(ref payload) => {
                        let payload_size = super::super::varint_size_from_value(payload.len() as \
                            u64);
                        super::super::non_varint_field_header_size(0, payload_size) +
                            payload_size
                    }
                    BarOut::QRequired(ref payload) => {
                        let payload_size = 8_u64 * (payload.len() as u64);
                        super::super::non_varint_field_header_size(1, payload_size) +
                            payload_size
                    }
                    BarOut::RRequired(ref payload) => {
                        let payload_size = payload.iter().fold(0_u64, |x, payload| x + \
                            super::super::varint_size_from_value(super::super::zigzag_encode(*\
                            payload)));
                        super::super::non_varint_field_header_size(2, payload_size) +
                            payload_size
                    }
                    BarOut::SRequired(ref payload) => {
                        let payload_size = payload.iter().fold(0_u64, |x, payload| { let \
                            payload_size = payload.iter().fold(0_u64, |x, payload| { let \
                            payload_size = payload.len() as u64; x + \
                            super::super::varint_size_from_value(payload_size) + \
                            payload_size }); x + \
                            super::super::varint_size_from_value(payload_size) + payload_size });
                        super::super::non_varint_field_header_size(3, payload_size) +
                            payload_size
                    }
                    BarOut::TRequired(ref payload) => {
                        let payload_size = 1_u64;
                        super::super::varint_field_header_size(4) +
                            payload_size
                    }
                    BarOut::URequired(ref payload) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(5, payload_size) +
                            payload_size
                    }
                    BarOut::VRequired(ref payload) => {
                        let payload_size = 8_u64;
                        super::super::non_varint_field_header_size(6, payload_size) +
                            payload_size
                    }
                    BarOut::WRequired(ref payload) => {
                        let payload_size = \
                            super::super::varint_size_from_value(super::super::zigzag_encode(*\
                            payload));
                        super::super::varint_field_header_size(7) +
                            payload_size
                    }
                    BarOut::XRequired(ref payload) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(8, payload_size) +
                            payload_size
                    }
                    BarOut::YRequired(ref payload) => {
                        let payload_size = super::super::varint_size_from_value(*payload);
                        super::super::varint_field_header_size(9) +
                            payload_size
                    }
                    BarOut::ZRequired => {
                        let payload_size = 0_u64;
                        super::super::non_varint_field_header_size(10, payload_size) +
                            payload_size
                    }
                    BarOut::PAsymmetric(ref payload, ref fallback) => {
                        let payload_size = super::super::varint_size_from_value(payload.len() as \
                            u64);
                        super::super::non_varint_field_header_size(11, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::QAsymmetric(ref payload, ref fallback) => {
                        let payload_size = 8_u64 * (payload.len() as u64);
                        super::super::non_varint_field_header_size(12, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::RAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_u64, |x, payload| x + \
                            super::super::varint_size_from_value(super::super::zigzag_encode(*\
                            payload)));
                        super::super::non_varint_field_header_size(13, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::SAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_u64, |x, payload| { let \
                            payload_size = payload.iter().fold(0_u64, |x, payload| { let \
                            payload_size = payload.len() as u64; x + \
                            super::super::varint_size_from_value(payload_size) + \
                            payload_size }); x + \
                            super::super::varint_size_from_value(payload_size) + payload_size });
                        super::super::non_varint_field_header_size(14, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::TAsymmetric(ref payload, ref fallback) => {
                        let payload_size = 1_u64;
                        super::super::varint_field_header_size(15) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::UAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(16, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::VAsymmetric(ref payload, ref fallback) => {
                        let payload_size = 8_u64;
                        super::super::non_varint_field_header_size(17, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::WAsymmetric(ref payload, ref fallback) => {
                        let payload_size = \
                            super::super::varint_size_from_value(super::super::zigzag_encode(*\
                            payload));
                        super::super::varint_field_header_size(18) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::XAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(19, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::YAsymmetric(ref payload, ref fallback) => {
                        let payload_size = super::super::varint_size_from_value(*payload);
                        super::super::varint_field_header_size(20) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::ZAsymmetric(ref fallback) => {
                        let payload_size = 0_u64;
                        super::super::non_varint_field_header_size(21, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::POptional(ref payload, ref fallback) => {
                        let payload_size = super::super::varint_size_from_value(payload.len() as \
                            u64);
                        super::super::non_varint_field_header_size(22, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::QOptional(ref payload, ref fallback) => {
                        let payload_size = 8_u64 * (payload.len() as u64);
                        super::super::non_varint_field_header_size(23, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::ROptional(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_u64, |x, payload| x + \
                            super::super::varint_size_from_value(super::super::zigzag_encode(*\
                            payload)));
                        super::super::non_varint_field_header_size(24, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::SOptional(ref payload, ref fallback) => {
                        let payload_size = payload.iter().fold(0_u64, |x, payload| { let \
                            payload_size = payload.iter().fold(0_u64, |x, payload| { let \
                            payload_size = payload.len() as u64; x + \
                            super::super::varint_size_from_value(payload_size) + \
                            payload_size }); x + \
                            super::super::varint_size_from_value(payload_size) + payload_size });
                        super::super::non_varint_field_header_size(25, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::TOptional(ref payload, ref fallback) => {
                        let payload_size = 1_u64;
                        super::super::varint_field_header_size(26) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::UOptional(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(27, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::VOptional(ref payload, ref fallback) => {
                        let payload_size = 8_u64;
                        super::super::non_varint_field_header_size(28, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::WOptional(ref payload, ref fallback) => {
                        let payload_size = \
                            super::super::varint_size_from_value(super::super::zigzag_encode(*\
                            payload));
                        super::super::varint_field_header_size(29) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::XOptional(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(30, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::YOptional(ref payload, ref fallback) => {
                        let payload_size = super::super::varint_size_from_value(*payload);
                        super::super::varint_field_header_size(31) +
                            payload_size +
                            fallback.size()
                    }
                    BarOut::ZOptional(ref fallback) => {
                        let payload_size = 0_u64;
                        super::super::non_varint_field_header_size(32, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                }
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                match *self {
                    BarOut::PRequired(ref payload) => {
                        super::super::serialize_non_varint_field_header(writer, 0, \
                            super::super::varint_size_from_value(payload.len() as u64))?;
                        super::super::serialize_varint(payload.len() as u64, writer)?;
                        Ok(())
                    }
                    BarOut::QRequired(ref payload) => {
                        super::super::serialize_non_varint_field_header(writer, 1, 8_u64 * \
                            (payload.len() as u64))?;
                        for payload in payload {
                            writer.write_all(&payload.to_le_bytes())?;
                        }
                        Ok(())
                    }
                    BarOut::RRequired(ref payload) => {
                        super::super::serialize_non_varint_field_header(writer, 2, \
                            payload.iter().fold(0_u64, |x, payload| x + \
                            super::super::varint_size_from_value(super::super::zigzag_encode(*\
                            payload))))?;
                        for payload in payload {
                            super::super::serialize_varint(super::super::zigzag_encode(*payload), \
                                writer)?;
                        }
                        Ok(())
                    }
                    BarOut::SRequired(ref payload) => {
                        super::super::serialize_non_varint_field_header(writer, 3, \
                            payload.iter().fold(0_u64, |x, payload| { let payload_size = \
                            payload.iter().fold(0_u64, |x, payload| { let payload_size = \
                            payload.len() as u64; x + \
                            super::super::varint_size_from_value(payload_size) + \
                            payload_size }); x + \
                            super::super::varint_size_from_value(payload_size) + payload_size }))?;
                        for payload in payload {
                            super::super::serialize_varint(payload.iter().fold(0_u64, |x, \
                                payload| { let payload_size = payload.len() as u64; x + \
                                super::super::varint_size_from_value(payload_size) + \
                                payload_size }), writer)?;
                            for payload in payload {
                                super::super::serialize_varint(payload.len() as u64, writer)?;
                                writer.write_all(payload.as_bytes())?;
                            }
                        }
                        Ok(())
                    }
                    BarOut::TRequired(ref payload) => {
                        super::super::serialize_varint_field_header(writer, 4)?;
                        super::super::serialize_varint(*payload as u64, writer)?;
                        Ok(())
                    }
                    BarOut::URequired(ref payload) => {
                        super::super::serialize_non_varint_field_header(writer, 5, payload.len() \
                            as u64)?;
                        writer.write_all(payload)?;
                        Ok(())
                    }
                    BarOut::VRequired(ref payload) => {
                        super::super::serialize_non_varint_field_header(writer, 6, 8_u64)?;
                        writer.write_all(&payload.to_le_bytes())?;
                        Ok(())
                    }
                    BarOut::WRequired(ref payload) => {
                        super::super::serialize_varint_field_header(writer, 7)?;
                        super::super::serialize_varint(super::super::zigzag_encode(*payload), \
                            writer)?;
                        Ok(())
                    }
                    BarOut::XRequired(ref payload) => {
                        super::super::serialize_non_varint_field_header(writer, 8, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        Ok(())
                    }
                    BarOut::YRequired(ref payload) => {
                        super::super::serialize_varint_field_header(writer, 9)?;
                        super::super::serialize_varint(*payload, writer)?;
                        Ok(())
                    }
                    BarOut::ZRequired => {
                        super::super::serialize_non_varint_field_header(writer, 10, 0_u64)?;
                        ();
                        Ok(())
                    }
                    BarOut::PAsymmetric(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 11, \
                            super::super::varint_size_from_value(payload.len() as u64))?;
                        super::super::serialize_varint(payload.len() as u64, writer)?;
                        fallback.serialize(writer)
                    }
                    BarOut::QAsymmetric(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 12, 8_u64 * \
                            (payload.len() as u64))?;
                        for payload in payload {
                            writer.write_all(&payload.to_le_bytes())?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::RAsymmetric(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 13, \
                            payload.iter().fold(0_u64, |x, payload| x + \
                            super::super::varint_size_from_value(super::super::zigzag_encode(*\
                            payload))))?;
                        for payload in payload {
                            super::super::serialize_varint(super::super::zigzag_encode(*payload), \
                                writer)?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::SAsymmetric(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 14, \
                            payload.iter().fold(0_u64, |x, payload| { let payload_size = \
                            payload.iter().fold(0_u64, |x, payload| { let payload_size = \
                            payload.len() as u64; x + \
                            super::super::varint_size_from_value(payload_size) + \
                            payload_size }); x + \
                            super::super::varint_size_from_value(payload_size) + payload_size }))?;
                        for payload in payload {
                            super::super::serialize_varint(payload.iter().fold(0_u64, |x, \
                                payload| { let payload_size = payload.len() as u64; x + \
                                super::super::varint_size_from_value(payload_size) + \
                                payload_size }), writer)?;
                            for payload in payload {
                                super::super::serialize_varint(payload.len() as u64, writer)?;
                                writer.write_all(payload.as_bytes())?;
                            }
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::TAsymmetric(ref payload, ref fallback) => {
                        super::super::serialize_varint_field_header(writer, 15)?;
                        super::super::serialize_varint(*payload as u64, writer)?;
                        fallback.serialize(writer)
                    }
                    BarOut::UAsymmetric(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 16, payload.len() \
                            as u64)?;
                        writer.write_all(payload)?;
                        fallback.serialize(writer)
                    }
                    BarOut::VAsymmetric(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 17, 8_u64)?;
                        writer.write_all(&payload.to_le_bytes())?;
                        fallback.serialize(writer)
                    }
                    BarOut::WAsymmetric(ref payload, ref fallback) => {
                        super::super::serialize_varint_field_header(writer, 18)?;
                        super::super::serialize_varint(super::super::zigzag_encode(*payload), \
                            writer)?;
                        fallback.serialize(writer)
                    }
                    BarOut::XAsymmetric(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 19, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    BarOut::YAsymmetric(ref payload, ref fallback) => {
                        super::super::serialize_varint_field_header(writer, 20)?;
                        super::super::serialize_varint(*payload, writer)?;
                        fallback.serialize(writer)
                    }
                    BarOut::ZAsymmetric(ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 21, 0_u64)?;
                        ();
                        fallback.serialize(writer)
                    }
                    BarOut::POptional(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 22, \
                            super::super::varint_size_from_value(payload.len() as u64))?;
                        super::super::serialize_varint(payload.len() as u64, writer)?;
                        fallback.serialize(writer)
                    }
                    BarOut::QOptional(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 23, 8_u64 * \
                            (payload.len() as u64))?;
                        for payload in payload {
                            writer.write_all(&payload.to_le_bytes())?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::ROptional(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 24, \
                            payload.iter().fold(0_u64, |x, payload| x + \
                            super::super::varint_size_from_value(super::super::zigzag_encode(*\
                            payload))))?;
                        for payload in payload {
                            super::super::serialize_varint(super::super::zigzag_encode(*payload), \
                                writer)?;
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::SOptional(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 25, \
                            payload.iter().fold(0_u64, |x, payload| { let payload_size = \
                            payload.iter().fold(0_u64, |x, payload| { let payload_size = \
                            payload.len() as u64; x + \
                            super::super::varint_size_from_value(payload_size) + \
                            payload_size }); x + \
                            super::super::varint_size_from_value(payload_size) + payload_size }))?;
                        for payload in payload {
                            super::super::serialize_varint(payload.iter().fold(0_u64, |x, \
                                payload| { let payload_size = payload.len() as u64; x + \
                                super::super::varint_size_from_value(payload_size) + \
                                payload_size }), writer)?;
                            for payload in payload {
                                super::super::serialize_varint(payload.len() as u64, writer)?;
                                writer.write_all(payload.as_bytes())?;
                            }
                        }
                        fallback.serialize(writer)
                    }
                    BarOut::TOptional(ref payload, ref fallback) => {
                        super::super::serialize_varint_field_header(writer, 26)?;
                        super::super::serialize_varint(*payload as u64, writer)?;
                        fallback.serialize(writer)
                    }
                    BarOut::UOptional(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 27, payload.len() \
                            as u64)?;
                        writer.write_all(payload)?;
                        fallback.serialize(writer)
                    }
                    BarOut::VOptional(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 28, 8_u64)?;
                        writer.write_all(&payload.to_le_bytes())?;
                        fallback.serialize(writer)
                    }
                    BarOut::WOptional(ref payload, ref fallback) => {
                        super::super::serialize_varint_field_header(writer, 29)?;
                        super::super::serialize_varint(super::super::zigzag_encode(*payload), \
                            writer)?;
                        fallback.serialize(writer)
                    }
                    BarOut::XOptional(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 30, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    BarOut::YOptional(ref payload, ref fallback) => {
                        super::super::serialize_varint_field_header(writer, 31)?;
                        super::super::serialize_varint(*payload, writer)?;
                        fallback.serialize(writer)
                    }
                    BarOut::ZOptional(ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 32, 0_u64)?;
                        ();
                        fallback.serialize(writer)
                    }
                }
            }
        }

        impl super::super::Deserialize for BarIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                loop {
                    let (index, size) = super::super::deserialize_field_header(&mut *reader)?;

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size);

                    match index {
                        0 => {
                            let payload = vec![(); super::super::deserialize_varint(&mut \
                                sub_reader)? as usize];
                            return Ok(BarIn::PRequired(payload));
                        }
                        1 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<f64> {
                                let mut buffer = [0; 8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                                let payload = f64::from_le_bytes(buffer);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }
                            return Ok(BarIn::QRequired(payload));
                        }
                        2 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<i64> {
                                let payload = \
                                    super::super::zigzag_decode(super::super::deserialize_varint(&\
                                    mut sub_reader)?);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }
                            return Ok(BarIn::RRequired(payload));
                        }
                        3 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size);
                                payload.push({
                                    let mut payload = Vec::new();

                                    loop {
                                        let element_size = match \
                                            super::super::deserialize_varint(&mut sub_reader) {
                                            Ok(element_size) => element_size,
                                            Err(err) => {
                                                if let std::io::ErrorKind::UnexpectedEof = \
                                                    err.kind() {
                                                    break;
                                                }

                                                return Err(err);
                                            }
                                        };

                                        let mut sub_reader = ::std::io::Read::take(&mut \
                                            sub_reader, element_size);
                                        payload.push({
                                            let mut buffer = vec![];
                                            ::std::io::Read::read_to_end(&mut sub_reader, &mut \
                                                buffer)?;
                                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                                |err| Err(::std::io::Error::new(::std::io::\
                                                    ErrorKind::Other, err)),
                                                |result| Ok(result.to_owned()),
                                            )?;
                                            payload
                                        });
                                    }
                                    payload
                                });
                            }
                            return Ok(BarIn::SRequired(payload));
                        }
                        4 => {
                            let mut buffer = [0_u8];
                            ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                            let payload = buffer[0] != 0b0000_0001;
                            return Ok(BarIn::TRequired(payload));
                        }
                        5 => {
                            let mut payload = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;
                            return Ok(BarIn::URequired(payload));
                        }
                        6 => {
                            let mut buffer = [0; 8];
                            ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                            let payload = f64::from_le_bytes(buffer);
                            return Ok(BarIn::VRequired(payload));
                        }
                        7 => {
                            let payload = \
                                super::super::zigzag_decode(super::super::deserialize_varint(&mut \
                                sub_reader)?);
                            return Ok(BarIn::WRequired(payload));
                        }
                        8 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(BarIn::XRequired(payload));
                        }
                        9 => {
                            let payload = super::super::deserialize_varint(&mut sub_reader)?;
                            return Ok(BarIn::YRequired(payload));
                        }
                        10 => {
                            let payload = ();
                            return Ok(BarIn::ZRequired);
                        }
                        11 => {
                            let payload = vec![(); super::super::deserialize_varint(&mut \
                                sub_reader)? as usize];
                            return Ok(BarIn::PAsymmetric(payload));
                        }
                        12 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<f64> {
                                let mut buffer = [0; 8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                                let payload = f64::from_le_bytes(buffer);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }
                            return Ok(BarIn::QAsymmetric(payload));
                        }
                        13 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<i64> {
                                let payload = \
                                    super::super::zigzag_decode(super::super::deserialize_varint(&\
                                    mut sub_reader)?);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }
                            return Ok(BarIn::RAsymmetric(payload));
                        }
                        14 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size);
                                payload.push({
                                    let mut payload = Vec::new();

                                    loop {
                                        let element_size = match \
                                            super::super::deserialize_varint(&mut sub_reader) {
                                            Ok(element_size) => element_size,
                                            Err(err) => {
                                                if let std::io::ErrorKind::UnexpectedEof = \
                                                    err.kind() {
                                                    break;
                                                }

                                                return Err(err);
                                            }
                                        };

                                        let mut sub_reader = ::std::io::Read::take(&mut \
                                            sub_reader, element_size);
                                        payload.push({
                                            let mut buffer = vec![];
                                            ::std::io::Read::read_to_end(&mut sub_reader, &mut \
                                                buffer)?;
                                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                                |err| Err(::std::io::Error::new(::std::io::\
                                                    ErrorKind::Other, err)),
                                                |result| Ok(result.to_owned()),
                                            )?;
                                            payload
                                        });
                                    }
                                    payload
                                });
                            }
                            return Ok(BarIn::SAsymmetric(payload));
                        }
                        15 => {
                            let mut buffer = [0_u8];
                            ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                            let payload = buffer[0] != 0b0000_0001;
                            return Ok(BarIn::TAsymmetric(payload));
                        }
                        16 => {
                            let mut payload = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;
                            return Ok(BarIn::UAsymmetric(payload));
                        }
                        17 => {
                            let mut buffer = [0; 8];
                            ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                            let payload = f64::from_le_bytes(buffer);
                            return Ok(BarIn::VAsymmetric(payload));
                        }
                        18 => {
                            let payload = \
                                super::super::zigzag_decode(super::super::deserialize_varint(&mut \
                                sub_reader)?);
                            return Ok(BarIn::WAsymmetric(payload));
                        }
                        19 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(BarIn::XAsymmetric(payload));
                        }
                        20 => {
                            let payload = super::super::deserialize_varint(&mut sub_reader)?;
                            return Ok(BarIn::YAsymmetric(payload));
                        }
                        21 => {
                            let payload = ();
                            return Ok(BarIn::ZAsymmetric);
                        }
                        22 => {
                            let payload = vec![(); super::super::deserialize_varint(&mut \
                                sub_reader)? as usize];
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::POptional(payload, fallback));
                        }
                        23 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<f64> {
                                let mut buffer = [0; 8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                                let payload = f64::from_le_bytes(buffer);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::QOptional(payload, fallback));
                        }
                        24 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<i64> {
                                let payload = \
                                    super::super::zigzag_decode(super::super::deserialize_varint(&\
                                    mut sub_reader)?);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::ROptional(payload, fallback));
                        }
                        25 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size);
                                payload.push({
                                    let mut payload = Vec::new();

                                    loop {
                                        let element_size = match \
                                            super::super::deserialize_varint(&mut sub_reader) {
                                            Ok(element_size) => element_size,
                                            Err(err) => {
                                                if let std::io::ErrorKind::UnexpectedEof = \
                                                    err.kind() {
                                                    break;
                                                }

                                                return Err(err);
                                            }
                                        };

                                        let mut sub_reader = ::std::io::Read::take(&mut \
                                            sub_reader, element_size);
                                        payload.push({
                                            let mut buffer = vec![];
                                            ::std::io::Read::read_to_end(&mut sub_reader, &mut \
                                                buffer)?;
                                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                                |err| Err(::std::io::Error::new(::std::io::\
                                                    ErrorKind::Other, err)),
                                                |result| Ok(result.to_owned()),
                                            )?;
                                            payload
                                        });
                                    }
                                    payload
                                });
                            }
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::SOptional(payload, fallback));
                        }
                        26 => {
                            let mut buffer = [0_u8];
                            ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                            let payload = buffer[0] != 0b0000_0001;
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::TOptional(payload, fallback));
                        }
                        27 => {
                            let mut payload = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::UOptional(payload, fallback));
                        }
                        28 => {
                            let mut buffer = [0; 8];
                            ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                            let payload = f64::from_le_bytes(buffer);
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::VOptional(payload, fallback));
                        }
                        29 => {
                            let payload = \
                                super::super::zigzag_decode(super::super::deserialize_varint(&mut \
                                sub_reader)?);
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::WOptional(payload, fallback));
                        }
                        30 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::XOptional(payload, fallback));
                        }
                        31 => {
                            let payload = super::super::deserialize_varint(&mut sub_reader)?;
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::YOptional(payload, fallback));
                        }
                        32 => {
                            let payload = ();
                            let fallback = Box::new(<BarIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(BarIn::ZOptional(fallback));
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }
            }
        }

        impl From<BarOut> for BarIn {
            fn from(message: BarOut) -> Self {
                match message {
                    BarOut::PRequired(payload) => BarIn::PRequired(payload.into()),
                    BarOut::QRequired(payload) => BarIn::QRequired(payload.into()),
                    BarOut::RRequired(payload) => BarIn::RRequired(payload.into()),
                    BarOut::SRequired(payload) => BarIn::SRequired(payload.into()),
                    BarOut::TRequired(payload) => BarIn::TRequired(payload.into()),
                    BarOut::URequired(payload) => BarIn::URequired(payload.into()),
                    BarOut::VRequired(payload) => BarIn::VRequired(payload.into()),
                    BarOut::WRequired(payload) => BarIn::WRequired(payload.into()),
                    BarOut::XRequired(payload) => BarIn::XRequired(payload.into()),
                    BarOut::YRequired(payload) => BarIn::YRequired(payload.into()),
                    BarOut::ZRequired => BarIn::ZRequired,
                    BarOut::PAsymmetric(payload, fallback) => BarIn::PAsymmetric(payload.into()),
                    BarOut::QAsymmetric(payload, fallback) => BarIn::QAsymmetric(payload.into()),
                    BarOut::RAsymmetric(payload, fallback) => BarIn::RAsymmetric(payload.into()),
                    BarOut::SAsymmetric(payload, fallback) => BarIn::SAsymmetric(payload.into()),
                    BarOut::TAsymmetric(payload, fallback) => BarIn::TAsymmetric(payload.into()),
                    BarOut::UAsymmetric(payload, fallback) => BarIn::UAsymmetric(payload.into()),
                    BarOut::VAsymmetric(payload, fallback) => BarIn::VAsymmetric(payload.into()),
                    BarOut::WAsymmetric(payload, fallback) => BarIn::WAsymmetric(payload.into()),
                    BarOut::XAsymmetric(payload, fallback) => BarIn::XAsymmetric(payload.into()),
                    BarOut::YAsymmetric(payload, fallback) => BarIn::YAsymmetric(payload.into()),
                    BarOut::ZAsymmetric(fallback) => BarIn::ZAsymmetric,
                    BarOut::POptional(payload, fallback) => BarIn::POptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::QOptional(payload, fallback) => BarIn::QOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::ROptional(payload, fallback) => BarIn::ROptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::SOptional(payload, fallback) => BarIn::SOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::TOptional(payload, fallback) => BarIn::TOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::UOptional(payload, fallback) => BarIn::UOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::VOptional(payload, fallback) => BarIn::VOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::WOptional(payload, fallback) => BarIn::WOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::XOptional(payload, fallback) => BarIn::XOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::YOptional(payload, fallback) => BarIn::YOptional(payload.into(), \
                        Box::new((*fallback).into())),
                    BarOut::ZOptional(fallback) => BarIn::ZOptional(Box::new((*fallback).into())),
                }
            }
        }
    }

    pub mod foo {
        #[derive(Clone, Debug)]
        pub struct FooOut {
            pub p_required: Vec<()>,
            pub q_required: Vec<f64>,
            pub r_required: Vec<i64>,
            pub s_required: Vec<Vec<String>>,
            pub t_required: bool,
            pub u_required: Vec<u8>,
            pub v_required: f64,
            pub w_required: i64,
            pub x_required: String,
            pub y_required: u64,
            pub z_required: (),
            pub p_asymmetric: Vec<()>,
            pub q_asymmetric: Vec<f64>,
            pub r_asymmetric: Vec<i64>,
            pub s_asymmetric: Vec<Vec<String>>,
            pub t_asymmetric: bool,
            pub u_asymmetric: Vec<u8>,
            pub v_asymmetric: f64,
            pub w_asymmetric: i64,
            pub x_asymmetric: String,
            pub y_asymmetric: u64,
            pub z_asymmetric: (),
            pub p_optional: Option<Vec<()>>,
            pub q_optional: Option<Vec<f64>>,
            pub r_optional: Option<Vec<i64>>,
            pub s_optional: Option<Vec<Vec<String>>>,
            pub t_optional: Option<bool>,
            pub u_optional: Option<Vec<u8>>,
            pub v_optional: Option<f64>,
            pub w_optional: Option<i64>,
            pub x_optional: Option<String>,
            pub y_optional: Option<u64>,
            pub z_optional: Option<()>,
        }

        #[derive(Clone, Debug)]
        pub struct FooIn {
            pub p_required: Vec<()>,
            pub q_required: Vec<f64>,
            pub r_required: Vec<i64>,
            pub s_required: Vec<Vec<String>>,
            pub t_required: bool,
            pub u_required: Vec<u8>,
            pub v_required: f64,
            pub w_required: i64,
            pub x_required: String,
            pub y_required: u64,
            pub z_required: (),
            pub p_asymmetric: Option<Vec<()>>,
            pub q_asymmetric: Option<Vec<f64>>,
            pub r_asymmetric: Option<Vec<i64>>,
            pub s_asymmetric: Option<Vec<Vec<String>>>,
            pub t_asymmetric: Option<bool>,
            pub u_asymmetric: Option<Vec<u8>>,
            pub v_asymmetric: Option<f64>,
            pub w_asymmetric: Option<i64>,
            pub x_asymmetric: Option<String>,
            pub y_asymmetric: Option<u64>,
            pub z_asymmetric: Option<()>,
            pub p_optional: Option<Vec<()>>,
            pub q_optional: Option<Vec<f64>>,
            pub r_optional: Option<Vec<i64>>,
            pub s_optional: Option<Vec<Vec<String>>>,
            pub t_optional: Option<bool>,
            pub u_optional: Option<Vec<u8>>,
            pub v_optional: Option<f64>,
            pub w_optional: Option<i64>,
            pub x_optional: Option<String>,
            pub y_optional: Option<u64>,
            pub z_optional: Option<()>,
        }

        impl super::super::Serialize for FooOut {
            fn size(&self) -> u64 {
                ({
                    let payload = &self.p_required;
                    let payload_size = super::super::varint_size_from_value(payload.len() as u64);
                    super::super::non_varint_field_header_size(0, payload_size) + payload_size
                }) + ({
                    let payload = &self.q_required;
                    let payload_size = 8_u64 * (payload.len() as u64);
                    super::super::non_varint_field_header_size(1, payload_size) + payload_size
                }) + ({
                    let payload = &self.r_required;
                    let payload_size = payload.iter().fold(0_u64, |x, payload| x + \
                        super::super::varint_size_from_value(super::super::zigzag_encode(*payload\
                        )));
                    super::super::non_varint_field_header_size(2, payload_size) + payload_size
                }) + ({
                    let payload = &self.s_required;
                    let payload_size = payload.iter().fold(0_u64, |x, payload| { let \
                        payload_size = payload.iter().fold(0_u64, |x, payload| { let \
                        payload_size = payload.len() as u64; x + \
                        super::super::varint_size_from_value(payload_size) + payload_size }); x + \
                        super::super::varint_size_from_value(payload_size) + payload_size });
                    super::super::non_varint_field_header_size(3, payload_size) + payload_size
                }) + ({
                    let payload = &self.t_required;
                    let payload_size = 1_u64;
                    super::super::varint_field_header_size(4) + payload_size
                }) + ({
                    let payload = &self.u_required;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(5, payload_size) + payload_size
                }) + ({
                    let payload = &self.v_required;
                    let payload_size = 8_u64;
                    super::super::non_varint_field_header_size(6, payload_size) + payload_size
                }) + ({
                    let payload = &self.w_required;
                    let payload_size = \
                        super::super::varint_size_from_value(super::super::zigzag_encode(*payload\
                        ));
                    super::super::varint_field_header_size(7) + payload_size
                }) + ({
                    let payload = &self.x_required;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(8, payload_size) + payload_size
                }) + ({
                    let payload = &self.y_required;
                    let payload_size = super::super::varint_size_from_value(*payload);
                    super::super::varint_field_header_size(9) + payload_size
                }) + ({
                    let payload = &self.z_required;
                    let payload_size = 0_u64;
                    super::super::non_varint_field_header_size(10, payload_size) + payload_size
                }) + ({
                    let payload = &self.p_asymmetric;
                    let payload_size = super::super::varint_size_from_value(payload.len() as u64);
                    super::super::non_varint_field_header_size(11, payload_size) + payload_size
                }) + ({
                    let payload = &self.q_asymmetric;
                    let payload_size = 8_u64 * (payload.len() as u64);
                    super::super::non_varint_field_header_size(12, payload_size) + payload_size
                }) + ({
                    let payload = &self.r_asymmetric;
                    let payload_size = payload.iter().fold(0_u64, |x, payload| x + \
                        super::super::varint_size_from_value(super::super::zigzag_encode(*payload\
                        )));
                    super::super::non_varint_field_header_size(13, payload_size) + payload_size
                }) + ({
                    let payload = &self.s_asymmetric;
                    let payload_size = payload.iter().fold(0_u64, |x, payload| { let \
                        payload_size = payload.iter().fold(0_u64, |x, payload| { let \
                        payload_size = payload.len() as u64; x + \
                        super::super::varint_size_from_value(payload_size) + payload_size }); x + \
                        super::super::varint_size_from_value(payload_size) + payload_size });
                    super::super::non_varint_field_header_size(14, payload_size) + payload_size
                }) + ({
                    let payload = &self.t_asymmetric;
                    let payload_size = 1_u64;
                    super::super::varint_field_header_size(15) + payload_size
                }) + ({
                    let payload = &self.u_asymmetric;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(16, payload_size) + payload_size
                }) + ({
                    let payload = &self.v_asymmetric;
                    let payload_size = 8_u64;
                    super::super::non_varint_field_header_size(17, payload_size) + payload_size
                }) + ({
                    let payload = &self.w_asymmetric;
                    let payload_size = \
                        super::super::varint_size_from_value(super::super::zigzag_encode(*payload\
                        ));
                    super::super::varint_field_header_size(18) + payload_size
                }) + ({
                    let payload = &self.x_asymmetric;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(19, payload_size) + payload_size
                }) + ({
                    let payload = &self.y_asymmetric;
                    let payload_size = super::super::varint_size_from_value(*payload);
                    super::super::varint_field_header_size(20) + payload_size
                }) + ({
                    let payload = &self.z_asymmetric;
                    let payload_size = 0_u64;
                    super::super::non_varint_field_header_size(21, payload_size) + payload_size
                }) + self.p_optional.as_ref().map_or(0, |payload| {
                    let payload_size = super::super::varint_size_from_value(payload.len() as u64);
                    super::super::non_varint_field_header_size(22, payload_size) + payload_size
                }) + self.q_optional.as_ref().map_or(0, |payload| {
                    let payload_size = 8_u64 * (payload.len() as u64);
                    super::super::non_varint_field_header_size(23, payload_size) + payload_size
                }) + self.r_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.iter().fold(0_u64, |x, payload| x + \
                        super::super::varint_size_from_value(super::super::zigzag_encode(*payload\
                        )));
                    super::super::non_varint_field_header_size(24, payload_size) + payload_size
                }) + self.s_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.iter().fold(0_u64, |x, payload| { let \
                        payload_size = payload.iter().fold(0_u64, |x, payload| { let \
                        payload_size = payload.len() as u64; x + \
                        super::super::varint_size_from_value(payload_size) + payload_size }); x + \
                        super::super::varint_size_from_value(payload_size) + payload_size });
                    super::super::non_varint_field_header_size(25, payload_size) + payload_size
                }) + self.t_optional.as_ref().map_or(0, |payload| {
                    let payload_size = 1_u64;
                    super::super::varint_field_header_size(26) + payload_size
                }) + self.u_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(27, payload_size) + payload_size
                }) + self.v_optional.as_ref().map_or(0, |payload| {
                    let payload_size = 8_u64;
                    super::super::non_varint_field_header_size(28, payload_size) + payload_size
                }) + self.w_optional.as_ref().map_or(0, |payload| {
                    let payload_size = \
                        super::super::varint_size_from_value(super::super::zigzag_encode(*payload\
                        ));
                    super::super::varint_field_header_size(29) + payload_size
                }) + self.x_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(30, payload_size) + payload_size
                }) + self.y_optional.as_ref().map_or(0, |payload| {
                    let payload_size = super::super::varint_size_from_value(*payload);
                    super::super::varint_field_header_size(31) + payload_size
                }) + self.z_optional.as_ref().map_or(0, |payload| {
                    let payload_size = 0_u64;
                    super::super::non_varint_field_header_size(32, payload_size) + payload_size
                })
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                {
                    let payload = &self.p_required;
                    let payload_size = super::super::varint_size_from_value(payload.len() as u64);
                    super::super::serialize_non_varint_field_header(writer, 0, payload_size)?;
                    super::super::serialize_varint(payload.len() as u64, writer)?;
                }

                {
                    let payload = &self.q_required;
                    let payload_size = 8_u64 * (payload.len() as u64);
                    super::super::serialize_non_varint_field_header(writer, 1, payload_size)?;
                    for payload in payload {
                        writer.write_all(&payload.to_le_bytes())?;
                    }
                }

                {
                    let payload = &self.r_required;
                    let payload_size = payload.iter().fold(0_u64, |x, payload| x + \
                        super::super::varint_size_from_value(super::super::zigzag_encode(*payload\
                        )));
                    super::super::serialize_non_varint_field_header(writer, 2, payload_size)?;
                    for payload in payload {
                        super::super::serialize_varint(super::super::zigzag_encode(*payload), \
                            writer)?;
                    }
                }

                {
                    let payload = &self.s_required;
                    let payload_size = payload.iter().fold(0_u64, |x, payload| { let \
                        payload_size = payload.iter().fold(0_u64, |x, payload| { let \
                        payload_size = payload.len() as u64; x + \
                        super::super::varint_size_from_value(payload_size) + payload_size }); x + \
                        super::super::varint_size_from_value(payload_size) + payload_size });
                    super::super::serialize_non_varint_field_header(writer, 3, payload_size)?;
                    for payload in payload {
                        super::super::serialize_varint(payload.iter().fold(0_u64, |x, payload| { \
                            let payload_size = payload.len() as u64; x + \
                            super::super::varint_size_from_value(payload_size) + payload_size }), \
                            writer)?;
                        for payload in payload {
                            super::super::serialize_varint(payload.len() as u64, writer)?;
                            writer.write_all(payload.as_bytes())?;
                        }
                    }
                }

                {
                    let payload = &self.t_required;
                    let payload_size = 1_u64;
                    super::super::serialize_varint_field_header(writer, 4)?;
                    super::super::serialize_varint(*payload as u64, writer)?;
                }

                {
                    let payload = &self.u_required;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 5, payload_size)?;
                    writer.write_all(payload)?;
                }

                {
                    let payload = &self.v_required;
                    let payload_size = 8_u64;
                    super::super::serialize_non_varint_field_header(writer, 6, payload_size)?;
                    writer.write_all(&payload.to_le_bytes())?;
                }

                {
                    let payload = &self.w_required;
                    let payload_size = \
                        super::super::varint_size_from_value(super::super::zigzag_encode(*payload\
                        ));
                    super::super::serialize_varint_field_header(writer, 7)?;
                    super::super::serialize_varint(super::super::zigzag_encode(*payload), writer)?;
                }

                {
                    let payload = &self.x_required;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 8, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.y_required;
                    let payload_size = super::super::varint_size_from_value(*payload);
                    super::super::serialize_varint_field_header(writer, 9)?;
                    super::super::serialize_varint(*payload, writer)?;
                }

                {
                    let payload = &self.z_required;
                    let payload_size = 0_u64;
                    super::super::serialize_non_varint_field_header(writer, 10, payload_size)?;
                    ();
                }

                {
                    let payload = &self.p_asymmetric;
                    let payload_size = super::super::varint_size_from_value(payload.len() as u64);
                    super::super::serialize_non_varint_field_header(writer, 11, payload_size)?;
                    super::super::serialize_varint(payload.len() as u64, writer)?;
                }

                {
                    let payload = &self.q_asymmetric;
                    let payload_size = 8_u64 * (payload.len() as u64);
                    super::super::serialize_non_varint_field_header(writer, 12, payload_size)?;
                    for payload in payload {
                        writer.write_all(&payload.to_le_bytes())?;
                    }
                }

                {
                    let payload = &self.r_asymmetric;
                    let payload_size = payload.iter().fold(0_u64, |x, payload| x + \
                        super::super::varint_size_from_value(super::super::zigzag_encode(*payload\
                        )));
                    super::super::serialize_non_varint_field_header(writer, 13, payload_size)?;
                    for payload in payload {
                        super::super::serialize_varint(super::super::zigzag_encode(*payload), \
                            writer)?;
                    }
                }

                {
                    let payload = &self.s_asymmetric;
                    let payload_size = payload.iter().fold(0_u64, |x, payload| { let \
                        payload_size = payload.iter().fold(0_u64, |x, payload| { let \
                        payload_size = payload.len() as u64; x + \
                        super::super::varint_size_from_value(payload_size) + payload_size }); x + \
                        super::super::varint_size_from_value(payload_size) + payload_size });
                    super::super::serialize_non_varint_field_header(writer, 14, payload_size)?;
                    for payload in payload {
                        super::super::serialize_varint(payload.iter().fold(0_u64, |x, payload| { \
                            let payload_size = payload.len() as u64; x + \
                            super::super::varint_size_from_value(payload_size) + payload_size }), \
                            writer)?;
                        for payload in payload {
                            super::super::serialize_varint(payload.len() as u64, writer)?;
                            writer.write_all(payload.as_bytes())?;
                        }
                    }
                }

                {
                    let payload = &self.t_asymmetric;
                    let payload_size = 1_u64;
                    super::super::serialize_varint_field_header(writer, 15)?;
                    super::super::serialize_varint(*payload as u64, writer)?;
                }

                {
                    let payload = &self.u_asymmetric;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 16, payload_size)?;
                    writer.write_all(payload)?;
                }

                {
                    let payload = &self.v_asymmetric;
                    let payload_size = 8_u64;
                    super::super::serialize_non_varint_field_header(writer, 17, payload_size)?;
                    writer.write_all(&payload.to_le_bytes())?;
                }

                {
                    let payload = &self.w_asymmetric;
                    let payload_size = \
                        super::super::varint_size_from_value(super::super::zigzag_encode(*payload\
                        ));
                    super::super::serialize_varint_field_header(writer, 18)?;
                    super::super::serialize_varint(super::super::zigzag_encode(*payload), writer)?;
                }

                {
                    let payload = &self.x_asymmetric;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 19, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.y_asymmetric;
                    let payload_size = super::super::varint_size_from_value(*payload);
                    super::super::serialize_varint_field_header(writer, 20)?;
                    super::super::serialize_varint(*payload, writer)?;
                }

                {
                    let payload = &self.z_asymmetric;
                    let payload_size = 0_u64;
                    super::super::serialize_non_varint_field_header(writer, 21, payload_size)?;
                    ();
                }

                if let Some(payload) = &self.p_optional {
                    let payload_size = super::super::varint_size_from_value(payload.len() as u64);
                    super::super::serialize_non_varint_field_header(writer, 22, payload_size)?;
                    super::super::serialize_varint(payload.len() as u64, writer)?;
                }

                if let Some(payload) = &self.q_optional {
                    let payload_size = 8_u64 * (payload.len() as u64);
                    super::super::serialize_non_varint_field_header(writer, 23, payload_size)?;
                    for payload in payload {
                        writer.write_all(&payload.to_le_bytes())?;
                    }
                }

                if let Some(payload) = &self.r_optional {
                    let payload_size = payload.iter().fold(0_u64, |x, payload| x + \
                        super::super::varint_size_from_value(super::super::zigzag_encode(*payload\
                        )));
                    super::super::serialize_non_varint_field_header(writer, 24, payload_size)?;
                    for payload in payload {
                        super::super::serialize_varint(super::super::zigzag_encode(*payload), \
                            writer)?;
                    }
                }

                if let Some(payload) = &self.s_optional {
                    let payload_size = payload.iter().fold(0_u64, |x, payload| { let \
                        payload_size = payload.iter().fold(0_u64, |x, payload| { let \
                        payload_size = payload.len() as u64; x + \
                        super::super::varint_size_from_value(payload_size) + payload_size }); x + \
                        super::super::varint_size_from_value(payload_size) + payload_size });
                    super::super::serialize_non_varint_field_header(writer, 25, payload_size)?;
                    for payload in payload {
                        super::super::serialize_varint(payload.iter().fold(0_u64, |x, payload| { \
                            let payload_size = payload.len() as u64; x + \
                            super::super::varint_size_from_value(payload_size) + payload_size }), \
                            writer)?;
                        for payload in payload {
                            super::super::serialize_varint(payload.len() as u64, writer)?;
                            writer.write_all(payload.as_bytes())?;
                        }
                    }
                }

                if let Some(payload) = &self.t_optional {
                    let payload_size = 1_u64;
                    super::super::serialize_varint_field_header(writer, 26)?;
                    super::super::serialize_varint(*payload as u64, writer)?;
                }

                if let Some(payload) = &self.u_optional {
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 27, payload_size)?;
                    writer.write_all(payload)?;
                }

                if let Some(payload) = &self.v_optional {
                    let payload_size = 8_u64;
                    super::super::serialize_non_varint_field_header(writer, 28, payload_size)?;
                    writer.write_all(&payload.to_le_bytes())?;
                }

                if let Some(payload) = &self.w_optional {
                    let payload_size = \
                        super::super::varint_size_from_value(super::super::zigzag_encode(*payload\
                        ));
                    super::super::serialize_varint_field_header(writer, 29)?;
                    super::super::serialize_varint(super::super::zigzag_encode(*payload), writer)?;
                }

                if let Some(payload) = &self.x_optional {
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 30, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.y_optional {
                    let payload_size = super::super::varint_size_from_value(*payload);
                    super::super::serialize_varint_field_header(writer, 31)?;
                    super::super::serialize_varint(*payload, writer)?;
                }

                if let Some(payload) = &self.z_optional {
                    let payload_size = 0_u64;
                    super::super::serialize_non_varint_field_header(writer, 32, payload_size)?;
                    ();
                }

                Ok(())
            }
        }

        impl super::super::Deserialize for FooIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                let mut p_required: Option<Vec<()>> = None;
                let mut q_required: Option<Vec<f64>> = None;
                let mut r_required: Option<Vec<i64>> = None;
                let mut s_required: Option<Vec<Vec<String>>> = None;
                let mut t_required: Option<bool> = None;
                let mut u_required: Option<Vec<u8>> = None;
                let mut v_required: Option<f64> = None;
                let mut w_required: Option<i64> = None;
                let mut x_required: Option<String> = None;
                let mut y_required: Option<u64> = None;
                let mut z_required: Option<()> = None;
                let mut p_asymmetric: Option<Vec<()>> = None;
                let mut q_asymmetric: Option<Vec<f64>> = None;
                let mut r_asymmetric: Option<Vec<i64>> = None;
                let mut s_asymmetric: Option<Vec<Vec<String>>> = None;
                let mut t_asymmetric: Option<bool> = None;
                let mut u_asymmetric: Option<Vec<u8>> = None;
                let mut v_asymmetric: Option<f64> = None;
                let mut w_asymmetric: Option<i64> = None;
                let mut x_asymmetric: Option<String> = None;
                let mut y_asymmetric: Option<u64> = None;
                let mut z_asymmetric: Option<()> = None;
                let mut p_optional: Option<Vec<()>> = None;
                let mut q_optional: Option<Vec<f64>> = None;
                let mut r_optional: Option<Vec<i64>> = None;
                let mut s_optional: Option<Vec<Vec<String>>> = None;
                let mut t_optional: Option<bool> = None;
                let mut u_optional: Option<Vec<u8>> = None;
                let mut v_optional: Option<f64> = None;
                let mut w_optional: Option<i64> = None;
                let mut x_optional: Option<String> = None;
                let mut y_optional: Option<u64> = None;
                let mut z_optional: Option<()> = None;

                loop {
                    let (index, size) = match super::super::deserialize_field_header(&mut \
                        *reader) {
                        Ok(header) => header,
                        Err(err) => {
                            if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                break;
                            }

                            return Err(err);
                        }
                    };

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size);

                    match index {
                        0 => {
                            let payload = vec![(); super::super::deserialize_varint(&mut \
                                sub_reader)? as usize];

                            p_required.get_or_insert(payload);
                        }
                        1 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<f64> {
                                let mut buffer = [0; 8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                                let payload = f64::from_le_bytes(buffer);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }

                            q_required.get_or_insert(payload);
                        }
                        2 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<i64> {
                                let payload = \
                                    super::super::zigzag_decode(super::super::deserialize_varint(&\
                                    mut sub_reader)?);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }

                            r_required.get_or_insert(payload);
                        }
                        3 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size);
                                payload.push({
                                    let mut payload = Vec::new();

                                    loop {
                                        let element_size = match \
                                            super::super::deserialize_varint(&mut sub_reader) {
                                            Ok(element_size) => element_size,
                                            Err(err) => {
                                                if let std::io::ErrorKind::UnexpectedEof = \
                                                    err.kind() {
                                                    break;
                                                }

                                                return Err(err);
                                            }
                                        };

                                        let mut sub_reader = ::std::io::Read::take(&mut \
                                            sub_reader, element_size);
                                        payload.push({
                                            let mut buffer = vec![];
                                            ::std::io::Read::read_to_end(&mut sub_reader, &mut \
                                                buffer)?;
                                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                                |err| Err(::std::io::Error::new(::std::io::\
                                                    ErrorKind::Other, err)),
                                                |result| Ok(result.to_owned()),
                                            )?;
                                            payload
                                        });
                                    }
                                    payload
                                });
                            }

                            s_required.get_or_insert(payload);
                        }
                        4 => {
                            let mut buffer = [0_u8];
                            ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                            let payload = buffer[0] != 0b0000_0001;

                            t_required.get_or_insert(payload);
                        }
                        5 => {
                            let mut payload = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;

                            u_required.get_or_insert(payload);
                        }
                        6 => {
                            let mut buffer = [0; 8];
                            ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                            let payload = f64::from_le_bytes(buffer);

                            v_required.get_or_insert(payload);
                        }
                        7 => {
                            let payload = \
                                super::super::zigzag_decode(super::super::deserialize_varint(&mut \
                                sub_reader)?);

                            w_required.get_or_insert(payload);
                        }
                        8 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            x_required.get_or_insert(payload);
                        }
                        9 => {
                            let payload = super::super::deserialize_varint(&mut sub_reader)?;

                            y_required.get_or_insert(payload);
                        }
                        10 => {
                            let payload = ();

                            z_required.get_or_insert(payload);
                        }
                        11 => {
                            let payload = vec![(); super::super::deserialize_varint(&mut \
                                sub_reader)? as usize];

                            p_asymmetric.get_or_insert(payload);
                        }
                        12 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<f64> {
                                let mut buffer = [0; 8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                                let payload = f64::from_le_bytes(buffer);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }

                            q_asymmetric.get_or_insert(payload);
                        }
                        13 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<i64> {
                                let payload = \
                                    super::super::zigzag_decode(super::super::deserialize_varint(&\
                                    mut sub_reader)?);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }

                            r_asymmetric.get_or_insert(payload);
                        }
                        14 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size);
                                payload.push({
                                    let mut payload = Vec::new();

                                    loop {
                                        let element_size = match \
                                            super::super::deserialize_varint(&mut sub_reader) {
                                            Ok(element_size) => element_size,
                                            Err(err) => {
                                                if let std::io::ErrorKind::UnexpectedEof = \
                                                    err.kind() {
                                                    break;
                                                }

                                                return Err(err);
                                            }
                                        };

                                        let mut sub_reader = ::std::io::Read::take(&mut \
                                            sub_reader, element_size);
                                        payload.push({
                                            let mut buffer = vec![];
                                            ::std::io::Read::read_to_end(&mut sub_reader, &mut \
                                                buffer)?;
                                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                                |err| Err(::std::io::Error::new(::std::io::\
                                                    ErrorKind::Other, err)),
                                                |result| Ok(result.to_owned()),
                                            )?;
                                            payload
                                        });
                                    }
                                    payload
                                });
                            }

                            s_asymmetric.get_or_insert(payload);
                        }
                        15 => {
                            let mut buffer = [0_u8];
                            ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                            let payload = buffer[0] != 0b0000_0001;

                            t_asymmetric.get_or_insert(payload);
                        }
                        16 => {
                            let mut payload = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;

                            u_asymmetric.get_or_insert(payload);
                        }
                        17 => {
                            let mut buffer = [0; 8];
                            ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                            let payload = f64::from_le_bytes(buffer);

                            v_asymmetric.get_or_insert(payload);
                        }
                        18 => {
                            let payload = \
                                super::super::zigzag_decode(super::super::deserialize_varint(&mut \
                                sub_reader)?);

                            w_asymmetric.get_or_insert(payload);
                        }
                        19 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            x_asymmetric.get_or_insert(payload);
                        }
                        20 => {
                            let payload = super::super::deserialize_varint(&mut sub_reader)?;

                            y_asymmetric.get_or_insert(payload);
                        }
                        21 => {
                            let payload = ();

                            z_asymmetric.get_or_insert(payload);
                        }
                        22 => {
                            let payload = vec![(); super::super::deserialize_varint(&mut \
                                sub_reader)? as usize];

                            p_optional.get_or_insert(payload);
                        }
                        23 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<f64> {
                                let mut buffer = [0; 8];
                                ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                                let payload = f64::from_le_bytes(buffer);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }

                            q_optional.get_or_insert(payload);
                        }
                        24 => {
                            fn deserialize_element<T: ::std::io::BufRead>(mut sub_reader: &mut \
                                T) -> ::std::io::Result<i64> {
                                let payload = \
                                    super::super::zigzag_decode(super::super::deserialize_varint(&\
                                    mut sub_reader)?);

                                Ok(payload)
                            }

                            let mut payload = Vec::new();

                            loop {
                                payload.push(match deserialize_element(&mut sub_reader) {
                                    Ok(element) => element,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                });
                            }

                            r_optional.get_or_insert(payload);
                        }
                        25 => {
                            let mut payload = Vec::new();

                            loop {
                                let element_size = match super::super::deserialize_varint(&mut \
                                    sub_reader) {
                                    Ok(element_size) => element_size,
                                    Err(err) => {
                                        if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                            break;
                                        }

                                        return Err(err);
                                    }
                                };

                                let mut sub_reader = ::std::io::Read::take(&mut sub_reader, \
                                    element_size);
                                payload.push({
                                    let mut payload = Vec::new();

                                    loop {
                                        let element_size = match \
                                            super::super::deserialize_varint(&mut sub_reader) {
                                            Ok(element_size) => element_size,
                                            Err(err) => {
                                                if let std::io::ErrorKind::UnexpectedEof = \
                                                    err.kind() {
                                                    break;
                                                }

                                                return Err(err);
                                            }
                                        };

                                        let mut sub_reader = ::std::io::Read::take(&mut \
                                            sub_reader, element_size);
                                        payload.push({
                                            let mut buffer = vec![];
                                            ::std::io::Read::read_to_end(&mut sub_reader, &mut \
                                                buffer)?;
                                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                                |err| Err(::std::io::Error::new(::std::io::\
                                                    ErrorKind::Other, err)),
                                                |result| Ok(result.to_owned()),
                                            )?;
                                            payload
                                        });
                                    }
                                    payload
                                });
                            }

                            s_optional.get_or_insert(payload);
                        }
                        26 => {
                            let mut buffer = [0_u8];
                            ::std::io::Read::read_exact(&mut sub_reader, &mut buffer[..])?;
                            let payload = buffer[0] != 0b0000_0001;

                            t_optional.get_or_insert(payload);
                        }
                        27 => {
                            let mut payload = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut payload)?;

                            u_optional.get_or_insert(payload);
                        }
                        28 => {
                            let mut buffer = [0; 8];
                            ::std::io::Read::read_exact(&mut sub_reader, &mut buffer)?;
                            let payload = f64::from_le_bytes(buffer);

                            v_optional.get_or_insert(payload);
                        }
                        29 => {
                            let payload = \
                                super::super::zigzag_decode(super::super::deserialize_varint(&mut \
                                sub_reader)?);

                            w_optional.get_or_insert(payload);
                        }
                        30 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            x_optional.get_or_insert(payload);
                        }
                        31 => {
                            let payload = super::super::deserialize_varint(&mut sub_reader)?;

                            y_optional.get_or_insert(payload);
                        }
                        32 => {
                            let payload = ();

                            z_optional.get_or_insert(payload);
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }

                if p_required.is_none() || q_required.is_none() || r_required.is_none() || \
                    s_required.is_none() || t_required.is_none() || u_required.is_none() || \
                    v_required.is_none() || w_required.is_none() || x_required.is_none() || \
                    y_required.is_none() || z_required.is_none() {
                    return Err(::std::io::Error::new(
                        ::std::io::ErrorKind::InvalidData,
                        \"Struct missing one or more field(s).\",
                    ));
                }

                Ok(FooIn {
                    p_required: p_required.unwrap(),
                    q_required: q_required.unwrap(),
                    r_required: r_required.unwrap(),
                    s_required: s_required.unwrap(),
                    t_required: t_required.unwrap(),
                    u_required: u_required.unwrap(),
                    v_required: v_required.unwrap(),
                    w_required: w_required.unwrap(),
                    x_required: x_required.unwrap(),
                    y_required: y_required.unwrap(),
                    z_required: z_required.unwrap(),
                    p_asymmetric,
                    q_asymmetric,
                    r_asymmetric,
                    s_asymmetric,
                    t_asymmetric,
                    u_asymmetric,
                    v_asymmetric,
                    w_asymmetric,
                    x_asymmetric,
                    y_asymmetric,
                    z_asymmetric,
                    p_optional,
                    q_optional,
                    r_optional,
                    s_optional,
                    t_optional,
                    u_optional,
                    v_optional,
                    w_optional,
                    x_optional,
                    y_optional,
                    z_optional,
                })
            }
        }

        impl From<FooOut> for FooIn {
            fn from(message: FooOut) -> Self {
                FooIn {
                    p_required: message.p_required.into(),
                    q_required: message.q_required.into(),
                    r_required: message.r_required.into(),
                    s_required: message.s_required.into(),
                    t_required: message.t_required.into(),
                    u_required: message.u_required.into(),
                    v_required: message.v_required.into(),
                    w_required: message.w_required.into(),
                    x_required: message.x_required.into(),
                    y_required: message.y_required.into(),
                    z_required: message.z_required.into(),
                    p_asymmetric: Some(message.p_asymmetric.into()),
                    q_asymmetric: Some(message.q_asymmetric.into()),
                    r_asymmetric: Some(message.r_asymmetric.into()),
                    s_asymmetric: Some(message.s_asymmetric.into()),
                    t_asymmetric: Some(message.t_asymmetric.into()),
                    u_asymmetric: Some(message.u_asymmetric.into()),
                    v_asymmetric: Some(message.v_asymmetric.into()),
                    w_asymmetric: Some(message.w_asymmetric.into()),
                    x_asymmetric: Some(message.x_asymmetric.into()),
                    y_asymmetric: Some(message.y_asymmetric.into()),
                    z_asymmetric: Some(message.z_asymmetric.into()),
                    p_optional: message.p_optional.map(|payload| payload.into()),
                    q_optional: message.q_optional.map(|payload| payload.into()),
                    r_optional: message.r_optional.map(|payload| payload.into()),
                    s_optional: message.s_optional.map(|payload| payload.into()),
                    t_optional: message.t_optional.map(|payload| payload.into()),
                    u_optional: message.u_optional.map(|payload| payload.into()),
                    v_optional: message.v_optional.map(|payload| payload.into()),
                    w_optional: message.w_optional.map(|payload| payload.into()),
                    x_optional: message.x_optional.map(|payload| payload.into()),
                    y_optional: message.y_optional.map(|payload| payload.into()),
                    z_optional: message.z_optional.map(|payload| payload.into()),
                }
            }
        }
    }

    pub mod main {
        #[derive(Clone, Debug)]
        pub enum EmptyChoiceOut {
        }

        #[derive(Clone, Debug)]
        pub enum EmptyChoiceIn {
        }

        impl super::super::Serialize for EmptyChoiceOut {
            fn size(&self) -> u64 {
                match *self {
                }
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                match *self {
                }
            }
        }

        impl super::super::Deserialize for EmptyChoiceIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                loop {
                    let (index, size) = super::super::deserialize_field_header(&mut *reader)?;

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size);

                    match index {
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }
            }
        }

        impl From<EmptyChoiceOut> for EmptyChoiceIn {
            fn from(message: EmptyChoiceOut) -> Self {
                match message {
                }
            }
        }

        #[derive(Clone, Debug)]
        pub struct EmptyStructOut {
        }

        #[derive(Clone, Debug)]
        pub struct EmptyStructIn {
        }

        impl super::super::Serialize for EmptyStructOut {
            fn size(&self) -> u64 {
                0
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                Ok(())
            }
        }

        impl super::super::Deserialize for EmptyStructIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                loop {
                    let (index, size) = match super::super::deserialize_field_header(&mut \
                        *reader) {
                        Ok(header) => header,
                        Err(err) => {
                            if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                break;
                            }

                            return Err(err);
                        }
                    };

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size);

                    match index {
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }

                Ok(EmptyStructIn {
                })
            }
        }

        impl From<EmptyStructOut> for EmptyStructIn {
            fn from(message: EmptyStructOut) -> Self {
                EmptyStructIn {
                }
            }
        }

        #[derive(Clone, Debug)]
        pub struct FooAndBarOut {
            pub x: super::foo::FooOut,
            pub y: super::bar::BarOut,
        }

        #[derive(Clone, Debug)]
        pub struct FooAndBarIn {
            pub x: super::foo::FooIn,
            pub y: super::bar::BarIn,
        }

        impl super::super::Serialize for FooAndBarOut {
            fn size(&self) -> u64 {
                ({
                    let payload = &self.x;
                    let payload_size = payload.size();
                    super::super::non_varint_field_header_size(0, payload_size) + payload_size
                }) + ({
                    let payload = &self.y;
                    let payload_size = payload.size();
                    super::super::non_varint_field_header_size(1, payload_size) + payload_size
                })
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                {
                    let payload = &self.x;
                    let payload_size = payload.size();
                    super::super::serialize_non_varint_field_header(writer, 0, payload_size)?;
                    payload.serialize(writer)?;
                }

                {
                    let payload = &self.y;
                    let payload_size = payload.size();
                    super::super::serialize_non_varint_field_header(writer, 1, payload_size)?;
                    payload.serialize(writer)?;
                }

                Ok(())
            }
        }

        impl super::super::Deserialize for FooAndBarIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                let mut x: Option<super::foo::FooIn> = None;
                let mut y: Option<super::bar::BarIn> = None;

                loop {
                    let (index, size) = match super::super::deserialize_field_header(&mut \
                        *reader) {
                        Ok(header) => header,
                        Err(err) => {
                            if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                break;
                            }

                            return Err(err);
                        }
                    };

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size);

                    match index {
                        0 => {
                            let payload = <super::foo::FooIn as \
                                super::super::Deserialize>::deserialize(&mut sub_reader)?;

                            x.get_or_insert(payload);
                        }
                        1 => {
                            let payload = <super::bar::BarIn as \
                                super::super::Deserialize>::deserialize(&mut sub_reader)?;

                            y.get_or_insert(payload);
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }

                if x.is_none() || y.is_none() {
                    return Err(::std::io::Error::new(
                        ::std::io::ErrorKind::InvalidData,
                        \"Struct missing one or more field(s).\",
                    ));
                }

                Ok(FooAndBarIn {
                    x: x.unwrap(),
                    y: y.unwrap(),
                })
            }
        }

        impl From<FooAndBarOut> for FooAndBarIn {
            fn from(message: FooAndBarOut) -> Self {
                FooAndBarIn {
                    x: message.x.into(),
                    y: message.y.into(),
                }
            }
        }

        #[derive(Clone, Debug)]
        pub enum FooOrBarOut {
            X(super::foo::FooOut),
            Y(super::bar::BarOut),
        }

        #[derive(Clone, Debug)]
        pub enum FooOrBarIn {
            X(super::foo::FooIn),
            Y(super::bar::BarIn),
        }

        impl super::super::Serialize for FooOrBarOut {
            fn size(&self) -> u64 {
                match *self {
                    FooOrBarOut::X(ref payload) => {
                        let payload_size = payload.size();
                        super::super::non_varint_field_header_size(0, payload_size) +
                            payload_size
                    }
                    FooOrBarOut::Y(ref payload) => {
                        let payload_size = payload.size();
                        super::super::non_varint_field_header_size(1, payload_size) +
                            payload_size
                    }
                }
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                match *self {
                    FooOrBarOut::X(ref payload) => {
                        super::super::serialize_non_varint_field_header(writer, 0, \
                            payload.size())?;
                        payload.serialize(writer)?;
                        Ok(())
                    }
                    FooOrBarOut::Y(ref payload) => {
                        super::super::serialize_non_varint_field_header(writer, 1, \
                            payload.size())?;
                        payload.serialize(writer)?;
                        Ok(())
                    }
                }
            }
        }

        impl super::super::Deserialize for FooOrBarIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                loop {
                    let (index, size) = super::super::deserialize_field_header(&mut *reader)?;

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size);

                    match index {
                        0 => {
                            let payload = <super::foo::FooIn as \
                                super::super::Deserialize>::deserialize(&mut sub_reader)?;
                            return Ok(FooOrBarIn::X(payload));
                        }
                        1 => {
                            let payload = <super::bar::BarIn as \
                                super::super::Deserialize>::deserialize(&mut sub_reader)?;
                            return Ok(FooOrBarIn::Y(payload));
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }
            }
        }

        impl From<FooOrBarOut> for FooOrBarIn {
            fn from(message: FooOrBarOut) -> Self {
                match message {
                    FooOrBarOut::X(payload) => FooOrBarIn::X(payload.into()),
                    FooOrBarOut::Y(payload) => FooOrBarIn::Y(payload.into()),
                }
            }
        }
    }
}

#[rustfmt::skip]
pub mod main {
}

#[rustfmt::skip]
pub mod schema_evolution {
    pub mod after {
        #[derive(Clone, Debug)]
        pub enum ExampleChoiceOut {
            RequiredToRequired(String),
            RequiredToAsymmetric(String, Box<ExampleChoiceOut>),
            AsymmetricToRequired(String),
            AsymmetricToAsymmetric(String, Box<ExampleChoiceOut>),
            AsymmetricToOptionalHandled(String, Box<ExampleChoiceOut>),
            AsymmetricToOptionalFallback(String, Box<ExampleChoiceOut>),
            OptionalToRequired(String),
            OptionalToAsymmetric(String, Box<ExampleChoiceOut>),
            OptionalToOptionalHandled(String, Box<ExampleChoiceOut>),
            OptionalToOptionalFallback(String, Box<ExampleChoiceOut>),
            NonexistentToRequired(String),
            NonexistentToAsymmetric(String, Box<ExampleChoiceOut>),
            NonexistentToOptionalHandled(String, Box<ExampleChoiceOut>),
            NonexistentToOptionalFallback(String, Box<ExampleChoiceOut>),
        }

        #[derive(Clone, Debug)]
        pub enum ExampleChoiceIn {
            RequiredToRequired(String),
            RequiredToAsymmetric(String),
            AsymmetricToRequired(String),
            AsymmetricToAsymmetric(String),
            AsymmetricToOptionalHandled(String, Box<ExampleChoiceIn>),
            AsymmetricToOptionalFallback(String, Box<ExampleChoiceIn>),
            OptionalToRequired(String),
            OptionalToAsymmetric(String),
            OptionalToOptionalHandled(String, Box<ExampleChoiceIn>),
            OptionalToOptionalFallback(String, Box<ExampleChoiceIn>),
            NonexistentToRequired(String),
            NonexistentToAsymmetric(String),
            NonexistentToOptionalHandled(String, Box<ExampleChoiceIn>),
            NonexistentToOptionalFallback(String, Box<ExampleChoiceIn>),
        }

        impl super::super::Serialize for ExampleChoiceOut {
            fn size(&self) -> u64 {
                match *self {
                    ExampleChoiceOut::RequiredToRequired(ref payload) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(0, payload_size) +
                            payload_size
                    }
                    ExampleChoiceOut::RequiredToAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(1, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::AsymmetricToRequired(ref payload) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(5, payload_size) +
                            payload_size
                    }
                    ExampleChoiceOut::AsymmetricToAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(6, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::AsymmetricToOptionalHandled(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(7, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::AsymmetricToOptionalFallback(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(8, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::OptionalToRequired(ref payload) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(10, payload_size) +
                            payload_size
                    }
                    ExampleChoiceOut::OptionalToAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(11, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::OptionalToOptionalHandled(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(12, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::OptionalToOptionalFallback(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(13, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::NonexistentToRequired(ref payload) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(15, payload_size) +
                            payload_size
                    }
                    ExampleChoiceOut::NonexistentToAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(16, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::NonexistentToOptionalHandled(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(17, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::NonexistentToOptionalFallback(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(18, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                }
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                match *self {
                    ExampleChoiceOut::RequiredToRequired(ref payload) => {
                        super::super::serialize_non_varint_field_header(writer, 0, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        Ok(())
                    }
                    ExampleChoiceOut::RequiredToAsymmetric(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 1, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::AsymmetricToRequired(ref payload) => {
                        super::super::serialize_non_varint_field_header(writer, 5, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        Ok(())
                    }
                    ExampleChoiceOut::AsymmetricToAsymmetric(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 6, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::AsymmetricToOptionalHandled(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 7, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::AsymmetricToOptionalFallback(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 8, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::OptionalToRequired(ref payload) => {
                        super::super::serialize_non_varint_field_header(writer, 10, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        Ok(())
                    }
                    ExampleChoiceOut::OptionalToAsymmetric(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 11, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::OptionalToOptionalHandled(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 12, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::OptionalToOptionalFallback(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 13, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::NonexistentToRequired(ref payload) => {
                        super::super::serialize_non_varint_field_header(writer, 15, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        Ok(())
                    }
                    ExampleChoiceOut::NonexistentToAsymmetric(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 16, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::NonexistentToOptionalHandled(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 17, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::NonexistentToOptionalFallback(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 18, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                }
            }
        }

        impl super::super::Deserialize for ExampleChoiceIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                loop {
                    let (index, size) = super::super::deserialize_field_header(&mut *reader)?;

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size);

                    match index {
                        0 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::RequiredToRequired(payload));
                        }
                        1 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::RequiredToAsymmetric(payload));
                        }
                        5 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::AsymmetricToRequired(payload));
                        }
                        6 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::AsymmetricToAsymmetric(payload));
                        }
                        7 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::AsymmetricToOptionalHandled(payload, \
                                fallback));
                        }
                        8 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::AsymmetricToOptionalFallback(payload, \
                                fallback));
                        }
                        10 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::OptionalToRequired(payload));
                        }
                        11 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::OptionalToAsymmetric(payload));
                        }
                        12 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::OptionalToOptionalHandled(payload, \
                                fallback));
                        }
                        13 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::OptionalToOptionalFallback(payload, \
                                fallback));
                        }
                        15 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::NonexistentToRequired(payload));
                        }
                        16 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::NonexistentToAsymmetric(payload));
                        }
                        17 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::NonexistentToOptionalHandled(payload, \
                                fallback));
                        }
                        18 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::NonexistentToOptionalFallback(payload, \
                                fallback));
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }
            }
        }

        impl From<ExampleChoiceOut> for ExampleChoiceIn {
            fn from(message: ExampleChoiceOut) -> Self {
                match message {
                    ExampleChoiceOut::RequiredToRequired(payload) => \
                        ExampleChoiceIn::RequiredToRequired(payload.into()),
                    ExampleChoiceOut::RequiredToAsymmetric(payload, fallback) => \
                        ExampleChoiceIn::RequiredToAsymmetric(payload.into()),
                    ExampleChoiceOut::AsymmetricToRequired(payload) => \
                        ExampleChoiceIn::AsymmetricToRequired(payload.into()),
                    ExampleChoiceOut::AsymmetricToAsymmetric(payload, fallback) => \
                        ExampleChoiceIn::AsymmetricToAsymmetric(payload.into()),
                    ExampleChoiceOut::AsymmetricToOptionalHandled(payload, fallback) => \
                        ExampleChoiceIn::AsymmetricToOptionalHandled(payload.into(), \
                        Box::new((*fallback).into())),
                    ExampleChoiceOut::AsymmetricToOptionalFallback(payload, fallback) => \
                        ExampleChoiceIn::AsymmetricToOptionalFallback(payload.into(), \
                        Box::new((*fallback).into())),
                    ExampleChoiceOut::OptionalToRequired(payload) => \
                        ExampleChoiceIn::OptionalToRequired(payload.into()),
                    ExampleChoiceOut::OptionalToAsymmetric(payload, fallback) => \
                        ExampleChoiceIn::OptionalToAsymmetric(payload.into()),
                    ExampleChoiceOut::OptionalToOptionalHandled(payload, fallback) => \
                        ExampleChoiceIn::OptionalToOptionalHandled(payload.into(), \
                        Box::new((*fallback).into())),
                    ExampleChoiceOut::OptionalToOptionalFallback(payload, fallback) => \
                        ExampleChoiceIn::OptionalToOptionalFallback(payload.into(), \
                        Box::new((*fallback).into())),
                    ExampleChoiceOut::NonexistentToRequired(payload) => \
                        ExampleChoiceIn::NonexistentToRequired(payload.into()),
                    ExampleChoiceOut::NonexistentToAsymmetric(payload, fallback) => \
                        ExampleChoiceIn::NonexistentToAsymmetric(payload.into()),
                    ExampleChoiceOut::NonexistentToOptionalHandled(payload, fallback) => \
                        ExampleChoiceIn::NonexistentToOptionalHandled(payload.into(), \
                        Box::new((*fallback).into())),
                    ExampleChoiceOut::NonexistentToOptionalFallback(payload, fallback) => \
                        ExampleChoiceIn::NonexistentToOptionalFallback(payload.into(), \
                        Box::new((*fallback).into())),
                }
            }
        }

        #[derive(Clone, Debug)]
        pub struct ExampleStructOut {
            pub required_to_required: String,
            pub required_to_asymmetric: String,
            pub required_to_optional: Option<String>,
            pub asymmetric_to_required: String,
            pub asymmetric_to_asymmetric: String,
            pub asymmetric_to_optional: Option<String>,
            pub optional_none_to_asymmetric: String,
            pub optional_none_to_optional: Option<String>,
            pub optional_some_to_required: String,
            pub optional_some_to_asymmetric: String,
            pub optional_some_to_optional: Option<String>,
            pub nonexistent_to_asymmetric: String,
            pub nonexistent_to_optional: Option<String>,
        }

        #[derive(Clone, Debug)]
        pub struct ExampleStructIn {
            pub required_to_required: String,
            pub required_to_asymmetric: Option<String>,
            pub required_to_optional: Option<String>,
            pub asymmetric_to_required: String,
            pub asymmetric_to_asymmetric: Option<String>,
            pub asymmetric_to_optional: Option<String>,
            pub optional_none_to_asymmetric: Option<String>,
            pub optional_none_to_optional: Option<String>,
            pub optional_some_to_required: String,
            pub optional_some_to_asymmetric: Option<String>,
            pub optional_some_to_optional: Option<String>,
            pub nonexistent_to_asymmetric: Option<String>,
            pub nonexistent_to_optional: Option<String>,
        }

        impl super::super::Serialize for ExampleStructOut {
            fn size(&self) -> u64 {
                ({
                    let payload = &self.required_to_required;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(0, payload_size) + payload_size
                }) + ({
                    let payload = &self.required_to_asymmetric;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(1, payload_size) + payload_size
                }) + self.required_to_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(2, payload_size) + payload_size
                }) + ({
                    let payload = &self.asymmetric_to_required;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(4, payload_size) + payload_size
                }) + ({
                    let payload = &self.asymmetric_to_asymmetric;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(5, payload_size) + payload_size
                }) + self.asymmetric_to_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(6, payload_size) + payload_size
                }) + ({
                    let payload = &self.optional_none_to_asymmetric;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(9, payload_size) + payload_size
                }) + self.optional_none_to_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(10, payload_size) + payload_size
                }) + ({
                    let payload = &self.optional_some_to_required;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(12, payload_size) + payload_size
                }) + ({
                    let payload = &self.optional_some_to_asymmetric;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(13, payload_size) + payload_size
                }) + self.optional_some_to_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(14, payload_size) + payload_size
                }) + ({
                    let payload = &self.nonexistent_to_asymmetric;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(17, payload_size) + payload_size
                }) + self.nonexistent_to_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(18, payload_size) + payload_size
                })
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                {
                    let payload = &self.required_to_required;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 0, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.required_to_asymmetric;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 1, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.required_to_optional {
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 2, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.asymmetric_to_required;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 4, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.asymmetric_to_asymmetric;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 5, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.asymmetric_to_optional {
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 6, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.optional_none_to_asymmetric;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 9, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.optional_none_to_optional {
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 10, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.optional_some_to_required;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 12, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.optional_some_to_asymmetric;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 13, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.optional_some_to_optional {
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 14, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.nonexistent_to_asymmetric;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 17, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.nonexistent_to_optional {
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 18, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                Ok(())
            }
        }

        impl super::super::Deserialize for ExampleStructIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                let mut required_to_required: Option<String> = None;
                let mut required_to_asymmetric: Option<String> = None;
                let mut required_to_optional: Option<String> = None;
                let mut asymmetric_to_required: Option<String> = None;
                let mut asymmetric_to_asymmetric: Option<String> = None;
                let mut asymmetric_to_optional: Option<String> = None;
                let mut optional_none_to_asymmetric: Option<String> = None;
                let mut optional_none_to_optional: Option<String> = None;
                let mut optional_some_to_required: Option<String> = None;
                let mut optional_some_to_asymmetric: Option<String> = None;
                let mut optional_some_to_optional: Option<String> = None;
                let mut nonexistent_to_asymmetric: Option<String> = None;
                let mut nonexistent_to_optional: Option<String> = None;

                loop {
                    let (index, size) = match super::super::deserialize_field_header(&mut \
                        *reader) {
                        Ok(header) => header,
                        Err(err) => {
                            if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                break;
                            }

                            return Err(err);
                        }
                    };

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size);

                    match index {
                        0 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            required_to_required.get_or_insert(payload);
                        }
                        1 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            required_to_asymmetric.get_or_insert(payload);
                        }
                        2 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            required_to_optional.get_or_insert(payload);
                        }
                        4 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            asymmetric_to_required.get_or_insert(payload);
                        }
                        5 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            asymmetric_to_asymmetric.get_or_insert(payload);
                        }
                        6 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            asymmetric_to_optional.get_or_insert(payload);
                        }
                        9 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_none_to_asymmetric.get_or_insert(payload);
                        }
                        10 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_none_to_optional.get_or_insert(payload);
                        }
                        12 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_some_to_required.get_or_insert(payload);
                        }
                        13 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_some_to_asymmetric.get_or_insert(payload);
                        }
                        14 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_some_to_optional.get_or_insert(payload);
                        }
                        17 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            nonexistent_to_asymmetric.get_or_insert(payload);
                        }
                        18 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            nonexistent_to_optional.get_or_insert(payload);
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }

                if required_to_required.is_none() || asymmetric_to_required.is_none() || \
                    optional_some_to_required.is_none() {
                    return Err(::std::io::Error::new(
                        ::std::io::ErrorKind::InvalidData,
                        \"Struct missing one or more field(s).\",
                    ));
                }

                Ok(ExampleStructIn {
                    required_to_required: required_to_required.unwrap(),
                    required_to_asymmetric,
                    required_to_optional,
                    asymmetric_to_required: asymmetric_to_required.unwrap(),
                    asymmetric_to_asymmetric,
                    asymmetric_to_optional,
                    optional_none_to_asymmetric,
                    optional_none_to_optional,
                    optional_some_to_required: optional_some_to_required.unwrap(),
                    optional_some_to_asymmetric,
                    optional_some_to_optional,
                    nonexistent_to_asymmetric,
                    nonexistent_to_optional,
                })
            }
        }

        impl From<ExampleStructOut> for ExampleStructIn {
            fn from(message: ExampleStructOut) -> Self {
                ExampleStructIn {
                    required_to_required: message.required_to_required.into(),
                    required_to_asymmetric: Some(message.required_to_asymmetric.into()),
                    required_to_optional: message.required_to_optional.map(|payload| \
                        payload.into()),
                    asymmetric_to_required: message.asymmetric_to_required.into(),
                    asymmetric_to_asymmetric: Some(message.asymmetric_to_asymmetric.into()),
                    asymmetric_to_optional: message.asymmetric_to_optional.map(|payload| \
                        payload.into()),
                    optional_none_to_asymmetric: Some(message.optional_none_to_asymmetric.into()),
                    optional_none_to_optional: message.optional_none_to_optional.map(|payload| \
                        payload.into()),
                    optional_some_to_required: message.optional_some_to_required.into(),
                    optional_some_to_asymmetric: Some(message.optional_some_to_asymmetric.into()),
                    optional_some_to_optional: message.optional_some_to_optional.map(|payload| \
                        payload.into()),
                    nonexistent_to_asymmetric: Some(message.nonexistent_to_asymmetric.into()),
                    nonexistent_to_optional: message.nonexistent_to_optional.map(|payload| \
                        payload.into()),
                }
            }
        }
    }

    pub mod before {
        #[derive(Clone, Debug)]
        pub enum ExampleChoiceOut {
            RequiredToRequired(String),
            RequiredToAsymmetric(String),
            AsymmetricToRequired(String, Box<ExampleChoiceOut>),
            AsymmetricToAsymmetric(String, Box<ExampleChoiceOut>),
            AsymmetricToOptionalHandled(String, Box<ExampleChoiceOut>),
            AsymmetricToOptionalFallback(String, Box<ExampleChoiceOut>),
            AsymmetricToNonexistent(String, Box<ExampleChoiceOut>),
            OptionalToRequired(String, Box<ExampleChoiceOut>),
            OptionalToAsymmetric(String, Box<ExampleChoiceOut>),
            OptionalToOptionalHandled(String, Box<ExampleChoiceOut>),
            OptionalToOptionalFallback(String, Box<ExampleChoiceOut>),
            OptionalToNonexistent(String, Box<ExampleChoiceOut>),
        }

        #[derive(Clone, Debug)]
        pub enum ExampleChoiceIn {
            RequiredToRequired(String),
            RequiredToAsymmetric(String),
            AsymmetricToRequired(String),
            AsymmetricToAsymmetric(String),
            AsymmetricToOptionalHandled(String),
            AsymmetricToOptionalFallback(String),
            AsymmetricToNonexistent(String),
            OptionalToRequired(String, Box<ExampleChoiceIn>),
            OptionalToAsymmetric(String, Box<ExampleChoiceIn>),
            OptionalToOptionalHandled(String, Box<ExampleChoiceIn>),
            OptionalToOptionalFallback(String, Box<ExampleChoiceIn>),
            OptionalToNonexistent(String, Box<ExampleChoiceIn>),
        }

        impl super::super::Serialize for ExampleChoiceOut {
            fn size(&self) -> u64 {
                match *self {
                    ExampleChoiceOut::RequiredToRequired(ref payload) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(0, payload_size) +
                            payload_size
                    }
                    ExampleChoiceOut::RequiredToAsymmetric(ref payload) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(1, payload_size) +
                            payload_size
                    }
                    ExampleChoiceOut::AsymmetricToRequired(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(5, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::AsymmetricToAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(6, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::AsymmetricToOptionalHandled(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(7, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::AsymmetricToOptionalFallback(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(8, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::AsymmetricToNonexistent(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(9, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::OptionalToRequired(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(10, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::OptionalToAsymmetric(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(11, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::OptionalToOptionalHandled(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(12, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::OptionalToOptionalFallback(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(13, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                    ExampleChoiceOut::OptionalToNonexistent(ref payload, ref fallback) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(14, payload_size) +
                            payload_size +
                            fallback.size()
                    }
                }
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                match *self {
                    ExampleChoiceOut::RequiredToRequired(ref payload) => {
                        super::super::serialize_non_varint_field_header(writer, 0, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        Ok(())
                    }
                    ExampleChoiceOut::RequiredToAsymmetric(ref payload) => {
                        super::super::serialize_non_varint_field_header(writer, 1, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        Ok(())
                    }
                    ExampleChoiceOut::AsymmetricToRequired(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 5, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::AsymmetricToAsymmetric(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 6, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::AsymmetricToOptionalHandled(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 7, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::AsymmetricToOptionalFallback(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 8, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::AsymmetricToNonexistent(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 9, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::OptionalToRequired(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 10, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::OptionalToAsymmetric(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 11, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::OptionalToOptionalHandled(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 12, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::OptionalToOptionalFallback(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 13, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                    ExampleChoiceOut::OptionalToNonexistent(ref payload, ref fallback) => {
                        super::super::serialize_non_varint_field_header(writer, 14, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        fallback.serialize(writer)
                    }
                }
            }
        }

        impl super::super::Deserialize for ExampleChoiceIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                loop {
                    let (index, size) = super::super::deserialize_field_header(&mut *reader)?;

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size);

                    match index {
                        0 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::RequiredToRequired(payload));
                        }
                        1 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::RequiredToAsymmetric(payload));
                        }
                        5 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::AsymmetricToRequired(payload));
                        }
                        6 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::AsymmetricToAsymmetric(payload));
                        }
                        7 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::AsymmetricToOptionalHandled(payload));
                        }
                        8 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::AsymmetricToOptionalFallback(payload));
                        }
                        9 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(ExampleChoiceIn::AsymmetricToNonexistent(payload));
                        }
                        10 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::OptionalToRequired(payload, fallback));
                        }
                        11 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::OptionalToAsymmetric(payload, fallback));
                        }
                        12 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::OptionalToOptionalHandled(payload, \
                                fallback));
                        }
                        13 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::OptionalToOptionalFallback(payload, \
                                fallback));
                        }
                        14 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            let fallback = Box::new(<ExampleChoiceIn as \
                                super::super::Deserialize>::deserialize(&mut *reader)?);
                            return Ok(ExampleChoiceIn::OptionalToNonexistent(payload, fallback));
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }
            }
        }

        impl From<ExampleChoiceOut> for ExampleChoiceIn {
            fn from(message: ExampleChoiceOut) -> Self {
                match message {
                    ExampleChoiceOut::RequiredToRequired(payload) => \
                        ExampleChoiceIn::RequiredToRequired(payload.into()),
                    ExampleChoiceOut::RequiredToAsymmetric(payload) => \
                        ExampleChoiceIn::RequiredToAsymmetric(payload.into()),
                    ExampleChoiceOut::AsymmetricToRequired(payload, fallback) => \
                        ExampleChoiceIn::AsymmetricToRequired(payload.into()),
                    ExampleChoiceOut::AsymmetricToAsymmetric(payload, fallback) => \
                        ExampleChoiceIn::AsymmetricToAsymmetric(payload.into()),
                    ExampleChoiceOut::AsymmetricToOptionalHandled(payload, fallback) => \
                        ExampleChoiceIn::AsymmetricToOptionalHandled(payload.into()),
                    ExampleChoiceOut::AsymmetricToOptionalFallback(payload, fallback) => \
                        ExampleChoiceIn::AsymmetricToOptionalFallback(payload.into()),
                    ExampleChoiceOut::AsymmetricToNonexistent(payload, fallback) => \
                        ExampleChoiceIn::AsymmetricToNonexistent(payload.into()),
                    ExampleChoiceOut::OptionalToRequired(payload, fallback) => \
                        ExampleChoiceIn::OptionalToRequired(payload.into(), \
                        Box::new((*fallback).into())),
                    ExampleChoiceOut::OptionalToAsymmetric(payload, fallback) => \
                        ExampleChoiceIn::OptionalToAsymmetric(payload.into(), \
                        Box::new((*fallback).into())),
                    ExampleChoiceOut::OptionalToOptionalHandled(payload, fallback) => \
                        ExampleChoiceIn::OptionalToOptionalHandled(payload.into(), \
                        Box::new((*fallback).into())),
                    ExampleChoiceOut::OptionalToOptionalFallback(payload, fallback) => \
                        ExampleChoiceIn::OptionalToOptionalFallback(payload.into(), \
                        Box::new((*fallback).into())),
                    ExampleChoiceOut::OptionalToNonexistent(payload, fallback) => \
                        ExampleChoiceIn::OptionalToNonexistent(payload.into(), \
                        Box::new((*fallback).into())),
                }
            }
        }

        #[derive(Clone, Debug)]
        pub struct ExampleStructOut {
            pub required_to_required: String,
            pub required_to_asymmetric: String,
            pub required_to_optional: String,
            pub required_to_nonexistent: String,
            pub asymmetric_to_required: String,
            pub asymmetric_to_asymmetric: String,
            pub asymmetric_to_optional: String,
            pub asymmetric_to_nonexistent: String,
            pub optional_none_to_asymmetric: Option<String>,
            pub optional_none_to_optional: Option<String>,
            pub optional_none_to_nonexistent: Option<String>,
            pub optional_some_to_required: Option<String>,
            pub optional_some_to_asymmetric: Option<String>,
            pub optional_some_to_optional: Option<String>,
            pub optional_some_to_nonexistent: Option<String>,
        }

        #[derive(Clone, Debug)]
        pub struct ExampleStructIn {
            pub required_to_required: String,
            pub required_to_asymmetric: String,
            pub required_to_optional: String,
            pub required_to_nonexistent: String,
            pub asymmetric_to_required: Option<String>,
            pub asymmetric_to_asymmetric: Option<String>,
            pub asymmetric_to_optional: Option<String>,
            pub asymmetric_to_nonexistent: Option<String>,
            pub optional_none_to_asymmetric: Option<String>,
            pub optional_none_to_optional: Option<String>,
            pub optional_none_to_nonexistent: Option<String>,
            pub optional_some_to_required: Option<String>,
            pub optional_some_to_asymmetric: Option<String>,
            pub optional_some_to_optional: Option<String>,
            pub optional_some_to_nonexistent: Option<String>,
        }

        impl super::super::Serialize for ExampleStructOut {
            fn size(&self) -> u64 {
                ({
                    let payload = &self.required_to_required;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(0, payload_size) + payload_size
                }) + ({
                    let payload = &self.required_to_asymmetric;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(1, payload_size) + payload_size
                }) + ({
                    let payload = &self.required_to_optional;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(2, payload_size) + payload_size
                }) + ({
                    let payload = &self.required_to_nonexistent;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(3, payload_size) + payload_size
                }) + ({
                    let payload = &self.asymmetric_to_required;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(4, payload_size) + payload_size
                }) + ({
                    let payload = &self.asymmetric_to_asymmetric;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(5, payload_size) + payload_size
                }) + ({
                    let payload = &self.asymmetric_to_optional;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(6, payload_size) + payload_size
                }) + ({
                    let payload = &self.asymmetric_to_nonexistent;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(7, payload_size) + payload_size
                }) + self.optional_none_to_asymmetric.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(9, payload_size) + payload_size
                }) + self.optional_none_to_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(10, payload_size) + payload_size
                }) + self.optional_none_to_nonexistent.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(11, payload_size) + payload_size
                }) + self.optional_some_to_required.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(12, payload_size) + payload_size
                }) + self.optional_some_to_asymmetric.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(13, payload_size) + payload_size
                }) + self.optional_some_to_optional.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(14, payload_size) + payload_size
                }) + self.optional_some_to_nonexistent.as_ref().map_or(0, |payload| {
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(15, payload_size) + payload_size
                })
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                {
                    let payload = &self.required_to_required;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 0, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.required_to_asymmetric;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 1, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.required_to_optional;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 2, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.required_to_nonexistent;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 3, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.asymmetric_to_required;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 4, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.asymmetric_to_asymmetric;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 5, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.asymmetric_to_optional;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 6, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                {
                    let payload = &self.asymmetric_to_nonexistent;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 7, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.optional_none_to_asymmetric {
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 9, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.optional_none_to_optional {
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 10, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.optional_none_to_nonexistent {
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 11, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.optional_some_to_required {
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 12, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.optional_some_to_asymmetric {
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 13, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.optional_some_to_optional {
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 14, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                if let Some(payload) = &self.optional_some_to_nonexistent {
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 15, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                Ok(())
            }
        }

        impl super::super::Deserialize for ExampleStructIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                let mut required_to_required: Option<String> = None;
                let mut required_to_asymmetric: Option<String> = None;
                let mut required_to_optional: Option<String> = None;
                let mut required_to_nonexistent: Option<String> = None;
                let mut asymmetric_to_required: Option<String> = None;
                let mut asymmetric_to_asymmetric: Option<String> = None;
                let mut asymmetric_to_optional: Option<String> = None;
                let mut asymmetric_to_nonexistent: Option<String> = None;
                let mut optional_none_to_asymmetric: Option<String> = None;
                let mut optional_none_to_optional: Option<String> = None;
                let mut optional_none_to_nonexistent: Option<String> = None;
                let mut optional_some_to_required: Option<String> = None;
                let mut optional_some_to_asymmetric: Option<String> = None;
                let mut optional_some_to_optional: Option<String> = None;
                let mut optional_some_to_nonexistent: Option<String> = None;

                loop {
                    let (index, size) = match super::super::deserialize_field_header(&mut \
                        *reader) {
                        Ok(header) => header,
                        Err(err) => {
                            if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                break;
                            }

                            return Err(err);
                        }
                    };

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size);

                    match index {
                        0 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            required_to_required.get_or_insert(payload);
                        }
                        1 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            required_to_asymmetric.get_or_insert(payload);
                        }
                        2 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            required_to_optional.get_or_insert(payload);
                        }
                        3 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            required_to_nonexistent.get_or_insert(payload);
                        }
                        4 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            asymmetric_to_required.get_or_insert(payload);
                        }
                        5 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            asymmetric_to_asymmetric.get_or_insert(payload);
                        }
                        6 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            asymmetric_to_optional.get_or_insert(payload);
                        }
                        7 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            asymmetric_to_nonexistent.get_or_insert(payload);
                        }
                        9 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_none_to_asymmetric.get_or_insert(payload);
                        }
                        10 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_none_to_optional.get_or_insert(payload);
                        }
                        11 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_none_to_nonexistent.get_or_insert(payload);
                        }
                        12 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_some_to_required.get_or_insert(payload);
                        }
                        13 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_some_to_asymmetric.get_or_insert(payload);
                        }
                        14 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_some_to_optional.get_or_insert(payload);
                        }
                        15 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            optional_some_to_nonexistent.get_or_insert(payload);
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }

                if required_to_required.is_none() || required_to_asymmetric.is_none() || \
                    required_to_optional.is_none() || required_to_nonexistent.is_none() {
                    return Err(::std::io::Error::new(
                        ::std::io::ErrorKind::InvalidData,
                        \"Struct missing one or more field(s).\",
                    ));
                }

                Ok(ExampleStructIn {
                    required_to_required: required_to_required.unwrap(),
                    required_to_asymmetric: required_to_asymmetric.unwrap(),
                    required_to_optional: required_to_optional.unwrap(),
                    required_to_nonexistent: required_to_nonexistent.unwrap(),
                    asymmetric_to_required,
                    asymmetric_to_asymmetric,
                    asymmetric_to_optional,
                    asymmetric_to_nonexistent,
                    optional_none_to_asymmetric,
                    optional_none_to_optional,
                    optional_none_to_nonexistent,
                    optional_some_to_required,
                    optional_some_to_asymmetric,
                    optional_some_to_optional,
                    optional_some_to_nonexistent,
                })
            }
        }

        impl From<ExampleStructOut> for ExampleStructIn {
            fn from(message: ExampleStructOut) -> Self {
                ExampleStructIn {
                    required_to_required: message.required_to_required.into(),
                    required_to_asymmetric: message.required_to_asymmetric.into(),
                    required_to_optional: message.required_to_optional.into(),
                    required_to_nonexistent: message.required_to_nonexistent.into(),
                    asymmetric_to_required: Some(message.asymmetric_to_required.into()),
                    asymmetric_to_asymmetric: Some(message.asymmetric_to_asymmetric.into()),
                    asymmetric_to_optional: Some(message.asymmetric_to_optional.into()),
                    asymmetric_to_nonexistent: Some(message.asymmetric_to_nonexistent.into()),
                    optional_none_to_asymmetric: \
                        message.optional_none_to_asymmetric.map(|payload| payload.into()),
                    optional_none_to_optional: message.optional_none_to_optional.map(|payload| \
                        payload.into()),
                    optional_none_to_nonexistent: \
                        message.optional_none_to_nonexistent.map(|payload| payload.into()),
                    optional_some_to_required: message.optional_some_to_required.map(|payload| \
                        payload.into()),
                    optional_some_to_asymmetric: \
                        message.optional_some_to_asymmetric.map(|payload| payload.into()),
                    optional_some_to_optional: message.optional_some_to_optional.map(|payload| \
                        payload.into()),
                    optional_some_to_nonexistent: \
                        message.optional_some_to_nonexistent.map(|payload| payload.into()),
                }
            }
        }
    }

    pub mod main {
        #[derive(Clone, Debug)]
        pub enum SingletonChoiceOut {
            X(String),
        }

        #[derive(Clone, Debug)]
        pub enum SingletonChoiceIn {
            X(String),
        }

        impl super::super::Serialize for SingletonChoiceOut {
            fn size(&self) -> u64 {
                match *self {
                    SingletonChoiceOut::X(ref payload) => {
                        let payload_size = payload.len() as u64;
                        super::super::non_varint_field_header_size(0, payload_size) +
                            payload_size
                    }
                }
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                match *self {
                    SingletonChoiceOut::X(ref payload) => {
                        super::super::serialize_non_varint_field_header(writer, 0, payload.len() \
                            as u64)?;
                        writer.write_all(payload.as_bytes())?;
                        Ok(())
                    }
                }
            }
        }

        impl super::super::Deserialize for SingletonChoiceIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                loop {
                    let (index, size) = super::super::deserialize_field_header(&mut *reader)?;

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size);

                    match index {
                        0 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;
                            return Ok(SingletonChoiceIn::X(payload));
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }
            }
        }

        impl From<SingletonChoiceOut> for SingletonChoiceIn {
            fn from(message: SingletonChoiceOut) -> Self {
                match message {
                    SingletonChoiceOut::X(payload) => SingletonChoiceIn::X(payload.into()),
                }
            }
        }

        #[derive(Clone, Debug)]
        pub struct SingletonStructOut {
            pub x: String,
        }

        #[derive(Clone, Debug)]
        pub struct SingletonStructIn {
            pub x: String,
        }

        impl super::super::Serialize for SingletonStructOut {
            fn size(&self) -> u64 {
                ({
                    let payload = &self.x;
                    let payload_size = payload.len() as u64;
                    super::super::non_varint_field_header_size(0, payload_size) + payload_size
                })
            }

            fn serialize<T: ::std::io::Write>(&self, writer: &mut T) -> ::std::io::Result<()> {
                {
                    let payload = &self.x;
                    let payload_size = payload.len() as u64;
                    super::super::serialize_non_varint_field_header(writer, 0, payload_size)?;
                    writer.write_all(payload.as_bytes())?;
                }

                Ok(())
            }
        }

        impl super::super::Deserialize for SingletonStructIn {
            fn deserialize<T>(reader: &mut T) -> ::std::io::Result<Self>
            where
                Self: Sized,
                T: ::std::io::BufRead,
            {
                let mut x: Option<String> = None;

                loop {
                    let (index, size) = match super::super::deserialize_field_header(&mut \
                        *reader) {
                        Ok(header) => header,
                        Err(err) => {
                            if let std::io::ErrorKind::UnexpectedEof = err.kind() {
                                break;
                            }

                            return Err(err);
                        }
                    };

                    let mut sub_reader = ::std::io::Read::take(&mut *reader, size);

                    match index {
                        0 => {
                            let mut buffer = vec![];
                            ::std::io::Read::read_to_end(&mut sub_reader, &mut buffer)?;
                            let payload = std::str::from_utf8(&buffer).map_or_else(
                                |err| Err(::std::io::Error::new(::std::io::ErrorKind::Other, err)),
                                |result| Ok(result.to_owned()),
                            )?;

                            x.get_or_insert(payload);
                        }
                        _ => {
                            super::super::skip(&mut sub_reader, size as usize)?;
                        }
                    }
                }

                if x.is_none() {
                    return Err(::std::io::Error::new(
                        ::std::io::ErrorKind::InvalidData,
                        \"Struct missing one or more field(s).\",
                    ));
                }

                Ok(SingletonStructIn {
                    x: x.unwrap(),
                })
            }
        }

        impl From<SingletonStructOut> for SingletonStructIn {
            fn from(message: SingletonStructOut) -> Self {
                SingletonStructIn {
                    x: message.x.into(),
                }
            }
        }
    }
}
",
        );
    }
}
