use serde::{Deserialize, Serialize};

pub struct TxlogClient {
    url: String,
    secret: String,
}

#[derive(Serialize, Deserialize)]
pub struct SubmitResponse {
    success: bool,
}

#[derive(Serialize, Deserialize)]
pub struct TxResponse {
    txid: String,
    status: u32,
    created: String,
    updated: String,
}

impl TxlogClient {
    pub fn new(url: String, secret: String) -> TxlogClient {
        return TxlogClient { url, secret };
    }

    pub async fn rawtx(&self, txid: &String) -> Option<String> {
        let client = reqwest::Client::new();
        let url = format!("{}/tx/{}/raw", &self.url, txid);
        let response = match client
            .get(url)
            .header("Authorization", format!("Bearer {}", self.secret))
            .send()
            .await
        {
            Err(_) => return None,
            Ok(v) => v,
        };

        let body = match response.text().await {
            Err(_) => return None,
            Ok(v) => v,
        };

        return Some(body);
    }

    pub async fn tx(&self, txid: &String) -> Option<TxResponse> {
        let client = reqwest::Client::new();
        let url = format!("{}/tx/{}", &self.url, txid);
        let response = match client
            .get(url)
            .header("Authorization", format!("Bearer {}", self.secret))
            .send()
            .await
        {
            Err(e) => {
                println!("{:?}", e);
                return None;
            }
            Ok(v) => v,
        };

        let body = match response.json::<TxResponse>().await {
            Err(e) => {
                println!("{:?}", e);
                return None;
            }
            Ok(v) => v,
        };

        return Some(body);
    }

    pub async fn submit(&self, rawtx: Vec<u8>, metadata: Option<String>) -> Option<SubmitResponse> {
        let client = reqwest::Client::new();
        let mut url = format!("{}/tx", &self.url);

        match metadata {
            Some(v) => {
                url = format!("{}?metadata={}", url, v);
            }
            None => (),
        }

        let response = match client
            .post(url)
            .header("Authorization", format!("Bearer {}", self.secret))
            .header("Content-Type", "application/octet-stream")
            .body(rawtx)
            .send()
            .await
        {
            Err(e) => {
                println!("{:?}", e);
                return None;
            }
            Ok(v) => v,
        };

        let body = match response.json::<SubmitResponse>().await {
            Err(e) => {
                println!("{:?}", e);
                return None;
            }
            Ok(v) => v,
        };

        return Some(body);
    }
}

#[cfg(test)]
mod tests {
    macro_rules! aw {
        ($e:expr) => {
            tokio_test::block_on($e)
        };
    }

    #[test]
    fn test_rawtx() {
        // Add variables to .env file
        let secret = dotenv::var("TXLOG_SECRET").unwrap();
        let url = dotenv::var("TXLOG_URL").unwrap();
        let rawtx = dotenv::var("RAWTX").unwrap();
        let txid = dotenv::var("TXID").unwrap();

        let client = crate::client::TxlogClient::new(url, secret);

        let response = aw!(client.rawtx(&txid)).unwrap();

        assert_eq!(response, rawtx);
    }

    #[test]
    fn test_submit() {
        // Add variables to .env file
        let secret = dotenv::var("TXLOG_SECRET").unwrap();
        let url = dotenv::var("TXLOG_URL").unwrap();
        let rawtx = dotenv::var("RAWTX").unwrap();

        let client = crate::client::TxlogClient::new(url, secret);

        let response = aw!(client.submit(hex::decode(&rawtx).unwrap(), None)).unwrap();

        assert_eq!(response.success, true);
    }

    #[test]
    fn test_tx() {
        // Add variables to .env file
        let secret = dotenv::var("TXLOG_SECRET").unwrap();
        let url = dotenv::var("TXLOG_URL").unwrap();
        let txid = dotenv::var("TXID").unwrap();

        let client = crate::client::TxlogClient::new(url, secret);

        let response = aw!(client.tx(&txid)).unwrap();

        assert_eq!(response.txid, txid);
    }
}
