use tokio::io::Error;

use env_logger::Builder;
use log::LevelFilter;
use tws::Client;
use tws::ws::Msg;
use async_channel::RecvError;

#[tokio::main]
async fn main() -> Result<(), Error> {
    // wss
    // real    7m8,429s
    // user    0m37,071s
    // sys     0m11,194s

    // ws
    // real    6m2,571s
    // user    0m25,745s
    // sys     0m7,258s


    let mut builder = Builder::new();
    builder.filter_level(LevelFilter::Trace);
    builder.init();

    let addr = "wss://localhost:9001".to_string();
    let agent = "twss";

    println!("Using fuzzingserver {}", addr);
    println!("Using agent {}", agent);

    println!("Running test suite...");

    let mut current_case_id = 1;
    let mut case_count = get_case_count(addr.clone()).await.unwrap();

    // case_count = 259;
    println!("count - {}", case_count);

    while current_case_id <= case_count {
        let case_id = current_case_id;
        current_case_id += 1;
        // tokio::time::sleep(tokio::time::Duration::from_millis(1000)).await;

        // println!("Executing test case: {}/{}", case_id, case_count);
        let url = addr.clone() + "/runCase?case=" + &case_id.to_string()[..] + "&agent=" + agent;
        println!("{}", url.as_str());

        let client = Client::new(url.as_str(), true);
        let mut socket = client.connect2().await?;

        while let msg = socket.recv().await {
            let message = match msg {
                Ok(msg) => { msg }
                Err(e) => {
                    println!("Error: {:?}", e);
                    // let _ = socket.send(Msg::Close(1002)).await;
                    break;
                }
            };

            match message {
                Msg::Binary(data) => {
                    let _ = socket.send(Msg::Binary(data)).await;
                }
                Msg::Text(data) => {
                    // println!("socket.recv Text");
                    let _ = socket.send(Msg::Text(data)).await;
                }
                Msg::Close(code) => {
                    // println!("socket.recv Close {}", code);
                    let _ = socket.send(Msg::Close(code)).await;
                }
                Msg::Ping(data) => {
                }
                Msg::Pong(_) => {}
                Msg::Clear() => {}
            }
        }
        // tokio::time::sleep(tokio::time::Duration::from_millis(1000)).await;
    }

    update_reports(addr, agent).await;

    Ok(())
}

async fn get_case_count(addr: String) -> Result<(i32), Error> {
    let url = addr + "/getCaseCount";

    let client = Client::new(url.as_str(), true);
    let mut socket = client.connect2().await?;

    let mut count = 0;

    while let msg = socket.recv().await {
        let message = match msg {
            Ok(msg) => { msg }
            Err(e) => {
                println!("Error: {:?}", e);
                // let _ = socket.send(Msg::Close(1002)).await;
                break;
            }
        };

        match message {
            Msg::Binary(_) => {}
            Msg::Text(data) => {
                count = String::from_utf8(data).unwrap().parse().unwrap();
            }
            Msg::Close(_) => {
                let _ = socket.send(Msg::Close(1002)).await;
            }
            Msg::Ping(data) => {
            }
            Msg::Pong(_) => {}
            Msg::Clear() => {}
        }
    }

    Ok(count)
}

async fn update_reports(addr: String, agent: &str) -> Result<(), Error> {
    let url = addr + "/updateReports?agent=" + agent;

    let client = Client::new(url.as_str(), true);
    let mut socket = client.connect2().await?;

    println!("Updating reports...");

    while let msg = socket.recv().await {
        let message = match msg {
            Ok(msg) => { msg }
            Err(e) => {
                println!("Error: {:?}", e);
                // let _ = socket.send(Msg::Close(1002)).await;
                break;
            }
        };

        match message {
            Msg::Binary(_) => {}
            Msg::Text(data) => {}
            Msg::Close(_) => {
                let _ = socket.send(Msg::Close(1000)).await;
            }
            Msg::Ping(data) => {
            }
            Msg::Pong(_) => {}
            Msg::Clear() => {}
        }
    }

    Ok(())
}