use std::collections::HashMap;

use tokio::net::TcpStream;
// use tokio_rustls::{TlsConnector, rustls::ClientConfig, client::TlsStream, webpki};
// use std::sync::Arc;
// use tokio_rustls::webpki::DNSNameRef;

use crate::http::Url;
use tokio::io::{AsyncWrite, AsyncRead, AsyncWriteExt, AsyncReadExt};
use crate::stream::Stream;
use crate::ws::{Msg, Reader, Writer};
use async_channel::{Sender, Receiver, RecvError, SendError, TrySendError};
// use tokio_rustls::rustls::{ServerCertVerifier, RootCertStore, Certificate, ServerCertVerified, TLSError, HandshakeSignatureValid};
// use tokio_rustls::rustls::internal::msgs::handshake::DigitallySignedStruct;
// use rustls::{ServerCertVerifier, RootCertStore, Certificate, ServerCertVerified, TLSError, HandshakeSignatureValid};
// use rustls::internal::msgs::handshake::DigitallySignedStruct;

pub mod frame;
pub mod http;
pub mod deflate;
pub mod stream;
pub mod ws;

use log::{trace, error};
use tokio_native_tls::{TlsStream, native_tls};
use tokio_native_tls::native_tls::TlsConnector;

/// Blindly trust any cert
/// Please DO NOT use in production
// struct AcceptAnyCertVerifier;
//
// impl ServerCertVerifier for AcceptAnyCertVerifier {
//     fn verify_server_cert(
//         &self,
//         _roots: &RootCertStore,
//         _presented_certs: &[Certificate],
//         _dns_name: DNSNameRef,
//         _ocsp_response: &[u8]
//     ) -> Result<ServerCertVerified, TLSError> {
//         // YOLO
//         Ok(ServerCertVerified::assertion())
//     }
//
//     fn verify_tls12_signature(
//         &self,
//         _message: &[u8],
//         _cert: &Certificate,
//         _dss: &DigitallySignedStruct,
//     ) -> Result<HandshakeSignatureValid, TLSError> {
//         // LOOKS GOOD
//         Ok(HandshakeSignatureValid::assertion())
//     }
//
//     fn verify_tls13_signature(
//         &self,
//         _message: &[u8],
//         _cert: &Certificate,
//         _dss: &DigitallySignedStruct,
//     ) -> Result<HandshakeSignatureValid, TLSError> {
//         // YUP VALID
//         Ok(HandshakeSignatureValid::assertion())
//     }
// }

async fn connect_tls(tcp_stream: TcpStream, url: &Url) -> Result<TlsStream<TcpStream>, tokio::io::Error> {
    // let mut config = rustls::ClientConfig::new();
    // config.dangerous().set_certificate_verifier(Arc::new(AcceptAnyCertVerifier));
    // config.root_store = match rustls_native_certs::load_native_certs() {
    //     Ok(store) => store,
    //     Err((Some(store), err)) => {
    //         log::warn!("Could not load all certificates: {:?}", err);
    //         store
    //     }
    //     Err((None, err)) => Err(err).expect("cannot access native cert store"),
    // };
    // if config.root_store.is_empty() {
    //     panic!("no CA certificates found");
    // }

    let cx = native_tls::TlsConnector::builder().danger_accept_invalid_certs(true).build().unwrap();
    let cx = tokio_native_tls::TlsConnector::from(cx);

    // let connector = TlsConnector::from(Arc::new(config));
    // let stream_tcp = TcpStream::connect(&url.addr).await?;

    // let dnsname = DNSNameRef::try_from_ascii_str(&url.domain).unwrap();
    // trace!("domain {:?}", url.domain);
    // let mut stream = connector.connect(dnsname, stream_tcp).await?;
    let mut stream = cx.connect(url.domain.as_str(), tcp_stream).await.unwrap();

    // if stream.is_err() {
    //     panic!("{}", stream.unwrap_err());
    //     Err(stream.unwrap_err())
    // } else {
    //     Ok(stream.unwrap())
    // }
    Ok(stream)
}

// async fn connect_stream<T>(
//     raw_stream: T,
//     url: &Url,
//     headers: HashMap<String, String>,
//     is_utf8_validate: bool,
// ) -> Result<Socket, tokio::io::Error>
//     where
//         T: AsyncWriteExt + AsyncReadExt + std::marker::Send + 'static,
// {
//     let (tx_close, mut rx_close): (Sender<u8>, Receiver<u8>) = async_channel::bounded::<u8>(25);
//     // tokio::spawn(async move {
//     //     if let e = rx_close.read().await {
//     //         error!("{:?}", e);
//     //     }
//     // });
//
//     let (mut stream_rh, mut stream_wh) = tokio::io::split(raw_stream);
//     let (stream_rh, stream_wh, deflate_supported, headers) = http::connect_http(stream_rh, stream_wh, &url, headers).await?;
//     let tx = Writer::spawn(stream_wh, true); // handle write half
//     let rx = Reader::spawn(stream_rh, tx.clone(), tx_close, deflate_supported, is_utf8_validate); // handle read half
//
//     return Ok(Socket { rx, tx, headers });
// }

async fn connect_stream<T>(
    raw_stream: T,
    url: &Url,
    headers: &HashMap<String, String>,
    is_utf8_validate: bool,
    ws_send: Sender<Msg>,
    app_send: Sender<Msg>,
    close_send: Sender<u8>,
    app_recv: Receiver<Msg>,
) -> Result<u8, tokio::io::Error>
    where
        T: AsyncWriteExt + AsyncReadExt + std::marker::Send + 'static,
{
    // tokio::spawn(async move {
    //     if let e = rx_close.read().await {
    //         error!("{:?}", e);
    //     }
    // });

    let (mut stream_rh, mut stream_wh) = tokio::io::split(raw_stream);
    let (stream_rh, stream_wh, deflate_supported, headers) = http::connect_http(stream_rh, stream_wh, &url, headers).await?;
    Writer::spawn(stream_wh, true, app_recv);
    Reader::spawn(stream_rh, ws_send, app_send, close_send, deflate_supported, is_utf8_validate);

    return Ok(0);
}

pub struct Client {
    url: String,
    pub headers: HashMap<String, String>,
    cookies: HashMap<String, String>,
    connect_send: Option<(Vec<Msg>, tokio::time::Duration)>,
    is_utf8_validate: bool,
}


impl Client {
    pub fn new(url: &str, is_utf8_validate: bool) -> Self {
        Client {
            url: url.to_owned(),
            headers: HashMap::new(),
            cookies: HashMap::new(),
            connect_send: None,
            is_utf8_validate,
        }
    }

    pub fn auto_reconnect(&mut self, msg: Vec<Msg>, wait: tokio::time::Duration) {
        self.connect_send = Some((msg, wait))
    }

    pub fn header(mut self, key: &str, value: &str) -> Client {
        self.headers.insert(key.to_owned(), value.to_owned());
        self
    }

    pub fn cookie(mut self, key: &str, value: &str) -> Client {
        self.cookies.insert(key.to_owned(), value.to_owned());
        self
    }

    fn cookies_to_header(&mut self) {
        if self.cookies.len() == 0 {
            return;
        }
        let mut cookie = String::new();
        for (key, value) in self.cookies.iter() {
            cookie.push_str(&key);
            cookie.push_str("=");
            cookie.push_str(&value);
            cookie.push_str("; ");
        }
        self.headers.insert("Cookie".to_owned(), cookie);
    }

    // pub async fn connect(mut self) -> Result<Socket, tokio::io::Error> {
    //     self.cookies_to_header();
    //     let url = http::parse_url(&self.url)?;
    //     let tcp_stream = TcpStream::connect(&url.addr).await?; // establish tcp connection
    //     tcp_stream.set_nodelay(true).unwrap();
    //     if url.wss {
    //         let tls_stream = connect_tls(tcp_stream, &url).await?; // tcp -> tls
    //         return Ok(connect_stream(tls_stream, &url, Some(self.headers), self.is_utf8_validate).await?);
    //     }
    //     Ok(connect_stream(tcp_stream, &url, Some(self.headers), self.is_utf8_validate).await?)
    // }


    pub async fn connect2(mut self) -> Result<Socket, tokio::io::Error> {
        self.cookies_to_header();
        let url = http::parse_url(&self.url)?;

        Socket::new(url, self.headers, self.is_utf8_validate, self.connect_send).await

        // let tcp_stream = TcpStream::connect(&url.addr).await?; // establish tcp connection
        // tcp_stream.set_nodelay(true).unwrap();
        // if url.wss {
        //     let tls_stream = connect_tls(tcp_stream, &url).await?; // tcp -> tls
        //     return Ok(connect_stream(tls_stream, &url, Some(self.headers), self.is_utf8_validate).await?);
        // }
        // Ok(connect_stream(tcp_stream, &url, Some(self.headers), self.is_utf8_validate).await?)
    }
}

pub struct Socket {
    url: Url,

    app_recv: Receiver<Msg>,
    app_send: Sender<Msg>,

    ws_recv: Receiver<Msg>,
    ws_send: Sender<Msg>,

    close_send: Sender<u8>,

    pub headers: HashMap<String, String>,
    is_utf8_validate: bool,
    reconnect_send: Vec<Msg>,
    reconnect_wait: tokio::time::Duration,
}


impl Socket {
    pub async fn new(url: Url, headers: HashMap<String, String>, is_utf8_validate: bool, connect_send: Option<(Vec<Msg>, tokio::time::Duration)>) -> Result<Socket, tokio::io::Error> {
        let (app_send, app_recv): (Sender<Msg>, Receiver<Msg>) = async_channel::bounded::<Msg>(25);
        let (ws_send, ws_recv): (Sender<Msg>, Receiver<Msg>) = async_channel::bounded::<Msg>(25);
        let (close_send, close_recv): (Sender<u8>, Receiver<u8>) = async_channel::bounded::<u8>(25);
        let mut reconnect_send = vec![];
        let mut reconnect_wait = tokio::time::Duration::from_millis(300);

        if connect_send.is_some() {
            let x = connect_send.unwrap();
            reconnect_send = x.0;
            reconnect_wait = x.1;
        }

        let mut socket = Socket {
            url,

            app_recv,
            app_send,

            ws_recv,
            ws_send,

            close_send,

            headers,
            is_utf8_validate,
            reconnect_send,
            reconnect_wait,
        };
        socket.reconnect(close_recv).await?;
        Ok(socket)
    }

    async fn reconnect(&mut self, mut close_recv: Receiver<u8>) -> Result<u8, tokio::io::Error> {
        let url = self.url.clone();
        let headers = self.headers.clone();
        let is_utf8_validate = self.is_utf8_validate;


        let reconnect_msg = self.reconnect_send.clone();
        let reconnect_wait = self.reconnect_wait.clone();

        if reconnect_msg.is_empty() {
            let tcp_stream = TcpStream::connect(&url.addr).await?; // establish tcp connection
            tcp_stream.set_nodelay(true).unwrap();
            if url.wss {
                let tls_stream = connect_tls(tcp_stream, &url).await?; // tcp -> tls
                connect_stream(tls_stream, &url, &headers, is_utf8_validate,
                               self.ws_send.clone(), self.app_send.clone(), self.close_send.clone(), self.app_recv.clone()).await.unwrap();
            } else {
                connect_stream(tcp_stream, &url, &headers, is_utf8_validate,
                               self.ws_send.clone(), self.app_send.clone(), self.close_send.clone(), self.app_recv.clone()).await.unwrap();
            }

            let ws_send = self.ws_send.clone();
            tokio::spawn(async move {
                let res = close_recv.recv().await;
                ws_send.close();
            });
        } else {
            let ws_send = self.ws_send.clone();
            let app_send = self.app_send.clone();
            let close_send = self.close_send.clone();
            let app_recv = self.app_recv.clone();

            tokio::spawn(async move {
                loop {
                    loop {
                        let tcp_stream = TcpStream::connect(&url.addr).await; // establish tcp connection
                        if tcp_stream.is_ok() {
                            let tcp_stream = tcp_stream.unwrap();
                            tcp_stream.set_nodelay(true).unwrap();
                            if url.wss {
                                let tls_stream = connect_tls(tcp_stream, &url).await; // tcp -> tls
                                if tls_stream.is_ok() {
                                    let tls_stream = tls_stream.unwrap();
                                    connect_stream(tls_stream, &url, &headers, is_utf8_validate,
                                                   ws_send.clone(), app_send.clone(), close_send.clone(), app_recv.clone()).await.unwrap();

                                    for msg in &reconnect_msg {
                                        app_send.send(msg.clone()).await;
                                        tokio::time::sleep(reconnect_wait).await;
                                    }
                                    break;
                                }
                            } else {
                                connect_stream(tcp_stream, &url, &headers, is_utf8_validate,
                                               ws_send.clone(), app_send.clone(), close_send.clone(), app_recv.clone()).await.unwrap();

                                for msg in &reconnect_msg {
                                    app_send.send(msg.clone()).await;
                                    tokio::time::sleep(reconnect_wait).await;
                                }
                                break;
                            }
                        }
                        tokio::time::sleep(tokio::time::Duration::from_millis(1000)).await;
                    }

                    let res = close_recv.recv().await;
                    if res.is_ok() {
                        let code = res.unwrap();
                        // println!("{}", code);
                        // if code == 0 {
                        app_send.send(Msg::Close(1000)).await;
                        // }
                        // else if code == 1 {
                        //     break
                        // }
                    } else {
                        break;
                    }
                }
            });
        }

        // let tcp_stream = TcpStream::connect(&self.url.addr).await?; // establish tcp connection
        // tcp_stream.set_nodelay(true).unwrap();
        // if self.url.wss {
        //     let tls_stream = connect_tls(tcp_stream, &self.url).await?; // tcp -> tls
        //     return self.connect_stream(tls_stream).await;
        // }
        //
        // return self.connect_stream(tcp_stream).await;

        // let (mut stream_rh, mut stream_wh) = tokio::io::split(tcp_stream);
        // let (stream_rh, stream_wh, deflate_supported, headers) = http::connect_http(stream_rh, stream_wh, &url, headers).await?;
        // let tx = Writer::spawn(stream_wh, true); // handle write half
        // let rx = Reader::spawn(stream_rh, tx.clone(), tx_close, deflate_supported, is_utf8_validate); // handle read half


        Ok(0)
    }


    pub async fn recv(&mut self) -> Result<Msg, RecvError> {
        self.ws_recv.recv().await
        // let res = self.ws_recv.recv().await;
        // if res.is_ok() {
        //     return res;
        // }
        // Ok(Msg::NetworkError())
    }

    pub async fn send(&mut self, msg: Msg) -> Result<(), SendError<Msg>> {
        self.app_send.send(msg).await
    }
}


#[derive(Debug)]
/// Definition of all errors returned from the library.
pub enum Error {
    InvalidUpgradeRequest,
    UrlParseError { url: String, error: url::ParseError },
}

