use std::collections::HashMap;

use tokio::net::TcpStream;
// use tokio_rustls::{TlsConnector, rustls::ClientConfig, client::TlsStream, webpki};
// use std::sync::Arc;
// use tokio_rustls::webpki::DNSNameRef;

use crate::http::Url;
use tokio::io::{AsyncWrite, AsyncRead, AsyncWriteExt, AsyncReadExt};
use crate::stream::Stream;
use crate::ws::{Msg, Reader, Writer};
use async_channel::{Sender, Receiver, RecvError, SendError};
// use tokio_rustls::rustls::{ServerCertVerifier, RootCertStore, Certificate, ServerCertVerified, TLSError, HandshakeSignatureValid};
// use tokio_rustls::rustls::internal::msgs::handshake::DigitallySignedStruct;
// use rustls::{ServerCertVerifier, RootCertStore, Certificate, ServerCertVerified, TLSError, HandshakeSignatureValid};
// use rustls::internal::msgs::handshake::DigitallySignedStruct;

pub mod frame;
pub mod http;
pub mod deflate;
pub mod stream;
pub mod ws;

use log::{trace, error};
use tokio_native_tls::{TlsStream, native_tls};
use tokio_native_tls::native_tls::TlsConnector;

/// Blindly trust any cert
/// Please DO NOT use in production
// struct AcceptAnyCertVerifier;
//
// impl ServerCertVerifier for AcceptAnyCertVerifier {
//     fn verify_server_cert(
//         &self,
//         _roots: &RootCertStore,
//         _presented_certs: &[Certificate],
//         _dns_name: DNSNameRef,
//         _ocsp_response: &[u8]
//     ) -> Result<ServerCertVerified, TLSError> {
//         // YOLO
//         Ok(ServerCertVerified::assertion())
//     }
//
//     fn verify_tls12_signature(
//         &self,
//         _message: &[u8],
//         _cert: &Certificate,
//         _dss: &DigitallySignedStruct,
//     ) -> Result<HandshakeSignatureValid, TLSError> {
//         // LOOKS GOOD
//         Ok(HandshakeSignatureValid::assertion())
//     }
//
//     fn verify_tls13_signature(
//         &self,
//         _message: &[u8],
//         _cert: &Certificate,
//         _dss: &DigitallySignedStruct,
//     ) -> Result<HandshakeSignatureValid, TLSError> {
//         // YUP VALID
//         Ok(HandshakeSignatureValid::assertion())
//     }
// }

async fn connect_tls(tcp_stream: TcpStream, url: &Url) -> Result<TlsStream<TcpStream>, tokio::io::Error> {
    // let mut config = rustls::ClientConfig::new();
    // config.dangerous().set_certificate_verifier(Arc::new(AcceptAnyCertVerifier));
    // config.root_store = match rustls_native_certs::load_native_certs() {
    //     Ok(store) => store,
    //     Err((Some(store), err)) => {
    //         log::warn!("Could not load all certificates: {:?}", err);
    //         store
    //     }
    //     Err((None, err)) => Err(err).expect("cannot access native cert store"),
    // };
    // if config.root_store.is_empty() {
    //     panic!("no CA certificates found");
    // }

    let cx = native_tls::TlsConnector::builder().danger_accept_invalid_certs(true).build().unwrap();
    let cx = tokio_native_tls::TlsConnector::from(cx);

    // let connector = TlsConnector::from(Arc::new(config));
    // let stream_tcp = TcpStream::connect(&url.addr).await?;

    // let dnsname = DNSNameRef::try_from_ascii_str(&url.domain).unwrap();
    // trace!("domain {:?}", url.domain);
    // let mut stream = connector.connect(dnsname, stream_tcp).await?;
    let mut stream = cx.connect(url.domain.as_str(), tcp_stream).await.unwrap();

    // if stream.is_err() {
    //     panic!("{}", stream.unwrap_err());
    //     Err(stream.unwrap_err())
    // } else {
    //     Ok(stream.unwrap())
    // }
    Ok(stream)
}

async fn connect_stream<T>(
    raw_stream: T,
    url: &Url,
    headers: Option<HashMap<String, String>>,
    is_utf8_validate: bool,
) -> Result<Socket, tokio::io::Error>
    where
        T: AsyncWriteExt + AsyncReadExt + std::marker::Send + 'static,
{
    let (mut stream_rh, mut stream_wh) = tokio::io::split(raw_stream);
    let (stream_rh, stream_wh, deflate_supported, headers) = http::connect_http(stream_rh, stream_wh, &url, headers).await?;
    let tx = Writer::spawn(stream_wh, true); // handle write half
    let rx = Reader::spawn(stream_rh, tx.clone(), deflate_supported, is_utf8_validate); // handle read half

    return Ok(Socket { rx, tx, headers });
}

pub struct Client {
    url: String,
    pub headers: HashMap<String, String>,
    cookies: HashMap<String, String>,
    is_utf8_validate: bool,
}


impl Client {
    pub fn new(url: &str, is_utf8_validate: bool) -> Self {
        Client {
            url: url.to_owned(),
            headers: HashMap::new(),
            cookies: HashMap::new(),
            is_utf8_validate,
        }
    }

    pub fn header(mut self, key: &str, value: &str) -> Client {
        self.headers.insert(key.to_owned(), value.to_owned());
        self
    }

    pub fn cookie(mut self, key: &str, value: &str) -> Client {
        self.cookies.insert(key.to_owned(), value.to_owned());
        self
    }

    fn cookies_to_header(&mut self) {
        if self.cookies.len() == 0 {
            return;
        }
        let mut cookie = String::new();
        for (key, value) in self.cookies.iter() {
            cookie.push_str(&key);
            cookie.push_str("=");
            cookie.push_str(&value);
            cookie.push_str("; ");
        }
        self.headers.insert("Cookie".to_owned(), cookie);
    }

    pub async fn connect(mut self) -> Result<Socket, tokio::io::Error> {
        self.cookies_to_header();
        let url = http::parse_url(&self.url)?;
        let tcp_stream = TcpStream::connect(&url.addr).await?; // establish tcp connection
        tcp_stream.set_nodelay(true).unwrap();
        if url.wss {
            let tls_stream = connect_tls(tcp_stream, &url).await?; // tcp -> tls
            return Ok(connect_stream(tls_stream, &url, Some(self.headers), self.is_utf8_validate).await?);
        }
        Ok(connect_stream(tcp_stream, &url, Some(self.headers), self.is_utf8_validate).await?)
    }
}

/// Represent a WebSocket connection. Used for sending and receiving messages.
#[derive(Debug)]
pub struct Socket {
    tx: Sender<Msg>,
    rx: Receiver<Msg>,
    pub headers: HashMap<String, String>,
}


impl Socket {
    pub async fn recv(&mut self) -> Result<Msg, RecvError> {
        self.rx.recv().await
    }

    pub async fn send(&mut self, msg: Msg) -> Result<(), SendError<Msg>> {
        self.tx.send(msg).await
    }
}


#[derive(Debug)]
/// Definition of all errors returned from the library.
pub enum Error {
    InvalidUpgradeRequest,
    UrlParseError { url: String, error: url::ParseError },
}

