use byteorder::{ByteOrder, NetworkEndian, ReadBytesExt, WriteBytesExt};
use std::io::{Cursor, ErrorKind, Read, Write};
use rand::Rng;
use std::fmt;
use crate::ws::Msg;

// data frame types
pub const CONTINUATION: u8 = 0;
pub const TEXT: u8 = 1;
pub const BINARY: u8 = 2;
// control frame types
pub const CLOSE: u8 = 8;
pub const PING: u8 = 9;
pub const PONG: u8 = 10;

pub struct FrameWriter {
    mask: Option<[u8; 4]>,
}

impl FrameWriter {
    pub fn new(mask: bool) -> Self {
        let res = if mask { Some(rand::random()) } else { None };
        Self {
            mask: res
        }
    }

    pub fn msg_vec(&self, msg: Msg) -> Vec<u8> {
        let mut opcode: u8 = 0;
        let mut payload: Vec<u8> = Vec::new();
        match msg {
            Msg::Binary(dt) => {
                payload = dt;
                opcode = BINARY;
            }
            Msg::Text(dt) => {
                payload = dt;
                opcode = TEXT;
            }
            Msg::Close(dt) => {
                if dt > 0 {
                    payload = dt.to_be_bytes().to_vec();
                    let bytes: [u8; 2] = [payload[0], payload[1]];
                    let status = u16::from_be_bytes(bytes);
                    // println!("payload status {}", status);
                }
                opcode = CLOSE;
            }
            Msg::Ping(dt) => {
                payload = dt;
                opcode = PING;
            }
            Msg::Pong(dt) => {
                payload = dt;
                opcode = PONG;
            }
        }

        let wtr = self.build(opcode, payload);


        wtr
    }

    fn ping(&self, payload: Vec<u8>) -> Vec<u8> {
        self.build(PING, payload)
    }

    pub fn pong(&self, payload: Vec<u8>) -> Vec<u8> {
        self.build(PONG, payload)
    }

    fn close(&self, status: u16) -> Vec<u8> {
        if status == 0 {
            self.build(CLOSE, Vec::new())
        } else {
            self.build(CLOSE, status.to_be_bytes().to_vec())
        }
    }

    fn binary(&self, payload: Vec<u8>) -> Vec<u8> {
        self.build(BINARY, payload)
    }

    fn text(&self, payload: Vec<u8>) -> Vec<u8> {
        self.build(TEXT, payload)
    }

    /*
     0                   1                   2                   3
     0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
    +-+-+-+-+-------+-+-------------+-------------------------------+
    |F|R|R|R| opcode|M| Payload len |    Extended payload length    |
    |I|S|S|S|  (4)  |A|     (7)     |             (16/64)           |
    |N|V|V|V|       |S|             |   (if payload len==126/127)   |
    | |1|2|3|       |K|             |                               |
    +-+-+-+-+-------+-+-------------+ - - - - - - - - - - - - - - - +
    |     Extended payload length continued, if payload len == 127  |
    + - - - - - - - - - - - - - - - +-------------------------------+
    |                               |Masking-key, if MASK set to 1  |
    +-------------------------------+-------------------------------+
    | Masking-key (continued)       |          Payload Data         |
    +-------------------------------- - - - - - - - - - - - - - - - +
    :                     Payload Data continued ...                :
    + - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - +
    |                     Payload Data continued ...                |
    +---------------------------------------------------------------+
    */
    fn build(&self, opcode: u8, mut payload: Vec<u8>) -> Vec<u8> {
        let mut wtr = Vec::new();
        let length: u64 = payload.len() as u64;

        let is_final: bool = true;
        let one = {
            opcode | if is_final { 0x80 } else { 0 }
                | if false { 0x40 } else { 0 }
                | if false { 0x20 } else { 0 }
                | if false { 0x10 } else { 0 }
        };

        let lenfmt = LengthFormat::for_length(length);

        let two = { lenfmt.length_byte() | if self.mask.is_some() { 0x80 } else { 0 } };

        wtr.write_all(&[one, two]).unwrap();
        match lenfmt {
            LengthFormat::U8(_) => (),
            LengthFormat::U16 => wtr.write_u16::<NetworkEndian>(length as u16).unwrap(),
            LengthFormat::U64 => wtr.write_u64::<NetworkEndian>(length).unwrap(),
        }

        if let Some(ref mask) = self.mask {
            wtr.write_all(mask).unwrap();
            mask_fn(&mut payload, mask);
        }

        wtr.extend_from_slice(payload.as_slice());

        wtr
    }
}

//Converts masked data into unmasked data, or vice versa.
//The same algorithm applies regardless of the direction of the translation,
//e.g., the same steps are applied to ask the data as to unmask the data.
pub fn mask_fn(payload: &mut Vec<u8>, key: &[u8; 4]) {
    // loop through the octets of ENCODED and XOR the octet with the (i modulo 4)th
    // octet of MASK ref: https://developer.mozilla.org/en-US/docs/Web/API/WebSockets_API/Writing_WebSocket_servers
    for (i, b) in payload.iter_mut().enumerate() {
        *b = *b ^ key[i % 4];
    }
}


/// Handling of the length format.
pub enum LengthFormat {
    U8(u8),
    U16,
    U64,
}

impl LengthFormat {
    /// Get length format for a given data size.
    #[inline]
    fn for_length(length: u64) -> Self {
        if length < 126 {
            LengthFormat::U8(length as u8)
        } else if length < 65536 {
            LengthFormat::U16
        } else {
            LengthFormat::U64
        }
    }

    /// Get the size of length encoding.
    #[inline]
    pub fn extra_bytes(&self) -> usize {
        match *self {
            LengthFormat::U8(_) => 0,
            LengthFormat::U16 => 2,
            LengthFormat::U64 => 8,
        }
    }

    /// Encode the givem length.
    #[inline]
    fn length_byte(&self) -> u8 {
        match *self {
            LengthFormat::U8(b) => b,
            LengthFormat::U16 => 126,
            LengthFormat::U64 => 127,
        }
    }

    /// Get length format for a given length byte.
    #[inline]
    pub fn for_byte(byte: u8) -> Self {
        match byte & 0x7F {
            126 => LengthFormat::U16,
            127 => LengthFormat::U64,
            b => LengthFormat::U8(b),
        }
    }
}


/// WebSocket message opcode as in RFC 6455.
#[derive(Debug, PartialEq, Eq, Clone, Copy)]
pub enum OpCode {
    /// Data (text or binary).
    Data(Data),
    /// Control message (close, ping, pong).
    Control(Control),
}

/// Data opcodes as in RFC 6455
#[derive(Debug, PartialEq, Eq, Clone, Copy)]
pub enum Data {
    /// 0x0 denotes a continuation frame
    Continue,
    /// 0x1 denotes a text frame
    Text,
    /// 0x2 denotes a binary frame
    Binary,
    /// 0x3-7 are reserved for further non-control frames
    Reserved(u8),
}

/// Control opcodes as in RFC 6455
#[derive(Debug, PartialEq, Eq, Clone, Copy)]
pub enum Control {
    /// 0x8 denotes a connection close
    Close,
    /// 0x9 denotes a ping
    Ping,
    /// 0xa denotes a pong
    Pong,
    /// 0xb-f are reserved for further control frames
    Reserved(u8),
}

impl fmt::Display for Data {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match *self {
            Data::Continue => write!(f, "CONTINUE"),
            Data::Text => write!(f, "TEXT"),
            Data::Binary => write!(f, "BINARY"),
            Data::Reserved(x) => write!(f, "RESERVED_DATA_{}", x),
        }
    }
}

impl fmt::Display for Control {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match *self {
            Control::Close => write!(f, "CLOSE"),
            Control::Ping => write!(f, "PING"),
            Control::Pong => write!(f, "PONG"),
            Control::Reserved(x) => write!(f, "RESERVED_CONTROL_{}", x),
        }
    }
}

impl fmt::Display for OpCode {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match *self {
            OpCode::Data(d) => d.fmt(f),
            OpCode::Control(c) => c.fmt(f),
        }
    }
}

impl Into<u8> for OpCode {
    fn into(self) -> u8 {
        use self::Control::{Close, Ping, Pong};
        use self::Data::{Binary, Continue, Text};
        use self::OpCode::*;
        match self {
            Data(Continue) => 0,
            Data(Text) => 1,
            Data(Binary) => 2,
            Data(self::Data::Reserved(i)) => i,

            Control(Close) => 8,
            Control(Ping) => 9,
            Control(Pong) => 10,
            Control(self::Control::Reserved(i)) => i,
        }
    }
}

impl From<u8> for OpCode {
    fn from(byte: u8) -> OpCode {
        use self::Control::{Close, Ping, Pong};
        use self::Data::{Binary, Continue, Text};
        use self::OpCode::*;
        match byte {
            0 => Data(Continue),
            1 => Data(Text),
            2 => Data(Binary),
            i @ 3..=7 => Data(self::Data::Reserved(i)),
            8 => Control(Close),
            9 => Control(Ping),
            10 => Control(Pong),
            i @ 11..=15 => Control(self::Control::Reserved(i)),
            _ => panic!("Bug: OpCode out of range"),
        }
    }
}

