use rand::Rng;

use crate::stream::Stream;
use std::collections::HashMap;
use tokio::io::{AsyncRead, ReadHalf, WriteHalf, AsyncWriteExt, AsyncBufReadExt, AsyncWrite, AsyncReadExt};
use sha1::{Digest, Sha1};
use base64;
use std::io::{Error, ErrorKind};
use std::io::{Read, Write};
use log::{trace};

pub fn connect_key() -> String {
    let buf = rand::thread_rng().gen::<[u8; 16]>();
    base64::encode(&buf)
}

pub fn connect_header(host: &str, path: &str, key: &str, headers: &HashMap<String, String>) -> String {
    let mut h = "GET ".to_owned()
        + path
        + " HTTP/1.1\r\n\
Connection: Upgrade\r\n\
Upgrade: websocket\r\n\
Sec-WebSocket-Version: 13\r\n\
Sec-WebSocket-Extensions: permessage-deflate; client_max_window_bits\r\n\
Sec-WebSocket-Key: ";
    h.push_str(key);
    h.push_str("\r\n");
    h.push_str("Host: ");
    h.push_str(host);
    h.push_str("\r\n");
    for (key, value) in headers.iter() {
        h.push_str(key);
        h.push_str(": ");
        h.push_str(value);
        h.push_str("\r\n");
    }
    h.push_str("\r\n");
    // println!("send {}", h);
    h
}

// Connects to the WebSocket server.
// It will send http upgrade request, wait for response and check whether
// upgrade request is accepted.
pub async fn connect_http<R, W>(
    mut stream_rh: R, mut stream_wh: W,
    url: &Url,
    headers: &HashMap<String, String>,
) -> Result<(tokio::io::BufReader<R>, W, bool, HashMap<String, String>), tokio::io::Error>
    where
        R: AsyncReadExt + std::marker::Unpin,
        W: AsyncWriteExt + std::marker::Unpin,
{
    let key = connect_key();

    let vec = connect_header(&url.addr, &url.path, &key, headers);
    let mut buf = vec.as_bytes();
    loop {
        let len = stream_wh.write(buf).await.unwrap();
        // trace!("header write len {}", len);
        if len == buf.len() {
            break;
        }
        buf = &buf[len..];
    }


    let mut inner = tokio::io::BufReader::with_capacity(1024, stream_rh);
    let mut lines: Vec<String> = Vec::new();
    loop {
        let mut line = String::new();
        // TODO: sta ako ovdje nikada nista ne posalje blokira mi thread !!!
        match inner.read_line(&mut line).await? {
            0 => break, // eof
            2 => break, // empty line \r\n = end of header line
            _n => {
                // println!("{}", line);
                lines.push(line.trim().to_owned())
            }
        }
    }


    let header = Header::from_lines(&lines);
    if header.is_valid_connect(&key) {
        return Ok((inner, stream_wh, header.is_deflate_supported(), header.lines));
    }
    Err(tokio::io::Error::new(ErrorKind::InvalidInput, "invalid upgrade request"))
}


#[derive(Debug)]
struct Header {
    connection: String,
    upgrade: String,
    version: String,
    key: String,
    extensions: String,
    accept: String,
    lines: HashMap<String, String>,
}

impl Header {
    fn new() -> Header {
        Header {
            connection: String::new(),
            upgrade: String::new(),
            version: String::new(),
            key: String::new(),
            extensions: String::new(),
            accept: String::new(),
            lines: HashMap::new(),
        }
    }

    fn from_lines(lines: &Vec<String>) -> Self {
        let mut header = Header::new();
        for line in lines {
            // trace!("{}", line);
            header.append(&line);
        }
        header
    }

    fn append(&mut self, line: &str) {
        if let Some((key, value)) = split_header_line(&line) {
            self.lines.insert(key.to_owned(), value.to_owned());
            match key.to_lowercase().as_str() {
                "connection" => self.connection = value.to_lowercase(),
                "upgrade" => self.upgrade = value.to_lowercase(),
                "sec-websocket-version" => self.version = value.to_string(),
                "sec-websocket-key" => self.key = value.to_string(),
                "sec-websocket-extensions" => self.add_extensions(value),
                "sec-websocket-accept" => self.accept = value.to_string(),
                _ => (),
            }
        }
    }

    fn add_extensions(&mut self, ex: &str) {
        if !self.extensions.is_empty() {
            self.extensions.push_str(", ");
        }
        self.extensions.push_str(ex);
    }

    fn is_deflate_supported(&self) -> bool {
        self.extensions.contains("permessage-deflate")
    }

    fn upgrade_response(&self) -> String {
        const HEADER: &str = "HTTP/1.1 101 Switching Protocols\r\n\
            Upgrade: websocket\r\n\
            Server: yarws\r\n\
            Connection: Upgrade\r\n\
            Sec-WebSocket-Accept: ";
        let mut s = HEADER.to_string();
        s.push_str(&ws_accept(&self.key));
        s.push_str(&"\r\n");
        if self.is_deflate_supported() {
            s.push_str(
                "Sec-WebSocket-Extensions: permessage-deflate;client_no_context_takeover;server_no_context_takeover",
            );
            s.push_str(&"\r\n");
        }
        s.push_str(&"\r\n");
        s
    }

    fn is_valid_upgrade(&self) -> bool {
        self.connection == "upgrade" && self.upgrade == "websocket" && self.version == "13" && self.key.len() > 0
    }

    fn is_valid_connect(&self, key: &str) -> bool {
        let accept = ws_accept(key);
        self.connection == "upgrade" && self.upgrade == "websocket" && self.accept == accept
    }
}

fn split_header_line(line: &str) -> Option<(&str, &str)> {
    let mut splitter = line.splitn(2, ':');
    let key = splitter.next()?;
    let value = splitter.next()?;
    Some((key, value.trim()))
}

// Calculate accept header value from |Sec-WebSocket-Key|.
// Ref: https://tools.ietf.org/html/rfc6455
//
// The server would append the string "258EAFA5-E914-47DA-95CA-C5AB0DC85B11" the
// value of the |Sec-WebSocket-Key| header field in the client's handshake. The
// server would then take the SHA-1 hash of this string. This value is then
// base64-encoded, to give the value which would be returned in the
// |Sec-WebSocket-Accept| header field.
fn ws_accept(key: &str) -> String {
    const WS_MAGIC_KEY: &str = "258EAFA5-E914-47DA-95CA-C5AB0DC85B11";
    let mut hasher = Sha1::new();
    let s = key.to_string() + WS_MAGIC_KEY;

    hasher.update(s.as_bytes());
    let hr = hasher.finalize();
    base64::encode(&hr)
}

#[derive(Clone)]
pub struct Url {
    pub addr: String,
    pub path: String,
    pub domain: String,
    pub wss: bool,
}

pub fn parse_url(u: &str) -> Result<Url, Error> {
    let url = url::Url::parse(u).unwrap();
    let host = url.host_str().unwrap_or("");
    let addr = format!("{}:{}", host, url.port_or_known_default().unwrap_or(0));
    let path = match url.query() {
        Some(q) => format!("{}?{}", url.path(), q),
        None => url.path().to_owned(),
    };
    let wss = url.scheme() == "wss";
    let u = Url {
        wss: wss,
        addr: addr,
        path: path,
        domain: host.to_string(),
    };
    Ok(u)
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_parse_url() {
        let url = parse_url("ws://localhost:9001/path?pero=zdero").unwrap();
        assert_eq!("localhost:9001", url.addr);
        assert_eq!("/path?pero=zdero", url.path);
        assert!(!url.wss);

        let url = parse_url("ws://localhost:9001/path").unwrap();
        assert_eq!("localhost:9001", url.addr);
        assert_eq!("/path", url.path);
        assert!(!url.wss);

        let url = parse_url("ws://localhost/path").unwrap();
        assert_eq!("localhost:80", url.addr);
        assert_eq!("/path", url.path);
        assert!(!url.wss);

        let url = parse_url("localhost/path").unwrap();
        assert_eq!("localhost:80", url.addr);
        assert_eq!("/path", url.path);
        assert!(!url.wss);

        let url = parse_url("pero://localhost/path").unwrap();
        assert_eq!("localhost:0", url.addr);
        assert_eq!("/path", url.path);
        assert!(!url.wss);

        let url = parse_url("wss://localhost:9001/path").unwrap();
        assert_eq!("localhost:9001", url.addr);
        assert_eq!("/path", url.path);
        assert!(url.wss);

        let url = parse_url("wss://echo.websocket.org/path").unwrap();
        assert_eq!("echo.websocket.org:443", url.addr);
        assert_eq!("echo.websocket.org", url.domain);
        assert_eq!("/path", url.path);
        assert!(url.wss);
    }
}
