use twmap::convert::TryTo;
use twmap::map_checks::MapError;
use twmap::*;

use clap::{Arg, Command};

use std::collections::HashMap;
use std::ffi::OsStr;
use std::ops::Not;

fn maybe_print_path(path: Option<&OsStr>) {
    if let Some(path) = path {
        print!("{:?}: ", path);
    }
}

#[derive(Default)]
struct HookthroughSummary {
    faulty: u64,
    combo: u64,
    new: u64,
}

fn map_check_hookthrough(
    map: &mut TwMap,
    path: Option<&OsStr>,
    results: &mut HookthroughSummary,
) -> Result<(), MapError> {
    if map.find_physics_layer::<FrontLayer>().is_none() {
        return Ok(());
    }
    map.groups
        .load_conditionally(|layer| [LayerKind::Game, LayerKind::Front].contains(&layer.kind()))?;

    let game_tiles = map
        .find_physics_layer::<GameLayer>()
        .unwrap()
        .tiles
        .unwrap_ref();
    let front_tiles = map
        .find_physics_layer::<FrontLayer>()
        .unwrap()
        .tiles
        .unwrap_ref();

    results.faulty += game_tiles
        .indexed_iter()
        .zip(front_tiles.iter())
        .filter(|((_, game), front)| front.id == 5 && (game.id != 1 && game.id != 3))
        .map(|(((y, x), _), _)| {
            maybe_print_path(path);
            println!(
                "x: {}, y: {} -> old hookthrough without a block underneath",
                x, y
            );
        })
        .count() as u64;
    results.combo += game_tiles
        .iter()
        .zip(front_tiles.iter())
        .filter(|(game, front)| front.id == 5 && (game.id == 1 || game.id == 3))
        .count() as u64;
    results.new += game_tiles
        .iter()
        .zip(front_tiles.iter())
        .filter(|(game, front)| front.id == 66 && (game.id == 1 || game.id == 3))
        .count() as u64;
    Ok(())
}

fn check_direction_sensitive_tile_map<T: TileMapLayer>(
    layer: &T,
    results: &mut HashMap<u8, u64>,
    path: Option<&OsStr>,
) -> Result<(), MapError> {
    layer
        .tiles()
        .unwrap_ref()
        .indexed_iter()
        .for_each(|((y, x), tile)| {
            let id = tile.id();
            let flags = tile.flags().unwrap();
            if is_sensitive(id)
                && (flags.contains(TileFlags::FLIP_H) ^ flags.contains(TileFlags::FLIP_V))
            {
                *results.entry(id).or_insert(0) += 1;
                maybe_print_path(path);
                println!("x: {}, y: {} -> {} is mirrored", x, y, sensitive_id(id));
            }
        });
    Ok(())
}

fn map_check_direction_sensitive(
    map: &mut TwMap,
    path: Option<&OsStr>,
    results: &mut HashMap<u8, u64>,
) -> Result<(), MapError> {
    map.groups.load_conditionally(|layer| {
        [LayerKind::Game, LayerKind::Front, LayerKind::Switch].contains(&layer.kind())
    })?;
    check_direction_sensitive_tile_map(
        map.find_physics_layer::<GameLayer>().unwrap(),
        results,
        path,
    )?;
    if let Some(front_layer) = map.find_physics_layer::<FrontLayer>() {
        check_direction_sensitive_tile_map(front_layer, results, path)?;
    }
    if let Some(switch_layer) = map.find_physics_layer::<SwitchLayer>() {
        check_direction_sensitive_tile_map(switch_layer, results, path)?;
    }
    Ok(())
}

const fn is_sensitive(id: u8) -> bool {
    matches!(id, 60 | 61 | 64 | 65 | 67 | 203..=209 | 224 | 225)
}

fn sensitive_id(id: u8) -> &'static str {
    match id {
        60 => "stopper",
        61 => "bi-directional stopper",
        64 => "arrow/speeder",
        65 => "double arrow/fast speeder",
        67 => "directional hookthrough",
        203..=209 => "spinning freezing laser",
        224 => "exploding bullet",
        225 => "freezing bullet",
        _ => unreachable!(),
    }
}

fn map_check_tele_tiles(map: &mut TwMap, path: Option<&OsStr>) -> Result<u64, MapError> {
    map.groups
        .load_conditionally(|layer| layer.kind() == LayerKind::Tele)?;
    let tele_tiles = match map.find_physics_layer::<TeleLayer>() {
        None => return Ok(0),
        Some(tele_layer) => tele_layer.tiles.unwrap_ref(),
    };
    let count = tele_tiles
        .indexed_iter()
        .filter(|(_, tele)| tele.id != 0 && tele.number == 0)
        .map(|((y, x), _)| {
            maybe_print_path(path);
            println!(
                "Tele at x: {}, y: {} has number 0, meaning it has no effect",
                x, y
            );
        })
        .count();
    Ok(count.try_to())
}

fn stderr_maybe_print_path(path: Option<&OsStr>) {
    if let Some(path) = path {
        eprint!("{:?}: ", path);
    }
}

fn is_valid_entity_tile(id: u8) -> bool {
    matches!(id, 192..=218 | 220..=229 | 233..=238 | 240)
}

fn is_valid_game_or_front_tile(id: u8) -> bool {
    matches!(id,
        2
        | 4..=6
        | 9
        | 11..=13
        | 16..=22
        | 32..=62
        | 64..=67
        | 71..=76
        | 88..=91
        | 96..=97
        | 104..=107
        | 112..=113
        | 128..=129
        | 144..=145
        | 190..=191
    ) || is_valid_entity_tile(id)
}

fn is_valid_game_tile(id: u8) -> bool {
    matches!(id, 1 | 3) || is_valid_game_or_front_tile(id)
}

fn is_valid_front_tile(id: u8) -> bool {
    matches!(id, 98..=99) || is_valid_game_or_front_tile(id)
}

fn is_valid_tele_tile(id: u8) -> bool {
    matches!(id, 10 | 14..=15 | 26..=27 | 29..=31 | 63)
}

fn is_valid_speedup_tile(id: u8) -> bool {
    id == 28
}

fn is_valid_switch_tile(id: u8) -> bool {
    match id {
        7 | 9 | 12..=13 | 19..=20 | 22..=25 | 79 | 95 | 98..=99 => true,
        _ if id >= 197 => is_valid_entity_tile(id),
        _ => false,
    }
}

fn is_valid_tune_tile(id: u8) -> bool {
    id == 68
}

macro_rules! check_invalid_tiles {
    ($layer: ident, $kind: ident, $path: ident, $check_fn: ident) => {{
        let tiles = $layer.tiles.unwrap_ref();
        tiles
            .indexed_iter()
            .filter(|(_, tile)| tile.id != 0 && !$check_fn(tile.id))
            .map(|((y, x), tile)| {
                maybe_print_path($path);
                println!(
                    "{:?} tile at x: {}, y: {} with an invalid id: {}",
                    $kind, x, y, tile.id
                )
            })
            .count() as u64
    }};
}

fn check_tile_ids(layer: &Layer, path: Option<&OsStr>) -> u64 {
    let kind = layer.kind();
    match layer {
        Layer::Game(layer) => check_invalid_tiles!(layer, kind, path, is_valid_game_tile),
        Layer::Front(layer) => check_invalid_tiles!(layer, kind, path, is_valid_front_tile),
        Layer::Tele(layer) => check_invalid_tiles!(layer, kind, path, is_valid_tele_tile),
        Layer::Speedup(layer) => check_invalid_tiles!(layer, kind, path, is_valid_speedup_tile),
        Layer::Switch(layer) => check_invalid_tiles!(layer, kind, path, is_valid_switch_tile),
        Layer::Tune(layer) => check_invalid_tiles!(layer, kind, path, is_valid_tune_tile),
        _ => 0,
    }
}

fn map_check_tile_ids(map: &mut TwMap, path: Option<&OsStr>) -> Result<u64, MapError> {
    map.groups
        .load_conditionally(|layer| layer.kind().is_physics_layer())?;
    let mut total = 0;
    for group in &map.groups {
        for layer in &group.layers {
            total += check_tile_ids(layer, path);
        }
    }
    Ok(total)
}

fn main() {
    let matches = Command::new("twmap_check_ddnet")
        .author("Patiga <dev@patiga.eu>")
        .about("Scans ddnet maps for faulty behavior. All checks are enabled by default and disabled if a specific check is enabled. If there are multiple maps passed, the map file name will be printed at the start of each line")
        .arg(Arg::new("files")
            .allow_invalid_utf8(true)
            .required(true)
            .min_values(1)
            .index(1))
        .arg(Arg::new("hookthrough")
            .help("Check for faulty hookthrough: combo ht with no solid block underneath")
            .long("hookthrough"))
        .arg(Arg::new("directional-blocks")
            .help("Check for direction sensitive blocks in the physics layer which have an invalid orientation")
            .long("directional-blocks"))
        .arg(Arg::new("zero-tele")
            .help("Checks for teleporter tiles with the number 0")
            .long("zero-tele"))
        .arg(Arg::new("tile-ids")
            .help("Checks if tile ids are actually valid (defined by code)")
            .long("tile-ids"))
        .arg(Arg::new("print-file-path")
            .long("print-file-path")
            .help("Print the file path of the inspected file, even if there is only one file"))
        .arg(Arg::new("summary")
            .long("summary")
            .help("Print a short summary at the end"))
        .get_matches();
    let files = matches.values_of_os("files").unwrap().collect::<Vec<_>>();
    let print_file_paths = files.len() > 1 || matches.is_present("print-file-path");
    let file_path_insert = match print_file_paths {
        true => Some(()),
        false => None,
    };

    let mut ht_results = HookthroughSummary::default();
    let mut directional_sensitive_results = HashMap::<u8, u64>::default();
    let mut total_faulty_teles = 0;
    let mut total_invalid_tile_ids = 0;

    let check_hookthrough = matches.is_present("hookthrough");
    let check_directional = matches.is_present("directional-blocks");
    let check_teles = matches.is_present("zero-tele");
    let check_tile_ids = matches.is_present("tile-ids");

    let check_everything = [
        check_hookthrough,
        check_directional,
        check_teles,
        check_tile_ids,
    ]
    .into_iter()
    .all(bool::not);

    for file_path in files.into_iter() {
        let print_path = file_path_insert.map(|_| file_path);
        let mut map = match TwMap::parse_path(&file_path) {
            Err(err) => {
                stderr_maybe_print_path(print_path);
                eprintln!("{}", err);
                continue;
            }
            Ok(map) => map,
        };
        if check_hookthrough || check_everything {
            if let Err(err) = map_check_hookthrough(&mut map, print_path, &mut ht_results) {
                stderr_maybe_print_path(print_path);
                eprintln!("{}", err);
            }
        }
        if check_directional || check_everything {
            if let Err(err) = map_check_direction_sensitive(
                &mut map,
                print_path,
                &mut directional_sensitive_results,
            ) {
                stderr_maybe_print_path(print_path);
                eprintln!("{}", err);
            }
        }
        if check_teles || check_everything {
            match map_check_tele_tiles(&mut map, print_path) {
                Ok(n) => total_faulty_teles += n,
                Err(err) => {
                    stderr_maybe_print_path(print_path);
                    eprintln!("{}", err);
                }
            }
        }
        if check_tile_ids || check_everything {
            match map_check_tile_ids(&mut map, print_path) {
                Ok(n) => total_invalid_tile_ids += n,
                Err(err) => {
                    stderr_maybe_print_path(print_path);
                    eprintln!("{}", err);
                }
            }
        }
    }
    if matches.is_present("summary") {
        println!("\nSummary:");
        if check_hookthrough || check_everything {
            println!(
                "Hookthrough combo (id 5) without blocks (faulty): {}",
                ht_results.faulty
            );
            println!(
                "Correctly placed hookthrough combo blocks: {}",
                ht_results.combo
            );
            println!(
                "Hookthrough tiles (id 66) with block underneath (no hookblock): {}.",
                ht_results.new
            );
        }
        if check_directional || check_everything {
            for (id, num) in directional_sensitive_results {
                if num != 0 {
                    println!("Mirrored {} (id {}) count: {}", sensitive_id(id), id, num);
                }
            }
        }
        if check_teles || check_everything {
            println!("Tele tiles with number 0: {}", total_faulty_teles);
        }
        if check_tile_ids || check_everything {
            println!("Tiles with an invalid id: {}", total_invalid_tile_ids);
        }
    }
}
