use anyhow::{anyhow, bail, Result};
use serde::{Deserialize, Serialize};
use tracing::debug;

// Request parameters for GET id.twitch.tv/oauth2/token
#[derive(Debug, Serialize)]
struct LoginQuery {
    client_id: String,
    client_secret: String,
    grant_type: String,
}

// Response to GET id.twitch.tv/oauth2/token
#[derive(Debug, Deserialize)]
struct LoginResponse {
    access_token: String,
    expires_in: usize,
    token_type: String,
}

#[derive(Debug, Deserialize)]
struct ErrorResponse {
    error: String,
    status: usize,
    message: String,
}

// Request parameters for GET api.twitch.tv/helix/users
#[derive(Debug, Serialize)]
struct GetUsersQuery {
    login: String,
}

// Response to GET api.twitch.tv/helix/users
#[derive(Debug, Deserialize)]
struct GetUsersResponse {
    data: Vec<GetUsersEntry>,
}

/*
"id":"689331234",
"login":"nextlander",
"display_name":"Nextlander",
"type":"",
"broadcaster_type":"partner",
"description":"Come watch Vinny, Brad, and Alex play through the latest games, a slew of classics, and chat with you, the viewer!",
"profile_image_url":"https://static-cdn.jtvnw.net/jtv_user_pictures/9b9f0a1e-6a0d-45bc-b42f-cfd09e297332-profile_image-300x300.png",
"offline_image_url":"",
"view_count":102251,
"created_at":"2021-05-20T22:34:07.072555Z"
*/
#[derive(Clone, Debug, Deserialize)]
pub struct GetUsersEntry {
    pub id: String,
    pub login: String,
    pub display_name: String,
    pub description: String,
    pub profile_image_url: String,
}

// Request parameters for GET api.twitch.tv/helix/videos
#[derive(Debug, Serialize)]
struct GetVideosQuery {
    user_id: String,
}

// Response to GET api.twitch.tv/helix/videos
#[derive(Debug, Deserialize)]
struct GetVideosResponse {
    data: Vec<GetVideosEntry>,
}

/*
"id": "1057165773",
"stream_id": "42361701644",
"user_id": "689331234",
"user_login": "nextlander",
"user_name": "Nextlander",
"title": "Nextlander @ E3: The Fresh Demos!",
"description": "",
"created_at": "2021-06-15T18:28:59Z",
"published_at": "2021-06-15T18:28:59Z",
"url": "https://www.twitch.tv/videos/1057165773",
"thumbnail_url": "https://static-cdn.jtvnw.net/cf_vods/dgeft87wbj63p/81255e5739ae02f7ccc8_nextlander_42361701644_1623781727//thumb/thumb0-%{width}x%{height}.jpg",
"viewable": "public",
"view_count": 27163,
"language": "en",
"type": "archive",
"duration": "3h11m2s",
"muted_segments": null
*/
#[derive(Clone, Debug, Deserialize)]
pub struct GetVideosEntry {
    pub id: String,
    pub title: String,
    pub description: String, // may be empty
    pub published_at: String,
    pub url: String,
    pub thumbnail_url: String,
    pub duration: String,
}

pub struct TwitchAuth {
    client_id: String,
    client_secret: String,
    access_token: Option<String>,
}

async fn send(request: surf::RequestBuilder) -> Result<surf::Response> {
    debug!("{:?}", request);
    let response = request.send().await.map_err(|err| anyhow!(err))?;
    debug!("{:?}", response);
    // Note: unable to log body here - body can only be read once
    Ok(response)
}

async fn helix_request(
    get_type: &str,
    query: &impl Serialize,
    auth: &mut TwitchAuth,
) -> Result<surf::RequestBuilder> {
    Ok(
        surf::get(format!("https://api.twitch.tv/helix/{}", get_type))
            .query(query)
            .map_err(|err| anyhow!(err))?
            .header("Client-Id", auth.client_id.as_str())
            .header("Authorization", format!("Bearer {}", auth.login().await?)),
    )
}

impl TwitchAuth {
    pub fn new(client_id: String, client_secret: String) -> TwitchAuth {
        TwitchAuth {
            client_id,
            client_secret,
            access_token: None,
        }
    }

    // see also https://dev.twitch.tv/docs/authentication/getting-tokens-oauth/#oauth-client-credentials-flow
    async fn login(&mut self) -> Result<String> {
        if let Some(token) = &self.access_token {
            // Token is already set, assume that it's fine...
            return Ok(token.clone());
        }

        let query = LoginQuery {
            client_id: self.client_id.clone(),
            client_secret: self.client_secret.clone(),
            grant_type: "client_credentials".to_string(),
        };
        let mut response = send(
            surf::post("https://id.twitch.tv/oauth2/token")
                .query(&query)
                .map_err(|err| anyhow!(err))?,
        )
        .await?;
        if !response.status().is_success() {
            let response_body = response.body_string().await.map_or_else(
                |err| format!("Error when fetching response body: {}", err),
                |body| body,
            );
            bail!("Failed to get Twitch auth token: {}", response_body);
        }
        let response_json: LoginResponse =
            response.body_json().await.map_err(|err| anyhow!(err))?;
        if response_json.access_token.is_empty() {
            bail!("Login response had empty access_token");
        }

        self.access_token = Some(response_json.access_token.clone());
        return Ok(response_json.access_token);
    }

    // Returns true if the error was auth-related and the token was reset
    async fn reset_if_auth_err(&mut self, response: &mut surf::Response) -> bool {
        // Example errors to treat as needing a new login():
        // {"error":"Unauthorized","status":401,"message":"Invalid OAuth token"}
        // {"error":"Unauthorized","status":401,"message":"OAuth token is missing"}
        // {"error":"Unauthorized","status":401,"message":"Client ID and OAuth token do not match"}
        if response.status() != 401 {
            return false;
        }
        if !response
            .body_string()
            .await
            .map_or_else(|_err| "".to_string(), |body| body)
            .contains("OAuth token")
        {
            return false;
        }
        self.access_token = None;
        return true;
    }
}

// see also https://dev.twitch.tv/docs/api/reference#get-users
// curl -v -H 'Client-Id: CLIENT_ID' -H 'Authorization: Bearer AUTH_TOKEN' "https://api.twitch.tv/helix/users?login=USER_NAME"
pub async fn get_user(auth: &mut TwitchAuth, user_name: &str) -> Result<GetUsersEntry> {
    let query = GetUsersQuery {
        login: user_name.to_string(),
    };
    let mut response = send(helix_request("users", &query, auth).await?).await?;
    if !response.status().is_success() {
        if auth.reset_if_auth_err(&mut response).await {
            // Assume auth token recently expired, try again once with new token
            response = send(helix_request("users", &query, auth).await?).await?;
        }
        if !response.status().is_success() {
            let response_body = response.body_string().await.map_or_else(
                |err| format!("Error when fetching user response body: {}", err),
                |body| body,
            );
            bail!(
                "Failed to get Twitch user id: {} {}",
                response.status(),
                response_body
            );
        }
    }
    let mut response_json: GetUsersResponse = response.body_json().await.map_err(|err| {
        anyhow!(
            "Failed to read {} user response body as JSON: {}",
            user_name,
            err
        )
    })?;
    if response_json.data.len() > 1 {
        bail!(
            "Get users response for {} had {} results, expected 1",
            user_name,
            response_json.data.len()
        );
    }
    if let Some(entry) = response_json.data.pop() {
        Ok(entry)
    } else {
        bail!("Missing entry in users response for {}", user_name)
    }
}

// see also https://dev.twitch.tv/docs/api/reference#get-videos
// curl -v -H 'Client-Id: CLIENT_ID' -H 'Authorization: Bearer AUTH_TOKEN' "https://api.twitch.tv/helix/videos?user_id=USER_ID"
pub async fn get_videos(auth: &mut TwitchAuth, user_id: &str) -> Result<Vec<GetVideosEntry>> {
    let query = GetVideosQuery {
        user_id: user_id.to_string(),
    };
    let mut response = send(helix_request("videos", &query, auth).await?).await?;
    if !response.status().is_success() {
        if auth.reset_if_auth_err(&mut response).await {
            // Assume auth token recently expired, try again once with new token
            response = send(helix_request("videos", &query, auth).await?).await?;
        }
        if !response.status().is_success() {
            let response_body = response.body_string().await.map_or_else(
                |err| format!("Error when fetching video response body: {}", err),
                |body| body,
            );
            bail!(
                "Failed to get Twitch user videos: {} {}",
                response.status(),
                response_body
            );
        }
    }
    let response_json: GetVideosResponse = response.body_json().await.map_err(|err| {
        anyhow!(
            "Failed to read {} video response body as JSON: {}",
            user_id,
            err
        )
    })?;
    return Ok(response_json.data);
}
