use ::rss::{ChannelBuilder, ImageBuilder, ItemBuilder};
use anyhow::{anyhow, Result};

use crate::twitch;

pub fn to_rss(
    user: &twitch::GetUsersEntry,
    videos: &Vec<twitch::GetVideosEntry>,
    thumbnail_width: &str,
    thumbnail_height: &str,
) -> Result<String> {
    let mut items = vec![];
    for video in videos {
        let mut item_builder = ItemBuilder::default();
        item_builder
            .title(format!("{} ({})", video.title, video.duration))
            .content(format!(
                "<p><img src=\"{}\"></p><p>{}</p>",
                video
                    .thumbnail_url
                    .replace("%{width}", thumbnail_width)
                    .replace("%{height}", thumbnail_height),
                video.duration
            ))
            .link(video.url.to_string())
            .pub_date(video.published_at.to_string());
        if !video.description.is_empty() {
            item_builder.description(video.description.to_string());
        }
        items.push(
            item_builder
                .build()
                .map_err(|e| anyhow!("Error when rendering RSS item: {}", e))?,
        );
    }
    let mut channel_builder = ChannelBuilder::default();
    channel_builder
        .title(user.display_name.to_string())
        .image(Some(
            ImageBuilder::default()
                .url(user.profile_image_url.as_str())
                .build()
                .map_err(|e| anyhow!("Error when rendering RSS channel image: {}", e))?,
        ))
        .link(format!("https://twitch.tv/{}", user.login))
        .generator("twitch-rss".to_string())
        .items(items);
    if !user.description.is_empty() {
        channel_builder.description(user.description.as_str());
    }
    Ok(channel_builder
        .build()
        .map_err(|e| anyhow!("Error when rendering RSS channel: {}", e))?
        .to_string())
}
