mod http;
mod logging;
mod rss;
mod twitch;

use anyhow::{bail, Context, Result};
use async_std::task;
use git_testament::{git_testament, render_testament};
use std::env;
use tracing::{error, info};

use crate::twitch::TwitchAuth;

git_testament!(TESTAMENT);

fn syntax() -> String {
    format!(
        "twitch-rss ({})

Syntax: {} [username]

If username is specified, prints the RSS data to stdout for writing to a file.
If no arguments are specified, starts an HTTP server that acts as a proxy for serving RSS.

{}",
        render_testament!(TESTAMENT),
        env::args().nth(0).unwrap(),
        env_help(),
    )
}

fn env_help() -> &'static str {
    return "
Environment variables:
  CLIENT_ID:     Twitch Client ID for querying feeds
  CLIENT_SECRET: Twitch Client Secret for querying feeds
  LOG_LEVEL:     error/warn/info/debug/trace/off

HTTP mode only:
  LISTEN:           Endpoint for HTTP service to listen on (default: '0.0.0.0:8080')
  USER_CACHE_SECS:  Duration in seconds to cache Twitch account info (default: 86400)
  VIDEO_CACHE_SECS: Duration in seconds to cache video lists (default: 600)";
}

fn main() -> Result<()> {
    logging::init_logging();

    match env::args().nth(1) {
        Some(username) => {
            if username.starts_with('-') {
                // Probably a commandline argument like '-h'/'--help', avoid parsing as a user
                bail!(syntax());
            }
            let mut twitch_auth = TwitchAuth::new(
                reqenv("CLIENT_ID", "Twitch Client ID for querying feeds")?,
                reqenv("CLIENT_SECRET", "Twitch Client Secret for querying feeds")?,
            );
            info!("version {}", render_testament!(TESTAMENT));
            task::block_on(async move {
                let user: twitch::GetUsersEntry;
                let videos: Vec<twitch::GetVideosEntry>;
                match twitch::get_user(&mut twitch_auth, &username).await {
                    Ok(u) => {
                        user = u;
                    }
                    Err(e) => {
                        error!("Failed to get info for username '{}': {}", username, e);
                        return;
                    }
                };
                match twitch::get_videos(&mut twitch_auth, &user.id).await {
                    Ok(v) => {
                        videos = v;
                    }
                    Err(e) => {
                        error!(
                            "Failed to get videos for user '{}'/{}: {}",
                            username, user.id, e
                        );
                        return;
                    }
                };
                match rss::to_rss(
                    &user,
                    &videos,
                    &optenv("THUMBNAIL_WIDTH", "640"),
                    &optenv("THUMBNAIL_HEIGHT", "360"),
                ) {
                    Ok(content) => println!("{}", content),
                    Err(e) => {
                        error!("Failed to render RSS: {}", e);
                        return;
                    }
                };
            });
            Ok(())
        }
        None => {
            info!("twitch-rss version {}", render_testament!(TESTAMENT));
            http::run(
                optenv("LISTEN", "0.0.0.0:8080"),
                TwitchAuth::new(
                    reqenv("CLIENT_ID", "Twitch Client ID for querying feeds")?,
                    reqenv("CLIENT_SECRET", "Twitch Client Secret for querying feeds")?,
                ),
                optenv("THUMBNAIL_WIDTH", "640"),
                optenv("THUMBNAIL_HEIGHT", "360"),
                optenv("USER_CACHE_SECS", "86400")
                    .parse::<u64>()
                    .with_context(|| "USER_CACHE_SECS must be an unsigned int")?,
                optenv("VIDEO_CACHE_SECS", "600")
                    .parse::<u64>()
                    .with_context(|| "VIDEO_CACHE_SECS must be an unsigned int")?,
            )
        }
    }
}

fn reqenv(name: &str, desc: &str) -> Result<String> {
    env::var(name).with_context(|| {
        format!(
            "Missing required envvar: {} ({})\n{}",
            name,
            desc,
            env_help()
        )
    })
}

fn optenv(name: &str, default: &str) -> String {
    match env::var(name) {
        Ok(val) => val,
        Err(_e) => default.to_string(),
    }
}
