use anyhow::{Context, Result};
use async_lock::Mutex;
use async_std::task;
use std::sync::Arc;
use std::time::Duration;
use tide::Request;
use tracing::{debug, error, info, warn};

use crate::{rss, twitch};

struct State {
    twitch_auth: twitch::TwitchAuth,
    thumbnail_width: String,
    thumbnail_height: String,
    user_cache_secs: u64,
    video_cache_secs: u64,
    username_to_user_cache: Arc<retainer::cache::Cache<String, twitch::GetUsersEntry>>,
    userid_to_videos_cache: Arc<retainer::cache::Cache<String, Vec<twitch::GetVideosEntry>>>,
    _cache_monitors: Vec<async_std::task::JoinHandle<()>>,
}

pub fn run(
    listen_endpoint: String,
    twitch_auth: twitch::TwitchAuth,
    thumbnail_width: String,
    thumbnail_height: String,
    user_cache_secs: u64,
    video_cache_secs: u64,
) -> Result<()> {
    let mut state = State {
        twitch_auth,
        thumbnail_width,
        thumbnail_height,
        user_cache_secs,
        video_cache_secs,
        username_to_user_cache: Arc::new(retainer::cache::Cache::new()),
        userid_to_videos_cache: Arc::new(retainer::cache::Cache::new()),
        _cache_monitors: vec![],
    };
    let user_cache_clone = state.username_to_user_cache.clone();
    state._cache_monitors.push(task::spawn(async move {
        // Every 3s, check 4 entries for expiration.
        // If >25% were expired, then repeat the check for another 4 entries
        user_cache_clone
            .monitor(4, 0.25, Duration::from_secs(3))
            .await
    }));
    let video_cache_clone = state.userid_to_videos_cache.clone();
    state._cache_monitors.push(task::spawn(async move {
        // Every 3s, check 4 entries for expiration.
        // If >25% were expired, then repeat the check for another 4 entries
        video_cache_clone
            .monitor(4, 0.25, Duration::from_secs(3))
            .await
    }));

    let mut app = tide::with_state(Arc::new(Mutex::new(state)));
    // Enable request logging if debug (or trace)
    app.with(LogMiddleware {});
    app.at("/").get(handle_index);
    app.at("/rss").get(handle_rss);
    info!("Listening at LISTEN={}", listen_endpoint);
    return task::block_on(app.listen(listen_endpoint)).context("HTTP listener exited");
}

async fn handle_index(req: Request<Arc<Mutex<State>>>) -> tide::Result {
    Ok(tide::Response::builder(200)
        .body(format!(
            "<html>
<head><title>twitch-rss</title></head>
<body text='#fff' bgcolor='#000'><table width=100% height=100%><tr><td valign=center align=center>
<h2>twitch-rss</h2>
<p>Add this URL to your RSS reader:</p>
<p>{}rss?account=<b>twitch-account</b></p>
</td></tr></table></body>
</html>
",
            req.url()
        ))
        .content_type(tide::http::mime::HTML)
        .build())
}

async fn handle_rss(req: Request<Arc<Mutex<State>>>) -> tide::Result {
    let mut query_account = None;
    for (key, val) in req.url().query_pairs() {
        if key == "account" {
            query_account = Some(val.to_string());
        }
    }

    match query_account {
        Some(account) => {
            let response_body: String;
            {
                let mut state = req.state().lock().await;
                let user: twitch::GetUsersEntry;
                let videos: Vec<twitch::GetVideosEntry>;
                if let Some(u) = state.username_to_user_cache.get(&account).await {
                    user = (*u).clone();
                } else {
                    match twitch::get_user(&mut state.twitch_auth, &account).await {
                        Ok(u) => {
                            state
                                .username_to_user_cache
                                .insert(
                                    account,
                                    u.clone(),
                                    Duration::from_secs(state.user_cache_secs),
                                )
                                .await;
                            user = u;
                        }
                        Err(e) => {
                            return Ok(tide::Response::builder(400)
                                .body(format!("400 Bad Request: {}", e))
                                .content_type(tide::http::mime::PLAIN)
                                .build());
                        }
                    };
                }
                if let Some(v) = state.userid_to_videos_cache.get(&user.id).await {
                    videos = (*v).clone();
                } else {
                    match twitch::get_videos(&mut state.twitch_auth, &user.id).await {
                        Ok(v) => {
                            state
                                .userid_to_videos_cache
                                .insert(
                                    user.id.clone(),
                                    v.clone(),
                                    Duration::from_secs(state.video_cache_secs),
                                )
                                .await;
                            videos = v;
                        }
                        Err(e) => {
                            return Ok(tide::Response::builder(400)
                                .body(format!("400 Bad Request: {}", e))
                                .content_type(tide::http::mime::PLAIN)
                                .build());
                        }
                    }
                };
                response_body = rss::to_rss(
                    &user,
                    &videos,
                    state.thumbnail_width.as_str(),
                    state.thumbnail_height.as_str(),
                )?;
            }
            Ok(tide::Response::builder(200)
                .body(response_body)
                .content_type("application/rss+xml")
                .build())
        }
        None => Ok(tide::Response::builder(400)
            .body("400 Bad Request: Missing 'account' parameter, e.g. /rss?account=x")
            .content_type(tide::http::mime::PLAIN)
            .build()),
    }
}

/// Copy/reimplementation of Tide's LogMiddleware, modified to instead work with the tracing library.
#[derive(Debug, Default, Clone)]
struct LogMiddleware {}

struct LogMiddlewareHasBeenRun;

#[async_trait::async_trait]
impl<State: Clone + Send + Sync + 'static> tide::Middleware<State> for LogMiddleware {
    async fn handle(&self, mut req: Request<State>, next: tide::Next<'_, State>) -> tide::Result {
        if req.ext::<LogMiddlewareHasBeenRun>().is_some() {
            return Ok(next.run(req).await);
        }
        req.set_ext(LogMiddlewareHasBeenRun);

        let path = req.url().path().to_owned();
        let method = req.method().to_string();
        debug!("<-- Request received: method={} path={}", method, path);
        let start = std::time::Instant::now();
        let response = next.run(req).await;
        let status = response.status();
        if status.is_server_error() {
            if let Some(error) = response.error() {
                error!(
                    "Internal error --> Response sent: method={} path={} duration={:?} status={}/{} error={:?}/{:?}",
                    method,
                    path,
                    start.elapsed(),
                    status as u16,
                    status.canonical_reason(),
                    error.type_name(),
                    error,
                );
            } else {
                error!(
                    "Internal error --> Response sent: method={} path={} duration={:?} status={}/{}",
                    method,
                    path,
                    start.elapsed(),
                    status as u16,
                    status.canonical_reason(),
                );
            }
        } else if status.is_client_error() {
            if let Some(error) = response.error() {
                warn!(
                    "Client error --> Response sent: method={} path={} duration={:?} status={}/{} error={:?}/{:?}",
                    method,
                    path,
                    start.elapsed(),
                    status as u16,
                    status.canonical_reason(),
                    error.type_name(),
                    error,
                );
            } else {
                warn!(
                    "Client error --> Response sent: method={} path={} duration={:?} status={}/{}",
                    method,
                    path,
                    start.elapsed(),
                    status as u16,
                    status.canonical_reason(),
                );
            }
        } else {
            info!(
                "--> Response sent: method={} path={} duration={:?} status={}/{}",
                method,
                path,
                start.elapsed(),
                status as u16,
                status.canonical_reason(),
            );
        }
        Ok(response)
    }
}
