// SPDX-FileCopyrightText: 2022 Declan Rixon <twisted.cubing@gmail.com>
//
// SPDX-License-Identifier: GPL-3.0-only

pub mod substep {
    use super::model::{
        edge::OrientedSliceEdges,
        orientation::{MultiAxisCO, CO, EO},
        permutation::{CP, EP},
    };
    use crate::{
        cube::{Axis, Face, Move},
        method::Substep,
    };

    #[derive(Copy, Clone)]
    pub struct Solved {
        pub co: CO,
        pub cp: CP,
        pub eo: EO,
        pub ep: EP,
    }

    impl Solved {
        pub fn corner_sticker(&self, piece: usize, sticker: i32) -> Option<Face> {
            if sticker > 2 {
                return None;
            };
            let corner = self.cp.0.get(piece)?;
            let co = self.co.0.get(piece)?;
            let sticker = (sticker + *co as i32).rem_euclid(3) as usize;
            use Face::*;
            Some(
                *match corner {
                    0 => Some([U, L, B]),
                    1 => Some([U, B, R]),
                    2 => Some([U, R, F]),
                    3 => Some([U, F, L]),
                    4 => Some([D, B, L]),
                    5 => Some([D, R, B]),
                    6 => Some([D, F, R]),
                    7 => Some([D, L, F]),
                    _ => None,
                }?
                .get(sticker)?,
            )
        }

        pub fn edge_sticker(&self, index: usize, top: bool) -> Option<Face> {
            let edge = self.ep.0.get(index)?;
            let eo = self.eo.0.get(index)?;
            let top = if *eo { top } else { !top };
            use Face::*;
            if top {
                match edge {
                    0 => Some(U),
                    1 => Some(U),
                    2 => Some(U),
                    3 => Some(U),
                    4 => Some(B),
                    5 => Some(B),
                    6 => Some(F),
                    7 => Some(F),
                    8 => Some(D),
                    9 => Some(D),
                    10 => Some(D),
                    11 => Some(D),
                    _ => None,
                }
            } else {
                match edge {
                    0 => Some(B),
                    1 => Some(R),
                    2 => Some(F),
                    3 => Some(L),
                    4 => Some(L),
                    5 => Some(R),
                    6 => Some(R),
                    7 => Some(L),
                    8 => Some(B),
                    9 => Some(R),
                    10 => Some(F),
                    11 => Some(L),
                    _ => None,
                }
            }
        }
    }

    impl Substep for Solved {
        fn new() -> Self {
            Solved {
                co: CO::new(),
                cp: CP::new(),
                eo: EO::new(),
                ep: EP::new(),
            }
        }

        fn after_moves(&self, moves: &[Move]) -> Self {
            Self {
                co: self.co.after_moves(&moves),
                cp: self.cp.after_moves(&moves),
                eo: self.eo.after_moves(&moves),
                ep: self.ep.after_moves(&moves),
            }
        }

        fn solved(&self) -> bool {
            self.co.solved() && self.cp.solved() && self.eo.solved() && self.ep.solved()
        }

        fn comment() -> &'static str {
            "Solved"
        }
    }

    pub struct DominoReduction {
        edges: OrientedSliceEdges,
        corners: MultiAxisCO,
    }

    impl DominoReduction {
        fn solved_for(&self, co_axis: Axis) -> bool {
            let edges = &self.edges.0;
            if !self.corners.for_axis(co_axis).iter().all(|c| *c == 0) {
                return false;
            }
            if !co_axis
                .belt()
                .map(|e| edges[e])
                .iter()
                .all(|e| e.from(co_axis))
            {
                return false;
            }
            for eo_axis in co_axis.complements() {
                if match eo_axis {
                    Axis::X => edges.iter().all(|e| e.eo.x),
                    Axis::Y => edges.iter().all(|e| e.eo.y),
                    Axis::Z => edges.iter().all(|e| e.eo.z),
                } {
                    return true;
                }
            }
            false
        }
    }

    impl Substep for DominoReduction {
        fn new() -> Self {
            Self {
                edges: OrientedSliceEdges::new(),
                corners: MultiAxisCO::new(),
            }
        }

        fn after_moves(&self, moves: &[Move]) -> Self {
            Self {
                edges: self.edges.after_moves(moves),
                corners: self.corners.after_moves(moves),
            }
        }

        fn solved(&self) -> bool {
            for co_axis in [Axis::X, Axis::Y, Axis::Z] {
                if self.solved_for(co_axis) {
                    return true;
                }
            }
            false
        }

        fn comment() -> &'static str {
            "Domino Reduction"
        }
    }

    pub struct DominoReductionCSep {
        dr: DominoReduction,
        cp: CP,
    }

    impl Substep for DominoReductionCSep {
        fn new() -> Self {
            Self {
                dr: DominoReduction::new(),
                cp: CP::new(),
            }
        }

        fn after_moves(&self, moves: &[Move]) -> Self {
            Self {
                dr: self.dr.after_moves(moves),
                cp: self.cp.after_moves(moves),
            }
        }

        fn solved(&self) -> bool {
            for axis in [Axis::X, Axis::Y, Axis::Z] {
                if self.dr.solved_for(axis) {
                    if [axis.top(), axis.bottom()].iter().all(|face| {
                        let cycle = face.corner_cycle();
                        let opposite_cycle = face.opposite().corner_cycle();
                        cycle
                            .iter()
                            .all(|index| cycle.contains(&(self.cp.0[*index as usize] as usize)))
                            || cycle.iter().all(|index| {
                                opposite_cycle.contains(&(self.cp.0[*index as usize] as usize))
                            })
                    }) {
                        return true;
                    }
                }
            }
            false
        }

        fn comment() -> &'static str {
            "Separate corners"
        }
    }

    pub struct DominoReductionCP {
        dr: DominoReduction,
        cp: CP,
    }

    impl Substep for DominoReductionCP {
        fn new() -> Self {
            Self {
                dr: DominoReduction::new(),
                cp: CP::new(),
            }
        }

        fn after_moves(&self, moves: &[Move]) -> Self {
            Self {
                dr: self.dr.after_moves(moves),
                cp: self.cp.after_moves(moves),
            }
        }

        fn solved(&self) -> bool {
            for axis in [Axis::X, Axis::Y, Axis::Z] {
                if self.dr.solved_for(axis) {
                    let top_corners = axis.top().corner_cycle().map(|c| self.cp.0[c] as usize);
                    let bottom_corners =
                        axis.bottom().corner_cycle().map(|c| self.cp.0[c] as usize);
                    let solved = (axis.top().corner_cycle_matches(&top_corners)
                        && axis.bottom().corner_cycle_matches(&bottom_corners))
                        || (axis.top().corner_cycle_matches(&bottom_corners)
                            && axis.bottom().corner_cycle_matches(&top_corners));
                    if solved {
                        return true;
                    }
                }
            }
            false
        }

        fn comment() -> &'static str {
            "Permute corners"
        }
    }
}

pub mod model {
    pub mod edge {
        use crate::cube::{Axis, Move};

        #[derive(Copy, Clone, Debug)]
        pub struct OrientedEdge {
            pub x: bool,
            pub y: bool,
            pub z: bool,
        }

        impl OrientedEdge {
            pub fn new() -> Self {
                Self {
                    x: true,
                    y: true,
                    z: true,
                }
            }
        }

        #[derive(Copy, Clone, Debug)]
        pub struct OrientedSliceEdge {
            pub slice: Axis,
            pub eo: OrientedEdge,
        }

        impl OrientedSliceEdge {
            pub fn new(slice: Axis) -> Self {
                Self {
                    slice,
                    eo: OrientedEdge::new(),
                }
            }

            pub fn flipped(&self, axis: Axis) -> Self {
                let mut new = self.clone();
                match axis {
                    Axis::X => {
                        new.eo.x = !new.eo.x;
                    }
                    Axis::Y => {
                        new.eo.y = !new.eo.y;
                    }
                    Axis::Z => {
                        new.eo.z = !new.eo.z;
                    }
                }
                new
            }

            pub fn from(&self, slice: Axis) -> bool {
                self.slice == slice
            }
        }

        pub struct OrientedSliceEdges(pub [OrientedSliceEdge; 12]);

        impl OrientedSliceEdges {
            pub fn new() -> Self {
                use Axis::*;
                Self([X, Z, X, Z, Y, Y, Y, Y, X, Z, X, Z].map(|a| OrientedSliceEdge::new(a)))
            }

            pub fn after_moves(&self, moves: &[Move]) -> Self {
                let Self(inner) = self;
                let mut new = inner.clone();
                for mv in moves {
                    let cycle = mv.face.edge_cycle();
                    let edges = cycle.map(|e| new[e]);
                    for from in 0..4 {
                        let to = (from as i32 + mv.amount).rem_euclid(4) as usize;
                        new[cycle[to]] = if mv.amount % 2 == 0 {
                            edges[from]
                        } else {
                            edges[from].flipped(mv.face.axis())
                        }
                    }
                }
                Self(new)
            }
        }
    }

    pub mod permutation {
        use crate::cube::Move;

        #[derive(Copy, Clone)]
        pub struct EP(pub [u8; 12]);

        impl EP {
            pub fn new() -> Self {
                EP([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11])
            }

            pub fn solved(&self) -> bool {
                let EP(arr) = self;
                *arr == [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11]
            }

            pub fn after_moves(&self, moves: &[Move]) -> Self {
                let mut new = self.clone();
                new.apply_moves(moves);
                new
            }

            pub fn apply_moves(&mut self, moves: &[Move]) {
                for mv in moves {
                    self.apply_move(mv);
                }
            }

            fn apply_move(&mut self, mv: &Move) {
                let cycle = mv.face.edge_cycle();
                let edges = cycle.map(|e| self.0[e]);
                for from in 0..4 {
                    let to = (from as i32 + mv.amount).rem_euclid(4) as usize;
                    self.0[cycle[to]] = edges[from]
                }
            }
        }

        #[derive(Copy, Clone)]
        pub struct CP(pub [u8; 8]);

        impl CP {
            pub fn new() -> Self {
                CP([0, 1, 2, 3, 4, 5, 6, 7])
            }

            pub fn solved(&self) -> bool {
                let CP(arr) = self;
                *arr == [0, 1, 2, 3, 4, 5, 6, 7]
            }

            pub fn after_moves(&self, moves: &[Move]) -> Self {
                let mut new = self.clone();
                new.apply_moves(moves);
                new
            }

            pub fn apply_moves(&mut self, moves: &[Move]) {
                for mv in moves {
                    self.apply_move(mv)
                }
            }

            fn apply_move(&mut self, mv: &Move) {
                let cycle = mv.face.corner_cycle();
                let corners = cycle.map(|c| self.0[c]);
                for from in 0..4 {
                    let to = (from as i32 + mv.amount).rem_euclid(4) as usize;
                    self.0[cycle[to]] = corners[from];
                }
            }
        }
    }

    pub mod orientation {
        use crate::cube::{Axis, Move};

        #[derive(Copy, Clone)]
        pub struct EO(pub [bool; 12]);

        impl EO {
            pub fn new() -> Self {
                EO([true; 12])
            }

            pub fn solved(&self) -> bool {
                self.0.iter().all(|b| *b)
            }

            pub fn after_moves(&self, moves: &[Move]) -> Self {
                let mut new = self.clone();
                new.apply_moves(moves);
                new
            }

            pub fn apply_moves(&mut self, moves: &[Move]) {
                for mv in moves {
                    self.apply_move(mv);
                }
            }

            pub fn apply_move(&mut self, mv: &Move) {
                let cycle = mv.face.edge_cycle();
                let edges = cycle.map(|e| self.0[e]);
                for from in 0..4 {
                    let to = (from as i32 + mv.amount).rem_euclid(4) as usize;
                    self.0[cycle[to]] = if mv.face.axis() == Axis::Z && mv.amount % 2 != 0 {
                        !edges[from]
                    } else {
                        edges[from]
                    }
                }
            }
        }

        #[derive(Copy, Clone)]
        pub struct CO(pub [u8; 8]);

        impl CO {
            pub fn new() -> Self {
                CO([0; 8])
            }

            pub fn solved(&self) -> bool {
                self.0 == [0; 8]
            }

            pub fn after_moves(&self, moves: &[Move]) -> Self {
                let mut new = self.clone();
                new.apply_moves(moves);
                new
            }

            pub fn apply_moves(&mut self, moves: &[Move]) {
                for mv in moves {
                    self.apply_move(mv);
                }
            }

            fn apply_move(&mut self, mv: &Move) {
                let u_layer = [0, 1, 2, 3];
                let cycle = mv.face.corner_cycle();
                let corners = cycle.map(|c| self.0[c]);
                for from in 0..4 {
                    let to = (from as i32 + mv.amount).rem_euclid(4) as usize;

                    self.0[cycle[to]] = if mv.face.axis() == Axis::Y || mv.amount % 2 == 0 {
                        corners[from]
                    } else {
                        let changes_layer =
                            u_layer.contains(&cycle[from]) == u_layer.contains(&cycle[to]);
                        let amount = mv.amount.rem_euclid(4);
                        let clockwise = amount == 1;
                        if changes_layer {
                            (corners[from] + if clockwise { 2 } else { 1 }) % 3
                        } else {
                            (corners[from] + if clockwise { 1 } else { 2 }) % 3
                        }
                    }
                }
            }
        }

        #[derive(Copy, Clone)]
        pub struct MultiAxisCO {
            pub co_x: [u8; 8],
            pub co_y: [u8; 8],
            pub co_z: [u8; 8],
        }

        impl MultiAxisCO {
            pub fn new() -> Self {
                Self {
                    co_x: [0; 8],
                    co_y: [0; 8],
                    co_z: [0; 8],
                }
            }

            pub fn for_axis(&self, axis: Axis) -> [u8; 8] {
                match axis {
                    Axis::X => self.co_x,
                    Axis::Y => self.co_y,
                    Axis::Z => self.co_z,
                }
            }

            pub fn apply_moves(&mut self, moves: &[Move]) {
                for mv in moves {
                    self.apply_move(mv);
                }
            }

            pub fn after_moves(&self, moves: &[Move]) -> Self {
                let mut after = self.clone();
                after.apply_moves(moves);
                after
            }

            pub fn subset_solved(&self, axis: Axis, subset: &[usize]) -> bool {
                let co = match axis {
                    Axis::X => &self.co_x,
                    Axis::Y => &self.co_y,
                    Axis::Z => &self.co_z,
                };
                subset.iter().all(|&p| co[p] == 0)
            }

            pub fn apply_move(&mut self, mv: &Move) {
                fn same_layer(pos1: usize, pos2: usize, axis: Axis) -> bool {
                    let layer = axis.top().corner_cycle();
                    layer.contains(&pos1) == layer.contains(&pos2)
                }
                for (co, axis) in [
                    (&mut self.co_x, Axis::X),
                    (&mut self.co_y, Axis::Y),
                    (&mut self.co_z, Axis::Y),
                ] {
                    let old = co.clone();
                    let cycle = mv.face.corner_cycle();
                    for from_cycle_index in 0..4 {
                        let to_cycle_index =
                            (from_cycle_index as i32 + mv.amount).rem_euclid(4) as usize;
                        let from = cycle[from_cycle_index];
                        let to = cycle[to_cycle_index];
                        co[to] = if mv.face.axis() == axis || mv.amount % 2 == 0 {
                            old[from]
                        } else {
                            let amount = mv.amount.rem_euclid(4);
                            let clockwise = amount == 1;
                            if same_layer(from, to, axis) {
                                (old[from] + if clockwise { 1 } else { 2 }) % 3
                            } else {
                                (old[from] + if clockwise { 2 } else { 1 }) % 3
                            }
                        }
                    }
                }
            }
        }
    }
}
