// SPDX-FileCopyrightText: 2022 Declan Rixon <twisted.cubing@gmail.com>
//
// SPDX-License-Identifier: GPL-3.0-only

//! A library for smart speedcubing.
//!
//! `twisted` models many of the common speedcubing concepts in addition to
//! supplying abstractions for interacting with smartcubes and an optional
//! module that can draw cube states using sdl2.
//!
//! # Goals
//! Currently, `twisted` has three main goals:
//! - Supply connection abstractions for working with smartcubes.
//! - Facilitate the development of FOSS speedcubing software like simulators
//! and training tools by making it easy to record and break down solves.
//! - Facilitate method development by supplying abstractions for methods and
//! substeps.
//!
//! In the future, we would like to support optimising solves and searching for
//! algorithms using IDA-Star, but this will require performance-optimised
//! models of substeps, and heuristic tables for many different substeps and
//! methods.
//!
//! Additionally, we would like to support more smartcubes and more of each
//! smartcube's additional functionality, such as gyroscopes and retrieving
//! things like battery level.
//!
//! # Smartcubes
//! `twisted' supports the following smartcubes:
//! - Moyu Weilong AI
//!
//! The following functionality exists for every cube:
//! - Dumping raw turn data to file.
//! - Loading raw turn data from file.
//! - Live recording and reconstruction of turns.
//! - Reconstructing moves from turn data.
//!
//! The following functionality is planned for each cube:
//! - Implementation of the [Stream] trait.
//! - Dumping data from recorded buffer(s) to file(s).
//!
//! # Supported Methods
//! 'twisted' contains models for the methods listed below, in addition to
//! models of their substeps. These are implementations of the [Method] and
//! [Substep] traits respectively.
//! - [Hollow Stairs & Columns (HSC)]
//!
//! # Getting Started
//! To add twisted to your project, simply add the library to the dependencies
//! section of your `Cargo.toml`.
//! ```toml
//! [dependencies]
//! ...
//! twisted =  "0.1.3"
//! ```
//! Remember to enable the `bluetooth` or `sdl2' modules if you need any of the
//! smartcube or gui functionality.
//! ```toml
//! [dependencies]
//! ...
//! twisted = { version = "0.1.3", features = ["bluetooth", "sdl2"] }
//! ```
//!
//! [Stream]: futures::stream::Stream
//! [Method]: crate::method::Method
//! [Substep]: crate::method::Substep
//! [Hollow Stairs & Columns (HSC)]: https://www.speedsolving.com/wiki/index.php/HSC

/// Bluetooth smartcubes.
#[cfg(feature = "bluetooth")]
pub mod bluetooth;

/// Models of (speed)cubing concepts.
pub mod cube;

/// Drawing cube nets to sdl2 canvases.
#[cfg(feature = "sdl2")]
pub mod gui;

/// Models of methods and their substeps.
pub mod method;
