// SPDX-FileCopyrightText: 2022 Declan Rixon <twisted.cubing@gmail.com>
//
// SPDX-License-Identifier: GPL-3.0-only

use crate::cube::Face;
use crate::gui::*;
use sdl2::rect::Rect;

pub struct Grid {
    pub width: i32,
    pub height: i32,
    pub sticker_gap: Option<i32>,
}

impl Grid {
    pub fn new(width: i32, height: i32, sticker_gap: Option<i32>) -> Option<Self> {
        if let Some(gap) = sticker_gap {
            if gap > width / 2 || gap > height / 2 {
                return None;
            }
        }
        Self {
            width,
            height,
            sticker_gap,
        }
        .into()
    }

    pub fn center_sq(&self, face: Face, x: i32, y: i32) -> Center {
        self.center_rect(face, x, y, 1, 1)
    }

    pub fn center_rect(&self, face: Face, x: i32, y: i32, width: i32, height: i32) -> Center {
        let gap = &self.sticker_gap.unwrap_or(0);
        let x = x * &self.width + gap;
        let y = y * &self.height + gap;
        let width = &self.width * width - 2 * gap;
        let height = &self.height * height - 2 * gap;
        Center {
            face,
            rect: Rect::new(x, y, width as u32, height as u32),
        }
    }

    pub fn corner_sq(&self, piece: usize, sticker: i32, x: i32, y: i32) -> Corner {
        self.corner_rect(piece, sticker, x, y, 1, 1)
    }

    pub fn corner_rect(
        &self,
        piece: usize,
        sticker: i32,
        x: i32,
        y: i32,
        width: i32,
        height: i32,
    ) -> Corner {
        let gap = &self.sticker_gap.unwrap_or(0);
        let x = x * &self.width + gap;
        let y = y * &self.height + gap;
        let width = &self.width * width - 2 * gap;
        let height = &self.height * height - 2 * gap;
        Corner {
            piece,
            sticker,
            rect: Rect::new(x, y, width as u32, height as u32),
        }
    }

    pub fn edge_sq(&self, piece: usize, top: bool, x: i32, y: i32) -> Edge {
        self.edge_rect(piece, top, x, y, 1, 1)
    }

    pub fn edge_rect(
        &self,
        piece: usize,
        top: bool,
        x: i32,
        y: i32,
        width: i32,
        height: i32,
    ) -> Edge {
        let gap = &self.sticker_gap.unwrap_or(0);
        let x = x * &self.width + gap;
        let y = y * &self.height + gap;
        let width = &self.width * width - 2 * gap;
        let height = &self.height * height - 2 * gap;
        Edge {
            piece,
            top,
            rect: Rect::new(x, y, width as u32, height as u32),
        }
    }
}
